/*
 * La Et Le — French Gender Classification Trainer
 *
 * Copyright (C) 2025 Seweryn Polec <sewerynpol@protonmail.com>
 *
 * This program is free software: you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation, either version 3 of the License, or
 * (at your option) any later version.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE. See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with this program. If not, see <https://www.gnu.org/licenses/>.
 */

package com.languageapp.laetle

import android.content.Intent
import android.os.Bundle
import android.text.Editable
import android.text.Html
import android.text.TextWatcher
import android.view.View
import android.view.ViewGroup
import android.widget.*
import androidx.appcompat.app.AlertDialog
import androidx.appcompat.app.AppCompatActivity
import androidx.appcompat.app.AppCompatDelegate
import com.google.android.material.materialswitch.MaterialSwitch
import com.google.android.material.snackbar.Snackbar
import android.text.method.LinkMovementMethod
import androidx.core.content.edit
import android.view.inputmethod.InputMethodManager // <-- NEW IMPORT
import android.content.Context // <-- NEW IMPORT
import android.view.WindowManager
import androidx.activity.enableEdgeToEdge
import androidx.core.view.ViewCompat
import androidx.core.view.WindowInsetsCompat


/**
 * Activity for managing user-configurable settings and app behavior.
 */
class SettingsActivity : AppCompatActivity() {

    // UI elements
    private lateinit var etLearnThreshold: EditText
    private lateinit var etReviewThreshold: EditText
    private lateinit var cbShowTranslation: MaterialSwitch
    private lateinit var cbShowExamplesFr: MaterialSwitch
    private lateinit var cbShowExamplesEn: MaterialSwitch
    private lateinit var cbEnableAudio: MaterialSwitch
    private lateinit var cbShowStreakPopup: MaterialSwitch
    private lateinit var cbEnableTutorial: MaterialSwitch
    private lateinit var rgTheme: RadioGroup
    private lateinit var rbLight: RadioButton
    private lateinit var rbDark: RadioButton
    private lateinit var rbSystem: RadioButton
    private lateinit var btnReminderSettings: Button
    private lateinit var btnLicenses: Button
    private lateinit var btnPrivacyPolicy: Button
    private lateinit var btnTutorialInfo: ImageButton
    private lateinit var btnRewatchTutorial: Button

    // Persistence
    private val progressPrefs by lazy { getSharedPreferences("progress", MODE_PRIVATE) }
    private val settingsPrefs by lazy { getSharedPreferences("settings", MODE_PRIVATE) }

    // Snackbar state management
    private var currentSnackbar: Snackbar? = null
    private var lastSnackTime = 0L
    private var userInteracting = false

    override fun onCreate(savedInstanceState: Bundle?) {
        super.onCreate(savedInstanceState)

        enableEdgeToEdge()

        setContentView(R.layout.activity_settings)

        ViewCompat.setOnApplyWindowInsetsListener(findViewById(R.id.rootLayout)) { v, insets ->
            val systemBars = insets.getInsets(WindowInsetsCompat.Type.systemBars())
            v.setPadding(systemBars.left, systemBars.top, systemBars.right, systemBars.bottom)
            insets
        }

        window.setSoftInputMode(WindowManager.LayoutParams.SOFT_INPUT_STATE_ALWAYS_HIDDEN)
        // Initialize UI components
        etLearnThreshold = findViewById(R.id.etLearnThreshold) // RENAMED ID
        etReviewThreshold = findViewById(R.id.etReviewThreshold) // NEW
        cbShowTranslation = findViewById(R.id.cbShowTranslation)
        cbShowExamplesFr = findViewById(R.id.cbShowExamplesFr)
        cbShowExamplesEn = findViewById(R.id.cbShowExamplesEn)
        cbEnableAudio = findViewById(R.id.cbEnableAudio)
        cbShowStreakPopup = findViewById(R.id.cbShowStreakPopup)
        cbEnableTutorial = findViewById(R.id.cbEnableTutorial)
        rgTheme = findViewById(R.id.rgTheme)
        rbLight = findViewById(R.id.rbLight)
        rbDark = findViewById(R.id.rbDark)
        rbSystem = findViewById(R.id.rbSystem)
        btnReminderSettings = findViewById(R.id.btnReminderSettings)
        btnLicenses = findViewById(R.id.btnLicenses)
        btnPrivacyPolicy = findViewById(R.id.btnPrivacyPolicy)
        btnTutorialInfo = findViewById(R.id.btnTutorialInfo)
        btnRewatchTutorial = findViewById(R.id.btnRewatchTutorial)

        loadSettings()
        setupListeners()

        // Setup tutorial button
        btnTutorialInfo.setOnClickListener {
            showTutorialDialog("Settings Screen Tutorial", R.raw.tutorial_settings)
        }
        clearFocusAndHideKeyboard()
    }

    /** Tracks user interaction to prevent snackbars from showing when the app is backgrounded. */
    override fun onUserInteraction() {
        userInteracting = true
    }

    /** Displays a short Snackbar message, preventing rapid or backgrounded calls. */
    private fun showSnack(message: String) {
        val now = System.currentTimeMillis()
        if (!userInteracting || currentSnackbar?.isShown == true || now - lastSnackTime < 1500) return

        val root = findViewById<View?>(R.id.rootLayout) ?: findViewById(android.R.id.content)
        currentSnackbar = Snackbar.make(root, message, Snackbar.LENGTH_SHORT).apply {
            // Force full width layout for consistency
            view.layoutParams = (view.layoutParams as? ViewGroup.MarginLayoutParams)?.apply {
                width = ViewGroup.LayoutParams.MATCH_PARENT
                marginStart = 0
                marginEnd = 0
            }
        }

        currentSnackbar?.show()
        lastSnackTime = now
        userInteracting = false
    }

    /** Sets up all UI event listeners. */
    private fun setupListeners() {

        // Master Threshold EditText listener
        etLearnThreshold.addTextChangedListener(object : TextWatcher {
            override fun afterTextChanged(s: Editable?) {
                val num = s?.toString()?.toIntOrNull()
                if (num != null && num > 0) {
                    progressPrefs.edit { putInt(PREF_LEARN_THRESHOLD, num) }
                    showSnack("Settings updated")
                }
            }
            override fun beforeTextChanged(s: CharSequence?, start: Int, count: Int, after: Int) {}
            override fun onTextChanged(s: CharSequence?, start: Int, before: Int, count: Int) {}
        })

        etReviewThreshold.addTextChangedListener(object : TextWatcher {
            override fun afterTextChanged(s: Editable?) {
                val num = s?.toString()?.toIntOrNull()
                if (num != null && num > 0) {
                    progressPrefs.edit { putInt(PREF_REVIEW_THRESHOLD, num) }
                    showSnack("Settings updated")
                }
            }
            override fun beforeTextChanged(s: CharSequence?, start: Int, count: Int, after: Int) {}
            override fun onTextChanged(s: CharSequence?, start: Int, before: Int, count: Int) {}
        })

        // Visibility Toggles listeners
        cbShowTranslation.setOnCheckedChangeListener { _, isChecked ->
            progressPrefs.edit { putBoolean(PREF_SHOW_TRANSLATION, isChecked) }
        }

        cbShowExamplesFr.setOnCheckedChangeListener { _, isChecked ->
            progressPrefs.edit { putBoolean(PREF_SHOW_EXAMPLES_FR, isChecked) }
        }

        cbShowExamplesEn.setOnCheckedChangeListener { _, isChecked ->
            progressPrefs.edit { putBoolean(PREF_SHOW_EXAMPLES_EN, isChecked) }
        }

        cbEnableAudio.setOnCheckedChangeListener { _, isChecked ->
            settingsPrefs.edit { putBoolean("enable_audio", isChecked) }
        }

        cbShowStreakPopup.setOnCheckedChangeListener { _, isChecked ->
            settingsPrefs.edit { putBoolean(PREF_SHOW_STREAK_POPUP, isChecked) }
        }

        cbEnableTutorial.setOnCheckedChangeListener { _, isChecked ->
            settingsPrefs.edit { putBoolean("enable_tutorial", isChecked) }
        }
        btnRewatchTutorial.setOnClickListener {
            TutorialManager.showTutorialDialog(this)
        }

        // Theme RadioGroup listener
        rgTheme.setOnCheckedChangeListener { _, checkedId ->
            val mode = when (checkedId) {
                R.id.rbLight -> AppCompatDelegate.MODE_NIGHT_NO
                R.id.rbDark -> AppCompatDelegate.MODE_NIGHT_YES
                else -> AppCompatDelegate.MODE_NIGHT_FOLLOW_SYSTEM
            }
            settingsPrefs.edit { putInt("night_mode", mode) }
            AppCompatDelegate.setDefaultNightMode(mode)
        }

        // Button listeners
        btnReminderSettings.setOnClickListener {
            startActivity(Intent(this, ReminderSettingsActivity::class.java))
        }
        btnLicenses.setOnClickListener {
            showLicensesDialog()
        }
        btnPrivacyPolicy.setOnClickListener {
            showPrivacyPolicyDialog()
        }
    }

    /** Loads and applies all saved settings to the UI controls. */
    private fun loadSettings() {
        // Load Learn Threshold with a set default
        etLearnThreshold.setText(progressPrefs.getInt(PREF_LEARN_THRESHOLD, 3).toString())
        // Load Review Threshold with a set default
        etReviewThreshold.setText(progressPrefs.getInt(PREF_REVIEW_THRESHOLD, 2).toString())
        cbShowTranslation.isChecked = progressPrefs.getBoolean(PREF_SHOW_TRANSLATION, true)
        cbShowExamplesFr.isChecked = progressPrefs.getBoolean(PREF_SHOW_EXAMPLES_FR, true)
        cbShowExamplesEn.isChecked = progressPrefs.getBoolean(PREF_SHOW_EXAMPLES_EN, true)
        cbEnableAudio.isChecked = settingsPrefs.getBoolean("enable_audio", true) // Ensure audio is also loaded
        cbShowStreakPopup.isChecked = settingsPrefs.getBoolean(PREF_SHOW_STREAK_POPUP, true)
        cbEnableTutorial.isChecked = settingsPrefs.getBoolean("enable_tutorial", true) // Default is ON

        when (settingsPrefs.getInt("night_mode", AppCompatDelegate.MODE_NIGHT_FOLLOW_SYSTEM)) {
            AppCompatDelegate.MODE_NIGHT_NO -> rbLight.isChecked = true
            AppCompatDelegate.MODE_NIGHT_YES -> rbDark.isChecked = true
            else -> rbSystem.isChecked = true
        }
    }

    /** Displays a scrollable dialog with tutorial content from a raw resource. */
    private fun showTutorialDialog(title: String, rawResId: Int) {
        val dialogView = layoutInflater.inflate(R.layout.dialog_licenses, null)
        val textView = dialogView.findViewById<TextView>(R.id.tvLicenses)

        val tutorialText = try {
            resources.openRawResource(rawResId)
                .bufferedReader()
                .use { it.readText() }
        } catch (e: Exception) {
            e.printStackTrace()
            "Could not load tutorial information for $title. (Resource $rawResId is missing in R.raw)"
        }

        // Render HTML content
        textView.text =
            Html.fromHtml(tutorialText, Html.FROM_HTML_MODE_COMPACT)

        AlertDialog.Builder(this)
            .setTitle(title)
            .setView(dialogView)
            .setPositiveButton("OK", null)
            .show()
    }

    /** Displays a scrollable dialog with license and attribution information. */
    private fun showLicensesDialog() {
        val dialogView = layoutInflater.inflate(R.layout.dialog_licenses, null)
        val textView = dialogView.findViewById<TextView>(R.id.tvLicenses)

        val licensesText = try {
            resources.openRawResource(R.raw.licenses)
                .bufferedReader()
                .use { it.readText() }
        } catch (e: Exception) {
            e.printStackTrace()
            "Could not load license information."
        }

        // Render HTML content
        textView.text =
            Html.fromHtml(licensesText, Html.FROM_HTML_MODE_COMPACT)


        textView.movementMethod = LinkMovementMethod.getInstance()

        AlertDialog.Builder(this)
            .setTitle("Licenses & Attribution")
            .setView(dialogView)
            .setPositiveButton("OK", null)
            .show()
    }

    private fun showPrivacyPolicyDialog() {
        // We reuse the 'dialog_licenses' layout since it just contains a ScrollView and TextView
        val dialogView = layoutInflater.inflate(R.layout.dialog_licenses, null)
        val textView = dialogView.findViewById<TextView>(R.id.tvLicenses)

        val policyText = try {
            // Assumes file is named privacy_policy.html in res/raw/
            resources.openRawResource(R.raw.privacy_policy)
                .bufferedReader()
                .use { it.readText() }
        } catch (e: Exception) {
            e.printStackTrace()
            "Could not load Privacy Policy."
        }

        textView.text = Html.fromHtml(policyText, Html.FROM_HTML_MODE_COMPACT)

        // This makes the links clickable
        textView.movementMethod = LinkMovementMethod.getInstance()

        AlertDialog.Builder(this)
            .setTitle("Privacy Policy")
            .setView(dialogView)
            .setPositiveButton("OK", null)
            .show()
    }

    private fun clearFocusAndHideKeyboard() {
        // 1. Clear focus from the EditText
        etLearnThreshold.clearFocus()

        // 2. Clear focus from the entire view hierarchy and hide the keyboard
        val view = this.currentFocus
        if (view != null) {
            val imm = getSystemService(Context.INPUT_METHOD_SERVICE) as InputMethodManager
            imm.hideSoftInputFromWindow(view.windowToken, 0)
        }
        // As a fallback, ensures the root view is focused
        val rootView = findViewById<View>(android.R.id.content)
        rootView.requestFocus()
    }
}