/*
 * La Et Le — French Gender Classification Trainer
 *
 * Copyright (C) 2025 Seweryn Polec <sewerynpol@protonmail.com>
 *
 * This program is free software: you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation, either version 3 of the License, or
 * (at your option) any later version.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE. See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with this program. If not, see <https://www.gnu.org/licenses/>.
 */

package com.languageapp.laetle

import android.content.Intent
import android.os.Bundle
import android.text.Html
import android.widget.ImageButton
import android.widget.TextView
import androidx.appcompat.app.AlertDialog
import androidx.appcompat.app.AppCompatActivity
import com.google.gson.Gson
import com.google.gson.reflect.TypeToken
import android.util.Log
import com.google.android.material.button.MaterialButton
import androidx.activity.enableEdgeToEdge
import androidx.core.view.ViewCompat
import androidx.core.view.WindowInsetsCompat

class SessionActivity : AppCompatActivity() {

    private lateinit var tvNewCardsCount: TextView
    private lateinit var tvReviewCardsCount: TextView
    private lateinit var tvTotalScore: TextView
    private val timeProvider: TimeProvider by lazy { RealTimeProvider() }
    private lateinit var btnLearnNewCards: MaterialButton
    private lateinit var btnReviewCurrentCards: MaterialButton
    private lateinit var btnTutorialInfo: ImageButton

    private val progressPrefs by lazy { getSharedPreferences("progress", MODE_PRIVATE) }
    private val gson = Gson()

    private lateinit var statsManager: StatsManager

    override fun onCreate(savedInstanceState: Bundle?) {
        super.onCreate(savedInstanceState)

        enableEdgeToEdge()

        setContentView(R.layout.activity_session)

        // Handle Insets
        ViewCompat.setOnApplyWindowInsetsListener(findViewById(android.R.id.content)) { v, insets ->
            val systemBars = insets.getInsets(WindowInsetsCompat.Type.systemBars())
            v.setPadding(systemBars.left, systemBars.top, systemBars.right, systemBars.bottom)
            insets
        }

        statsManager = StatsManager(this)

        tvNewCardsCount = findViewById(R.id.tvNewCardsCount)
        tvReviewCardsCount = findViewById(R.id.tvReviewCardsCount)
        btnLearnNewCards = findViewById<MaterialButton>(R.id.btnLearnNewCards)
        btnReviewCurrentCards = findViewById<MaterialButton>(R.id.btnReviewCurrentCards)
        btnTutorialInfo = findViewById(R.id.btnTutorialInfo)

        btnLearnNewCards.setOnClickListener { startSession(REVIEW_MODE_NEW) }
        btnReviewCurrentCards.setOnClickListener { startSession(REVIEW_MODE_REVIEW) }
        tvTotalScore = findViewById(R.id.tvTotalScore)

        btnTutorialInfo.setOnClickListener {
            showTutorialDialog("Session Hub Tutorial", R.raw.tutorial_session)
        }
    }

    override fun onResume() {
        super.onResume()
        statsManager = StatsManager(this)
        updateCardCounts()
    }

    private fun startSession(mode: Int) {
        val intent = Intent(this, StartActivity::class.java).apply {
            putExtra("SESSION_MODE", mode)
        }
        startActivity(intent)
    }

    private fun updateCardCounts() {
        val totalWords = loadFullNounsListSize()
        val masteredWordsCount = progressPrefs.getInt("learnedCount", 0)
        val newCardsRemaining = totalWords - masteredWordsCount

        val reviewDeckJson = progressPrefs.getString(PREF_REVIEW_DECK, "[]")
        val reviewDeck = gson.fromJson<MutableList<ReviewItem>>(
            reviewDeckJson,
            object : TypeToken<MutableList<ReviewItem>>() {}.type
        )
        val today = timeProvider.getCurrentTimeMillis()
        val cardsToReviewToday = reviewDeck.count { it.nextReviewDate <= today }

        tvNewCardsCount.text = getString(R.string.session_new_cards_count, newCardsRemaining)
        tvReviewCardsCount.text = getString(R.string.session_review_cards_count, cardsToReviewToday)

        btnLearnNewCards.isEnabled = newCardsRemaining > 0
        btnReviewCurrentCards.isEnabled = cardsToReviewToday > 0

        tvTotalScore.text = "Score: ${statsManager.totalScore}"
    }

    private fun showTutorialDialog(title: String, rawResId: Int) {
        val dialogView = layoutInflater.inflate(R.layout.dialog_licenses, null)
        val textView = dialogView.findViewById<TextView>(R.id.tvLicenses)
        val tutorialText = try {
            resources.openRawResource(rawResId).bufferedReader().use { it.readText() }
        } catch (_: Exception) {
            "Could not load tutorial."
        }
        // The corrected line:
        textView.text = Html.fromHtml(tutorialText, Html.FROM_HTML_MODE_COMPACT)

        AlertDialog.Builder(this)
            .setTitle(title)
            .setView(dialogView)
            .setPositiveButton("OK", null)
            .show()
    }

    private fun loadFullNounsListSize(): Int {
        return try {
            resources.openRawResource(R.raw.french_nouns_frequency)
                .bufferedReader()
                .readLines()
                .size - 1
        } catch (_: Exception) {
            Log.e("SessionActivity", "Failed to load nouns list")
            DEFAULT_NOUN_COUNT
        }
    }
}