/*
 * La Et Le — French Gender Classification Trainer
 *
 * Copyright (C) 2025 Seweryn Polec <sewerynpol@protonmail.com>
 *
 * This program is free software: you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation, either version 3 of the License, or
 * (at your option) any later version.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE. See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with this program. If not, see <https://www.gnu.org/licenses/>.
 */

package com.languageapp.laetle

import android.Manifest
import android.app.TimePickerDialog
import android.content.SharedPreferences
import android.os.Build
import android.os.Bundle
import android.text.Html
import android.view.View
import android.widget.Button
import android.widget.ImageButton
import android.widget.TextView
import android.widget.ToggleButton
import androidx.activity.result.contract.ActivityResultContracts
import androidx.appcompat.app.AlertDialog
import androidx.appcompat.app.AppCompatActivity
import com.google.android.material.materialswitch.MaterialSwitch
import com.google.android.material.snackbar.Snackbar
import androidx.core.content.edit
import java.util.Locale
import androidx.activity.enableEdgeToEdge
import androidx.core.view.ViewCompat
import androidx.core.view.WindowInsetsCompat

/**
 * Activity for managing daily practice reminder settings.
 */
class ReminderSettingsActivity : AppCompatActivity() {

    // UI elements
    private lateinit var switchEnable: MaterialSwitch
    private lateinit var btnPickTime: Button
    private lateinit var tvSelectedTime: TextView
    private lateinit var dayButtons: List<ToggleButton>
    private lateinit var btnTutorialInfo: ImageButton

    // Persistence and state
    private lateinit var prefs: SharedPreferences
    private val settingsPrefs by lazy { getSharedPreferences("settings", MODE_PRIVATE) }
    private var selectedHour = 9
    private var selectedMinute = 0
    private val dayStates = BooleanArray(7) // Stores last state of day buttons when switch is off

    /** Displays a short Snackbar message. */
    private fun showSnack(message: String) {
        val root = findViewById<View>(android.R.id.content)
        Snackbar.make(root, message, Snackbar.LENGTH_SHORT).show()
    }

    /** Launcher for requesting the POST_NOTIFICATIONS permission on Android 13+. */
    private val requestPermissionLauncher =
        registerForActivityResult(ActivityResultContracts.RequestPermission()) { granted ->
            if (granted) {
                showSnack("Permission granted. Reminders scheduled!")
                ReminderScheduler.scheduleReminder(this)
            } else {
                showSnack("Notifications permission denied. Reminders disabled.")
                // Disable UI and persistence if permission is denied
                switchEnable.isChecked = false
                prefs.edit { putBoolean("enabled", false) }
                updateUIState(false)
            }
        }

    override fun onCreate(savedInstanceState: Bundle?) {
        super.onCreate(savedInstanceState)

        enableEdgeToEdge()

        setContentView(R.layout.activity_reminder_settings)

        ViewCompat.setOnApplyWindowInsetsListener(findViewById(R.id.rootLayout)) { v, insets ->
            val systemBars = insets.getInsets(WindowInsetsCompat.Type.systemBars())
            v.setPadding(systemBars.left, systemBars.top, systemBars.right, systemBars.bottom)
            insets
        }

        prefs = getSharedPreferences("reminder_prefs", MODE_PRIVATE)

        // Initialize UI components
        switchEnable = findViewById(R.id.switchEnableReminders)
        btnPickTime = findViewById(R.id.btnPickTime)
        tvSelectedTime = findViewById(R.id.tvSelectedTime)
        btnTutorialInfo = findViewById(R.id.btnTutorialInfo)

        // Group day ToggleButtons
        dayButtons = listOf(
            findViewById(R.id.tbMon), findViewById(R.id.tbTue), findViewById(R.id.tbWed),
            findViewById(R.id.tbThu), findViewById(R.id.tbFri), findViewById(R.id.tbSat),
            findViewById(R.id.tbSun)
        )

        loadPreferences()
        setupListeners()

        // Set initial state
        updateUIState(switchEnable.isChecked)

        // Setup tutorial button
        btnTutorialInfo.setOnClickListener {
            showTutorialDialog("Notification Settings Tutorial", R.raw.tutorial_notifications)
        }
    }

    override fun onResume() {
        super.onResume()
        // Control tutorial button visibility based on global setting
    }

    /** Sets up all UI event listeners. */
    private fun setupListeners() {
        // Enable switch listener
        switchEnable.setOnCheckedChangeListener { _, isChecked ->
            prefs.edit { putBoolean("enabled", isChecked) }
            updateUIState(isChecked)

            if (isChecked) {
                checkAndSchedule()
            } else {
                ReminderScheduler.cancelReminder(this)
                showSnack("Reminders disabled")
            }
        }

        // Time picker button listener
        btnPickTime.setOnClickListener {
            if (!switchEnable.isChecked) {
                showSnack("Enable Daily Practice Reminders first!")
                return@setOnClickListener
            }

            val picker = TimePickerDialog(
                this,
                { _, hour, minute ->
                    selectedHour = hour
                    selectedMinute = minute
                    updateTimeDisplay()
                    prefs.edit {
                        putInt("hour", hour)
                            .putInt("minute", minute)
                    }
                    // Reschedule with the new time
                    if (switchEnable.isChecked) ReminderScheduler.scheduleReminder(this)
                },
                selectedHour,
                selectedMinute,
                true
            )
            picker.show()
        }

        // Day buttons listeners
        dayButtons.forEachIndexed { index, button ->
            button.setOnCheckedChangeListener { _, isChecked ->
                if (!switchEnable.isChecked) {
                    // Revert state and notify user if reminders are disabled
                    button.isChecked = !isChecked
                    showSnack("Enable Daily Practice Reminders first!")
                    return@setOnCheckedChangeListener
                }
                // Save state and reschedule if reminders are enabled
                dayStates[index] = isChecked
                prefs.edit { putBoolean("day_$index", isChecked) }
                if (switchEnable.isChecked) ReminderScheduler.scheduleReminder(this)
            }
        }
    }

    /** Updates the visual state of controls based on the enable switch state. */
    private fun updateUIState(isEnabled: Boolean) {
        val alpha = if (isEnabled) 1.0f else 0.5f
        btnPickTime.alpha = alpha
        tvSelectedTime.alpha = alpha

        if (isEnabled) {
            // Restore saved states to day buttons
            dayButtons.forEachIndexed { index, button ->
                button.isChecked = dayStates[index]
            }
        } else {
            // Save current states and visually uncheck day buttons
            dayButtons.forEachIndexed { index, button ->
                dayStates[index] = button.isChecked
                button.isChecked = false
            }
        }
    }

    /** Loads time and day preferences from SharedPreferences. */
    private fun loadPreferences() {
        switchEnable.isChecked = prefs.getBoolean("enabled", false)
        selectedHour = prefs.getInt("hour", 9)
        selectedMinute = prefs.getInt("minute", 0)

        // Load day states into the temporary array
        for (i in dayButtons.indices) {
            dayStates[i] = prefs.getBoolean("day_$i", false)
        }

        updateTimeDisplay()
    }

    /** Updates the TextView with the currently selected time. */
    private fun updateTimeDisplay() {
        val formatted = String.format(Locale.getDefault(), "%02d:%02d", selectedHour, selectedMinute)
        tvSelectedTime.text = getString(R.string.reminder_time_format, formatted)
    }

    /** Checks for notification permission and schedules the alarm if granted or unnecessary. */
    private fun checkAndSchedule() {
        if (Build.VERSION.SDK_INT >= Build.VERSION_CODES.TIRAMISU) {
            requestPermissionLauncher.launch(Manifest.permission.POST_NOTIFICATIONS)
        } else {
            ReminderScheduler.scheduleReminder(this)
            showSnack("Reminders enabled and scheduled!")
        }
    }

    /** Displays the tutorial dialog by reading and rendering HTML from a raw resource. */
    private fun showTutorialDialog(title: String, rawResId: Int) {
        val dialogView = layoutInflater.inflate(R.layout.dialog_licenses, null)
        val textView = dialogView.findViewById<TextView>(R.id.tvLicenses)

        // Read the text from the raw resource file
        val tutorialText = try {
            resources.openRawResource(rawResId)
                .bufferedReader()
                .use { it.readText() }
        } catch (e: Exception) {
            e.printStackTrace()
            "Could not load tutorial information for $title. (Resource $rawResId is missing in R.raw)"
        }

        // Render HTML content for bolding and lists
        textView.text =
            Html.fromHtml(tutorialText, Html.FROM_HTML_MODE_COMPACT)

        AlertDialog.Builder(this)
            .setTitle(title)
            .setView(dialogView)
            .setPositiveButton("OK", null)
            .show()
    }
}