/*
 * La Et Le — French Gender Classification Trainer
 *
 * Copyright (C) 2025 Seweryn Polec <sewerynpol@protonmail.com>
 *
 * This program is free software: you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation, either version 3 of the License, or
 * (at your option) any later version.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE. See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with this program. If not, see <https://www.gnu.org/licenses/>.
 */

package com.languageapp.laetle

import android.app.*
import android.content.*
import android.util.Log
import java.util.*

/**
 * Manages scheduling and cancelling of weekly recurring alarms for practice reminders.
 */
object ReminderScheduler {

    /** Schedules alarms for all selected days at the chosen time. Cancels previous alarms first. */
    fun scheduleReminder(context: Context) {
        val prefs = context.getSharedPreferences("reminder_prefs", Context.MODE_PRIVATE)
        if (!prefs.getBoolean("enabled", false)) return

        val hour = prefs.getInt("hour", 9)
        val minute = prefs.getInt("minute", 0)
        val alarmManager = context.getSystemService(Context.ALARM_SERVICE) as AlarmManager

        // Cancel existing alarms before scheduling new ones
        cancelReminder(context)

        // Schedule one alarm per selected day
        for (dayIndex in 0 until 7) {
            if (!prefs.getBoolean("day_$dayIndex", false)) continue

            // Map the 0-6 index (Mon-Sun) to Calendar.DAY_OF_WEEK (Sun=1...Sat=7)
            val calendarDayOfWeek = when (dayIndex) {
                0 -> Calendar.MONDAY
                1 -> Calendar.TUESDAY
                2 -> Calendar.WEDNESDAY
                3 -> Calendar.THURSDAY
                4 -> Calendar.FRIDAY
                5 -> Calendar.SATURDAY
                6 -> Calendar.SUNDAY
                else -> throw IllegalStateException("Invalid dayIndex: $dayIndex")
            }

            val calendar = Calendar.getInstance().apply {
                set(Calendar.DAY_OF_WEEK, calendarDayOfWeek)
                set(Calendar.HOUR_OF_DAY, hour)
                set(Calendar.MINUTE, minute)
                set(Calendar.SECOND, 0)
                set(Calendar.MILLISECOND, 0)

                // If the scheduled time/day has already passed this week, schedule for the next week
                if (before(Calendar.getInstance())) {
                    add(Calendar.WEEK_OF_YEAR, 1)
                }
            }

            val intent = Intent(context, ReminderReceiver::class.java)
            val pendingIntent = PendingIntent.getBroadcast(
                context,
                dayIndex, // Unique request code per day
                intent,
                PendingIntent.FLAG_UPDATE_CURRENT or PendingIntent.FLAG_IMMUTABLE
            )

            // Set a weekly repeating alarm
            alarmManager.setRepeating(
                AlarmManager.RTC_WAKEUP,
                calendar.timeInMillis,
                AlarmManager.INTERVAL_DAY * 7,
                pendingIntent
            )

            Log.d("ReminderScheduler", "Scheduled reminder for day $dayIndex (Calendar constant: $calendarDayOfWeek) at $hour:$minute")
        }
    }

    /** Cancels all 7 potential pending intents for reminders. */
    fun cancelReminder(context: Context) {
        val alarmManager = context.getSystemService(Context.ALARM_SERVICE) as AlarmManager
        for (dayIndex in 0 until 7) {
            val intent = Intent(context, ReminderReceiver::class.java)
            val pendingIntent = PendingIntent.getBroadcast(
                context,
                dayIndex,
                intent,
                PendingIntent.FLAG_UPDATE_CURRENT or PendingIntent.FLAG_IMMUTABLE
            )
            alarmManager.cancel(pendingIntent)
        }
        Log.d("ReminderScheduler", "All reminders canceled")
    }
}