package com.kylecorry.trail_sense.tools.climate.infrastructure.temperatures.calculators

import com.kylecorry.sol.math.Range
import com.kylecorry.sol.science.astronomy.SunTimesMode
import com.kylecorry.sol.time.Time.getClosestFutureTime
import com.kylecorry.sol.time.Time.getClosestPastTime
import com.kylecorry.sol.time.Time.toZonedDateTime
import com.kylecorry.sol.units.Coordinate
import com.kylecorry.sol.units.Reading
import com.kylecorry.sol.units.Temperature
import com.kylecorry.trail_sense.tools.astronomy.domain.AstronomyService
import java.time.Duration
import java.time.LocalDate
import java.time.LocalTime
import java.time.ZonedDateTime

internal class DailyTemperatureCalculator(
    private val location: Coordinate,
    private val dailyTemperatureRangeProvider: suspend (location: Coordinate, date: LocalDate) -> Range<Temperature>
) : ITemperatureCalculator {

    private val calculators = mutableListOf<Pair<Range<ZonedDateTime>, ITemperatureCalculator>>()
    private val offset = Duration.ofHours(3)
    private val astronomy = AstronomyService()

    override suspend fun calculate(time: ZonedDateTime): Temperature {
        val existing = calculators.firstOrNull { it.first.contains(time) }
        val calculator = if (existing == null) {
            val newCalculator = createCalculator(time)
            calculators.add(newCalculator)
            newCalculator.second
        } else {
            existing.second
        }
        return calculator.calculate(time)
    }

    private suspend fun createCalculator(time: ZonedDateTime): Pair<Range<ZonedDateTime>, ITemperatureCalculator> {
        val range = dailyTemperatureRangeProvider.invoke(location, time.toLocalDate())
        val nextHigh = getNextHighTime(time)
        val nextLow = getNextLowTime(time)
        val previousHigh = getPreviousHighTime(time)
        val previousLow = getPreviousLowTime(time)

        // Low High x Low High
        if (nextLow.isBefore(nextHigh)) {

            val start = if (nextLow.toLocalDate() != time.toLocalDate()) {
                dailyTemperatureRangeProvider.invoke(location, nextLow.toLocalDate()).start
            } else {
                range.start
            }

            val end = if (previousHigh.toLocalDate() != time.toLocalDate()) {
                dailyTemperatureRangeProvider.invoke(location, previousHigh.toLocalDate()).end
            } else {
                range.end
            }

            return Range(previousHigh, nextLow) to QuadraticTemperatureCalculator(
                Reading(start, nextLow.toInstant()),
                Reading(end, previousHigh.toInstant())
            )
        }

        val start = if (previousLow.toLocalDate() != time.toLocalDate()) {
            dailyTemperatureRangeProvider.invoke(location, previousLow.toLocalDate()).start
        } else {
            range.start
        }

        val end = if (nextHigh.toLocalDate() != time.toLocalDate()) {
            dailyTemperatureRangeProvider.invoke(location, nextHigh.toLocalDate()).end
        } else {
            range.end
        }

        // High Low x High Low
        return Range(previousLow, nextHigh) to SineTemperatureCalculator(
            Reading(start, previousLow.toInstant()),
            Reading(end, nextHigh.toInstant())
        )
    }


    private fun getNextHighTime(time: ZonedDateTime): ZonedDateTime {
        val today = astronomy.getSunTimes(location, SunTimesMode.Actual, time.toLocalDate())
        val tomorrow =
            astronomy.getSunTimes(location, SunTimesMode.Actual, time.toLocalDate().plusDays(1))
        return getClosestFutureTime(
            time,
            listOf(today.transit?.plus(offset), tomorrow.transit?.plus(offset))
        ) ?: getClosestFutureTime(
            time,
            listOf(
                ZonedDateTime.of(time.toLocalDate(), LocalTime.NOON.plus(offset), time.zone),
                ZonedDateTime.of(
                    time.toLocalDate().plusDays(1),
                    LocalTime.NOON.plus(offset),
                    time.zone
                )
            )
        )!!
    }

    private fun getNextLowTime(time: ZonedDateTime): ZonedDateTime {
        return astronomy.getNextSunrise(location, SunTimesMode.Actual, time)?.toZonedDateTime()
            ?: getClosestFutureTime(
                time,
                listOf(
                    ZonedDateTime.of(time.toLocalDate(), LocalTime.MIN, time.zone),
                    ZonedDateTime.of(time.toLocalDate().plusDays(1), LocalTime.MIN, time.zone)
                )
            )!!
    }

    private fun getPreviousHighTime(time: ZonedDateTime): ZonedDateTime {
        val today = astronomy.getSunTimes(location, SunTimesMode.Actual, time.toLocalDate())
        val yesterday =
            astronomy.getSunTimes(location, SunTimesMode.Actual, time.toLocalDate().minusDays(1))
        return getClosestPastTime(
            time,
            listOf(today.transit?.plus(offset), yesterday.transit?.plus(offset))
        ) ?: getClosestPastTime(
            time,
            listOf(
                ZonedDateTime.of(time.toLocalDate(), LocalTime.NOON.plus(offset), time.zone),
                ZonedDateTime.of(
                    time.toLocalDate().minusDays(1),
                    LocalTime.NOON.plus(offset),
                    time.zone
                )
            )
        )!!
    }

    private fun getPreviousLowTime(time: ZonedDateTime): ZonedDateTime {
        val today = astronomy.getSunTimes(location, SunTimesMode.Actual, time.toLocalDate())
        val yesterday =
            astronomy.getSunTimes(location, SunTimesMode.Actual, time.toLocalDate().minusDays(1))
        return getClosestPastTime(
            time,
            listOf(today.rise, yesterday.rise)
        ) ?: getClosestPastTime(
            time,
            listOf(
                ZonedDateTime.of(time.toLocalDate(), LocalTime.MIN, time.zone),
                ZonedDateTime.of(
                    time.toLocalDate().minusDays(1),
                    LocalTime.MIN,
                    time.zone
                )
            )
        )!!
    }
}