package com.kylecorry.trail_sense.tools.photo_maps.infrastructure.tiles

import android.content.Context
import com.kylecorry.andromeda.bitmaps.operations.BitmapOperation
import com.kylecorry.sol.math.SolMath
import com.kylecorry.sol.science.geology.CoordinateBounds
import com.kylecorry.trail_sense.shared.andromeda_temp.intersects2
import com.kylecorry.trail_sense.shared.map_layers.tiles.IGeographicImageRegionLoader
import com.kylecorry.trail_sense.shared.map_layers.tiles.ITileSourceSelector
import com.kylecorry.trail_sense.tools.photo_maps.domain.PhotoMap

class PhotoMapTileSourceSelector(
    private val context: Context,
    maps: List<PhotoMap>,
    private val maxLayers: Int = 4,
    private val loadPdfs: Boolean = true,
    private val isPixelPerfect: Boolean = false,
    private val operations: List<BitmapOperation> = emptyList()
) : ITileSourceSelector {

    private val sortedMaps = maps
        .filter { it.isCalibrated }
        .sortedBy { it.distancePerPixel() }

    // TODO: Factor in rotation by using projection to see if the bounds intersect/are contained
    override fun getRegionLoaders(bounds: CoordinateBounds): List<IGeographicImageRegionLoader> {
        val minArea = bounds.width().meters().value.toDouble() * bounds.height()
            .meters().value.toDouble() * 0.25

        val possibleMaps = sortedMaps.filter {
            val boundary = it.boundary() ?: return@filter false
            if (boundary == CoordinateBounds.world) {
                return@filter true
            }
            val area = boundary.width().meters().value.toDouble() *
                    boundary.height().meters().value.toDouble()
            area >= minArea
        }

        val firstContained = possibleMaps.firstOrNull {
            contains(
                it.boundary() ?: return@firstOrNull false,
                bounds,
                fullyContained = true
            )
        }

        val containedMaps = possibleMaps.filter {
            contains(
                it.boundary() ?: return@filter false,
                bounds
            )
        }.take(maxLayers).toMutableList()


        val maps = if (firstContained != null && !containedMaps.contains(firstContained)) {
            if (containedMaps.size == maxLayers) {
                containedMaps.removeLastOrNull()
            }
            containedMaps.add(firstContained)
            containedMaps
        } else if (firstContained != null && SolMath.isZero(
                firstContained.baseRotation() - firstContained.calibration.rotation,
                0.5f
            )
        ) {
            // The contained map isn't really rotated so only include a map after it
            val index = containedMaps.indexOf(firstContained)
            containedMaps.subList(
                0,
                minOf(index + 2, containedMaps.size)
            )
        } else {
            containedMaps
        }

        return maps.map {
            PhotoMapRegionLoader(
                context,
                it,
                loadPdfs,
                isPixelPerfect,
                operations
            )
        }
    }

    // TODO: Extract to sol
    private fun contains(
        bounds: CoordinateBounds,
        subBounds: CoordinateBounds,
        fullyContained: Boolean = false
    ): Boolean {

        return if (fullyContained) {
            val corners = listOf(
                bounds.contains(subBounds.northWest),
                bounds.contains(subBounds.northEast),
                bounds.contains(subBounds.southWest),
                bounds.contains(subBounds.southEast),
                bounds.contains(subBounds.center)
            )
            corners.all { it }
        } else {
            bounds.intersects2(subBounds)
        }
    }

}