import 'package:flutter/material.dart';

import 'package:flutter_form_builder/flutter_form_builder.dart';

/// Single Checkbox field
class FormBuilderCheckbox extends FormBuilderFieldDecoration<bool> {
  /// The primary content of the CheckboxListTile.
  ///
  /// Typically a [Text] widget.
  final Widget title;

  /// Additional content displayed below the title.
  ///
  /// Typically a [Text] widget.
  final Widget? subtitle;

  /// A widget to display on the opposite side of the tile from the checkbox.
  ///
  /// Typically an [Icon] widget.
  final Widget? secondary;

  /// The color to use when this checkbox is checked.
  ///
  /// Defaults to accent color of the current [Theme].
  final Color? activeColor;

  /// The color to use for the check icon when this checkbox is checked.
  ///
  /// Defaults to Color(0xFFFFFFFF).
  final Color? checkColor;

  /// Where to place the control relative to its label.
  final ListTileControlAffinity controlAffinity;

  /// Defines insets surrounding the tile's contents.
  ///
  /// This value will surround the [Checkbox], [title], [subtitle], and [secondary]
  /// widgets in [CheckboxListTile].
  ///
  /// When the value is null, the `contentPadding` is `EdgeInsets.symmetric(horizontal: 16.0)`.
  final EdgeInsets contentPadding;

  /// Defines how compact the list tile's layout will be.
  ///
  /// {@macro flutter.material.themedata.visualDensity}
  final VisualDensity? visualDensity;

  /// {@macro flutter.widgets.Focus.autofocus}
  final bool autofocus;

  /// If true the checkbox's [value] can be true, false, or null.
  ///
  /// Checkbox displays a dash when its value is null.
  ///
  /// When a tri-state checkbox ([tristate] is true) is tapped, its [onChanged]
  /// callback will be applied to true if the current value is false, to null if
  /// value is true, and to false if value is null (i.e. it cycles through false
  /// => true => null => false when tapped).
  ///
  /// If tristate is false (the default), [value] must not be null.
  final bool tristate;

  /// Whether to render icons and text in the [activeColor].
  ///
  /// No effort is made to automatically coordinate the [selected] state and the
  /// [value] state. To have the list tile appear selected when the checkbox is
  /// checked, pass the same value to both.
  ///
  /// Normally, this property is left to its default value, false.
  final bool selected;

  /// {@macro flutter.material.checkbox.shape}
  ///
  /// If this property is null then [CheckboxThemeData.shape] of [ThemeData.checkboxTheme]
  /// is used. If that's null then the shape will be a [RoundedRectangleBorder]
  /// with a circular corner radius of 1.0.
  final OutlinedBorder? shape;

  /// {@macro flutter.material.checkbox.side}
  ///
  /// The given value is passed directly to [Checkbox.side].
  ///
  /// If this property is null, then [CheckboxThemeData.side] of
  /// [ThemeData.checkboxTheme] is used. If that is also null, then the side
  /// will be width 2.
  final BorderSide? side;

  /// Creates a single Checkbox field
  FormBuilderCheckbox({
    super.key,
    required super.name,
    super.validator,
    super.initialValue,
    super.decoration = const InputDecoration(
      border: InputBorder.none,
      focusedBorder: InputBorder.none,
      enabledBorder: InputBorder.none,
      errorBorder: InputBorder.none,
      disabledBorder: InputBorder.none,
    ),
    super.onChanged,
    super.valueTransformer,
    super.enabled,
    super.onSaved,
    super.autovalidateMode = AutovalidateMode.disabled,
    super.onReset,
    super.focusNode,
    super.restorationId,
    super.errorBuilder,
    required this.title,
    this.activeColor,
    this.autofocus = false,
    this.checkColor,
    this.contentPadding = EdgeInsets.zero,
    this.visualDensity,
    this.controlAffinity = ListTileControlAffinity.leading,
    this.secondary,
    this.selected = false,
    this.subtitle,
    this.tristate = false,
    this.shape,
    this.side,
  }) : super(
         builder: (FormFieldState<bool?> field) {
           final state = field as _FormBuilderCheckboxState;

           return InputDecorator(
             decoration: state.decoration,
             isFocused: state.effectiveFocusNode.hasFocus,
             child: CheckboxListTile(
               dense: true,
               isThreeLine: false,
               focusNode: state.effectiveFocusNode,
               title: title,
               subtitle: subtitle,
               value: tristate ? state.value : (state.value ?? false),
               onChanged: state.enabled
                   ? (value) {
                       state.didChange(value);
                     }
                   : null,
               checkColor: checkColor,
               activeColor: activeColor,
               secondary: secondary,
               controlAffinity: controlAffinity,
               autofocus: autofocus,
               tristate: tristate,
               contentPadding: contentPadding,
               visualDensity: visualDensity,
               selected: selected,
               checkboxShape: shape,
               side: side,
             ),
           );
         },
       );

  @override
  FormBuilderFieldDecorationState<FormBuilderCheckbox, bool> createState() =>
      _FormBuilderCheckboxState();
}

class _FormBuilderCheckboxState
    extends FormBuilderFieldDecorationState<FormBuilderCheckbox, bool> {
  void handleFocusChange() {
    setState(() {});
  }

  @override
  void initState() {
    super.initState();
    effectiveFocusNode.addListener(handleFocusChange);
  }

  @override
  void dispose() {
    effectiveFocusNode.removeListener(handleFocusChange);
    super.dispose();
  }

  @override
  void didChange(bool? value) {
    focus();
    super.didChange(value);
  }
}
