// Copyright 2023 Insco. All rights reserved.
// This source code is licensed under the GNU General Public License v3.0.
// See the LICENSE file in the project root for full license information.

import 'package:flutter/material.dart';

import 'package:picguard/generated/colors.gen.dart';

typedef BaseFormItemCallback = void Function();

///
class BaseFormItem extends StatefulWidget {
  ///
  const BaseFormItem({
    required this.child,
    this.title,
    this.required = true,
    this.showTip = true,
    this.padding = const EdgeInsets.only(top: 10),
    this.mainAxisAlignment = MainAxisAlignment.start,
    this.mainAxisSize = MainAxisSize.max,
    this.tipWidget,
    this.onTipTap,
    super.key,
  });

  final Widget child;
  final String? title;
  final bool required;
  final bool showTip;
  final EdgeInsetsGeometry padding;
  final MainAxisAlignment mainAxisAlignment;
  final MainAxisSize mainAxisSize;
  final Widget? tipWidget;
  final BaseFormItemCallback? onTipTap;

  @override
  State<BaseFormItem> createState() => _BaseFormItemState();
}

class _BaseFormItemState extends State<BaseFormItem> {
  @override
  Widget build(BuildContext context) {
    return widget.title != null
        ? Padding(
            padding: widget.padding,
            child: Column(
              crossAxisAlignment: CrossAxisAlignment.start,
              mainAxisSize: widget.mainAxisSize,
              children: [
                if (!widget.showTip)
                  content
                else
                  Row(
                    mainAxisAlignment: widget.mainAxisAlignment,
                    children: [
                      content,
                      if (widget.tipWidget == null)
                        tips
                      else
                        GestureDetector(
                          onTap: () {
                            widget.onTipTap?.call();
                          },
                          child: widget.tipWidget,
                        ),
                    ],
                  ),
                widget.child,
              ],
            ),
          )
        : Padding(
            padding: widget.padding,
            child: widget.child,
          );
  }

  Widget get content {
    final brightness = Theme.of(context).brightness;
    final isDark = brightness == Brightness.dark;
    return Text.rich(
      TextSpan(
        children: [
          if (widget.required)
            const TextSpan(
              text: '*',
              style: TextStyle(
                color: PGColors.errorTextColor,
                fontSize: 14,
              ),
            ),
          TextSpan(
            text: widget.title,
            style: TextStyle(
              color: isDark ? Colors.white : PGColors.primaryTextColor,
              fontSize: 14,
            ),
          ),
        ],
      ),
      maxLines: 2,
      overflow: TextOverflow.ellipsis,
    );
  }

  Widget get tips {
    return Padding(
      padding: const EdgeInsets.only(left: 4),
      child: IconButton(
        style: ButtonStyle(
          padding: WidgetStateProperty.all(EdgeInsets.zero),
          elevation: WidgetStateProperty.all(0),
          minimumSize: WidgetStateProperty.all(Size.zero),
          tapTargetSize: MaterialTapTargetSize.shrinkWrap,
        ),
        onPressed: widget.onTipTap,
        icon: const Icon(
          Icons.info,
          size: 16,
          color: PGColors.primaryColor,
        ),
      ),
    );
  }
}
