import 'package:slang/src/builder/generator/helper.dart';
import 'package:slang/src/builder/model/build_model_config.dart';
import 'package:slang/src/builder/model/generate_config.dart';
import 'package:slang/src/builder/model/i18n_data.dart';
import 'package:slang/src/builder/model/i18n_locale.dart';
import 'package:slang/src/builder/model/node.dart';
import 'package:slang/src/builder/utils/path_utils.dart';

String generateHeader(
  GenerateConfig config,
  List<I18nData> allLocales,
) {
  const String pluralResolverType = 'PluralResolver';
  const String pluralResolverMapCardinal = '_pluralResolversCardinal';
  const String pluralResolverMapOrdinal = '_pluralResolversOrdinal';

  final buffer = StringBuffer();

  _generateHeaderComment(
    buffer: buffer,
    config: config,
    translations: allLocales,
    now: DateTime.now().toUtc(),
  );

  _generateImports(config, buffer);

  _generateLocaleImports(
    buffer: buffer,
    config: config,
    locales: allLocales,
  );

  if (config.translationOverrides) {
    _generateBuildConfig(
      buffer: buffer,
      config: config.buildConfig,
    );
  }

  _generateEnum(
    buffer: buffer,
    config: config,
    allLocales: allLocales,
  );

  if (config.localeHandling) {
    _generateTranslationGetter(
      buffer: buffer,
      config: config,
    );

    _generateLocaleSettings(
      buffer: buffer,
      config: config,
      allLocales: allLocales,
      pluralResolverType: pluralResolverType,
      pluralResolverCardinal: pluralResolverMapCardinal,
      pluralResolverOrdinal: pluralResolverMapOrdinal,
    );
  }

  _generateUtil(
    buffer: buffer,
    config: config,
  );

  _generateContextEnums(buffer: buffer, config: config);

  _generateInterfaces(buffer: buffer, config: config);

  return buffer.toString();
}

void _generateHeaderComment({
  required StringBuffer buffer,
  required GenerateConfig config,
  required List<I18nData> translations,
  required DateTime now,
}) {
  final int count = translations.fold(
    0,
    (prev, curr) => prev + _countTranslations(curr.root),
  );

  String countPerLocale = '';
  if (translations.length != 1) {
    countPerLocale = ' (${(count / translations.length).floor()} per locale)';
  }

  String twoDigits(int value) => value.toString().padLeft(2, '0');

  final String statisticsStr;
  if (config.renderStatistics) {
    statisticsStr = '''

///
/// Locales: ${translations.length}
/// Strings: $count$countPerLocale''';
  } else {
    statisticsStr = '';
  }

  final String timestampStr;
  if (config.renderTimestamp) {
    final date = '${now.year}-${twoDigits(now.month)}-${twoDigits(now.day)}';
    final time = '${twoDigits(now.hour)}:${twoDigits(now.minute)}';
    timestampStr = '''

///
/// Built on $date at $time UTC''';
  } else {
    timestampStr = '';
  }

  buffer.writeln('''
/// Generated file. Do not edit.
///
/// Source: ${config.inputDirectoryHint}
/// To regenerate, run: `dart run slang`$statisticsStr$timestampStr

// coverage:ignore-file
// ignore_for_file: type=lint, unused_import''');
}

void _generateImports(GenerateConfig config, StringBuffer buffer) {
  buffer.writeln();
  final imports = [
    ...config.imports,
    'package:intl/intl.dart',
    'package:slang/generated.dart',
    if (config.obfuscation.enabled) 'package:slang/secret.dart',
    if (config.translationOverrides) 'package:slang/overrides.dart',
    if (config.flutterIntegration) ...[
      'package:flutter/widgets.dart',
      'package:slang_flutter/slang_flutter.dart',
    ] else
      'package:slang/slang.dart'
  ]..sort((a, b) => a.compareTo(b));

  for (final i in imports) {
    buffer.writeln('import \'$i\';');
  }

  // export statements
  if (config.flutterIntegration) {
    buffer.writeln('export \'package:slang_flutter/slang_flutter.dart\';');
  } else {
    buffer.writeln('export \'package:slang/slang.dart\';');
  }
}

void _generateLocaleImports({
  required StringBuffer buffer,
  required GenerateConfig config,
  required List<I18nData> locales,
}) {
  buffer.writeln();
  for (final locale in locales.skip(1)) {
    final localeImportName = getImportName(
      locale: locale.locale,
    );
    final deferred = config.lazy ? ' deferred' : '';
    buffer.writeln(
        'import \'${BuildResultPaths.localePath(outputPath: config.outputFileName, locale: locale.locale)}\'$deferred as $localeImportName;');
  }
  buffer.writeln(
      'part \'${BuildResultPaths.localePath(outputPath: config.outputFileName, locale: locales.first.locale)}\';');
}

void _generateBuildConfig({
  required StringBuffer buffer,
  required BuildModelConfig config,
}) {
  buffer.writeln();
  buffer.writeln('/// Generated by the "Translation Overrides" feature.');
  buffer.writeln(
      '/// This config is needed to recreate the translation model exactly');
  buffer.writeln('/// the same way as this file was created.');
  buffer.writeln('final _buildConfig = BuildModelConfig(');
  buffer.writeln(
      '\tfallbackStrategy: FallbackStrategy.${config.fallbackStrategy.name},');
  buffer.writeln(
      '\tkeyCase: ${config.keyCase != null ? 'CaseStyle.${config.keyCase!.name}' : 'null'},');
  buffer.writeln(
      '\tkeyMapCase: ${config.keyMapCase != null ? 'CaseStyle.${config.keyMapCase!.name}' : 'null'},');
  buffer.writeln(
      '\tparamCase: ${config.paramCase != null ? 'CaseStyle.${config.paramCase!.name}' : 'null'},');
  buffer.writeln(
      '\tsanitization: SanitizationConfig(enabled: ${config.sanitization.enabled}, prefix: \'${config.sanitization.prefix}\', caseStyle: ${config.sanitization.caseStyle}),');
  buffer.writeln(
      '\tstringInterpolation: StringInterpolation.${config.stringInterpolation.name},');
  buffer.writeln('\tmaps: [${config.maps.map((m) => "'$m'").join(', ')}],');
  buffer.writeln('\tpluralAuto: PluralAuto.${config.pluralAuto.name},');
  buffer.writeln('\tpluralParameter: \'${config.pluralParameter}\',');
  buffer.writeln(
      '\tpluralCardinal: [${config.pluralCardinal.map((e) => '\'$e\'').join(', ')}],');
  buffer.writeln(
      '\tpluralOrdinal: [${config.pluralOrdinal.map((e) => '\'$e\'').join(', ')}],');
  buffer.write('\tcontexts: [');
  for (final context in config.contexts) {
    buffer.write(
        'ContextType(enumName: \'${context.enumName}\', defaultParameter: \'${context.defaultParameter}\', generateEnum: ${context.generateEnum}),');
  }
  buffer.writeln('],');
  buffer.writeln('\tinterfaces: [], // currently not supported');
  buffer.writeln('\tgenerateEnum: ${config.generateEnum},');
  buffer.writeln(');');
}

void _generateEnum({
  required StringBuffer buffer,
  required GenerateConfig config,
  required List<I18nData> allLocales,
}) {
  final String enumName = config.enumName;
  final String baseLocaleEnumConstant =
      '$enumName.${config.baseLocale.enumConstant}';

  buffer.writeln();
  buffer.writeln('/// Supported locales.');
  buffer.writeln('///');
  buffer.writeln('/// Usage:');
  buffer.writeln(
      '/// - LocaleSettings.setLocale($baseLocaleEnumConstant) // set locale');
  buffer.writeln(
      '/// - Locale locale = $baseLocaleEnumConstant.flutterLocale // get flutter locale from enum');
  buffer.writeln(
      '/// - if (LocaleSettings.currentLocale == $baseLocaleEnumConstant) // locale check');

  buffer.writeln(
      'enum $enumName with BaseAppLocale<$enumName, ${config.className}> {');
  for (int i = 0; i < allLocales.length; i++) {
    final I18nLocale locale = allLocales[i].locale;

    buffer
        .write('\t${locale.enumConstant}(languageCode: \'${locale.language}\'');
    if (locale.script != null) {
      buffer.write(', scriptCode: \'${locale.script}\'');
    }
    if (locale.country != null) {
      buffer.write(', countryCode: \'${locale.country}\'');
    }
    buffer.write(')');

    if (i != allLocales.length - 1) {
      buffer.writeln(',');
    } else {
      buffer.writeln(';');
    }
  }

  buffer.writeln();
  buffer.writeln('\tconst $enumName({');
  buffer.writeln('\t\trequired this.languageCode,');
  buffer.writeln(
    '\t\tthis.scriptCode, // ignore: unused_element, unused_element_parameter',
  );
  buffer.writeln(
    '\t\tthis.countryCode, // ignore: unused_element, unused_element_parameter',
  );
  buffer.writeln('\t});');

  buffer.writeln();
  buffer.writeln('\t@override final String languageCode;');
  buffer.writeln('\t@override final String? scriptCode;');
  buffer.writeln('\t@override final String? countryCode;');

  void generateBuildMethod(StringBuffer buffer,
      {required bool sync, required bool proxySync}) {
    buffer.writeln();
    buffer.writeln('\t@override');
    buffer.writeln(
        '\t${sync ? config.className : 'Future<${config.className}>'} build${sync ? 'Sync' : ''}({');
    buffer.writeln('\t\tMap<String, Node>? overrides,');
    buffer.writeln('\t\tPluralResolver? cardinalResolver,');
    buffer.writeln('\t\tPluralResolver? ordinalResolver,');
    buffer.writeln('\t}) ${sync ? '' : 'async '}{');

    if (proxySync) {
      buffer.writeln('\t\treturn buildSync(');
      buffer.writeln('\t\t\toverrides: overrides,');
      buffer.writeln('\t\t\tcardinalResolver: cardinalResolver,');
      buffer.writeln('\t\t\tordinalResolver: ordinalResolver,');
      buffer.writeln('\t\t);');
    } else {
      buffer.writeln('\t\tswitch (this) {');
      for (final locale in allLocales) {
        final localeImportName = getImportName(
          locale: locale.locale,
        );
        final className = getClassNameRoot(
          className: config.className,
          locale: locale.locale,
        );

        buffer.writeln('\t\t\tcase $enumName.${locale.locale.enumConstant}:');
        if (!locale.base && !sync) {
          buffer.writeln('\t\t\t\tawait $localeImportName.loadLibrary();');
        }
        buffer.writeln(
            '\t\t\t\treturn ${locale.base ? '' : '$localeImportName.'}$className(');
        buffer.writeln('\t\t\t\t\toverrides: overrides,');
        buffer.writeln('\t\t\t\t\tcardinalResolver: cardinalResolver,');
        buffer.writeln('\t\t\t\t\tordinalResolver: ordinalResolver,');
        buffer.writeln('\t\t\t\t);');
      }
      buffer.writeln('\t\t}');
    }

    buffer.writeln('\t}');
  }

  generateBuildMethod(buffer, sync: false, proxySync: !config.lazy);
  generateBuildMethod(buffer, sync: true, proxySync: false);

  if (config.localeHandling) {
    buffer.writeln();
    buffer.writeln('\t/// Gets current instance managed by [LocaleSettings].');
    buffer.writeln(
        '\t${config.className} get translations => LocaleSettings.instance.getTranslations(this);');
  }

  buffer.writeln('}');
}

void _generateTranslationGetter({
  required StringBuffer buffer,
  required GenerateConfig config,
}) {
  final String translationsClass = config.className;
  final String translateVar = config.translateVariable;
  final String enumName = config.enumName;

  // t getter
  buffer.writeln();
  buffer.writeln('/// Method A: Simple');
  buffer.writeln('///');
  buffer.writeln('/// No rebuild after locale change.');
  buffer.writeln(
      '/// Translation happens during initialization of the widget (call of $translateVar).');
  buffer.writeln('/// Configurable via \'translate_var\'.');
  buffer.writeln('///');
  buffer.writeln('/// Usage:');
  buffer.writeln('/// String a = $translateVar.someKey.anotherKey;');
  if (config.renderFlatMap) {
    buffer.writeln(
        '/// String b = $translateVar[\'someKey.anotherKey\']; // Only for edge cases!');
  }
  buffer.writeln(
      '${config.className} get $translateVar => LocaleSettings.instance.currentTranslations;');

  // t getter (advanced)
  if (config.flutterIntegration) {
    buffer.writeln();
    buffer.writeln('/// Method B: Advanced');
    buffer.writeln('///');
    buffer.writeln(
        '/// All widgets using this method will trigger a rebuild when locale changes.');
    buffer.writeln(
        '/// Use this if you have e.g. a settings page where the user can select the locale during runtime.');
    buffer.writeln('///');
    buffer.writeln('/// Step 1:');
    buffer.writeln('/// wrap your App with');
    buffer.writeln('/// TranslationProvider(');
    buffer.writeln('/// \tchild: MyApp()');
    buffer.writeln('/// );');
    buffer.writeln('///');
    buffer.writeln('/// Step 2:');
    buffer.writeln(
        '/// final $translateVar = $translationsClass.of(context); // Get $translateVar variable.');
    buffer.writeln(
        '/// String a = $translateVar.someKey.anotherKey; // Use $translateVar variable.');
    if (config.renderFlatMap) {
      buffer.writeln(
          '/// String b = $translateVar[\'someKey.anotherKey\']; // Only for edge cases!');
    }
    buffer.writeln(
        'class TranslationProvider extends BaseTranslationProvider<$enumName, ${config.className}> {');
    buffer.writeln(
        '\tTranslationProvider({required super.child}) : super(settings: LocaleSettings.instance);');
    buffer.writeln();
    buffer.writeln(
        '\tstatic InheritedLocaleData<$enumName, ${config.className}> of(BuildContext context) => InheritedLocaleData.of<$enumName, ${config.className}>(context);');
    buffer.writeln('}');

    // BuildContext extension for provider
    buffer.writeln();
    buffer
        .writeln('/// Method B shorthand via [BuildContext] extension method.');
    buffer.writeln('/// Configurable via \'translate_var\'.');
    buffer.writeln('///');
    buffer.writeln('/// Usage (e.g. in a widget\'s build method):');
    buffer.writeln('/// context.$translateVar.someKey.anotherKey');
    buffer.writeln(
        'extension BuildContextTranslationsExtension on BuildContext {');
    buffer.writeln(
        '\t${config.className} get $translateVar => TranslationProvider.of(this).translations;');
    buffer.writeln('}');
  }
}

void _generateLocaleSettings({
  required StringBuffer buffer,
  required GenerateConfig config,
  required List<I18nData> allLocales,
  required String pluralResolverType,
  required String pluralResolverCardinal,
  required String pluralResolverOrdinal,
}) {
  const String settingsClass = 'LocaleSettings';
  final String enumName = config.enumName;
  final String baseClass = config.flutterIntegration
      ? 'BaseFlutterLocaleSettings'
      : 'BaseLocaleSettings';

  buffer.writeln();
  buffer
      .writeln('/// Manages all translation instances and the current locale');
  buffer.writeln(
      'class $settingsClass extends $baseClass<$enumName, ${config.className}> {');
  buffer.writeln('\t$settingsClass._() : super(');
  buffer.writeln('\t\tutils: AppLocaleUtils.instance,');
  buffer.writeln('\t\tlazy: ${config.lazy},');
  buffer.writeln('\t);');
  buffer.writeln();
  buffer.writeln('\tstatic final instance = $settingsClass._();');

  buffer.writeln();
  buffer
      .writeln('\t// static aliases (checkout base methods for documentation)');
  buffer.writeln(
      '\tstatic $enumName get currentLocale => instance.currentLocale;');
  buffer.writeln(
      '\tstatic Stream<$enumName> getLocaleStream() => instance.getLocaleStream();');
  buffer.writeln(
      '\tstatic Future<$enumName> setLocale($enumName locale, {bool? listenToDeviceLocale = false}) => instance.setLocale(locale, listenToDeviceLocale: listenToDeviceLocale);');
  buffer.writeln(
      '\tstatic Future<$enumName> setLocaleRaw(String rawLocale, {bool? listenToDeviceLocale = false}) => instance.setLocaleRaw(rawLocale, listenToDeviceLocale: listenToDeviceLocale);');

  if (config.flutterIntegration) {
    buffer.writeln(
        '\tstatic Future<$enumName> useDeviceLocale() => instance.useDeviceLocale();');
  }

  buffer.writeln(
      '\tstatic Future<void> setPluralResolver({String? language, $enumName? locale, PluralResolver? cardinalResolver, PluralResolver? ordinalResolver}) => instance.setPluralResolver(');
  buffer.writeln('\t\tlanguage: language,');
  buffer.writeln('\t\tlocale: locale,');
  buffer.writeln('\t\tcardinalResolver: cardinalResolver,');
  buffer.writeln('\t\tordinalResolver: ordinalResolver,');
  buffer.writeln('\t);');
  if (config.translationOverrides) {
    buffer.writeln(
        '\tstatic Future<void> overrideTranslations({required $enumName locale, required FileType fileType, required String content}) => instance.overrideTranslations(locale: locale, fileType: fileType, content: content);');
    buffer.writeln(
        '\tstatic Future<void> overrideTranslationsFromMap({required $enumName locale, required bool isFlatMap, required Map map}) => instance.overrideTranslationsFromMap(locale: locale, isFlatMap: isFlatMap, map: map);');
  }

  // sync versions
  buffer.writeln();
  buffer.writeln('\t// synchronous versions');
  buffer.writeln(
      '\tstatic $enumName setLocaleSync($enumName locale, {bool? listenToDeviceLocale = false}) => instance.setLocaleSync(locale, listenToDeviceLocale: listenToDeviceLocale);');
  buffer.writeln(
      '\tstatic $enumName setLocaleRawSync(String rawLocale, {bool? listenToDeviceLocale = false}) => instance.setLocaleRawSync(rawLocale, listenToDeviceLocale: listenToDeviceLocale);');
  if (config.flutterIntegration) {
    buffer.writeln(
        '\tstatic $enumName useDeviceLocaleSync() => instance.useDeviceLocaleSync();');
  }

  buffer.writeln(
      '\tstatic void setPluralResolverSync({String? language, $enumName? locale, PluralResolver? cardinalResolver, PluralResolver? ordinalResolver}) => instance.setPluralResolverSync(');
  buffer.writeln('\t\tlanguage: language,');
  buffer.writeln('\t\tlocale: locale,');
  buffer.writeln('\t\tcardinalResolver: cardinalResolver,');
  buffer.writeln('\t\tordinalResolver: ordinalResolver,');
  buffer.writeln('\t);');

  if (config.translationOverrides) {
    buffer.writeln(
        '\tstatic void overrideTranslationsSync({required $enumName locale, required FileType fileType, required String content}) => instance.overrideTranslationsSync(locale: locale, fileType: fileType, content: content);');
    buffer.writeln(
        '\tstatic void overrideTranslationsFromMapSync({required $enumName locale, required bool isFlatMap, required Map map}) => instance.overrideTranslationsFromMapSync(locale: locale, isFlatMap: isFlatMap, map: map);');
  }

  buffer.writeln('}');
}

void _generateUtil({
  required StringBuffer buffer,
  required GenerateConfig config,
}) {
  const String utilClass = 'AppLocaleUtils';
  final String enumName = config.enumName;

  buffer.writeln();
  buffer.writeln('/// Provides utility functions without any side effects.');
  buffer.writeln(
      'class $utilClass extends BaseAppLocaleUtils<$enumName, ${config.className}> {');
  buffer.writeln('\t$utilClass._() : super(');
  buffer
      .writeln('\t\tbaseLocale: $enumName.${config.baseLocale.enumConstant},');
  buffer.writeln('\t\tlocales: $enumName.values,');
  if (config.translationOverrides) {
    buffer.writeln('\t\tbuildConfig: _buildConfig,');
  }
  buffer.writeln('\t);');
  buffer.writeln();
  buffer.writeln('\tstatic final instance = $utilClass._();');

  buffer.writeln();
  buffer
      .writeln('\t// static aliases (checkout base methods for documentation)');
  buffer.writeln(
      '\tstatic $enumName parse(String rawLocale) => instance.parse(rawLocale);');
  buffer.writeln(
      '\tstatic $enumName parseLocaleParts({required String languageCode, String? scriptCode, String? countryCode}) => instance.parseLocaleParts(languageCode: languageCode, scriptCode: scriptCode, countryCode: countryCode);');
  if (config.flutterIntegration) {
    buffer.writeln(
        '\tstatic $enumName findDeviceLocale() => instance.findDeviceLocale();');
    buffer.writeln(
        '\tstatic List<Locale> get supportedLocales => instance.supportedLocales;');
  }
  buffer.writeln(
      '\tstatic List<String> get supportedLocalesRaw => instance.supportedLocalesRaw;');
  if (config.translationOverrides) {
    buffer.writeln(
        '\tstatic Future<${config.className}> buildWithOverrides({required $enumName locale, required FileType fileType, required String content, PluralResolver? cardinalResolver, PluralResolver? ordinalResolver}) => instance.buildWithOverrides(locale: locale, fileType: fileType, content: content, cardinalResolver: cardinalResolver, ordinalResolver: ordinalResolver);');
    buffer.writeln(
        '\tstatic Future<${config.className}> buildWithOverridesFromMap({required $enumName locale, required bool isFlatMap, required Map map, PluralResolver? cardinalResolver, PluralResolver? ordinalResolver}) => instance.buildWithOverridesFromMap(locale: locale, isFlatMap: isFlatMap, map: map, cardinalResolver: cardinalResolver, ordinalResolver: ordinalResolver);');
    buffer.writeln(
        '\tstatic ${config.className} buildWithOverridesSync({required $enumName locale, required FileType fileType, required String content, PluralResolver? cardinalResolver, PluralResolver? ordinalResolver}) => instance.buildWithOverridesSync(locale: locale, fileType: fileType, content: content, cardinalResolver: cardinalResolver, ordinalResolver: ordinalResolver);');
    buffer.writeln(
        '\tstatic ${config.className} buildWithOverridesFromMapSync({required $enumName locale, required bool isFlatMap, required Map map, PluralResolver? cardinalResolver, PluralResolver? ordinalResolver}) => instance.buildWithOverridesFromMapSync(locale: locale, isFlatMap: isFlatMap, map: map, cardinalResolver: cardinalResolver, ordinalResolver: ordinalResolver);');
  }

  buffer.writeln('}');
}

void _generateContextEnums({
  required StringBuffer buffer,
  required GenerateConfig config,
}) {
  final contexts = config.contexts.where((c) => c.generateEnum);

  if (contexts.isNotEmpty) {
    buffer.writeln();
    buffer.writeln('// context enums');
  }

  for (final contextType in contexts) {
    buffer.writeln();
    buffer.writeln('enum ${contextType.enumName} {');
    for (final enumValue in contextType.enumValues) {
      buffer.writeln('\t$enumValue,');
    }
    buffer.writeln('}');
  }
}

void _generateInterfaces({
  required StringBuffer buffer,
  required GenerateConfig config,
}) {
  if (config.interface.isNotEmpty) {
    buffer.writeln();
    buffer.writeln('// interfaces generated as mixins');
  }

  const fieldsVar = r'$fields';

  for (final interface in config.interface) {
    buffer.writeln();
    buffer.writeln('mixin ${interface.name} {');
    for (final attribute in interface.attributes) {
      // If this attribute is optional, then these 2 modifications will be added
      final nullable = attribute.optional ? '?' : '';
      final defaultNull = attribute.optional ? ' => null' : '';

      if (attribute.parameters.isEmpty) {
        buffer.writeln(
            '\t${attribute.returnType}$nullable get ${attribute.attributeName}$defaultNull;');
      } else {
        buffer.write(
            '\t${attribute.returnType}$nullable ${attribute.attributeName}({');
        bool first = true;
        for (final param in attribute.parameters) {
          if (!first) buffer.write(', ');
          buffer.write('required ${param.type} ${param.parameterName}');
          first = false;
        }
        buffer.writeln('})$defaultNull;');
      }
    }

    // equals override
    buffer.writeln();
    buffer.writeln('\t@override');
    buffer.writeln('\tbool operator ==(Object other) {');
    buffer.writeln('\t\tif (identical(this, other)) return true;');
    buffer.writeln('\t\tif (other is! ${interface.name}) return false;');

    buffer.writeln();
    buffer.writeln('\t\tfinal fields = $fieldsVar;');
    buffer.writeln('\t\tfinal otherFields = other.$fieldsVar;');
    buffer.writeln('\t\tfor (int i = 0; i < fields.length; i++) {');
    buffer.writeln('\t\t\tif (fields[i] != otherFields[i]) return false;');
    buffer.writeln('\t\t}');

    buffer.writeln();
    buffer.writeln('\t\treturn true;');
    buffer.writeln('\t}');

    // hashCode override
    buffer.writeln();
    buffer.writeln('\t@override');
    buffer.writeln('\tint get hashCode {');
    buffer.writeln('\t\tfinal fields = $fieldsVar;');
    buffer.writeln('\t\tint result = fields.first.hashCode;');
    buffer.writeln('\t\tfor (final element in fields.skip(1)) {');
    buffer.writeln('\t\t\tresult *= element.hashCode;');
    buffer.writeln('\t\t}');

    buffer.writeln();
    buffer.writeln('\t\treturn result;');
    buffer.writeln('\t}');

    // fields
    buffer.writeln();
    buffer.writeln('\tList<Object?> get $fieldsVar => [');
    for (final attribute in interface.attributes) {
      buffer.writeln('\t\t${attribute.attributeName},');
    }
    buffer.writeln('\t];');

    buffer.writeln('}');
  }
}

int _countTranslations(Node node) {
  if (node is TextNode) {
    return 1;
  } else if (node is ListNode) {
    int sum = 0;
    for (Node entry in node.entries) {
      sum += _countTranslations(entry);
    }
    return sum;
  } else if (node is ObjectNode) {
    int sum = 0;
    for (Node entry in node.entries.values) {
      sum += _countTranslations(entry);
    }
    return sum;
  } else if (node is PluralNode) {
    return node.quantities.entries.length;
  } else if (node is ContextNode) {
    return node.entries.entries.length;
  } else {
    return 0;
  }
}
