/* Copyright 2024 Tusky Contributors
 *
 * This file is a part of Tusky.
 *
 * This program is free software; you can redistribute it and/or modify it under the terms of the
 * GNU General Public License as published by the Free Software Foundation; either version 3 of the
 * License, or (at your option) any later version.
 *
 * Tusky is distributed in the hope that it will be useful, but WITHOUT ANY WARRANTY; without even
 * the implied warranty of MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE. See the GNU General
 * Public License for more details.
 *
 * You should have received a copy of the GNU General Public License along with Tusky; if not,
 * see <http://www.gnu.org/licenses>. */

package com.keylesspalace.tusky.components.notifications.requests.details

import android.content.Context
import android.content.Intent
import android.content.SharedPreferences
import android.os.Bundle
import android.util.Log
import android.view.View
import androidx.fragment.app.activityViewModels
import androidx.lifecycle.lifecycleScope
import androidx.paging.LoadState
import androidx.recyclerview.widget.DividerItemDecoration
import androidx.recyclerview.widget.LinearLayoutManager
import androidx.recyclerview.widget.SimpleItemAnimator
import at.connyduck.calladapter.networkresult.onFailure
import at.connyduck.sparkbutton.SparkButton
import com.google.android.material.snackbar.BaseTransientBottomBar.LENGTH_LONG
import com.google.android.material.snackbar.Snackbar
import com.keylesspalace.tusky.R
import com.keylesspalace.tusky.components.notifications.NotificationActionListener
import com.keylesspalace.tusky.components.notifications.NotificationsPagingAdapter
import com.keylesspalace.tusky.databinding.FragmentNotificationRequestDetailsBinding
import com.keylesspalace.tusky.entity.Status
import com.keylesspalace.tusky.fragment.SFragment
import com.keylesspalace.tusky.interfaces.AccountActionListener
import com.keylesspalace.tusky.interfaces.LoadMoreActionListener
import com.keylesspalace.tusky.interfaces.StatusActionListener
import com.keylesspalace.tusky.settings.PrefKeys
import com.keylesspalace.tusky.util.CardViewMode
import com.keylesspalace.tusky.util.StatusDisplayOptions
import com.keylesspalace.tusky.util.getErrorString
import com.keylesspalace.tusky.util.hide
import com.keylesspalace.tusky.util.openLink
import com.keylesspalace.tusky.util.show
import com.keylesspalace.tusky.util.viewBinding
import com.keylesspalace.tusky.util.visible
import com.keylesspalace.tusky.view.ConfirmationBottomSheet.Companion.confirmFavourite
import com.keylesspalace.tusky.view.ConfirmationBottomSheet.Companion.confirmReblog
import com.keylesspalace.tusky.viewdata.AttachmentViewData
import com.keylesspalace.tusky.viewdata.NotificationViewData
import dagger.hilt.android.AndroidEntryPoint
import javax.inject.Inject
import kotlinx.coroutines.flow.collectLatest
import kotlinx.coroutines.launch

@AndroidEntryPoint
class NotificationRequestDetailsFragment :
    SFragment<NotificationViewData.Concrete>(R.layout.fragment_notification_request_details),
    StatusActionListener<NotificationViewData.Concrete>,
    LoadMoreActionListener<NotificationViewData.LoadMore>,
    NotificationActionListener,
    AccountActionListener {

    @Inject
    lateinit var preferences: SharedPreferences

    private val viewModel: NotificationRequestDetailsViewModel by activityViewModels()

    private val binding by viewBinding(FragmentNotificationRequestDetailsBinding::bind)

    private var adapter: NotificationsPagingAdapter? = null

    private var buttonToAnimate: SparkButton? = null

    override fun onViewCreated(view: View, savedInstanceState: Bundle?) {
        super.onViewCreated(view, savedInstanceState)

        setupAdapter().let { adapter ->
            this.adapter = adapter
            setupRecyclerView(adapter)

            lifecycleScope.launch {
                viewModel.pager.collectLatest { pagingData ->
                    adapter.submitData(pagingData)
                }
            }
        }

        lifecycleScope.launch {
            viewModel.error.collect { error ->
                Snackbar.make(
                    binding.root,
                    error.getErrorString(requireContext()),
                    LENGTH_LONG
                ).show()
            }
        }
    }

    private fun setupRecyclerView(adapter: NotificationsPagingAdapter) {
        binding.recyclerView.adapter = adapter
        binding.recyclerView.setHasFixedSize(true)
        binding.recyclerView.layoutManager = LinearLayoutManager(requireContext())
        binding.recyclerView.addItemDecoration(
            DividerItemDecoration(requireContext(), DividerItemDecoration.VERTICAL)
        )
        (binding.recyclerView.itemAnimator as SimpleItemAnimator).supportsChangeAnimations = false
    }

    private fun setupAdapter(): NotificationsPagingAdapter {
        val activeAccount = accountManager.activeAccount!!
        val statusDisplayOptions = StatusDisplayOptions(
            animateAvatars = preferences.getBoolean(PrefKeys.ANIMATE_GIF_AVATARS, false),
            mediaPreviewEnabled = activeAccount.mediaPreviewEnabled,
            useAbsoluteTime = preferences.getBoolean(PrefKeys.ABSOLUTE_TIME_VIEW, false),
            showBotOverlay = preferences.getBoolean(PrefKeys.SHOW_BOT_OVERLAY, true),
            useBlurhash = preferences.getBoolean(PrefKeys.USE_BLURHASH, true),
            cardViewMode = if (preferences.getBoolean(PrefKeys.SHOW_CARDS_IN_TIMELINES, false)) {
                CardViewMode.INDENTED
            } else {
                CardViewMode.NONE
            },
            hideStats = preferences.getBoolean(PrefKeys.WELLBEING_HIDE_STATS_POSTS, false),
            animateEmojis = preferences.getBoolean(PrefKeys.ANIMATE_CUSTOM_EMOJIS, false),
            showStatsInline = preferences.getBoolean(PrefKeys.SHOW_STATS_INLINE, false),
            showSensitiveMedia = activeAccount.alwaysShowSensitiveMedia,
            openSpoiler = activeAccount.alwaysOpenSpoiler
        )

        return NotificationsPagingAdapter(
            accountId = activeAccount.accountId,
            statusDisplayOptions = statusDisplayOptions,
            statusListener = this,
            loadMoreListener = this,
            notificationActionListener = this,
            accountActionListener = this,
            instanceName = activeAccount.domain
        ).apply {
            addLoadStateListener { loadState ->
                binding.progressBar.visible(
                    loadState.refresh == LoadState.Loading && itemCount == 0
                )

                if (loadState.refresh is LoadState.Error) {
                    binding.recyclerView.hide()
                    binding.statusView.show()
                    val errorState = loadState.refresh as LoadState.Error
                    binding.statusView.setup(errorState.error) { retry() }
                    Log.w(TAG, "error loading notifications for user ${viewModel.accountId}", errorState.error)
                } else {
                    binding.recyclerView.show()
                    binding.statusView.hide()
                }
            }
        }
    }

    override fun onLoadMore(loadMore: NotificationViewData.LoadMore) {
        // not relevant here
    }

    override fun onReply(viewData: NotificationViewData.Concrete) {
        val status = viewData.asStatusOrNull() ?: return
        super.reply(status.status)
    }

    override fun removeItem(viewData: NotificationViewData.Concrete) {
        viewModel.remove(viewData)
    }

    override fun onReblog(viewData: NotificationViewData.Concrete, reblog: Boolean, visibility: Status.Visibility?, button: SparkButton?) {
        val status = viewData.asStatusOrNull() ?: return
        buttonToAnimate = button

        if (reblog && visibility == null) {
            confirmReblog(preferences) { visibility ->
                viewModel.reblog(true, status, visibility)
                buttonToAnimate?.playAnimation()
                buttonToAnimate?.isChecked = true
            }
        } else {
            viewModel.reblog(reblog, status, visibility ?: Status.Visibility.PUBLIC)
            if (reblog) {
                buttonToAnimate?.playAnimation()
            }
            buttonToAnimate?.isChecked = reblog
        }
    }

    override val onMoreTranslate: ((Boolean, NotificationViewData.Concrete) -> Unit)?
        get() = { translate: Boolean, viewData: NotificationViewData.Concrete ->
            if (translate) {
                onTranslate(viewData)
            } else {
                onUntranslate(viewData)
            }
        }

    override fun onFavourite(viewData: NotificationViewData.Concrete, favourite: Boolean, button: SparkButton?) {
        val status = viewData.asStatusOrNull() ?: return
        buttonToAnimate = button

        if (favourite) {
            confirmFavourite(preferences) {
                viewModel.favorite(true, status)
                buttonToAnimate?.playAnimation()
                buttonToAnimate?.isChecked = true
            }
        } else {
            viewModel.favorite(false, status)
            buttonToAnimate?.isChecked = false
        }
    }

    override fun onBookmark(viewData: NotificationViewData.Concrete, bookmark: Boolean) {
        val status = viewData.asStatusOrNull() ?: return
        viewModel.bookmark(bookmark, status)
    }

    override fun onMore(viewData: NotificationViewData.Concrete, view: View) {
        super.more(viewData, view)
    }

    override fun onViewMedia(viewData: NotificationViewData.Concrete, attachmentIndex: Int, view: View?) {
        val status = viewData.asStatusOrNull() ?: return
        super.viewMedia(attachmentIndex, AttachmentViewData.list(status), view)
    }

    override fun onViewThread(viewData: NotificationViewData.Concrete) {
        val status = viewData.asStatusOrNull()?.status ?: return
        super.viewThread(status.id, status.url)
    }

    override fun onOpenReblog(viewData: NotificationViewData.Concrete) {
        val status = viewData.asStatusOrNull() ?: return
        super.openReblog(status.status)
    }

    override fun onExpandedChange(viewData: NotificationViewData.Concrete, expanded: Boolean) {
        val status = viewData.asStatusOrNull() ?: return
        viewModel.changeExpanded(expanded, status)
    }

    override fun onContentHiddenChange(viewData: NotificationViewData.Concrete, isShowing: Boolean) {
        val status = viewData.asStatusOrNull() ?: return
        viewModel.changeContentShowing(isShowing, status)
    }

    override fun onContentCollapsedChange(viewData: NotificationViewData.Concrete, isCollapsed: Boolean) {
        val status = viewData.asStatusOrNull() ?: return
        viewModel.changeContentCollapsed(isCollapsed, status)
    }

    override fun onVoteInPoll(viewData: NotificationViewData.Concrete, choices: List<Int>) {
        val status = viewData.asStatusOrNull() ?: return
        viewModel.voteInPoll(choices, status)
    }

    override fun onShowPollResults(viewData: NotificationViewData.Concrete) {
        val status = viewData.asStatusOrNull() ?: return
        viewModel.showPollResults(status)
    }

    override fun changeFilter(filtered: Boolean, viewData: NotificationViewData.Concrete) {
        // not applicable here
    }

    private fun onTranslate(viewData: NotificationViewData.Concrete) {
        val status = viewData.asStatusOrNull() ?: return
        viewLifecycleOwner.lifecycleScope.launch {
            viewModel.translate(status)
                .onFailure {
                    Snackbar.make(
                        requireView(),
                        getString(R.string.ui_error_translate, it.message),
                        LENGTH_LONG
                    ).show()
                }
        }
    }

    override fun onUntranslate(viewData: NotificationViewData.Concrete) {
        val status = viewData.asStatusOrNull() ?: return
        viewModel.untranslate(status)
    }

    override fun onViewTag(tag: String) {
        super.viewTag(tag)
    }

    override fun onViewAccount(id: String) {
        super.viewAccount(id)
    }

    override fun onViewReport(reportId: String) {
        requireContext().openLink(
            "https://${accountManager.activeAccount!!.domain}/admin/reports/$reportId"
        )
    }

    override fun onMute(mute: Boolean, id: String, position: Int, notifications: Boolean) {
        // not needed, muting via the more menu on statuses is handled in SFragment
    }

    override fun onBlock(block: Boolean, id: String, position: Int) {
        // not needed, blocking via the more menu on statuses is handled in SFragment
    }

    override fun onRespondToFollowRequest(accept: Boolean, accountIdRequestingFollow: String, position: Int) {
        val notification = adapter?.peek(position) ?: return
        viewModel.respondToFollowRequest(accept, accountId = accountIdRequestingFollow, notification = notification)
    }

    override fun onDestroyView() {
        adapter = null
        buttonToAnimate = null
        super.onDestroyView()
    }

    companion object {
        private const val TAG = "NotificationRequestsDetailsFragment"
        private const val EXTRA_ACCOUNT_ID = "accountId"
        fun newIntent(accountId: String, context: Context) = Intent(context, NotificationRequestDetailsActivity::class.java).apply {
            putExtra(EXTRA_ACCOUNT_ID, accountId)
        }
    }
}
