import 'package:flutter/material.dart';

import '../../flex_color_picker.dart';

/// Draws recently used colors.
///
/// Not library exposed, private to the library.
class RecentColors extends StatelessWidget {
  /// Default const constructor.
  const RecentColors({
    super.key,
    required this.spacing,
    required this.runSpacing,
    required this.recentColors,
    required this.selectedColor,
    required this.onSelectColor,
    required this.includeIndex850,
    required this.width,
    required this.height,
    this.borderRadius,
    required this.hasBorder,
    this.borderColor,
    required this.elevation,
    required this.selectedColorIcon,
    this.selectedRequestsFocus = false,
  });

  /// The spacing between the color pick items.
  final double spacing;

  /// The run spacing between the color pick items when wrapped on several rows.
  final double runSpacing;

  /// List of recently selected colors.
  final List<Color> recentColors;

  /// The selected color.
  final Color selectedColor;

  /// Void callback called when a color is selected.
  final ValueChanged<Color> onSelectColor;

  /// Set to trued if index 850 is to be included in the main shades.
  final bool includeIndex850;

  /// Width of the color pick item.
  final double width;

  /// Height of the color pick item.
  final double height;

  /// Border radius of the color pick item.
  final double? borderRadius;

  /// Set to true if pick item should have a border.
  final bool hasBorder;

  /// Color of the border when one is used.
  final Color? borderColor;

  /// Material elevation of the color pick item.
  final double elevation;

  /// Icon used to mark selected color.
  final IconData selectedColorIcon;

  /// Set to true, if a an indicator should request focus if it is selected.
  ///
  /// The indicator will always request focus when it clicked and selected,
  /// setting this value to true is to make it request focus when it is drawn.
  /// This is used to set focus to the selected color, but only when
  /// the piker is redrawn.
  ///
  /// Defaults to false.
  final bool selectedRequestsFocus;

  @override
  Widget build(BuildContext context) {
    final double effectiveBorderRadius = borderRadius ?? width / 4.0;
    return Wrap(
      spacing: spacing,
      runSpacing: runSpacing,
      children: <Widget>[
        for (final Color color in recentColors)
          ColorIndicator(
            isSelected: selectedColor == color ||
                selectedColor.value32bit == color.value32bit,
            color: color,
            width: width,
            height: height,
            borderRadius: effectiveBorderRadius,
            hasBorder: hasBorder,
            borderColor: borderColor,
            elevation: elevation,
            selectedIcon: selectedColorIcon,
            onSelect: () {
              onSelectColor(color);
            },
            selectedRequestsFocus: selectedRequestsFocus,
          ),
      ],
    );
  }
}
