package com.jksalcedo.passvault.importer

import com.jksalcedo.passvault.data.BitwardenExport
import com.jksalcedo.passvault.data.ImportRecord
import com.jksalcedo.passvault.utils.Utility.toEpochMillis
import com.jksalcedo.passvault.utils.Utility.toPasswordEntry
import kotlinx.serialization.json.Json


/**
 * An importer for Bitwarden.
 * This class is responsible for parsing a JSON export from Bitwarden
 * and converting it into a list of [ImportRecord] objects.
 */
class BitwardenImporter(
    private val json: Json = Json { ignoreUnknownKeys = true }
) : VaultImporter {

    override suspend fun parse(raw: String): List<ImportRecord> {
        try {
            val export = json.decodeFromString<BitwardenExport>(raw)
            return export.items
                .filter { it.type == 1 && (it.name.isNotBlank() || it.login?.password?.isNotBlank() == true) }
                .map { item ->
                    ImportRecord(
                        title = item.name,
                        username = item.login?.username,
                        password = item.login?.password.orEmpty(),
                        email = null,
                        url = item.login?.uris?.first()?.uri,
                        category = null,
                        notes = item.notes,
                        createdAt = item.creationDate?.toEpochMillis(),
                        updatedAt = item.revisionDate?.toEpochMillis()
                    )
                }
        } catch (e: Exception) {
            e.printStackTrace()
            throw e
        }
    }

    override fun mapToPasswordEntries(records: List<ImportRecord>) =
        records.map { it.toPasswordEntry() }
}