package com.jksalcedo.passvault.ui.auth

import android.content.Context
import android.widget.Toast
import androidx.appcompat.app.AppCompatActivity
import androidx.biometric.BiometricManager
import androidx.biometric.BiometricPrompt
import androidx.core.content.ContextCompat
import androidx.fragment.app.Fragment
import com.jksalcedo.passvault.R

/**
 * A class for handling biometric authentication.
 */
class BiometricAuthenticator {

    companion object {
        fun canAuthenticate(context: Context): Boolean {
            val biometricManager = BiometricManager.from(context)
            val authenticators =
                BiometricManager.Authenticators.BIOMETRIC_STRONG or BiometricManager.Authenticators.BIOMETRIC_WEAK
            return biometricManager.canAuthenticate(authenticators) == BiometricManager.BIOMETRIC_SUCCESS
        }
    }

    /**
     * Shows a biometric prompt.
     * @param activity The activity to show the prompt in.
     * @param onSuccess A callback to be invoked when authentication is successful.
     * @param onFailure A callback to be invoked when authentication fails.
     */
    fun showBiometricPrompt(
        activity: AppCompatActivity,
        onSuccess: (BiometricPrompt.AuthenticationResult) -> Unit,
        onFailure: (errorCode: Int, errString: CharSequence) -> Unit
    ) {
        val executor = ContextCompat.getMainExecutor(activity)
        val authenticators =
            BiometricManager.Authenticators.BIOMETRIC_STRONG or BiometricManager.Authenticators.BIOMETRIC_WEAK

        val prompt =
            BiometricPrompt(activity, executor, object : BiometricPrompt.AuthenticationCallback() {
                override fun onAuthenticationSucceeded(result: BiometricPrompt.AuthenticationResult) {
                    super.onAuthenticationSucceeded(result)
                    onSuccess(result)
                }

                override fun onAuthenticationError(errorCode: Int, errString: CharSequence) {
                    super.onAuthenticationError(errorCode, errString)
                    if (errorCode != BiometricPrompt.ERROR_NEGATIVE_BUTTON && errorCode != BiometricPrompt.ERROR_CANCELED) {
                        onFailure(errorCode, errString)
                    }
                }

                override fun onAuthenticationFailed() {
                    super.onAuthenticationFailed()
                    Toast.makeText(
                        activity,
                        activity.getString(R.string.biometric_failed),
                        Toast.LENGTH_SHORT
                    ).show()
                }
            })

        val promptInfo = BiometricPrompt.PromptInfo.Builder()
            .setTitle(activity.getString(R.string.biometric_prompt_title))
            .setSubtitle(activity.getString(R.string.biometric_prompt_subtitle))
            .setNegativeButtonText(activity.getString(R.string.biometric_negative))
            .setAllowedAuthenticators(authenticators)
            .build()

        prompt.authenticate(promptInfo)
    }

    /**
     * Shows a biometric prompt.
     * @param fragment The fragment to show the prompt in.
     * @param onSuccess A callback to be invoked when authentication is successful.
     * @param onFailure A callback to be invoked when authentication fails.
     */
    fun showBiometricPrompt(
        fragment: Fragment,
        onSuccess: (BiometricPrompt.AuthenticationResult) -> Unit,
        onFailure: (errorCode: Int, errString: CharSequence) -> Unit
    ) {
        showBiometricPrompt(fragment.requireActivity() as AppCompatActivity, onSuccess, onFailure)
    }
}