package com.jherkenhoff.qalculate.ui.calculator

import androidx.compose.animation.AnimatedContent
import androidx.compose.foundation.layout.Arrangement
import androidx.compose.foundation.layout.Box
import androidx.compose.foundation.layout.Row
import androidx.compose.foundation.layout.Spacer
import androidx.compose.foundation.layout.fillMaxHeight
import androidx.compose.foundation.layout.height
import androidx.compose.foundation.layout.size
import androidx.compose.foundation.layout.width
import androidx.compose.foundation.lazy.LazyRow
import androidx.compose.foundation.lazy.items
import androidx.compose.material.icons.Icons
import androidx.compose.material.icons.filled.Close
import androidx.compose.material.icons.filled.Keyboard
import androidx.compose.material3.FilledIconToggleButton
import androidx.compose.material3.Icon
import androidx.compose.material3.IconButton
import androidx.compose.material3.MaterialTheme
import androidx.compose.material3.SuggestionChip
import androidx.compose.material3.Text
import androidx.compose.runtime.Composable
import androidx.compose.ui.Alignment
import androidx.compose.ui.Modifier
import androidx.compose.ui.draw.drawWithContent
import androidx.compose.ui.graphics.BlendMode
import androidx.compose.ui.graphics.Brush
import androidx.compose.ui.graphics.Color
import androidx.compose.ui.graphics.CompositingStrategy
import androidx.compose.ui.graphics.graphicsLayer
import androidx.compose.ui.unit.dp
import com.jherkenhoff.qalculate.domain.AutocompleteResult
import com.jherkenhoff.qalculate.model.Action
import com.jherkenhoff.qalculate.model.AutocompleteItem
import com.jherkenhoff.qalculate.model.ActionLabel
import com.jherkenhoff.qalculate.model.Keys

@Composable
fun AuxiliaryBar(
    autocompleteResult: AutocompleteResult,
    keyboardEnable: Boolean,
    auxiliaryActions: List<Action>,
    modifier: Modifier = Modifier,
    onAutocompleteClick: (AutocompleteItem) -> Unit = { },
    onKeyboardEnableChange: (Boolean) -> Unit = { },
    onAction: (Action) -> Unit = { },
    onAutocompleteDismiss: () -> Unit = { },
) {
    val fadeWidth = 40f

    Box(
        modifier.height(48.dp),
        contentAlignment = Alignment.Center
    ) {
        AnimatedContent(autocompleteResult.items.isNotEmpty()) {
            if (it) {
                Row {
                    LazyRow(
                        modifier = Modifier.fillMaxHeight().weight(1f)
                            .graphicsLayer(compositingStrategy = CompositingStrategy.Offscreen)
                            .drawWithContent {
                                drawContent()
                                drawRect(brush = Brush.horizontalGradient(0f to Color.White, 1f to Color.Transparent, startX = this.size.width-fadeWidth, endX = this.size.width), blendMode = BlendMode.DstIn)
                                drawRect(brush = Brush.horizontalGradient(0f to Color.Transparent, 1f to Color.White, startX = 0f, endX = fadeWidth), blendMode = BlendMode.DstIn)
                            },
                        horizontalArrangement = Arrangement.spacedBy(8.dp),
                    ) {
                        item{
                            Spacer(Modifier.width(fadeWidth.toDp()-8.dp))
                        }
                        items(autocompleteResult.items) { it ->
                            SuggestionChip(
                                label = { Text(it.title) },
                                onClick = { onAutocompleteClick(it) }
                            )
                        }
                        item{
                            Spacer(Modifier.width(fadeWidth.toDp()-8.dp))
                        }
                    }

                    IconButton(onAutocompleteDismiss) {
                        Icon(Icons.Default.Close, "Dismiss autocomplete suggestions")
                    }
                }
            } else {
                Row(
                    horizontalArrangement = Arrangement.spacedBy(4.dp),
                    verticalAlignment = Alignment.CenterVertically,
                ) {
                    FilledIconToggleButton(keyboardEnable, onKeyboardEnableChange) {
                        Icon(Icons.Filled.Keyboard, null)
                    }
                    for (action in auxiliaryActions) {
                        IconButton({ onAction(action) }, enabled = action.enabled) {
                            when (val label = action.label) {
                                is ActionLabel.Text -> Text(
                                    label.text,
                                    style = MaterialTheme.typography.labelLarge
                                )

                                is ActionLabel.Icon -> Icon(
                                    label.icon,
                                    label.description,
                                    modifier = Modifier.size(MaterialTheme.typography.labelLarge.lineHeight.toDp())
                                )

                                null -> null
                            }
                        }
                    }
                }
            }
        }
    }
}