package com.ivarna.finalbenchmark2.cpuBenchmark.algorithms

/**
 * XorShift128+ Random Number Generator
 * 
 * PERFORMANCE: 2-3x faster than java.util.Random and kotlin.random.Random
 * - Uses XorShift128+ algorithm (high-quality, fast PRNG)
 * - Period: 2^128 - 1 (extremely long before repeating)
 * - Thread-safe when each thread has its own instance
 * - Generates true random sequences with proper entropy seeding
 * 
 * ALGORITHM:
 * - Based on Sebastiano Vigna's xorshift128+ implementation
 * - Passes BigCrush statistical tests (high quality randomness)
 * - Optimized for CPU cache efficiency
 * 
 * USAGE:
 * ```kotlin
 * val rng = XorShift128Plus()  // Auto-seeds with true entropy
 * val randomDouble = rng.nextDouble()  // [0.0, 1.0)
 * ```
 * 
 * @see <a href="http://xorshift.di.unimi.it/">XorShift Reference</a>
 */
class XorShift128Plus {
    private var s0: Long
    private var s1: Long

    /**
     * Constructor with automatic true random seeding
     * Combines multiple entropy sources for maximum randomness
     */
    constructor() {
        // TRUE RANDOM SEEDING: Combine multiple entropy sources
        val entropy1 = System.nanoTime()
        val entropy2 = System.currentTimeMillis()
        val entropy3 = Thread.currentThread().threadId()
        val entropy4 = hashCode().toLong()
        val entropy5 = Runtime.getRuntime().freeMemory()
        
        // Mix entropy sources using splitmix64 algorithm for better distribution
        s0 = splitmix64(entropy1 xor entropy2 xor entropy3)
        s1 = splitmix64(entropy4 xor entropy5 xor (entropy1 shr 32))
        
        // Ensure non-zero state (required for XorShift)
        if (s0 == 0L && s1 == 0L) {
            s0 = 0x123456789ABCDEF0L
            s1 = -0x1234567899ABCDF0L
        }
    }

    /**
     * Constructor with explicit seed (for reproducible benchmarks)
     * @param seed Initial seed value
     */
    constructor(seed: Long) {
        // Use splitmix64 to generate two independent state values from single seed
        s0 = splitmix64(seed)
        s1 = splitmix64(s0)
        
        // Ensure non-zero state
        if (s0 == 0L && s1 == 0L) {
            s0 = 0x123456789ABCDEF0L
            s1 = -0x1234567899ABCDF0L
        }
    }

    /**
     * Constructor with explicit state (for advanced use cases)
     * @param state0 First state value
     * @param state1 Second state value
     */
    constructor(state0: Long, state1: Long) {
        s0 = state0
        s1 = state1
        
        // Ensure non-zero state
        if (s0 == 0L && s1 == 0L) {
            s0 = 0x123456789ABCDEF0L
            s1 = -0x1234567899ABCDF0L
        }
    }

    /**
     * Generate next random Long value
     * @return Random 64-bit signed integer
     */
    fun nextLong(): Long {
        var x = s0
        val y = s1
        s0 = y
        x = x xor (x shl 23) // a
        s1 = x xor y xor (x ushr 17) xor (y ushr 26) // b, c
        return s1 + y
    }

    /**
     * Generate next random Double in range [0.0, 1.0)
     * @return Random double-precision floating point number
     */
    fun nextDouble(): Double {
        // Use upper 53 bits for double precision (IEEE 754 double has 53-bit mantissa)
        val value = (nextLong() ushr 11) and 0x1FFFFFFFFFFFFFL
        return value.toDouble() / (1L shl 53).toDouble()
    }

    /**
     * Generate next random Int value
     * @return Random 32-bit signed integer
     */
    fun nextInt(): Int {
        return (nextLong() ushr 32).toInt()
    }

    /**
     * Generate next random Int in range [0, bound)
     * @param bound Upper bound (exclusive)
     * @return Random integer in [0, bound)
     */
    fun nextInt(bound: Int): Int {
        require(bound > 0) { "bound must be positive" }
        
        // Use rejection sampling for uniform distribution
        val mask = (1 shl (32 - Integer.numberOfLeadingZeros(bound - 1))) - 1
        var result: Int
        do {
            result = nextInt() and mask
        } while (result >= bound)
        return result
    }

    /**
     * SplitMix64 algorithm for seed mixing
     * Ensures good distribution of entropy across state bits
     * 
     * @param seed Input seed value
     * @return Mixed seed value with better bit distribution
     */
    private fun splitmix64(seed: Long): Long {
        var z = seed + -0x61C8864680B583EBL
        z = (z xor (z ushr 30)) * -0x4A7BE9E3A1C71C47L
        z = (z xor (z ushr 27)) * -0x6BF2FBB4ACC67C15L
        return z xor (z ushr 31)
    }

    companion object {
        /**
         * Create XorShift128+ instance with mixed entropy from multiple sources
         * Each call generates different random sequence
         * 
         * @param additionalEntropy Additional entropy to mix in (e.g., thread ID, iteration count)
         * @return New XorShift128+ instance with unique seed
         */
        fun withEntropy(vararg additionalEntropy: Long): XorShift128Plus {
            val baseEntropy = System.nanoTime() xor System.currentTimeMillis()
            val mixedEntropy = additionalEntropy.fold(baseEntropy) { acc, value -> acc xor value }
            return XorShift128Plus(mixedEntropy)
        }
    }
}
