# Table of Contents <!-- omit in toc --> 
- [**Breaking changes**](#Breaking-changes)  
  - [**state injection**](#state-injection)  
  - [**Builders (widget listeners)**](#Builders-widget-listeners)  
  - [**setState method**](#setState-method)
  - [**Input field and form validation**](#Input-field-and-form-validation)  
  - [**middleSnapState**](#middleSnapState)

- [**New Features**](#new-features)  
  - [**Auth token refreshing**](#auth-token-refreshing)  
  - [**TopStatelessWidget**](#topstatelesswidget)  
  - [**Working with tabs and pages (InjectedTabPageView)**](#working-with-tabs-and-pages-injectedtabpageview)  
  - [**OnFormFieldBuilder**](#onformfieldBuilder)  
  - [**add isReadOnly and isEnable to `InjectedTextEditing`, and `OnFormBuilder`**](#add-isreadOnly-and-isenable-to-injectedtextediting-and-onformbuilder)  


# Breaking changes
The goal of this version 5.0.0 is to clean up the API and make it consistent and more intuitive. To this end, we have a lot of deprecated features.

The overall API image will look like this:
* Inject your state using: `RM.inject`, `RM.injectFuture`, `RM.injectStream`, or `RM.injectFoo` (Foo replaces `Auth`, `Crud`, `Animation`, ...).
* Listen to your state to rebuild the UI using: `ReactiveStatelessWidget`, `OnReactive`, `OnBuilder`, or `OnFooBuilder` (Foo replaces `Auth`, `Crud`, `Animation`, ...).        
* Notify your state using: direct immutable mutation (myState.state = newState), `setState`, or `toggle` methods.
* Whenever you want to invoke side effects, on (state /or widget) initialization, mutation, or disposition, use `sideEffects` parameter.

## state injection:
### Before:

The API of side effects is not that clear.

```dart
final myState = RM.inject<MyState>(
  () => MyState(),
  onData: (data) {
    print(data);
  },
  onError: (err, refresh) {
    print(err);
  },
  onWaiting: () {
    print('onWaiting');
  },
  onDisposed: (_) {
    print('disposed');
  },
  onSetState: On(
    () {
      print('onSetState');
    },
  ),
);
```
### After
```dart
final myState = RM.inject<MyState>(
  () => MyState(),
  sideEffects: SideEffects(
    onSetState: (snap) {
      print('onSetState');
      if (snap.isWaiting) {
        print('onWaiting');
      } else if (snap.hasError) {
        print('error');
      } else if (snap.hasData) {
        print(snap.data);
      }

      //or simply use onAll or onOrElse
      // snap.onAll(
      //   onWaiting: () => print('onWaiting'),
      //   onError: (err, refresh) => print('Error'),
      //   onData: (data) => print(data),
      // );
    },
    dispose: () {
      print('disposed');
    },
  ),
);
```

`SideEffects` has named constructor to reduce the boilerplate in some use cases: `SideEffects.onData`, `SideEffects.onWaiting`, `SideEffects.onError`, `SideEffects.onAll`, `SideEffects.onOrElse`.

The same `SideEffects` class will be used in `OnReactive`, `onBuilder` and `setState` method.

`onDate`, `onError`, `onWaiting`, `onSetState` are `deprecated` in all injected methods. Use `SideEffects` instead.

## Builders (widget listeners)
To listen to an injected state you can:
* use `ReactiveStatelessWidget` or `OnReactive` for implicit subscription.
* use `OnBuilder` for explicit subscription (or use `rebuild` method equivalent of `OnBuilder` for explicit subscription).
* use `OnFooBuilder` for `injectedFoo` (Foo replaces `Auth`, `Crud`, `Animation`, ...).

All other `On` widget subscription is deprecated.

### Before
```dart
On(
  () => MyWidget(),
).listenTo(
  myState,
),
//
On.data(
  () => MyWidget(),
).listenTo(
  myState,
),
//
On.animation(
  (animate) => MyWidget(),
).listenTo(
  myAnimation,
),
```
### after
```dart
OnBuilder(
  listenTo: myState,
  builder: ()=> MyWidget(),
)
//
OnBuilder.data(
  listenTo: myState,
  builder: (data)=> MyWidget(),
)
//
OnAnimationBuilder(
  listenTo: myAnimation,
  builder: (animate) => MyWidget(),
)
```

## setState method

### before
```dart
myState.setState(
  (s) => newState,
  onData: (data) {
    print(data);
  },
  onError: (error) {
    print(error);
  },
  onSetState: On(
    () {
      print('onSetState');
    },
  ),
  onRebuildState: () {
    print('onRebuildState');
  },
);
```

Side effects defined here can override side effects defined globally in `RM.inject` and other equivalent methods. The logic of overridden is implicitly built in states_rebuilder.

### after
```dart
myState.setState(
  (s) => newState,
  sideEffects: SideEffects(
    onSetState: (snap) {
      print('onSetState');
      if (snap.hasError) {
        print('error');
      } else if (snap.hasData) {
        print(snap.data);
      }
    },
    onAfterBuild: () {
      print('onRebuildState');
    },
  ),
  //Optionally
  shouldOverrideDefaultSideEffects: (snap)=> true,
);
``` 

In this version, you have to manual decide when to override the global default side effects using the `shouldOverrideDefaultSideEffects` parameter.

## Input field and form validation
`validator` is deprecated in favor to `validators` parameter:
### before
```dart
final password = RM.injectTextEditing(
  validator: (String? value) {
      if (value!.length < 6) {
        return "Password must have at least 6 characters";
      }
      return null;
    },
);
```
### after
```dart
final password = RM.injectTextEditing(
  validators: [
    (String? value) {
      if (value!.length < 6) {
        return "Password must have at least 6 characters";
      }
      return null;
    },
  ],
);
```

With `validators` is a list, you can add more than on validator. You can also use external libraries dedicated for input validation.

## middleSnapState
`middleSnapState` is deprecated in favor to `stateInterceptor`

### before
```dart
final myState = RM.inject(
   () => [],
   middleSnapState: (middleSnap) {
     return middleSnap.nextSnap.copyTo(data: [
       ...middleSnap.currentSnap.state,
       ...middleSnap.nextSnap.state,
     ]);
   },
  );
```
### after
```dart
  final myState = RM.inject(
   () => [],
   stateInterceptor: (currentSnap, nextSnap) {
     return nextSnap.copyTo(data: [
       ...currentSnap.state,
       ...nextSnap.state,
     ]);
   },
  );
```
Also, before `middleSnapState` is triggered when state is first initialized and when it is disposed of. In contrast `stateInterceptor` is not. it is triggered only when state is mutated.

# New Features

## Auth token refreshing
Tokens are data used to authorize an application to query a server on behalf of an authenticated user.

For security reasons, tokens have a short lifespan and a refresh token is used to generate new tokens.

With the upcoming 5.0.0 update, the refresh token capability is added to `InjectedAuth`.

To do this, you must override the `refreshToken` method of the` IAuth` interface.

```dart

class UserRepository implements IAuth<User?, UserParam> {
  
  @override
  Future<User?>? refreshToken(User? currentUser) async {
    final url = 'https://securetoken.googleapis.com/v1/token?key=$webApiKey';
       
       //The refresh toke is obtained from the exposed currentUser
       final response = await http.post(
         Uri.parse(url),
         body: json.encode(
           {
             'grant_type': 'refresh_token',
             'refresh_token': currentUser!.token.refreshToken,
           },
         ),
       );
   
       if (response.statusCode == 200) {
         final responseData = json.decode(response.body);
   
         return currentUser!.copyWith(
          token: responseData['id_token'],
          refreshToken: responseData['refresh_token'],
          tokenExpiration: DateTime.now().add(
              Duration(seconds: responseData[expires_in] ),
          ),
        );
       }
  }

  @override
  Future<void> init() async {}

  @override
  Future<User?> signUp(UserParam param) {
    // Sign up implementation
  }

  @override
  Future<User?> signIn(UserParam param) {
    // Sign in implementation
  }

  @override
  Future<void> signOut(UserParam param) async {
    // Sign out implementation
  }

  @override
  void dispose() {
    // TODO: implement dispose
  }




    .
    .
    .
}
```
The `refreshToken` methods handle the token refresh logic as prescribed by the authentication server.

`refreshToken` exposes the currently authenticated user where to get the refresh token.

It should return a new copy of the current user with the new token and the refresh token.

If it returns null, the user is signed out.

`refreshToken` is automatically by the library, or it can be manually triggered by the developer.

If you defined the `autoRefreshTokenOrSignOut` parameter in `RM.injectAuth`, the refreshToken is invoked after the returned duration:

```dart
final userRM = RM.injectAuth<User?, UserParam>(
    () => AuthRepository(),
    //
    autoRefreshTokenOrSignOut: (user) {
      final timeToExpiry = user!.token.expiryDate!
          .difference( DateTime.new()).inSeconds;

      return Duration(seconds: timeToExpiry);
    },
);
```
`autoRefreshTokenOrSignOut` is a new parameter that is used instead of the old `autoSignOut` parameter, which will be deprecated.

After the returned duration form`autoRefreshTokenOrSignOut`, the library will invoke `refreshToken` and if the token is refreshed successfully, the `userRM` will hold the new user with the refreshed token.

If the refresh token is expired, the user is signed out.

If the user is persisted, and when the app starts, it will automatically refresh the token if it is expired.

You can also refresh the token manually by invoking the new added method `userRM.auth.refreshToken()`

In other repositories that request secure date form the server using the token, you should use getter to get the valid auth token and to be sure that it is always the refreshed token:

```dart
class MyRepository implements IMyReposInterface {
  //Use getter to bu sure to use the refreshed token
  String get authToken => authBloc.user!.token!;

  @override
  Future<List<Items>> getItems() async {
      final response = await http.get(
        Uri.parse('$baseUrl/$userId.json?auth=$authToken'),
      );
      ..... 
      ..... 
  }
}
```
[See more details here.](https://github.com/GIfatahTH/states_rebuilder/wiki/injected_auth_api)

## TopStatelessWidget

In previous releases we have `TopAppWidget` useful for:
- Initialize plugins; display splashScreen while initialization, and error screen in case of failure.
- Subscribe to `InjectedI18N` and `InjectedTheme` to internationalize the app and manage its theming.
- Invoke side effects when app life cycle changes.

 `TopAppWidget` may look cumbersome and `InjectedI18N` and `InjectedTheme` must be explicitly specified.

`TopStatelessWidget` is a rewrite of `TopAppWidget` to make it less boilerplate.

In all cases just use `TopStatelessWidget` instead of `StatelessWidget` on top of `MaterialApp` widget.

### Plugins initialization
In Flutter it is common to initialize plugins inside the main method:
```dart
void main()async{
 WidgetsFlutterBinding.ensureInitialized();
 await initializeFirstPlugin();
 await initializeSecondPlugin();
 runApp(MyApp());
}
```
If you want to initialize plugins and display splash screen while waiting for them to initialize and display an error screen if any of them fails to initialize or request for permission with the ability to retry the initialization you can use `TopStatelessWidget`:
```dart
class MyApp extends TopStatelessWidget {
  const MyApp({Key? key}) : super(key: key);
  @override
  List<Future<void>>? ensureInitialization() {
    return [
      initializeFirstPlugin(),
      initializeSecondPlugin(),
    ];
  }
  @override
  Widget? splashScreen() {
    return Material(
      child: Scaffold(
        body: Center(
          child: CircularProgressIndicator(),
        ),
      ),
    );
  }
  @override
  Widget? errorScreen(error, void Function() refresh) {
    return ElevatedButton.icon(
      onPressed: () => refresh(),
      icon: Icon(Icons.refresh),
      label: Text('Retry again'),
    );
  }
  @override
  Widget build(BuildContext context) {
    return MyHomePage();
  }
}
```

### App internationalization and theme swishing

```dart
 void main() {
   runApp(MyApp());
 }
 final themeRM = RM.injectTheme(
   ...
 );

 final i18nRM = RM.injectedI18N(
   ...
 );

 class MyApp extends TopStatelessWidget {
   // This widget is the root of your application.
   @override
   Widget build(BuildContext context) {
     return MaterialApp(
       //
       theme: themeRM.lightTheme, //light theme
       darkTheme: themeRM.darkTheme, //dark theme
       themeMode: themeRM.themeMode, //theme mode
       //
       locale: i18nRM.locale,
       localeResolutionCallback: i18nRM.localeResolutionCallback,
       localizationsDelegates: i18nRM.localizationsDelegates,
       home: HomePage(),
     );
   }
 }
```

### App lifecycle

To invoke side effects depending on the app life cycle, use `didChangeAppLifecycleState` hook
```dart
class MyApp extends TopStatelessWidget {
  @override
  void didChangeAppLifecycleState(AppLifecycleState state) {
    print(state);
  }
  const MyApp({Key? key}) : super(key: key);
  @override
  Widget build(BuildContext context) {
    return Container();
  }
}
```
[See more details here.](https://github.com/GIfatahTH/states_rebuilder/wiki/widget_listener_api)

## Working with tabs and pages (InjectedTabPageView)

In Flutter to work with scrollable view, you use:
* `TabBarView`, `TabBar`, `TabController` and `DefaultTabController`, or
* `PageView` and `pageController`.

`InjectedTabPageView` state encapsulates the functionality of TabController and pageController to control tab and page views.


Here is the API of InjectedTabPageView.
```dart
InjectedTabPageView injectTabPageView({
  // The number of tabs / pages to display. It can be dynamically changes later.
  required int length,
  //  The index of the tab / page to start with.
  int initialIndex = 0,
  // The duration the tab / page transition takes.
  Duration duration = kTabScrollDuration,
  // The duration the tab / page transition takes.
  Curve curve = Curves.ease,
})
```

InjectedTabPageView exposes the following api:
- `tabController`: get the associated `TabController`.
- `pageController`: get the associated `PageController`.
- `index`: get the current index of the active tab/page. When the `index` is set, The tab / page will animate to the target index.
- `length`: get  and dynamically set the length of views
- `animateTo` method: 
    ```dart
        void animateTo(
          int index, {
          Duration duration = kTabScrollDuration,
          Curve curve = Curves.ease,
        })
    ```
    Immediately sets index and `previousIndex` and then plays the animation from its current value to `index`.
- `nextView` method: 
    ```dart
        void nextView()
    ```
    Animates the controlled pages/tabs to the next page/tab
- `previousView` method: 
    ```dart
        void previousView()
    ```
    Animates the controlled pages/tabs to the previous page/tab


To listen to an `InjectedTabPageView`, we use `OnTabPageViewBuilder`:

```dart
OnTabPageViewBuilder({
  // Optional, In most cases it can be omitted
  listenTo: myInjectedTabPageView,

  builder: (int currentIndex) {
      
      return TabView( ... )
  },
})
```
By default, `OnTabPageBuilder` deduces the `InjectedTabPageView` it must listen to. So listenTo parameter is optional. 


## Example
First we define our injectedTabPage
```dart
final injectedTabPage = RM.injectTabPageView(
  length: 3,
  //Optional
  initialIndex: 0,
  curve: Curves.ease,
  duration: Duration(milliseconds: 300),
);

/// List of DataIcon to be used in pages an tabs
final icons = [
  Icons.directions_car,
  Icons.directions_transit,
  Icons.directions_bike,
  Icons.directions_boat,
  Icons.directions_bus_filled,
];

class MyApp extends StatelessWidget {
  const MyApp({Key? key}) : super(key: key);

  @override
  Widget build(BuildContext context) {
    return Scaffold(
      appBar: AppBar(
        // Display the active index 
        title: OnReactive(
          () => Text(
            'Page ${injectedTabPage.index} is displayed',
          ),
        ),
        bottom: PreferredSize(
          preferredSize: Size(0, 40),
          child: Row(
            children: [
              IconButton(
                onPressed: () {
                  // Go to the previous view
                  injectedTabPage.previousView();
                },
                icon: Icon(Icons.arrow_back_ios_rounded),
              ),
              IconButton(
                onPressed: () {
                  // Go to the next view
                  injectedTabPage.nextView();
                },
                icon: Icon(Icons.arrow_forward_ios_rounded),
              ),
              Spacer(),
              IconButton(
                onPressed: () {
                  // Dynamically shrink the number of tabs / pages
                  if (injectedTabPage.length > 1) injectedTabPage.length--;
                },
                icon: Icon(Icons.cancel),
              ),
              IconButton(
                onPressed: () {
                  // Dynamically extend the number of tabs / pages
                  if (injectedTabPage.length < icons.length)
                    injectedTabPage.length++;
                  injectedTabPage.animateTo(injectedTabPage.length - 1);
                },
                icon: Icon(Icons.add_circle),
              ),
            ],
          ),
        ),
      ),
      // In the body and the bottomNavigationBar we can use Tabs only or pages only or use both.
      // The above code remains unchangeable.
      // See below the core of the two methods depending on different use cases
      body: _body() ,
      bottomNavigationBar: _bottomNavigationBar(),
      ),
    );
  }
}
```
1. `TabBarView` controlled by `TabBar`
    Here we use `TabBarView` and `TabBar` controlled by `TabController`.
    ```dart
    Widget _body(){
        return OnTabPageViewBuilder(
            builder: (_) => TabBarView(
              controller: injectedTabPage.tabController,
              children: icons
                  .getRange(0, injectedTabPage.length)
                  .map((icon) => Icon(icon, size: 50))
                  .toList(),
            ),
          )
    }

    Widget __bottomNavigationBar(){
        return OnTabPageViewBuilder(
            builder: (index) {
              return TabBar(
                controller: injectedTabPage.tabController,
                tabs: icons
                  .getRange(0, injectedTabPage.length)
                  .map((icon) => Icon(icon, color: Colors.blue))
                  .toList(),
              );
            },
          ),
    }
    ```

2. `PageView` controlled by with a list of `OutlinedButton`:

    Here we use `PageView` controlled with a list of `OutlinedButton`.
    ```dart
    Widget _body(){
        return OnTabPageViewBuilder(
            builder: (_) => PageView.builder(
              controller: injectedTabPage.pageController,
              itemCount: injectedTabPage.length,
              itemBuilder: (_, i) {
                return Icon(
                  icons[i],
                  size: 50,
                );
              },
            ),
          )
    }

    Widget __bottomNavigationBar(){
        return OnTabPageViewBuilder(
            listenTo: injectedTabPage,
            builder: (index) {
              return Row(
                children: icons
                    .getRange(0, injectedTabPage.length)
                    .toList()
                    .asMap()
                    .map((i, icon) {
                      return MapEntry(
                        i,
                        OutlinedButton(
                          onPressed: () => injectedTabPage.index = i,
                          child: Transform.scale(
                            scale: animate(i == index ? 1.2 : 0.8, '$i')!,
                            child: Icon(icon),
                          ),
                          style: OutlinedButton.styleFrom(
                            backgroundColor:
                                animate(i == index ? Colors.blue : null, '$i'),
                            primary: animate(
                                i != index ? Colors.blue : Colors.white,
                                'primary$i'),
                          ),
                        ),
                      );
                    })
                    .values
                    .toList(),
                mainAxisAlignment: MainAxisAlignment.center,
              );
            },
          ),
    }
    ```

3. `PageView` controlled by `TabBar`

    Here we use `PageView` controlled by `TabBar`.

    ```dart
    Widget _body(){
        return OnTabPageViewBuilder(
            builder: (_) => PageView.builder(
              controller: injectedTabPage.pageController,
              itemCount: injectedTabPage.length,
              itemBuilder: (_, i) {
                return Icon(
                  icons[i],
                  size: 50,
                );
              },
            ),
          )
    }

    Widget __bottomNavigationBar(){
        return OnTabPageViewBuilder(
            builder: (index) {
              return TabBar(
                controller: injectedTabPage.tabController,
                tabs: icons
                  .getRange(0, injectedTabPage.length)
                  .map((icon) => Icon(icon, color: Colors.blue))
                  .toList(),
              );
            },
          ),
    }
    ```


4. `TabBarView` controlled by `BottomNavigationBar`

    Here we use `PageView` controlled by `TabBar`.

    ```dart
    Widget _body(){
        return OnTabPageViewBuilder(
            builder: (_) => PageView.builder(
              controller: injectedTabPage.pageController,
              itemCount: injectedTabPage.length,
              itemBuilder: (_, i) {
                return Icon(
                  icons[i],
                  size: 50,
                );
              },
            ),
          )
    }

    Widget __bottomNavigationBar(){
        return OnTabPageViewBuilder(
            listenTo: injectedTabPage,
            builder: (index) {
              return BottomNavigationBar(
                currentIndex: index,
                onTap: (i)=> injectedTabPage.index = i,
                selectedItemColor: Colors.amber[800],
                tabs: icons
                  .getRange(0, injectedTabPage.length)
                  .map((icon) => 
                    BottomNavigationBarItem(
                        icon: Icon(icon),
                    ) 
                  ,)
                  .toList(),
              );
            },
          ),
    }
    ```
[See more details here.](https://github.com/GIfatahTH/states_rebuilder/wiki/injected_tab_page_view_api)


## OnFormFieldBuilder

## add isReadOnly and isEnable to `InjectedTextEditing`, and `OnFormBuilder`
In some scenarios, we need to disable an input field or just make it read only.

Now `InjectedTextEditing` and `InjectedFormField` have:
- `isEnabled`: If set to false, it will make the field non-selectable; non-focusable, non-editable.
- `isReadOnly`: If set to true, the field is selectable, focusable but not editable.

Later on, you can change value of the `isEnabled`, `isReadOnly` properties:
```dart
final myText = RM.injectedTextEditing(
  isEnabled: false,
  isReadOnly: true,
);
final myCheckBox = RM.injectedFormField(
  isEnabled: false,
  isReadOnly: true,
);

// In the widget tree
Column(
  children: [
    // For TextFields you have to explicitly define enabled and readOnly parameters.
    TextField(
      controller: myText.controller,
      enabled: myText.isEnabled,
      readOnly: myText.isReadOnly,
    ),
    //In contrast in OnFormFieldBuilder enabled and readOnly are implicitly assigned.
    OnFormFieldBuilder<bool>(
      listenTo: myCheckBox,
      builder: (value, onChanged){
        return CheckBoxListTile(
          value: value,
          onChanged: onChanged,
          title: Text('Accept me'),
        );
      }
    )
  ]
)

// To toggle isEnabled and isReadOnly:
myText.isEnabled = true;
myText.isReadOnly = false;
myCheckBox.isEnabled = true;
myCheckBox.isReadOnly = false;
```
In case we have many input fields we can enable or disable each field individually but this my be very tedious. `InjectedForm` and `OnFormBuilder` are here to help us.

Let's imagine a case where we want to disable all input fields (or just make them read only) while the form is submitting.
```dart
final isEnabledRM = true.inj();
final formRM =  RM.injectForm(
  submissionSideEffects: SideEffects.onOrElse(
    onWaiting: ()=> isEnabledRM = false,
    orElse: (_)=> isEnabledRM = true,
    submit: () => repository.submitForm( ... ),
  ),
);

// In the widget tree
OnFormBuilder(
  listenTo: formRM,
  // Adding this all child input's enabled and readOnly properties are controlled.
  isEnabledRM: isEnabledRM,

  // // Similar if you want to make it readOnly
  //isReadOnlyRM: isReadOnlyRM,

  builder: () => Column(
      children: [
        TextField(
          controller: myText.controller,
          enabled: myText.isEnabled,
          readOnly: myText.isReadOnly,
        ),
        OnFormFieldBuilder<bool>(
          listenTo: myCheckBox,
          builder: (value, onChanged){
            return CheckBoxListTile(
              value: value,
              onChanged: onChanged,
              title: Text('Accept me'),
            );
          }
        )
      ]
  ),
)
```

As you can use many nested `OnFormFieldBuilder` for the same `formRM` you can disable (or make readOnly) only a part of the field inputs.For example, sometimes there is a checkbox as an agreement to continue to the next section, before that all/some fields on the next section are disabled.

[See more details here.](https://github.com/GIfatahTH/states_rebuilder/wiki/injected_text_editing_api)
