## OnReactive widget:

`OnReactive` widget is a new widget used to rebuild a part of the widget tree in response to state change.

`OnReactive` implicitly subscribes to injected ReactiveModels based on the on the getter `ReactiveModel.state` called during rebuild.

Example:

```dart
final counter1 = RM.inject(()=> 0) // Or just use extension: 0.inj()
final counter2 = 0.inj();
int get sum => counter1.state + counter2.state;

//In the widget tree:
Column(
    children: [
        OnReactive( // Will listen to counter1
            ()=> Text('${counter1.state}');
        ),
        OnReactive( // Will listen to counter2
            ()=> Text('${counter2.state}');
        ),
        OnReactive(// Will listen to both counter1 and counter2
            ()=> Text('$sum');
        )
    ]
)
```

Note that counter1 and counter2 are global final variable that holds the state. They are disposed automatically when not use (have no listener).

You can scope the counter1 and counter2 variable and put them inside a class:
```dart
class CounterModel {
    final counter1 = RM.inject(()=> 0) // Or just use extension: 0.inj()
    final counter2 = 0.inj();
    int get sum => counter1.state + counter2.state;

    void increment1() => counter1.state++;
    void increment2() => counter2.state++;

    void asyncMethod() => counter1.setState((s) async => asyncRep())
}
//Just instantiate a global instance of the CounterModel and use it throughout your app. The CounterModel instance is not a global state rather it acts like a container that contains the counter1 and counter2 states.

//You can easily test the app and make sure the all states are reset to their initial state between tests.
final counterModel = CounterModel();

//In the widget tree:
Column(
    children: [
        OnReactive( // Will listen to counter1
            ()=> Text('${counterModel.counter1.state}');
        ),
        OnReactive( // Will listen to counter2
            ()=> Text('${counterModel.counter2.state}');
        ),
        OnReactive(// Will listen to both counter1 and counter2
            ()=> Text('${counterModel.sum}');
        )
    ]
)
```

`OnReactive` can listen to any state called form its child widget tree no matter how deep the widget tree is.

```dart
OnReactive(
    ()=> DeepWidgetTree(),
)

class DeepWidgetTree extends StatelessWidget{
    Widget builder (BuildContext context){
        return Column(
            children: [
                //Will look up the widget tree and subscribe (if not already subscribed) to the first found OnReactive widget
                Text('${counter1.state}'),
                AnOtherChildWidget();
            ]
        );
    }
}

class DeepWidgetTree extends StatelessWidget {
    Widget builder (BuildContext context){
        //Will look up the widget tree and subscribe (if not already subscribed) to the first found OnReactive widget
        return Text('${counter2.state}');
    }
}
```
Inside OnReactive you can call any of the available state status flags (isWaiting, hasError, hasData, ...) or just use onAll and onOr methods:
```dart
OnReactive(
    ()=> {
        if(myModel.isWaiting){
            return WaitingWidget();
        }
        if(myModel.hasError){
            return ErrorWidget();
        }
        return DataWidget();
    }
)
//Or use onAll method:
OnReactive(
    ()=> {
        myModel.onAll(
            onWaiting: ()=> WaitingWidget(),
            onError: (err, refreshErr)=> ErrorWidget(),
            onDate: (data)=> DataWidget(),
        );
    }
)

//Or use onOr method:
OnReactive(
    ()=> {
        myModel.onAll(
            onWaiting: ()=> WaitingWidget(),
            or: (data)=> DataWidget(),
        );
    }
)
```

`OnReactive` is a lightweight widget that behaves as it is a const widget. That is, once `OnReactive` is created it will not recrate after parent widgets rebuild. `OnReactive` is rebuild only after any observable state emits a notification or after hot restart.

This is the full API of `OnReactive`:

```dart
OnReactive(
    (){
        //Widget to rebuild
    }, 
    initState: (){
        // Side effect to call when the widget is first inserted into the widget tree
    },
    dispose: (){
        // Side effect to call when the widget is removed from the widget tree
    },
    onSetState: (snapState){
        // Side effect to call when is notified to rebuild

        //if the OnReactive listens to many states, the exposed snapState is that of the state that emits the notification
    },
    shouldRebuild: (oldSnap, newSnap){
        // return bool to whether rebuild the widget or not.

        //if the OnReactive listens to many states, the exposed snapState is that of the state that emits the notification
    },
);
```


## OnBuild, OnFooBuild widgets and myState.rebuild and myState.rebuild.onFoo methods:

Beside the `OnReactive`, there are other widgets used to explicitly listen to one or more states.

The first widget is `OnBuilder` widget.

```dart
OnBuilder(
    listenTo: myState,
    onBuilder: On(
         () => Text('${myState.state}'),
    ),
),

//This is exactly equivalent to:
On(
    () => Text('${myState.state}'),
).listenTo(myState)
//This last form may be deprecated in the future (We are in version v.4.4.0)
```

With `OnBuilder` you can perform side effects defining the sideEffects parameter

```dart
       OnBuilder(
           listenTo: counter,
           onBuilder: On(() => Text('${counter.state}')),
           sideEffect: SideEffect(
            initState: () => print('initState'),
            onSetState: On(() => print('onSetState')),
            onAfterBuild: On(() => print('onAfterBuild')),
            dispose: () => print('dispose'),
          ),
          shouldRebuild: (oldSnap, newSnap) {
            return true;
          },
          debugPrintWhenRebuild: 'counter',
        ),
```

You can control when to build and what to display depending on the state status using the On named constructors
```dart
OnBuilder(
    listenTo: counter,
    onBuilder: On.data( // Will be rebuilt onData only
        () => Text('${counter.state}'),
    ),
),

OnBuilder(
    listenTo: counter,
    onBuilder: On.all(
        onIdle: () => Text('onIdle'),//This is optional
        onWaiting: () => Text('onWaiting'),
        onError: (err, errorRefresh) => Text('onError'),
        onData: () => Text('{counter.state}'),
    ),
),

OnBuilder(
    listenTo: counter,
    onBuilder: On.all(
        onWaiting: () => Text('onWaiting'),
        or: () => Text('{counter.state}'),
    ),
),
```











```dart
        OnAuthBuilder(
          listenTo: user,
          onUnsigned: () => Text('Sign in/ up page'),
          onSigned: () => Text('Home page'),
          useRouteNavigation: true,
        ),
```

```dart
        OnCRUDBuilder(
          listenTo: products,
          onWaiting: () => Text('onWaiting'),
          onError: (err, refreshError) => Text('onError'),
          onResult: (result) => Text('onResult'),
        ),
```

```dart
        OnAnimationBuilder(
            listenTo: myAnimation
            builder: (animate) {
                return Container(
                    width: animate.fromTween((_) => Tween(begin: 0, end: 100)),
                );
           }
        ),
```

```dart
        OnFormBuilder(
          listenTo: myForm
          builder:() => TextField(
             controller: userName.controller,
           ),
        ),
        //
        OnFormSubmissionBuilder(
          listenTo: myForm
          onSubmitting: () => Text('onSubmitting'),
          child: Text('child'),
        ),
```

```dart
        OnScrollBuilder(
          listenTo: scroller,
          builder: (scroll) => Text('onScroll'),
        ),
```

```dart
        Scaffold(
          body: OnTabBuilder(
              listenTo: myTab,
              builder : () {
              return TabBarView(
                controller: myTab.controller,
                children: screens,
              );
            },
          ),
          bottomNavigationBar: 
          OnTabBuilder(
              listenTo: myTab,
              builder :() => TabBar(
              controller: myTab.controller,
              tabs: tabs,
            ),
          ),
        ),
```
