# Table of Contents <!-- omit in toc --> 
- [**Breaking changes**](#Breaking-changes)  
- [**`On` for all widget listeners**](#`On`-for-all-widget-listeners)  
- [**middleSnapState**](#middleSnapState)  
- [**`RM.injectCRUD` for easier CREATE READ UPDATE DELETE operation**](#`RM.injectCRUD`-for-easier-CREATE-READ-UPDATE-DELETE-operation)  
- [**RM.injectAuth: Authentication has never been easier**](#RM.injectAuth:-Authentication-has-never-been-easier)  
- [**RM.injectTheme for themes management**](#RM.injectTheme-for-themes-management)  
- [**RM.injectI18N for internationalization and localization**](#RM.injectI18N-for-internationalization-and-localization)  
- [**Page route transition animation**](#Page-route-transition-animation)  
- [**Nested Routes and dynamic segments**](#Nested-Routes-and-dynamic-segments)  


## Breaking changes

* The class RMKey is removed. Because the functional injection is the preferred approach and RMKey adds in the code complexity and has no use case with function injection.
* `RM.injectComputed `is removed. From version 4.0.0 any state can be set to depend on another state without using `RM.injectComputed`.

Example:
```dart
final counter = RM.inject(() => 0);

final counter1 = RM.inject(
  () => 0,
  dependsOn: DependsOn(
    {counter},
    shouldNotify: (state) => state < 10,
    debounceDelay: 500,
    throttleDelay: 500,
  ),
);

final counter2 = RM.injectFuture(
  () => Future.value(0),
  dependsOn: DependsOn({counter}),
);

final counter3 = RM.injectStream(
  () => Stream.fromIterable([1, 2, 3]),
  dependsOn: DependsOn({counter, counter1, counter2}),
);
```
* `injector.env` is deprecated in favor of `RM.env`
You can set flavors without using `RM.env`.

```dart
enum Env { dev, prod }

//Change here and all dependencies are changed
Env currentEnv = Env.dev;

final 

final InjectedAuth<User, UserParam> user = RM.injectAuth<User, UserParam>(
  () {
    assert(currentEnv != null);
    return {
      Env.dev: FakeUserRepository(),
      Env.prod: UserRepository(),
    }[currentEnv];
  },
);

final canSignInWithApple = RM.injectFuture(
  () => {
    Env.dev: FakeAppleSignInChecker().check(),
    Env.prod: AppleSignInChecker().check(),
  }[currentEnv],
  autoDisposeWhenNotUsed: false,
);
```
## `On` for all widget listeners

Before version 4.0.0, to listen to a ReactiveModel or an Injected model, we use:
`StateBuilder`, `WhenRebuilder`, `WhenRebuilderOr`, `model.rebuilder()`, model.whenRebuilder() `model.whenRebuilderOr()` `model.fututreBuilder()`, `model.streamBuilder()`.

I was looking for a widget that simplifies the Listening API and makes it easier to add and remove from the widget tree. I tried many alternatives and ended up going with the “On” and “OnCombined” widgets.


To make a part of the widget tree reactive, simply wrap it with `On` and use the` listenTo` extension method to indicate which injected model it is listening to.
example of a  counter app:
```dart
final counter = RM.inject(()=> 0);

On(
  ()=> Text(counter.state.toString()),
).listenTo(counter);
```
`On` has many named constructors to control when the widget rebuilds:

```dart
final model = RM.inject(()=> Model());

// in the widget tree.

//On() constructor: the widget is rebuild for each notification regardless the its state status
On(()=> MyWidget()).listenTo(model);


//On.data() constructor: the widget is rebuild only if notification has `hasData` state status
On.data(()=> MyWidget()).listenTo(model);


//On.waiting() constructor: the widget is rebuild only if notification has `isWaiting` or `hasData` state status
On.waiting(()=> MyWidget()).listenTo(model);

//On.error() constructor: the widget is rebuild only if notification has `hasError` or `hasData` state status
On.error((err, refresh)=> MyWidget()).listenTo(model);


//On.all() constructor: for each state status there is a corresponding widget to render
On.all(
    onIdle: ()=> Text('Idle'),
    onWaiting: ()=> Text('Waiting..'),
    onError: (error, refresh) => Text('Error'),
    onData: ()=> MyWidget(),
).listenTo(model);

//On.or() constructor: similar to On.all but state status callbacks 
// are optional with one required default one
On.or(
    onWaiting: ()=> Text('Waiting..'),
    or: => MyWidget(),
).listenTo(model);

```

With On, we can run side effects: once the widget is inserted or removed from the widget tree, after notification and before rebuild or after build. You have the full control on when to fire side effects
```dart
 On<Widget>(
  ()=> Text(counter.state.toString()),
).listenTo<T>(
    counter,
    onSetState : On(()=> print(..)), 
    onAfterBuild : On(()=> print(..)), 
    initState: () { ..}, 
    dispose: () { ..}, 
    didUpdateWidget: (oldWidget) { ..}, 
    shouldRebuild: () { ..}, 
    watch: () { ..}, 
    key: Key(),
  }
)
```

All onError callback exposes a refresher to be used to refresh the error and try another attempt to invoke the future.
Example: 
```dart
final model = RM.injectFuture<String>(
  () => Future.delayed(
    Duration(seconds: 1),
    () {
      return Random().nextBool() ? 'Data' : throw Exception('UnKnown Error');
    },
  ),
);

class App extends StatelessWidget {
 
  @override
  Widget build(BuildContext context) {
    return Scaffold(
      appBar: AppBar(),
      body: Center(
        child: On.or(
          onWaiting: () => CircularProgressIndicator(),
          onError: (err, refresh) => ElevatedButton(
            onPressed: () => refresh(), //Retry calling the future
            child: Text('Refresh the error'),
          ),
          or: () => Text('${model.state}'),
        ).listenTo(model),
      ),
    );
  }
}
```
It is similar to the “On” widget. It allows to listen to several injected states and to react and expose a merged state.

```dart
OnCombined.all(
    onWaiting: ()=> Text('Waiting..'),//if any is waiting
    onError: (error, refresh) => Text('Error'),//if any has error
    onIdle: ()=> Text('Idle'),//if any is idle
    onData: ()=> MyWidget(),// if all have data
).listenTo(
  [model1, model2, model3],
   onSetState: On.error(//If any has error
        (err)=> //show snack bar or navigate to an other page
    ),
),
```
[See here for a detailed overview of listeners](https://github.com/GIfatahTH/states_rebuilder/wiki/widget_listener_api)

## middleSnapState
In all Injected, there is a new parameter called `middleSnapState`. It is a callback that is used to track the state's life and transitions. It exposes a `MiddleSnapState<T>` object and returns a `SnapState<T>`.

The `MiddleSnapState<T>` objects, holds a snap of the state before state calculation (called `currentSnap`) and a snap of the state after new calculation and just before state mutation.

It is the right palace to log some valuable information about the state for debugging purposes.

Inside the middleSnapSate callback, you can change how the state will be mutated.
Here is an example of a count-down timer that counts from 60 to 0 seconds

```dart
final timer = RM.injectStream<int>(
  //stream emits 0, 1, 2, ... infinitely
  () => Stream.periodic(Duration(seconds: 1), (tick) => tick + 1),
  middleSnapState: (middleSnap) {
    print(middleSnap.currentSnap); //snap state before calculation
    print(middleSnap.nextSnap); //snap state after calculation
    //
    if(middleSnap.nextSnap.hasError) {
        //Error and stackTraces can be sent to a an error tracking service
        print(middleSnap.nextSnap.error);
        print(middleSnap.nextSnap.stackTrace);
    }
    //
    middleSnap.print(); //Build-in logger


    if (middleSnap.nextSnap.data > 60) {
      //cancel subscription and return the currentSnap (holds 0)
      timer.subscription.cancel();
      return middleSnap.currentSnap;
    }
    return middleSnap.nextSnap.copyToHasData(
      // It will return a state of 60, 59, 58, ... 0
      60 - middleSnap.nextSnap.data,
    );
  },
)

class App extends StatelessWidget {
  @override
  Widget build(BuildContext context) {
    return MaterialApp(
      home: Scaffold(
        body: Center(
          child: On(
            () => Text(timer.state.toString()),
          ).listenTo(timer),
        ),
      ),
    );
  }
}
```

This is another example of email field validation
```dart
final email = RM.inject<String>(
  () => '',
  middleSnapState: (middleSnap) {
    if (middleSnap.nextSnap.hasData) {
      if (!middleSnap.nextSnap.data.contains('@')) {
        return middleSnap.nextSnap.copyToHasError(
          Exception('Enter a valid Email'),
        );
      }
    }
  },
);

class App extends StatelessWidget {
  @override
  Widget build(BuildContext context) {
    return MaterialApp(
      home: Scaffold(
        body: On(
          () => TextField(
            onChanged: (value) => email.state = value,
            decoration: InputDecoration(
              errorText: email.error?.message,
            ),
          ),
        ).listenTo(email),
      ),
    );
  }
}
```

[See here for more details on middleSnapState](https://github.com/GIfatahTH/states_rebuilder/wiki/rm_injected_api)



## `RM.injectCRUD` for easier CREATE READ UPDATE DELETE operation

Form this update, states_rebuilder offers four custom Injected state to deal with the most common required tasks in app development, which are:
1. Fetch, create, update and delete data from a backend database service
2. Authenticate and authorize a user.
3. Manage app themes.
4. App internationalization.

To set states_rebuilder to use A`RM.injectCRUD` you have to do is define:
- A data class that represents your data (a row in a table) and,
- A parameter class which contains the parameters to be used in the backend query. It can be a simple primitive (String, int ...)
- A class that implements an interface to tell states_rebuilder how to query your backend service. states_rebuilder doesn't know and shouldn't know anything about your backend service.

You have six methods to implement:
- `init`: to initialize plugins
- `create`,` read`, `update`,` delete` for CRUD operations
- "dispose": to dispose of resources and do some cleaning if necessary.
- You can add other custom methods, for example, to count items.

The `ItemRepository` here is a fake implementation, you can replace it with any real implementation e.g. for Sqflite, firebase, ....

```dart
class ProductsRepository implements ICRUD<Products, ProductsParam> {

  @override
  Future<void> init() async {
    //Initialize the plugins
  }

  @override
  Future<Products> create(Products number, ProductsParam param) async {
  }

  @override
  Future<List<Products>> read(ProductsParam param) async {

  }

  @override
  Future update(List<Products> numbers, ProductsParam param) async {
  }

  @override
  Future delete(List<Products> numbers, ProductsParam param) async {

  }

  Future<int> count(ProductsParam param) async {
  ///Your custom method non defined in the ICRUD interface
  }

  @override
  void dispose() {}
}
```

Now you can use `RM.injectCRUD`


 ```dart
  final products = RM.injectCRUD<Product, Param>(
      ()=> MyProductRepository(),//Implements ICRUD<Product, Param>
      readOnInitialization: true,// Optional (Default is false)
  );
  ```

This is the hardest part in the UI to:
  ```dart
  //READ
  products.crud.read(param: (param)=> NewParam());
  //CREATE
  products.crud.create(NewProduct());
  //UPDATE
  products.crud.update(
    where: (product) => product.id == 1,
    set: (product)=> product.copyWith(...),
  );
  //DELETE
  products.crud.update(
    where: (product) => product.id == 1,
    isOptimistic: false, // Optional (Default is true)
  );
  ```
[See the docs for more information about the InjectecCRUD api](https://github.com/GIfatahTH/states_rebuilder/wiki/injected_crud_api)

See the full working example  [here](https://github.com/GIfatahTH/states_rebuilder/tree/master/examples/ex_006_1_crud_app) and  [here](https://github.com/GIfatahTH/states_rebuilder/tree/master/examples/ex_006_2_infinite_scroll_list)


## RM.injectAuth: Authentication has never been easier

To set `RM.injectAuth`, you have to define:
* Data class for the User,
* A class that extends the User to represent unsigned user
* Implement the IAuth interface

Here is an example of UserRepository that implemenets IAuth.
```dart
class UserRepository implements IAuth<User, UserParam> {
  Map<String, User> _signedUser;
  @override
  Future<void> init() async {
    //Initialize plugins
  }

  @override
  Future<User> signUp(UserParam param) {
    switch (param.signUp) {
      case SignUp.withEmailAndPassword:
        return _createUserWithEmailAndPassword(
          param.email,
          param.password,
        );
      default:
        throw UnimplementedError();
    }
  }

  @override
  Future<User> signIn(UserParam param) {
    switch (param.signIn) {
      case SignIn.anonymously:
        return _signInAnonymously();
      case SignIn.withApple:
        return _signInWithApple();
      case SignIn.withGoogle:
        return _signInWithGoogle();
      case SignIn.withEmailAndPassword:
        return _signInWithEmailAndPassword(
          param.email,
          param.password,
        );

      default:
        throw UnimplementedError();
    }
  }

  @override
  Future<void> signOut(UserParam param) {
    // TODO: implement signOut
    throw UnimplementedError();
  } 


 /// Other methods not imposed by the IAuth interface
  Future<User> currentUSer() async {
    //get the current signed user
  }

  Stream<User> onAuthStateChange() {
    // Stream to listen to user auth state change
  }

  @override
  void dispose() {}
}
```

Now we are ready to use the `RM.injectAuth`
* Minimal required parameter
```dart
final user = RM.injectAuth(
  () => UserRepository(), 
  unsignedUser: UnsignedUSer(),//Our class for unsigned user
);
```
* Auto sign out if token expired
```dart
final user = RM.injectAuth(
  () => UserRepository(),
  unsignedUser: UnsignedUSer(),
  //
  autoSignOut: (user) => Duration(seconds: user.takenExpiryData),
);
```
* Listen to the auth state changed stream
```dart
final user = RM.injectAuth(
  () => UserRepository(),
  unsignedUser: UnsignedUSer(),
  //
  onAuthStream: (repo) => (repo as UserRepository).onAuthStateChange(),
);
```
 * fetch current user on app start up
```dart
final user = RM.injectAuth(
  () => UserRepository(),
  unsignedUser: UnsignedUSer(),
  //
  onAuthStream: (repo) => (repo as UserRepository).currentUSer().asStream(),
);
```
* Persist signed user
```dart
final user1 = RM.injectAuth<User, UserParam>(
  () => UserRepository(),
  unsignedUser: UnsignedUSer(),
 //
  persist: () => PersistState(
    key: '__user__',
    toJson: (user) => user.toJson(),
    fromJson: (json) => User.fromJson(json),
  ),
);
```
Now the app on startup will see if there is a persistent user and then direct to the onSigned page

In  UI we use `On.auth`:
```dart
class App extends StatelessWidget {
  @override
  Widget build(BuildContext context) {
    return MaterialApp(
      home: Scaffold(
        body: On.auth(
        //If the user state is set to find the current user or is set to listen for 
       //the authentication state change, we will wait for it when the app starts.
          onWaiting: () => CircularProgressIndicator(),
          onUnsigned: () => AuthPage(),
          onSigned: () => HomeUserPage(),
        ).listenTo(user,
            //This is optional and its default value is false.
            //
            //Set it to true if you want switch between `onSinged` and `onUnsigned` page using
            //the route page transition animation
            useRouteNavigation: true),
      ),
    );
  }
}
```
By default, the switch between the `onSinged` and the `onUnsigned` pages is a simple widget replacement. To use the navigation page transition animation, set `userRouteNavigation` to true. In this case, you need to set the `RM.navigate.navigatorKey`.


To sign up
```dart
  user.auth.signUp(
    (param) => UserParam(
      signUp: SignUp.withEmailAndPassword,
      email: emailController.text,
      password: passwordController.text,
    ),
  );
```
To sign in.
```dart
  user.auth.signIn(
    (param) => UserParam(
      signIn: SignIn.withGoogle,
    ),
  );
```
 To sign out
```dart
  user.auth.signOut();
```

[See the docs for more information about the InjectecCRUD api](https://github.com/GIfatahTH/states_rebuilder/wiki/injected_auth_api)

See a full working example [here](https://github.com/GIfatahTH/states_rebuilder/tree/master/examples/ex_008_clean_architecture_firebase_login)

## RM.injectTheme for themes management

`RM.injectTheme` is created to simplify app theme management.

  ```dart
  final theme = RM.injectTheme<String>(
      lightThemes : {
        'simple': ThemeData.light( ... ),
        'solarized': ThemeData.light( ...),
      },
      darkThemes: {
        'simple': ThemeData.dark( ... ),
        'solarized': ThemeData.dark( ...),
      };
      themeMode: ThemeMode.system;
      persistKey: '__theme__',
  );
  ```

  ```dart
  //choose the theme
  theme.state = 'solarized'
  //toggle between dark and light mode of the chosen them
  theme.toggle();
  ```

This is a working example : 
```dart
final theme = RM.injectTheme(
  lightThemes: {
    'simple': ThemeData.light(),
  },
  darkThemes: {
    'simple': ThemeData.dark(),
  },
);

main() {
  runApp(MyApp());
}

class MyApp extends StatelessWidget {
  @override
  Widget build(BuildContext context) {
    return TopAppWidget(
      injectedTheme: theme,
      builder: (context) {
        return MaterialApp(
          theme: theme.lightTheme,
          darkTheme: theme.darkTheme,
          themeMode: theme.themeMode,
          home: Scaffold(
            appBar: AppBar(
              title: Text(theme.isDarkTheme ? 'Dark mode' : 'Light mode'),
            ),
            body: Center(
              child: ElevatedButton(
                onPressed: () => theme.toggle(),
                child: Text('Toggle Theme'),
              ),
            ),
          ),
        );
      },
    );
  }
}
```
[See more detailed information about `InjectedTheme`](https://github.com/GIfatahTH/states_rebuilder/wiki/injected_theme_api).

See a full working example [here](https://github.com/GIfatahTH/states_rebuilder/tree/master/examples/ex_005_theme_switching)

## RM.injectI18N for internationalization and localization

`RM.injectI18N` is yet another custom inject that will simplify the developer's life and gain him much time.

To internationalize and localize your app:
  ```dart
  //US english
  class EnUS {
    final helloWorld = 'Hello world';
  }
  //Spanish
  class EsEs implements EnUs{
    final helloWorld = 'Hola Mondo';
  }
  ```
  ```dart
  final i18n = RM.injectI18N<EnUS>(
      {
        Local('en', 'US'): ()=> EnUS();//can be async
        Local('es', 'ES'): ()=> EsES();
      };
      persistKey: '__lang__', //local persistance of language 
  );
  ```
  In the UI:
  ```dart
  Text(i18n.of(context).helloWorld);
  ```

  ```dart
  //choose the language
  i18n.locale = Local('es', 'Es');
  //Or choose the system language
  i18n.locale = SystemLocale();
  ```

This is a simple working example.

```dart
class EnUS {
  final helloWorld = 'Hello world';
}

class EsES extends EnUS {
  final helloWorld = 'Holà mondo';
}

final i18n = RM.injectI18N({
  Locale('en', 'US'): () => EnUS(),
  Locale('es', 'ES'): () => EsES(),
});

main() {
  runApp(MyApp());
}

class MyApp extends StatelessWidget {
  @override
  Widget build(BuildContext context) {
    return TopAppWidget(
      injectedI18N: i18n,
      builder: (context) {
        return MaterialApp(
          locale: i18n.locale,
          localeResolutionCallback: i18n.localeResolutionCallback,
          home: const MyHomePage(),
        );
      },
    );
  }
}

class MyHomePage extends StatelessWidget {
  const MyHomePage();
  @override
  Widget build(BuildContext context) {
    return Scaffold(
      body: Center(
        child: Column(
          children: [
            Text(i18n.of(context).helloWorld),
            ElevatedButton(
              onPressed: () => i18n.locale = Locale('en', 'US'),
              child: Text('To English'),
            ),
            ElevatedButton(
              onPressed: () => i18n.locale = Locale('es'),
              child: Text('To Spanish'),
            ),
            ElevatedButton(
              onPressed: () => i18n.locale = SystemLocale(),
              child: Text('To System locale'),
            ),
          ],
          mainAxisAlignment: MainAxisAlignment.center,
        ),
      ),
    );
  }
}

class Home extends StatelessWidget {
  @override
  Widget build(BuildContext context) {
    return TopAppWidget(
      builder: (_) => MaterialApp(
        navigatorKey: RM.navigate.navigatorKey,
        onGenerateRoute: RM.navigate.onGenerateRoute(
          {
            App.routeName: (_) => App(),
            Page1.routeName: (arg) => Page1(data: arg),
            Page2.routeName: (_) => Page2(),
            Page3.routeName: (_) => Page3(),
          },
          unknownRoute: Scaffold(
            appBar: AppBar(),
            body: Center(
              child: Text('404'),
            ),
          ),
        ),
      ),
    );
  }
}
```
[🗎 See more detailed information about InjectedI18N](https://github.com/GIfatahTH/states_rebuilder/wiki/injected_i18n_api).

See a full working example [here](https://github.com/GIfatahTH/states_rebuilder/tree/master/examples/ex_005_theme_switching)

## Page route transition animation

Now, states_rebuilder offers four predefined page transition animations.
For example to animate the page from bottom to up, use : (This is the default Flutter animation):

```dart
main(){
  //In the main app and before runApp method:
  RM.navigate.transitionsBuilder = RM.transitions.bottomToUP();

  runApp(MyApp());
}
```
The above code works for widget routing as well as for named routing provided you use the `RM.navigate.onGenerateRoute` for named routes.

In the latter case (named routing), you can use the `transitionBuilder` parameter of `RM.navigate.onGenerateRoute` to set the transition animation.

```dart
 final widget_ = MaterialApp(
      navigatorKey: RM.navigate.navigatorKey,
      onGenerateRoute: RM.navigate.onGenerateRoute(
        {
          '/': (_) => HomePage('Home'),
          'page1': (param) => Route1(param as String),
          'Route2': (param) => Route2(param as int),
        },
        //Optional argument
        transitionsBuilder:  RM.transitions.bottomToUP();//uses the default flutter param
      ),
    );
```

Page transition animation consists of to animation: translation and opacity animations.

You can set the tween, the curve, and the duration of the position and opacity animation:

```dart
RM.navigate.transitionsBuilder = RM.transitions.bottomToUP(
    //Default: Duration(milliseconds: 300)
    duration: Duration(milliseconds: 500), 
    //Default: Tween<Offset>( begin: const Offset(0.0, 0.25), end: Offset.zero)
    positionTween: Tween<Offset>( begin: const Offset(0.0, 1), end: Offset.zero),
    //Default:  Curves.fastOutSlowIn;
    positionCurve : Curves.bounceInOut,
    //Default: Tween<double>(begin: 0.0, end: 1.0)
    opacityTween: Tween<double>(begin: 0.0, end: 1.0),
    //Curves.easeIn
    opacityCurve : Curves.easeOut,
  );
```

There are four predefined and configurable page transition animations:

```dart
RM.transitions.bottomToUP();
RM.transitions.upToBottom();
RM.transitions.leftToRight();
RM.transitions.rightToLeft();
```

If your animation cannot be done with one of the four predefined animations, you can define your own using `transitionsBuilder` or` pageRouteBuilder` for full options


[See here for more detailed on page route animation](https://github.com/GIfatahTH/states_rebuilder/wiki/navigation_dialog_scaffold_without_BuildContext_api)


## Nested Routes and dynamic segments

you can use nested routes using `RouteWidget` like this:

```dart
return MaterialApp(
    navigatorKey: RM.navigate.navigatorKey,
    onGenerateRoute: RM.navigate.onGenerateRoute({
      '/': (_) => LoginPage(),
      '/posts': (_) => RouteWidget(
            routes: {
              '/:author': (RouteData data) {
                  final queryParams = data.queryParams;
                  final pathParams = data.pathParams;
                  final arguments = data.arguments;

                  //OR
                  //Inside a child widget of AuthorWidget :
                  //
                  //context.routeQueryParams;
                  //context.routePathParams;
                  //context.routeArguments;
                  return  AuthorWidget();

              },
              '/postDetails': (_) => PostDetailsWidget(),
            },
          ),
      '/settings': (_) => SettingsPage(),
    }),
  );
```


In the UI:
```dart
RM.navigate.to('/'); // => renders LoginPage()
RM.navigate.to('/posts'); // => 404 error
RM.navigate.to('/posts/foo'); // =>  renders AuthorWidget(), with pathParams = {'author' : 'foo' }
RM.navigate.to('/posts/postDetails'); // =>  renders PostDetailsWidget(),
//If you are in AuthorWidget you can use relative path (name without the back slash at the beginning)
RM.navigate.to('postDetails'); // =>  renders PostDetailsWidget(),
RM.navigate.to('postDetails', queryParams : {'postId': '1'}); // =>  renders PostDetailsWidget(),
```

`RouteWidget` has `builder` parameters. it allows to insert widgets above the route.


```dart
return MaterialApp(
    navigatorKey: RM.navigate.navigatorKey,
    onGenerateRoute: RM.navigate.onGenerateRoute({
      '/dashboard': (_) => RouteWidget(
          builder: (Widget child) {
            //child is the route output
            //
            //Instead you can set the entry point where router will display
            //using context.routeWidget
            return DashboardPage(child: child);
          } ,
          routes: {
            '/': (_) => OverviewWidget(),
            newUsers: (_) => RouteWidget(
                  builder: (_) => NewUsersWidget(),
                  routes: {
                    '/:id': (_) => UserDetailWidget(),
                  },
                ),
            sales: (_) => SalesWidget(),
          },
        ),
    }),
);
```

### SubRoute transition animation
What is interesting is that if you define the `builder` and set the entry point where the router will display, only that part of the widget tree will be animated during page transition.

You can override the default transition animation for a particular route using transitionsBuilder of the RouteWidget:

```dart
return MaterialApp(
    navigatorKey: RM.navigate.navigatorKey,
    onGenerateRoute: RM.navigate.onGenerateRoute({
      '/dashboard': (_) => RouteWidget(
          builder: (Widget child) {
            return DashboardPage(child: child);
          } ,
          routes: {
            '/': (_) => OverviewWidget(),
            newUsers: (_) => RouteWidget(
                  builder: (_) => NewUsersWidget(),
                  routes: {
                    '/:id': (_) => UserDetailWidget(),
                  },
                  transitionsBuilder: RM.transitions.upToBottom(),
                ),
            sales: (_) => SalesWidget(),
          },
          //The default custom animation.
          transitionsBuilder: RM.transitions.leftToRight(),

        ),
    }),
);
```

### Protected routes
```dart
return MaterialApp(
    navigatorKey: RM.navigate.navigatorKey,
    onGenerateRoute: RM.navigate.onGenerateRoute({
      '/': (_) => Home(),
      '/about': (_)=> About(),
      '/profile': (data) {
          return data.arguments != null ? Profile(data.arguments as User) : Login(),
      }
    }),
  );
```

If you push the profile route with a valid user object the `Profile` page will be rendered, in the opposite case a `Login` page is displayed instead.



