// Copyright 2018 the Charts project authors. Please see the AUTHORS file
// for details.
//
// Licensed under the Apache License, Version 2.0 (the "License");
// you may not use this file except in compliance with the License.
// You may obtain a copy of the License at
//
// http://www.apache.org/licenses/LICENSE-2.0
//
// Unless required by applicable law or agreed to in writing, software
// distributed under the License is distributed on an "AS IS" BASIS,
// WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
// See the License for the specific language governing permissions and
// limitations under the License.

import 'package:nimble_charts_common/src/chart/cartesian/axis/scale.dart';
import 'package:nimble_charts_common/src/chart/cartesian/axis/simple_ordinal_scale.dart';
import 'package:nimble_charts_common/src/common/style/material_style.dart';
import 'package:nimble_charts_common/src/common/style/style_factory.dart';

import 'package:test/test.dart';

const EPSILON = 0.001;

class TestStyle extends MaterialStyle {
  @override
  late double rangeBandSize;
}

void main() {
  late SimpleOrdinalScale scale;

  setUp(() {
    scale = SimpleOrdinalScale()
      ..addDomain('a')
      ..addDomain('b')
      ..addDomain('c')
      ..addDomain('d')
      ..range = const ScaleOutputExtent(2000, 1000);
  });

  group('conversion', () {
    test('with duplicate keys', () {
      scale
        ..addDomain('c')
        ..addDomain('a');

      // Current RangeBandConfig.styleAssignedPercent sets size to 0.65 percent.
      expect(scale.rangeBand, closeTo(250 * 0.65, EPSILON));
      expect(scale['a'], closeTo(2000 - 125, EPSILON));
      expect(scale['b'], closeTo(2000 - 375, EPSILON));
      expect(scale['c'], closeTo(2000 - 625, EPSILON));
    });

    test('invalid domain does not throw exception', () {
      expect(scale['e'], 0);
    });

    test('invalid domain can translate is false', () {
      expect(scale.canTranslate('e'), isFalse);
    });
  });

  group('copy', () {
    test('can convert domain', () {
      final copied = scale.copy();
      expect(copied['c'], closeTo(2000 - 625, EPSILON));
    });

    test('does not affect original', () {
      final copied = scale.copy()..addDomain('bar');

      expect(copied.canTranslate('bar'), isTrue);
      expect(scale.canTranslate('bar'), isFalse);
    });
  });

  group('reset', () {
    test('clears domains', () {
      scale
        ..resetDomain()
        ..addDomain('foo')
        ..addDomain('bar');

      expect(scale['foo'], closeTo(2000 - 250, EPSILON));
    });
  });

  group('set RangeBandConfig', () {
    test('fixed pixel range band changes range band', () {
      scale.rangeBandConfig = const RangeBandConfig.fixedPixel(123);

      expect(scale.rangeBand, closeTo(123.0, EPSILON));

      // Adding another domain to ensure it still doesn't change.
      scale.addDomain('foo');
      expect(scale.rangeBand, closeTo(123.0, EPSILON));
    });

    test('percent range band changes range band', () {
      scale.rangeBandConfig = RangeBandConfig.percentOfStep(0.5);
      // 125 = 0.5f * 1000pixels / 4domains
      expect(scale.rangeBand, closeTo(125.0, EPSILON));
    });

    test('space from step changes range band', () {
      scale.rangeBandConfig =
          const RangeBandConfig.fixedPixelSpaceBetweenStep(50);
      // 200 = 1000pixels / 4domains) - 50
      expect(scale.rangeBand, closeTo(200.0, EPSILON));
    });

    test('fixed domain throws argument exception', () {
      expect(
        () => scale.rangeBandConfig = const RangeBandConfig.fixedDomain(5),
        throwsArgumentError,
      );
    });

    test('type of none throws argument exception', () {
      expect(
        () => scale.rangeBandConfig = const RangeBandConfig.none(),
        throwsArgumentError,
      );
    });

    // set to null throws argument exception removed because
    // it is not possible to set to null.

    test('range band size used from style', () {
      final oldStyle = StyleFactory.style;
      StyleFactory.style = TestStyle()..rangeBandSize = 0.4;

      scale.rangeBandConfig = RangeBandConfig.styleAssignedPercent();
      // 100 = 0.4f * 1000pixels / 4domains
      expect(scale.rangeBand, closeTo(100, EPSILON));

      // Restore style for other tests.
      StyleFactory.style = oldStyle;
    });
  });

  group('set step size config', () {
    test('to null does not throw', () {
      scale.stepSizeConfig = null;
    });

    test('to auto does not throw', () {
      scale.stepSizeConfig = const StepSizeConfig.auto();
    });

    test('to fixed domain throw argument exception', () {
      expect(
        () => scale.stepSizeConfig = const StepSizeConfig.fixedDomain(1),
        throwsArgumentError,
      );
    });

    test('to fixed pixel throw argument exception', () {
      expect(
        () => scale.stepSizeConfig = const StepSizeConfig.fixedPixels(1),
        throwsArgumentError,
      );
    });
  });

  group('set range persists', () {
    test('', () {
      expect(scale.range.start, equals(2000));
      expect(scale.range.end, equals(1000));
      expect(scale.range.min, equals(1000));
      expect(scale.range.max, equals(2000));
      expect(scale.rangeWidth, equals(1000));

      expect(scale.isRangeValueWithinViewport(1500), isTrue);
      expect(scale.isRangeValueWithinViewport(1000), isTrue);
      expect(scale.isRangeValueWithinViewport(2000), isTrue);

      expect(scale.isRangeValueWithinViewport(500), isFalse);
      expect(scale.isRangeValueWithinViewport(2500), isFalse);
    });
  });

  group('scale factor', () {
    test('sets horizontally', () {
      scale
        ..range = const ScaleOutputExtent(1000, 2000)
        ..setViewportSettings(2, -700);

      expect(scale.viewportScalingFactor, closeTo(2.0, EPSILON));
      expect(scale.viewportTranslatePx, closeTo(-700.0, EPSILON));
    });

    test('sets vertically', () {
      scale
        ..range = const ScaleOutputExtent(2000, 1000)
        ..setViewportSettings(2, 700);

      expect(scale.viewportScalingFactor, closeTo(2.0, EPSILON));
      expect(scale.viewportTranslatePx, closeTo(700.0, EPSILON));
    });

    test('rangeband is scaled horizontally', () {
      scale
        ..range = const ScaleOutputExtent(1000, 2000)
        ..setViewportSettings(2, -700)
        ..rangeBandConfig = RangeBandConfig.percentOfStep(1);

      expect(scale.rangeBand, closeTo(500.0, EPSILON));
    });

    test('rangeband is scaled vertically', () {
      scale
        ..range = const ScaleOutputExtent(2000, 1000)
        ..setViewportSettings(2, 700)
        ..rangeBandConfig = RangeBandConfig.percentOfStep(1);

      expect(scale.rangeBand, closeTo(500.0, EPSILON));
    });

    test('translate to pixels is scaled horizontally', () {
      scale
        ..rangeBandConfig = RangeBandConfig.percentOfStep(1)
        ..range = const ScaleOutputExtent(1000, 2000)
        ..setViewportSettings(2, -700);

      const scaledStepWidth = 500.0;
      const scaledInitialShift = 250.0;

      expect(scale['a'], closeTo(1000 + scaledInitialShift - 700, EPSILON));

      expect(
        scale['b'],
        closeTo(1000 + scaledInitialShift - 700 + scaledStepWidth, EPSILON),
      );
    });

    test('translate to pixels is scaled vertically', () {
      scale
        ..rangeBandConfig = RangeBandConfig.percentOfStep(1)
        ..range = const ScaleOutputExtent(2000, 1000)
        ..setViewportSettings(2, 700);

      const scaledStepWidth = 500.0;
      const scaledInitialShift = 250.0;

      expect(scale['a'], closeTo(2000 - scaledInitialShift + 700, EPSILON));

      expect(
        scale['b'],
        closeTo(
          2000 - scaledInitialShift + 700 - (scaledStepWidth * 1),
          EPSILON,
        ),
      );
    });

    test('only b and c should be within the viewport horizontally', () {
      scale
        ..rangeBandConfig = RangeBandConfig.percentOfStep(1)
        ..range = const ScaleOutputExtent(1000, 2000)
        ..setViewportSettings(2, -700);

      expect(scale.compareDomainValueToViewport('a'), equals(-1));
      expect(scale.compareDomainValueToViewport('c'), equals(0));
      expect(scale.compareDomainValueToViewport('d'), equals(1));
      expect(scale.compareDomainValueToViewport('f'), isNot(0));
    });

    test('only b and c should be within the viewport vertically', () {
      scale
        ..rangeBandConfig = RangeBandConfig.percentOfStep(1)
        ..range = const ScaleOutputExtent(2000, 1000)
        ..setViewportSettings(2, 700);

      expect(scale.compareDomainValueToViewport('a'), equals(1));
      expect(scale.compareDomainValueToViewport('c'), equals(0));
      expect(scale.compareDomainValueToViewport('d'), equals(-1));
      expect(scale.compareDomainValueToViewport('f'), isNot(0));
    });

    test('applies in reverse horizontally', () {
      scale
        ..range = const ScaleOutputExtent(1000, 2000)
        ..setViewportSettings(2, -700);

      expect(scale.reverse(scale['d']), 'd');
      expect(scale.reverse(scale['b']), 'b');
      expect(scale.reverse(scale['c']), 'c');
      expect(scale.reverse(scale['d']), 'd');
    });

    test('applies in reverse vertically', () {
      scale
        ..range = const ScaleOutputExtent(2000, 1000)
        ..setViewportSettings(2, 700);

      expect(scale.reverse(scale['d']), 'd');
      expect(scale.reverse(scale['b']), 'b');
      expect(scale.reverse(scale['c']), 'c');
      expect(scale.reverse(scale['d']), 'd');
    });
  });

  group('viewport', () {
    test('set adjust scale to show viewport horizontally', () {
      scale
        ..range = const ScaleOutputExtent(1000, 2000)
        ..rangeBandConfig = RangeBandConfig.percentOfStep(0.5)
        ..setViewport(2, 'b');

      expect(scale['a'], closeTo(750, EPSILON));
      expect(scale['b'], closeTo(1250, EPSILON));
      expect(scale['c'], closeTo(1750, EPSILON));
      expect(scale['d'], closeTo(2250, EPSILON));
      expect(scale.compareDomainValueToViewport('a'), equals(-1));
      expect(scale.compareDomainValueToViewport('b'), equals(0));
      expect(scale.compareDomainValueToViewport('c'), equals(0));
      expect(scale.compareDomainValueToViewport('d'), equals(1));
    });

    test('set adjust scale to show viewport vertically', () {
      scale
        ..range = const ScaleOutputExtent(2000, 1000)
        ..rangeBandConfig = RangeBandConfig.percentOfStep(0.5)
        // Bottom up as domain values are usually reversed.
        ..setViewport(2, 'c');

      expect(scale['a'], closeTo(2250, EPSILON));
      expect(scale['b'], closeTo(1750, EPSILON));
      expect(scale['c'], closeTo(1250, EPSILON));
      expect(scale['d'], closeTo(750, EPSILON));
      expect(scale.compareDomainValueToViewport('a'), equals(1));
      expect(scale.compareDomainValueToViewport('b'), equals(0));
      expect(scale.compareDomainValueToViewport('c'), equals(0));
      expect(scale.compareDomainValueToViewport('d'), equals(-1));
    });

    test('illegal to set window size less than one', () {
      expect(() => scale.setViewport(0, 'b'), throwsArgumentError);
    });

    test(
        'set starting value if starting domain is not in domain list '
        'horizontally', () {
      scale
        ..range = const ScaleOutputExtent(1000, 2000)
        ..rangeBandConfig = RangeBandConfig.percentOfStep(0.5)
        ..setViewport(2, 'f');

      expect(scale['a'], closeTo(1250, EPSILON));
      expect(scale['b'], closeTo(1750, EPSILON));
      expect(scale['c'], closeTo(2250, EPSILON));
      expect(scale['d'], closeTo(2750, EPSILON));
    });

    test(
        'set starting value if starting domain is not in domain list '
        'vertically', () {
      scale
        ..range = const ScaleOutputExtent(2000, 1000)
        ..rangeBandConfig = RangeBandConfig.percentOfStep(0.5)
        ..setViewport(2, 'f');

      expect(scale['a'], closeTo(2750, EPSILON));
      expect(scale['b'], closeTo(2250, EPSILON));
      expect(scale['c'], closeTo(1750, EPSILON));
      expect(scale['d'], closeTo(1250, EPSILON));
    });

    test(
        'get size returns number of full steps that fit scale range '
        'horizontally ', () {
      scale
        ..range = const ScaleOutputExtent(1000, 2000)
        ..setViewportSettings(2, 0);
      expect(scale.viewportDataSize, equals(2));

      scale.setViewportSettings(5, 0);
      expect(scale.viewportDataSize, equals(0));
    });

    test(
        'get size returns number of full steps that fit scale range '
        'vertically ', () {
      scale
        ..range = const ScaleOutputExtent(2000, 1000)
        ..setViewportSettings(2, 0);
      expect(scale.viewportDataSize, equals(2));

      scale.setViewportSettings(5, 0);
      expect(scale.viewportDataSize, equals(0));
    });

    test('get starting viewport gets first fully visible domain horizontally',
        () {
      scale
        ..range = const ScaleOutputExtent(1000, 2000)
        ..setViewportSettings(2, -500);
      expect(scale.viewportStartingDomain, equals('b'));

      scale.setViewportSettings(2, -100);
      expect(scale.viewportStartingDomain, equals('b'));
    });

    test('get starting viewport gets first fully visible domain vertically',
        () {
      scale
        ..range = const ScaleOutputExtent(2000, 1000)
        ..setViewportSettings(2, 500);
      expect(scale.viewportStartingDomain, equals('c'));

      scale.setViewportSettings(2, 500);
      expect(scale.viewportStartingDomain, equals('c'));
    });
  });
}
