// Copyright 2018 the Charts project authors. Please see the AUTHORS file
// for details.
//
// Licensed under the Apache License, Version 2.0 (the "License");
// you may not use this file except in compliance with the License.
// You may obtain a copy of the License at
//
// http://www.apache.org/licenses/LICENSE-2.0
//
// Unless required by applicable law or agreed to in writing, software
// distributed under the License is distributed on an "AS IS" BASIS,
// WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
// See the License for the specific language governing permissions and
// limitations under the License.

import 'dart:math' show Point, Rectangle, min, sqrt;

import 'package:meta/meta.dart' show protected;
import 'package:nimble_charts_common/src/chart/common/chart_canvas.dart'
    show ChartCanvas, FillPatternType;
import 'package:nimble_charts_common/src/common/color.dart' show Color;
import 'package:nimble_charts_common/src/common/style/style_factory.dart'
    show StyleFactory;

/// Strategy for rendering a symbol.
abstract class BaseSymbolRenderer {
  bool shouldRepaint(covariant BaseSymbolRenderer oldRenderer);
}

/// Strategy for rendering a symbol bounded within a box.
abstract class SymbolRenderer extends BaseSymbolRenderer {
  SymbolRenderer({required this.isSolid});

  /// Whether the symbol should be rendered as a solid shape, or a hollow shape.
  ///
  /// If this is true, then fillColor and strokeColor will be used to fill in
  /// the shape, and draw a border, respectively. The stroke (border) will only
  /// be visible if a non-zero strokeWidthPx is configured.
  ///
  /// If this is false, then the shape will be filled in with a white color
  /// (overriding fillColor). strokeWidthPx will default to 2 if none was
  /// configured.
  final bool isSolid;

  void paint(
    ChartCanvas canvas,
    Rectangle<num> bounds, {
    List<int>? dashPattern,
    Color? fillColor,
    FillPatternType? fillPattern,
    Color? strokeColor,
    double? strokeWidthPx,
  });

  @protected
  double? getSolidStrokeWidthPx(double? strokeWidthPx) =>
      isSolid ? strokeWidthPx : (strokeWidthPx ?? 2.0);

  @protected
  Color? getSolidFillColor(Color? fillColor) =>
      isSolid ? fillColor : StyleFactory.style.white;

  @override
  bool operator ==(Object other) =>
      other is SymbolRenderer && other.isSolid == isSolid;

  @override
  int get hashCode => isSolid.hashCode;
}

/// Strategy for rendering a symbol centered around a point.
///
/// An optional second point can describe an extended symbol.
abstract class PointSymbolRenderer extends BaseSymbolRenderer {
  void paint(
    ChartCanvas canvas,
    Point<double> p1,
    double radius, {
    required Point<double> p2,
    Color? fillColor,
    Color? strokeColor,
  });
}

/// Rounded rectangular symbol with corners having [radius].
class RoundedRectSymbolRenderer extends SymbolRenderer {
  RoundedRectSymbolRenderer({super.isSolid = true, double? radius})
      : radius = radius ?? 1.0;
  final double radius;

  @override
  void paint(
    ChartCanvas canvas,
    Rectangle<num> bounds, {
    List<int>? dashPattern,
    Color? fillColor,
    FillPatternType? fillPattern,
    Color? strokeColor,
    double? strokeWidthPx,
  }) {
    canvas.drawRRect(
      bounds,
      fill: getSolidFillColor(fillColor),
      fillPattern: fillPattern,
      stroke: strokeColor,
      radius: radius,
      roundTopLeft: true,
      roundTopRight: true,
      roundBottomRight: true,
      roundBottomLeft: true,
    );
  }

  @override
  bool shouldRepaint(RoundedRectSymbolRenderer oldRenderer) =>
      this != oldRenderer;

  @override
  bool operator ==(Object other) =>
      other is RoundedRectSymbolRenderer &&
      other.radius == radius &&
      super == other;

  @override
  int get hashCode {
    var hashcode = super.hashCode;
    hashcode = (hashcode * 37) + radius.hashCode;
    return hashcode;
  }
}

/// Line symbol renderer.
class LineSymbolRenderer extends SymbolRenderer {
  LineSymbolRenderer({
    List<int>? dashPattern,
    super.isSolid = true,
    double? strokeWidth,
  })  : strokeWidth = strokeWidth ?? strokeWidthForRoundEndCaps,
        _dashPattern = dashPattern;
  static const roundEndCapsPixels = 2;
  static const minLengthToRoundCaps = (roundEndCapsPixels * 2) + 1;
  static const strokeWidthForRoundEndCaps = 4.0;
  static const strokeWidthForNonRoundedEndCaps = 2.0;

  /// Thickness of the line stroke.
  final double strokeWidth;

  /// Dash pattern for the line.
  final List<int>? _dashPattern;

  @override
  void paint(
    ChartCanvas canvas,
    Rectangle<num> bounds, {
    List<int>? dashPattern,
    Color? fillColor,
    FillPatternType? fillPattern,
    Color? strokeColor,
    double? strokeWidthPx,
  }) {
    final centerHeight = (bounds.bottom - bounds.top) / 2;

    // If we have a dash pattern, do not round the end caps, and set
    // strokeWidthPx to a smaller value. Using round end caps makes smaller
    // patterns blurry.
    final localDashPattern = dashPattern ?? _dashPattern;
    final roundEndCaps = localDashPattern == null;

    // If we have a dash pattern, the normal stroke width makes them look
    // strangely tall.
    final localStrokeWidthPx = localDashPattern == null
        ? getSolidStrokeWidthPx(strokeWidthPx ?? strokeWidth)
        : strokeWidthForNonRoundedEndCaps;

    // Adjust the length so the total width includes the rounded pixels.
    // Otherwise the cap is drawn past the bounds and appears to be cut off.
    // If bounds is not long enough to accommodate the line, do not adjust.
    var left = bounds.left;
    var right = bounds.right;

    if (roundEndCaps && bounds.width >= minLengthToRoundCaps) {
      left += roundEndCapsPixels;
      right -= roundEndCapsPixels;
    }

    // TODO: Pass in strokeWidth, roundEndCaps, and dashPattern from
    // line renderer config.
    canvas.drawLine(
      points: [Point(left, centerHeight), Point(right, centerHeight)],
      dashPattern: localDashPattern,
      fill: getSolidFillColor(fillColor),
      roundEndCaps: roundEndCaps,
      stroke: strokeColor,
      strokeWidthPx: localStrokeWidthPx,
    );
  }

  @override
  bool shouldRepaint(LineSymbolRenderer oldRenderer) => this != oldRenderer;

  @override
  bool operator ==(Object other) =>
      other is LineSymbolRenderer &&
      other.strokeWidth == strokeWidth &&
      super == other;

  @override
  int get hashCode {
    var hashcode = super.hashCode;
    hashcode = (hashcode * 37) + strokeWidth.hashCode;
    return hashcode;
  }
}

/// Circle symbol renderer.
class CircleSymbolRenderer extends SymbolRenderer {
  CircleSymbolRenderer({super.isSolid = true});

  @override
  void paint(
    ChartCanvas canvas,
    Rectangle<num> bounds, {
    List<int>? dashPattern,
    Color? fillColor,
    FillPatternType? fillPattern,
    Color? strokeColor,
    double? strokeWidthPx,
  }) {
    final center = Point(
      bounds.left + (bounds.width / 2),
      bounds.top + (bounds.height / 2),
    );
    final radius = min(bounds.width, bounds.height) / 2;
    canvas.drawPoint(
      point: center,
      radius: radius,
      fill: getSolidFillColor(fillColor),
      stroke: strokeColor,
      strokeWidthPx: getSolidStrokeWidthPx(strokeWidthPx),
    );
  }

  @override
  bool shouldRepaint(CircleSymbolRenderer oldRenderer) => this != oldRenderer;

  @override
  bool operator ==(Object other) =>
      other is CircleSymbolRenderer && super == other;

  @override
  int get hashCode {
    var hashcode = super.hashCode;
    hashcode = (hashcode * 37) + runtimeType.hashCode;
    return hashcode;
  }
}

/// Rectangle symbol renderer.
class RectSymbolRenderer extends SymbolRenderer {
  RectSymbolRenderer({super.isSolid = true});

  @override
  void paint(
    ChartCanvas canvas,
    Rectangle<num> bounds, {
    List<int>? dashPattern,
    Color? fillColor,
    FillPatternType? fillPattern,
    Color? strokeColor,
    double? strokeWidthPx,
  }) {
    canvas.drawRect(
      bounds,
      fill: getSolidFillColor(fillColor),
      stroke: strokeColor,
      strokeWidthPx: getSolidStrokeWidthPx(strokeWidthPx),
    );
  }

  @override
  bool shouldRepaint(RectSymbolRenderer oldRenderer) => this != oldRenderer;

  @override
  bool operator ==(Object other) =>
      other is RectSymbolRenderer && super == other;

  @override
  int get hashCode {
    var hashcode = super.hashCode;
    hashcode = (hashcode * 37) + runtimeType.hashCode;
    return hashcode;
  }
}

/// This [SymbolRenderer] renders an upward pointing equilateral triangle.
class TriangleSymbolRenderer extends SymbolRenderer {
  TriangleSymbolRenderer({super.isSolid = true});

  @override
  void paint(
    ChartCanvas canvas,
    Rectangle<num> bounds, {
    List<int>? dashPattern,
    Color? fillColor,
    FillPatternType? fillPattern,
    Color? strokeColor,
    double? strokeWidthPx,
  }) {
    // To maximize the size of the triangle in the available space, we can use
    // the width as the length of each size. Set the bottom edge to be the full
    // width, and then calculate the height based on the 30/60/90 degree right
    // triangle whose tall side is the height of our equilateral triangle.
    final dy = sqrt(3) / 2 * bounds.width;
    final centerX = (bounds.left + bounds.right) / 2;
    canvas.drawPolygon(
      points: [
        Point(bounds.left, bounds.top + dy),
        Point(bounds.right, bounds.top + dy),
        Point(centerX, bounds.top),
      ],
      fill: getSolidFillColor(fillColor),
      stroke: strokeColor,
      strokeWidthPx: getSolidStrokeWidthPx(strokeWidthPx),
    );
  }

  @override
  bool shouldRepaint(TriangleSymbolRenderer oldRenderer) => this != oldRenderer;

  @override
  // ignore: hash_and_equals
  bool operator ==(Object other) =>
      other is TriangleSymbolRenderer && super == other;
}

/// Draws a cylindrical shape connecting two points.
class CylinderSymbolRenderer extends PointSymbolRenderer {
  CylinderSymbolRenderer();

  @override
  void paint(
    ChartCanvas canvas,
    Point<double> p1,
    double radius, {
    required Point<double> p2,
    Color? fillColor,
    Color? strokeColor,
    double? strokeWidthPx,
  }) {
    final adjustedP1 = Point<double>(p1.x, p1.y);
    final adjustedP2 = Point<double>(p2.x, p2.y);

    canvas.drawLine(
      points: [adjustedP1, adjustedP2],
      stroke: strokeColor,
      roundEndCaps: true,
      strokeWidthPx: radius * 2,
    );
  }

  @override
  bool shouldRepaint(CylinderSymbolRenderer oldRenderer) => this != oldRenderer;

  @override
  bool operator ==(Object other) => other is CylinderSymbolRenderer;

  @override
  int get hashCode => runtimeType.hashCode;
}

/// Draws a rectangular shape connecting two points.
class RectangleRangeSymbolRenderer extends PointSymbolRenderer {
  RectangleRangeSymbolRenderer();

  @override
  void paint(
    ChartCanvas canvas,
    Point<double> p1,
    double radius, {
    required Point<double> p2,
    Color? fillColor,
    Color? strokeColor,
    double? strokeWidthPx,
  }) {
    final adjustedP1 = Point<double>(p1.x, p1.y);
    final adjustedP2 = Point<double>(p2.x, p2.y);

    canvas.drawLine(
      points: [adjustedP1, adjustedP2],
      stroke: strokeColor,
      roundEndCaps: false,
      strokeWidthPx: radius * 2,
    );
  }

  @override
  bool shouldRepaint(RectangleRangeSymbolRenderer oldRenderer) =>
      this != oldRenderer;

  @override
  bool operator ==(Object other) => other is RectangleRangeSymbolRenderer;

  @override
  int get hashCode => runtimeType.hashCode;
}
