// Copyright 2018 the Charts project authors. Please see the AUTHORS file
// for details.
//
// Licensed under the Apache License, Version 2.0 (the "License");
// you may not use this file except in compliance with the License.
// You may obtain a copy of the License at
//
// http://www.apache.org/licenses/LICENSE-2.0
//
// Unless required by applicable law or agreed to in writing, software
// distributed under the License is distributed on an "AS IS" BASIS,
// WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
// See the License for the specific language governing permissions and
// limitations under the License.

import 'dart:collection' show LinkedHashMap;
import 'dart:math' show Rectangle, max;

import 'package:nimble_charts_common/src/chart/cartesian/axis/axis.dart'
    show ImmutableAxis;
import 'package:nimble_charts_common/src/chart/cartesian/cartesian_chart.dart'
    show CartesianChart;
import 'package:nimble_charts_common/src/chart/common/base_chart.dart'
    show BaseChart;
import 'package:nimble_charts_common/src/chart/common/chart_canvas.dart'
    show ChartCanvas;
import 'package:nimble_charts_common/src/chart/common/processed_series.dart'
    show ImmutableSeries, MutableSeries;
import 'package:nimble_charts_common/src/chart/layout/layout_view.dart'
    show
        LayoutPosition,
        LayoutView,
        LayoutViewConfig,
        LayoutViewPaintOrder,
        LayoutViewPositionOrder,
        ViewMeasuredSizes;
import 'package:nimble_charts_common/src/chart/scatter_plot/point_renderer.dart'
    show AnimatedPoint, DatumPoint, PointRenderer;
import 'package:nimble_charts_common/src/chart/scatter_plot/symbol_annotation_renderer_config.dart'
    show SymbolAnnotationRendererConfig;
import 'package:nimble_charts_common/src/common/graphics_factory.dart'
    show GraphicsFactory;

/// Series renderer which draws a row of symbols for each series below the
/// drawArea but above the bottom axis.
///
/// This renderer can draw point annotations and range annotations. Point
/// annotations are drawn at the location of the domain along the chart's domain
/// axis, in the row for its series. Range annotations are drawn as a range
/// shape between the domainLowerBound and domainUpperBound positions along the
/// chart's domain axis. Point annotations are drawn on top of range
/// annotations.
///
/// Limitations:
/// Does not handle horizontal bars.
class SymbolAnnotationRenderer<D> extends PointRenderer<D>
    implements LayoutView {
  SymbolAnnotationRenderer({
    String? rendererId,
    SymbolAnnotationRendererConfig<D>? config,
  }) : super(rendererId: rendererId ?? 'symbolAnnotation', config: config);
  late Rectangle<int> _componentBounds;

  @override
  GraphicsFactory? graphicsFactory;

  late CartesianChart<D> _chart;

  var _currentHeight = 0;

  // ignore: prefer_collection_literals, https://github.com/dart-lang/linter/issues/1649
  final _seriesInfo = LinkedHashMap<String, _SeriesInfo<D>>();

  //
  // Renderer methods
  //
  /// Symbol annotations do not use any measure axes, or draw anything in the
  /// main draw area associated with them.
  @override
  void configureMeasureAxes(List<MutableSeries<D>> seriesList) {}

  @override
  void preprocessSeries(List<MutableSeries<D>> seriesList) {
    final localConfig = config as SymbolAnnotationRendererConfig;

    _seriesInfo.clear();

    var offset = 0.0;

    for (final series in seriesList) {
      final seriesKey = series.id;

      // Default to the configured radius if none was defined by the series.
      series.radiusPxFn ??= (_) => config.radiusPx;

      var maxRadius = 0.0;
      for (var index = 0; index < series.data.length; index++) {
        // Default to the configured radius if none was returned by the
        // accessor function.
        var radiusPx = series.radiusPxFn?.call(index)?.toDouble();
        radiusPx ??= config.radiusPx;

        maxRadius = max(maxRadius, radiusPx);
      }

      final rowInnerHeight = maxRadius * 2;

      final rowHeight = localConfig.verticalSymbolBottomPaddingPx +
          localConfig.verticalSymbolTopPaddingPx +
          rowInnerHeight;

      final symbolCenter = offset +
          localConfig.verticalSymbolTopPaddingPx +
          (rowInnerHeight / 2);

      series.measureFn = (index) => 0;
      // ignore: cascade_invocations
      series.measureOffsetFn = (index) => 0;

      // Override the key function to allow for range annotations that start at
      // the same point. This is a necessary hack because every annotation has a
      // measure value of 0, so the key generated in [PointRenderer] is not
      // unique enough.
      // ignore: cascade_invocations
      series.keyFn ??= (index) => '${series.id}__${series.domainFn(index)}__'
          '${series.domainLowerBoundFn!(index)}__'
          '${series.domainUpperBoundFn!(index)}';

      _seriesInfo[seriesKey] = _SeriesInfo<D>(
        rowHeight: rowHeight,
        rowStart: offset,
        symbolCenter: symbolCenter,
      );

      offset += rowHeight;
    }

    _currentHeight = offset.ceil();

    super.preprocessSeries(seriesList);
  }

  @override
  DatumPoint<D> getPoint(
    Object? datum,
    D? domainValue,
    D? domainLowerBoundValue,
    D? domainUpperBoundValue,
    ImmutableSeries<D> series,
    ImmutableAxis<D> domainAxis,
    num? measureValue,
    num? measureLowerBoundValue,
    num? measureUpperBoundValue,
    num? measureOffsetValue,
    ImmutableAxis<num> measureAxis,
  ) {
    final domainPosition = domainAxis.getLocation(domainValue);

    final domainLowerBoundPosition = domainLowerBoundValue != null
        ? domainAxis.getLocation(domainLowerBoundValue)
        : null;

    final domainUpperBoundPosition = domainUpperBoundValue != null
        ? domainAxis.getLocation(domainUpperBoundValue)
        : null;

    final seriesKey = series.id;
    final seriesInfo = _seriesInfo[seriesKey]!;

    final measurePosition = _componentBounds.top + seriesInfo.symbolCenter;

    final measureLowerBoundPosition =
        domainLowerBoundPosition != null ? measurePosition : null;

    final measureUpperBoundPosition =
        domainUpperBoundPosition != null ? measurePosition : null;

    return DatumPoint<D>(
      datum: datum,
      domain: domainValue,
      series: series,
      x: domainPosition,
      xLower: domainLowerBoundPosition,
      xUpper: domainUpperBoundPosition,
      y: measurePosition,
      yLower: measureLowerBoundPosition,
      yUpper: measureUpperBoundPosition,
    );
  }

  @override
  void onAttach(BaseChart<D> chart) {
    if (chart is! CartesianChart<D>) {
      throw ArgumentError(
        'SymbolAnnotationRenderer can only be attached to a CartesianChart<D>',
      );
    }

    _chart = chart;

    // Only vertical rendering is supported by this behavior.
    assert(_chart.vertical);

    super.onAttach(chart);
    _chart.addView(this);
  }

  @override
  void onDetach(BaseChart<D> chart) {
    chart.removeView(this);
  }

  @override
  void paint(ChartCanvas canvas, double animationPercent) {
    super.paint(canvas, animationPercent);

    // Use the domain axis of the attached chart to render the separator lines
    // to keep the same overall style.
    if ((config as SymbolAnnotationRendererConfig).showSeparatorLines) {
      seriesPointMap.forEach((key, points) {
        final seriesInfo = _seriesInfo[key]!;

        final y = componentBounds.top + seriesInfo.rowStart;

        final domainAxis = _chart.domainAxis!;
        final bounds = Rectangle<int>(
          componentBounds.left,
          y.round(),
          componentBounds.width,
          0,
        );
        domainAxis.tickDrawStrategy!
            .drawAxisLine(canvas, domainAxis.axisOrientation!, bounds);
      });
    }
  }

  //
  // Layout methods
  //

  @override
  LayoutViewConfig get layoutConfig => LayoutViewConfig(
        paintOrder: LayoutViewPaintOrder.point,
        position: LayoutPosition.Bottom,
        positionOrder: LayoutViewPositionOrder.symbolAnnotation,
      );

  @override
  ViewMeasuredSizes measure(int maxWidth, int maxHeight) =>
      // The sizing of component is not flexible. It's height is always a
      // multiple of the number of series rendered, even if that ends up taking
      // all of the available margin space.
      ViewMeasuredSizes(
        preferredWidth: maxWidth,
        preferredHeight: _currentHeight,
      );

  @override
  void layout(Rectangle<int> componentBounds, Rectangle<int> drawAreaBounds) {
    _componentBounds = componentBounds;

    super.layout(componentBounds, drawAreaBounds);
  }

  @override
  Rectangle<int> get componentBounds => _componentBounds;
}

class _SeriesInfo<D> {
  _SeriesInfo({
    required this.rowHeight,
    required this.rowStart,
    required this.symbolCenter,
  });
  double rowHeight;
  double rowStart;
  double symbolCenter;
}
