import 'package:flutter/material.dart';
import 'package:navigation_builder/navigation_builder.dart';

// This is an example of declarative navigation using NavigationBuilder.setRouteStack
//
void main() {
  runApp(const BooksApp());
}

List<Book> books = [
  Book('Left Hand of Darkness', 'Ursula K. Le Guin'),
  Book('Too Like the Lightning', 'Ada Palmer'),
  Book('Kindred', 'Octavia E. Butler'),
];

class Book {
  final String title;
  final String author;

  Book(this.title, this.author);
}

final navigator = NavigationBuilder.create(
  // At least one route must be defined. This is the home page of the app
  // In this example deep link wont't work (See the next example)
  routes: {
    '/': (data) => BooksListScreen(books: books),
  },
);

class BooksApp extends StatelessWidget {
  const BooksApp({Key? key}) : super(key: key);

  @override
  Widget build(BuildContext context) {
    return MaterialApp.router(
      title: 'Books App',
      routerConfig: navigator.routerConfig,
    );
  }
}

class BooksListScreen extends StatelessWidget {
  final List<Book> books;

  const BooksListScreen({
    Key? key,
    required this.books,
  }) : super(key: key);

  @override
  Widget build(BuildContext context) {
    return Scaffold(
      appBar: AppBar(),
      body: ListView(
        children: [
          for (var book in books)
            ListTile(
              title: Text(book.title),
              subtitle: Text(book.author),
              onTap: () {
                final bookId = books.indexOf(book);
                navigator.setRouteStack(
                  (pages) {
                    return [
                      PageSettings(
                        name: '/books',
                        child: BooksListScreen(books: books),
                      ),
                      if (bookId < books.length)
                        PageSettings(
                          name: '/books/$bookId',
                          child: BookDetailsScreen(book: books[bookId]),
                        )
                      else
                        const PageSettings(
                          name: '/404',
                          child: UnknownScreen(),
                        ),
                    ];
                  },
                );
              },
            )
        ],
      ),
    );
  }
}

class BookDetailsScreen extends StatelessWidget {
  final Book book;
  const BookDetailsScreen({
    required this.book,
    Key? key,
  }) : super(key: key);

  @override
  Widget build(BuildContext context) {
    return Scaffold(
      appBar: AppBar(),
      body: Padding(
        padding: const EdgeInsets.all(8.0),
        child: Column(
          crossAxisAlignment: CrossAxisAlignment.start,
          children: [
            Text(book.title, style: Theme.of(context).textTheme.headline6),
            Text(book.author, style: Theme.of(context).textTheme.subtitle1),
          ],
        ),
      ),
    );
  }
}

class UnknownScreen extends StatelessWidget {
  const UnknownScreen({Key? key}) : super(key: key);

  @override
  Widget build(BuildContext context) {
    return Scaffold(
      appBar: AppBar(),
      body: const Center(
        child: Text('404!'),
      ),
    );
  }
}
