---
title: Fade-In GPT Content
---
ChatGPT-style large language models (LLM) produces text and content in snippets.
The answer to a single question might includes 50 pieces of information that are
generated by the LLM. The chat-style experience adds each new piece to the existing
answer, making it appear that the AI is "writing" the answer in real-time.

It's common in these experiences to fade-in the content as its generated.
Super Editor includes a couple of tools that can be used to achieve this
effect, when feeding LLM content to a `SuperEditor` or `SuperReader`.

## How to Fade-In Content
The following example demonstrates how to fade content into a `SuperReader`.
The `SuperReader` widget is typically used for LLMs because LLM output isn't
typically meant to be editable.

First, create a `SuperReader` with a `Document` that will hold the LLM output.
Include a `FadeInStyler`, which will control the opacity of new content.

```dart
class MyState extends State<MyWidget> {
  late final MutableDocument _document;
  late final MutableDocumentComposer _composer;
  late final Editor _editor;
  late final FadeInStyler _fadeInStylePhase;

  @override
  void initState() {
    super.initState();

    _document = MutableDocument.empty();
    _composer = MutableDocumentComposer(
      initialSelection: DocumentSelection.collapsed(
        position: DocumentPosition(
          nodeId: _document.first.id,
          nodePosition: TextNodePosition(offset: 0),
        ),
      ),
    );
    _editor = createDefaultDocumentEditor(document: _document, composer: _composer);
    _fadeInStylePhase = FadeInStyler(this);
  }

  @override
  void dispose() {
    _fadeInStylePhase.dispose();
    _composer.dispose();
    _editor.dispose();
    _document.dispose();

    super.dispose();
  }

  @override
  Widget build(BuildContext context) {
    return SuperReader(
      editor: _editor,
      customStylePhases: [
        _fadeInStylePhase,
      ],
    );
  }
}
```

With a `SuperReader` in place, take the output from an LLM and feed it into the
`Editor` however you'd like. The way you feed content into the `Editor` depends on 
the format of the output from the LLM. You can feed the LLM text into the `Editor`
as plain text, or you can parse the LLM text into styled `AttributedText` and
feed that to the `Editor`, or you can also parse out entire blocks like images and
horizontal rules from the LLM and insert those as nodes in the `Editor`.

The most important part about feeding content from the LLM to the `Editor` is to
include the current timestamp when inserted. This timestamp is used by the `FadeInStyler`
to choose the opacity for the content.

```dart
// Insert plain text.
_editor.execute([
  InsertPlainTextAtEndOfDocumentRequest(
    "...this is a snippet from the LLM...",
    createdAt: DateTime.now(),
  ),
]);

// Insert styled text.
_editor.execute([
  InsertStyledTextAtEndOfDocumentRequest(
    AttributedText(
      "...this is a bold snippet...",
      AttributedSpans(attributions: [
        const SpanMarker(
          attribution: boldAttribution,
          offset: 0,
          markerType: SpanMarkerType.start,
        ),
        const SpanMarker(
          attribution: boldAttribution,
          offset: 27,
          markerType: SpanMarkerType.end,
        ),
      ]),
    ),
    createdAt: DateTime.now(),
  ),
]);

// Insert a block node.
_editor.execute([
  InsertNodeAtEndOfDocumentRequest(
    ImageNode( 
      id: Editor.createNodeId(),
      imageUrl: "https://somewhere.com/image.png",
      metadata: {
        NodeMetadata.createdAt: DateTime.now(),
      },
    ),
  ),
]);
```

With a `FadeInStyler` plugged into `SuperReader`, and content insertions that
include a `CreatedAtAttribution`, the inserted content will fade-in within the
`SuperReader` widget.

## How to Control the Fade-In
When using the `FadeInStyler` to fade-in content, the fade-in effect is configurable.

```dart
// Customize fade-in times and curves.
final styler = FadeInStyler(
  tickerProvider,
  blockNodeFadeInDuration: const Duration(milliseconds: 1500),
  textSnippetFadeInDuration: const Duration(milliseconds: 250),
  fadeCurve: Curves.easeInOut,
);
```

The `FadeInStyler` supports different timing for text snippets vs blocks, like images.
The fade animation curve is also configurable.
