/*
 * Infomaniak Core - Android
 * Copyright (C) 2025 Infomaniak Network SA
 *
 * This program is free software: you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation, either version 3 of the License, or
 * (at your option) any later version.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with this program.  If not, see <http://www.gnu.org/licenses/>.
 */
package com.infomaniak.core.crossapplogin.front.components

import androidx.compose.foundation.layout.PaddingValues
import androidx.compose.foundation.layout.Row
import androidx.compose.foundation.layout.Spacer
import androidx.compose.foundation.layout.fillMaxWidth
import androidx.compose.foundation.layout.heightIn
import androidx.compose.foundation.layout.padding
import androidx.compose.foundation.layout.width
import androidx.compose.material3.ExperimentalMaterial3Api
import androidx.compose.material3.Icon
import androidx.compose.material3.LocalRippleConfiguration
import androidx.compose.material3.MaterialTheme
import androidx.compose.material3.RippleConfiguration
import androidx.compose.material3.Surface
import androidx.compose.material3.TextButton
import androidx.compose.runtime.Composable
import androidx.compose.runtime.CompositionLocalProvider
import androidx.compose.ui.Alignment
import androidx.compose.ui.Modifier
import androidx.compose.ui.graphics.RectangleShape
import androidx.compose.ui.semantics.semantics
import androidx.compose.ui.semantics.toggleableState
import androidx.compose.ui.state.ToggleableState
import androidx.compose.ui.tooling.preview.Preview
import androidx.compose.ui.tooling.preview.PreviewParameter
import com.infomaniak.core.compose.basics.Dimens
import com.infomaniak.core.compose.margin.Margin
import com.infomaniak.core.crossapplogin.back.ExternalAccount
import com.infomaniak.core.crossapplogin.front.data.CrossLoginCustomization
import com.infomaniak.core.crossapplogin.front.data.CrossLoginDefaults
import com.infomaniak.core.crossapplogin.front.icons.Checkmark
import com.infomaniak.core.crossapplogin.front.previews.AccountsPreviewParameter

@OptIn(ExperimentalMaterial3Api::class)
@Composable
internal fun BottomSheetItem(
    account: ExternalAccount,
    customization: CrossLoginCustomization,
    isSelected: () -> Boolean,
    contentPadding: PaddingValues = PaddingValues(horizontal = Margin.Medium),
    onClick: () -> Unit,
) {
    CompositionLocalProvider(
        LocalRippleConfiguration provides RippleConfiguration(color = customization.colors.primaryColor)
    ) {
        TextButton(
            modifier = Modifier
                .fillMaxWidth()
                .heightIn(min = Dimens.buttonHeight)
                .semantics { toggleableState = if (isSelected()) ToggleableState.On else ToggleableState.Off },
            shape = RectangleShape,
            onClick = onClick,
            contentPadding = contentPadding,
        ) {
            Row(
                modifier = Modifier
                    .fillMaxWidth()
                    .padding(vertical = Margin.Mini),
                verticalAlignment = Alignment.CenterVertically,
            ) {

                SingleAccount(account, customization, Modifier.weight(1.0f))

                Spacer(Modifier.width(Margin.Mini))

                if (isSelected()) {
                    Icon(
                        imageVector = Checkmark,
                        tint = customization.colors.primaryColor,
                        contentDescription = null,
                    )
                }
            }
        }
    }
}

@Preview
@Composable
private fun Preview(@PreviewParameter(AccountsPreviewParameter::class) accounts: List<ExternalAccount>) {
    MaterialTheme {
        Surface {
            BottomSheetItem(
                account = accounts.first(),
                customization = CrossLoginDefaults.customize(),
                isSelected = { true },
                onClick = {},
            )
        }
    }
}
