/*
 * Infomaniak kDrive - Android
 * Copyright (C) 2022-2025 Infomaniak Network SA
 *
 * This program is free software: you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation, either version 3 of the License, or
 * (at your option) any later version.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with this program.  If not, see <http://www.gnu.org/licenses/>.
 */
package com.infomaniak.drive.ui.dropbox

import android.os.Bundle
import android.view.View
import androidx.core.view.isGone
import androidx.navigation.fragment.findNavController
import androidx.navigation.fragment.navArgs
import com.infomaniak.core.legacy.utils.hideProgressCatching
import com.infomaniak.core.legacy.utils.initProgress
import com.infomaniak.core.legacy.utils.showProgressCatching
import com.infomaniak.core.network.models.ApiResponse
import com.infomaniak.core.network.utils.ApiErrorCode.Companion.translateError
import com.infomaniak.drive.R
import com.infomaniak.drive.data.cache.FileController
import com.infomaniak.drive.data.models.DropBox
import com.infomaniak.drive.data.models.File
import com.infomaniak.drive.utils.Utils
import com.infomaniak.drive.utils.showSnackbar

class ConvertToDropboxFragment : ManageDropboxFragment() {

    private val navigationArgs: ManageDropboxFragmentArgs by navArgs()
    override var isManageDropBox = false

    override fun onViewCreated(view: View, savedInstanceState: Bundle?): Unit = with(binding) {
        super.onViewCreated(view, savedInstanceState)

        fileShareCollapsingToolbarLayout.title = getString(R.string.convertToDropboxTitle, navigationArgs.fileName)

        shareLinkCardView.isGone = true
        disableButton.isGone = true

        val file = FileController.getFileById(navigationArgs.fileId) ?: return

        updateUi(file)

        settings.expirationDateInput.init(fragmentManager = parentFragmentManager)

        setupSaveButton(file)
    }

    private fun setupSaveButton(file: File) = with(binding) {
        enableSaveButton()
        saveButton.initProgress(viewLifecycleOwner)
        saveButton.setOnClickListener {
            settings.apply {
                val limitFileSize = if (limitStorageSwitch.isChecked) {
                    limitStorageValue.text.toString().toDoubleOrNull()
                } else {
                    null
                }

                saveButton.showProgressCatching()
                dropboxViewModel.createDropBoxFolder(
                    file = file,
                    emailWhenFinished = emailWhenFinishedSwitch.isChecked,
                    password = if (passwordSwitch.isChecked) passwordTextInput.text?.toString() else null,
                    limitFileSize = limitFileSize?.let { Utils.convertGigaByteToBytes(it) },
                    validUntil = if (expirationDateSwitch.isChecked) expirationDateInput.getCurrentTimestampValue() else null
                ).observe(viewLifecycleOwner) { apiResponse -> onDropboxCreated(apiResponse) }
            }
        }
    }

    private fun onDropboxCreated(apiResponse: ApiResponse<DropBox>) {
        if (apiResponse.isSuccess()) {
            dropboxViewModel.createDropBoxSuccess.value = apiResponse.data
            findNavController().popBackStack()
        } else {
            showSnackbar(apiResponse.translateError())
        }
        binding.saveButton.hideProgressCatching(R.string.buttonSave)
    }
}
