package com.imcys.bilibilias

import android.content.Intent
import android.os.Build
import android.os.Bundle
import android.view.WindowManager
import androidx.activity.ComponentActivity
import androidx.activity.compose.setContent
import androidx.activity.enableEdgeToEdge
import androidx.activity.result.ActivityResult
import androidx.activity.result.contract.ActivityResultContracts
import androidx.compose.foundation.layout.Box
import androidx.compose.material3.AlertDialog
import androidx.compose.material3.SnackbarDuration
import androidx.compose.material3.SnackbarHost
import androidx.compose.material3.SnackbarHostState
import androidx.compose.material3.SnackbarResult
import androidx.compose.material3.Text
import androidx.compose.runtime.Composable
import androidx.compose.runtime.LaunchedEffect
import androidx.compose.runtime.getValue
import androidx.compose.runtime.mutableStateOf
import androidx.compose.runtime.remember
import androidx.compose.runtime.setValue
import androidx.compose.ui.Alignment
import androidx.compose.ui.Modifier
import androidx.compose.ui.res.stringResource
import androidx.compose.ui.tooling.preview.Preview
import androidx.lifecycle.lifecycleScope
import com.imcys.bilibilias.common.event.AnalysisEvent
import com.imcys.bilibilias.common.event.sendAnalysisEvent
import com.imcys.bilibilias.common.utils.Manufacturers.XIAOMI
import com.imcys.bilibilias.common.utils.createDownloadNotificationChannel
import com.imcys.bilibilias.data.repository.AppSettingsRepository
import com.imcys.bilibilias.datastore.AppSettings
import com.imcys.bilibilias.ui.BILIBILIASAppScreen
import com.imcys.bilibilias.ui.theme.BILIBILIASTheme
import kotlinx.coroutines.Dispatchers
import kotlinx.coroutines.flow.Flow
import kotlinx.coroutines.flow.MutableStateFlow
import kotlinx.coroutines.launch
import org.koin.android.ext.android.inject
import androidx.compose.runtime.collectAsState
import com.imcys.bilibilias.common.data.CommonBuildConfig
import com.imcys.bilibilias.common.utils.analyticsSafe
import com.imcys.bilibilias.common.utils.baiduAnalyticsSafe
import com.imcys.bilibilias.ui.weight.ASTextButton

class MainActivity : ComponentActivity() {
    private val appSettingsRepository: AppSettingsRepository by inject()

    private val appSettingsFlow: Flow<AppSettings> = appSettingsRepository.appSettingsFlow

    // 更新相关
    private var showUpdateSnackBar = MutableStateFlow(false)
    private var showSkipVersion = MutableStateFlow(false)
    private var googlePlaySkipVersionListener: () -> Unit = {}
    private var performedInstallListen = {}

    private var agreePrivacyPolicyState: AppSettings.AgreePrivacyPolicyState =
        AppSettings.AgreePrivacyPolicyState.Default


    override fun onCreate(savedInstanceState: Bundle?) {
        super.onCreate(savedInstanceState)
        enableEdgeToEdge()
        setContent {
            var enabledDynamicColor by remember { mutableStateOf(false) }
            val updateSnackBarHostState = remember { SnackbarHostState() }
            val showSkipVersionState by showSkipVersion.collectAsState()

            LaunchedEffect(Unit) {
                appSettingsFlow.collect {
                    enabledDynamicColor = it.enabledDynamicColor
                }
            }

            LaunchedEffect(showUpdateSnackBar) {
                if (!showUpdateSnackBar.value) return@LaunchedEffect
                val result = updateSnackBarHostState.showSnackbar(
                    message = getString(R.string.update_downloaded),
                    actionLabel = getString(R.string.update_action),
                    duration = SnackbarDuration.Short
                )
                when (result) {
                    SnackbarResult.ActionPerformed -> {
                        showUpdateSnackBar.value = false
                        performedInstallListen.invoke()
                    }

                    SnackbarResult.Dismissed -> {
                        showUpdateSnackBar.value = false
                    }
                }
            }

            BILIBILIASTheme(dynamicColor = enabledDynamicColor) {
                Box {
                    BILIBILIASAppScreen()
                    SnackbarHost(
                        hostState = updateSnackBarHostState,
                        modifier = Modifier.align(Alignment.BottomCenter)
                    )
                    SkipVersionDialog(showSkipVersionState, onConfirm = {
                        lifecycleScope.launch {
                            googlePlaySkipVersionListener.invoke()
                            showSkipVersion.value = false
                        }
                    }, onDismiss = {
                        showSkipVersion.value = false
                    })
                }
            }
        }
        // 处理特殊厂商的适配选项
        specialManufacturersOption()
        // 初始化设置
        initAppSetting()
        // 初始化通知渠道
        initNotificationChannel()
        // 更新检查
        handleShareInfo(intent)
    }

    @Composable
    fun SkipVersionDialog(value: Boolean, onConfirm: () -> Unit, onDismiss: () -> Unit = {}) {
        if (value) {
            AlertDialog(
                onDismissRequest = {},
                title = { Text(stringResource(R.string.update_skip_title)) },
                text = { Text(stringResource(R.string.update_skip_message)) },
                confirmButton = {
                    ASTextButton(onClick = onConfirm) {
                        Text(stringResource(R.string.update_skip_this_version))
                    }

                },
                dismissButton = {
                    ASTextButton(onClick = onDismiss) {
                        Text(stringResource(R.string.update_later))
                    }
                }
            )
        }
    }

    override fun onNewIntent(intent: Intent) {
        super.onNewIntent(intent)
        setIntent(intent)
        handleShareInfo(intent)
    }

    private fun handleShareInfo(incoming: Intent?) {
        if (incoming == null) return
        val action = incoming.action
        val type = incoming.type
        when (action) {
            Intent.ACTION_SEND -> {
                if (type?.startsWith("text/") == true) {
                    val sharedText = incoming.getStringExtra(Intent.EXTRA_TEXT).orEmpty()
                    sendAnalysisEvent(AnalysisEvent(analysisText = sharedText))
                }
            }

            Intent.ACTION_MAIN -> {
                // 正常启动
            }
        }
    }


    private fun specialManufacturersOption() {
        val manufacturer = Build.MANUFACTURER.lowercase()
        when {
            manufacturer.contains(XIAOMI) -> {
                // 设置沉浸式虚拟键，在MIUI系统中，虚拟键背景透明。原生系统中，虚拟键背景半透明。
                window.addFlags(WindowManager.LayoutParams.FLAG_TRANSLUCENT_NAVIGATION)
            }
        }
    }

    /**
     * 初始化APP设置
     */
    private fun initAppSetting() {
        lifecycleScope.launch(Dispatchers.IO) {
            appSettingsFlow.collect {
                agreePrivacyPolicyState = it.agreePrivacyPolicy
                CommonBuildConfig.agreedPrivacyPolicy = it.agreePrivacyPolicy ==
                        AppSettings.AgreePrivacyPolicyState.Agreed
                initBaiduAnalytics(it.agreePrivacyPolicy)
            }
        }
    }

    /**
     * 初始化百度统计
     */
    fun initBaiduAnalytics(state: AppSettings.AgreePrivacyPolicyState) {
        baiduAnalyticsSafe {
        }
    }

    /**
     */


    private fun initNotificationChannel() {
        // 创建文件下载进度渠道
        createDownloadNotificationChannel()
    }


    /**
     * 初始化更新检查
     */

    private val appUpdateLauncher =
        registerForActivityResult(ActivityResultContracts.StartIntentSenderForResult()) { result: ActivityResult ->
            // handle callback
            if (result.resultCode != RESULT_OK) {
                // 详见：https://developer.android.google.cn/guide/playcore/in-app-updates/kotlin-java?hl=zh-cn#setup
                showSkipVersion.value = true
            }
        }

    /**
     * 处理Google Play更新
     */

    /**
     * 显示Google Play更新完成提示
     */

    override fun onResume() {
        super.onResume()
        baiduAnalyticsSafe {
            if (agreePrivacyPolicyState != AppSettings.AgreePrivacyPolicyState.Agreed) return
        }
    }

    override fun onPause() {
        super.onPause()
        baiduAnalyticsSafe {
            if (agreePrivacyPolicyState != AppSettings.AgreePrivacyPolicyState.Agreed) return
        }
    }


}

@Preview(showBackground = true)
@Composable
fun GreetingPreview() {
    BILIBILIASTheme {
        BILIBILIASAppScreen()
    }
}