/* **************************************************************
 * Name:
 * Purpose:   Folder access permission manager for OpenMoneyBox
 * Author:    Igor Calì (igorcali@disroot.org)
 * Created:   2025-12-20
 * Copyright: Igor Calì (igorcali@disroot.org)
 * License:		GNU
 **************************************************************/

package com.igisw.openmoneybox;

import android.app.Activity;
import android.content.Context;
import android.content.Intent;
import android.content.SharedPreferences;
import android.net.Uri;
import android.os.Build;
import android.os.Environment;
import android.provider.DocumentsContract;
import android.widget.Toast;

import androidx.activity.result.ActivityResultLauncher;
import androidx.activity.result.contract.ActivityResultContracts;
import androidx.appcompat.app.AppCompatActivity;
import androidx.core.content.ContextCompat;
import androidx.preference.PreferenceManager;

import org.jetbrains.annotations.NonNls;

public class DirectoryAccessManager {

    private final AppCompatActivity activity;
    private final SharedPreferences sharedPreferences;
    private ActivityResultLauncher<Intent> documentTreeLauncher;
    private ActivityResultLauncher<String[]> requestPermissionLauncher;

    // URI of the selected directory (saved for future access)
    private static final @NonNls String PREF_SELECTED_DIRECTORY_URI = "selected_directory_uri";

    public DirectoryAccessManager(AppCompatActivity activity, Context context) {
        this.activity = activity;
        this.sharedPreferences = PreferenceManager.getDefaultSharedPreferences(context);

        // Initialize the ActivityResultLaunchers
        initializeLaunchers();
    }

    // Initialize the ActivityResultLaunchers for handling permissions and directory selection
    private void initializeLaunchers() {
        // Launcher for requesting permissions
        requestPermissionLauncher = activity.registerForActivityResult(
                new ActivityResultContracts.RequestMultiplePermissions(),
                result -> {
                    boolean allGranted = true;
                    for (Boolean granted : result.values()) {
                        if (!granted) {
                            allGranted = false;
                            break;
                        }
                    }
                    if (allGranted) {
                        launchDocumentTreeIntent();
                    } else {
                        Toast.makeText(activity, activity.getResources().getString(R.string.folder_access_permission_required), Toast.LENGTH_SHORT).show();
                    }
                });

        // Launcher for selecting a directory
        documentTreeLauncher = activity.registerForActivityResult(
                new ActivityResultContracts.StartActivityForResult(),
                result -> {
                    if (result.getResultCode() == Activity.RESULT_OK && result.getData() != null) {
                        Uri treeUri = result.getData().getData();
                        if (treeUri != null) {
                            // Save the URI for future access
                            saveDirectoryUri(treeUri);
                            Toast.makeText(activity, activity.getResources().getString(R.string.folder_access_permission_granted), Toast.LENGTH_SHORT).show();
                        }
                    }
                });
    }

    // Check and request permissions if not granted
    public void requestDirectoryAccess() {
        if (Build.VERSION.SDK_INT >= Build.VERSION_CODES.R) {
            // For Android 11 (API 30) and above, no runtime permissions are required for ACTION_OPEN_DOCUMENT_TREE
            launchDocumentTreeIntent();
        } else {
            // For Android 10 (API 29) and below, request legacy permissions
            if (ContextCompat.checkSelfPermission(activity, android.Manifest.permission.READ_EXTERNAL_STORAGE) != android.content.pm.PackageManager.PERMISSION_GRANTED ||
                    ContextCompat.checkSelfPermission(activity, android.Manifest.permission.WRITE_EXTERNAL_STORAGE) != android.content.pm.PackageManager.PERMISSION_GRANTED) {
                requestPermissionLauncher.launch(new String[]{
                        android.Manifest.permission.READ_EXTERNAL_STORAGE,
                        android.Manifest.permission.WRITE_EXTERNAL_STORAGE
                });
            } else {
                launchDocumentTreeIntent();
            }
        }
    }

    // Launch ACTION_OPEN_DOCUMENT_TREE to select a directory
    private void launchDocumentTreeIntent() {
        Intent intent = new Intent(Intent.ACTION_OPEN_DOCUMENT_TREE);
        // Optionally, set the initial directory (e.g., Downloads)
        if (Build.VERSION.SDK_INT >= Build.VERSION_CODES.O) {
            intent.putExtra(DocumentsContract.EXTRA_INITIAL_URI, Uri.parse(Environment.getExternalStoragePublicDirectory(Environment.DIRECTORY_DOWNLOADS).toString()));
        }
        documentTreeLauncher.launch(intent);
    }

    // Save the selected directory URI for future access
    private void saveDirectoryUri(Uri uri) {
        activity.getContentResolver().takePersistableUriPermission(
                uri,
                Intent.FLAG_GRANT_READ_URI_PERMISSION | Intent.FLAG_GRANT_WRITE_URI_PERMISSION
        );

        sharedPreferences.edit().putString(PREF_SELECTED_DIRECTORY_URI, uri.toString()).apply();
    }

    // Retrieve the saved directory URI
    public Uri getSavedDirectoryUri() {
        String uriString = sharedPreferences.getString(PREF_SELECTED_DIRECTORY_URI, null);
        return uriString != null ? Uri.parse(uriString) : null;
    }

    // Check if directory access is already granted
    public boolean isDirectoryAccessGranted() {
        return getSavedDirectoryUri() != null;
    }
}