import 'package:camera/camera.dart';
import 'package:file_picker/file_picker.dart';
import 'package:flutter/material.dart';
import 'package:librecamera/l10n/app_localizations.dart';
import 'package:librecamera/l10n/l10n.dart';
import 'package:librecamera/src/pages/onboarding_page.dart';
import 'package:librecamera/src/provider/locale_provider.dart';
import 'package:librecamera/src/provider/theme_provider.dart';
import 'package:librecamera/src/utils/preferences.dart';
import 'package:librecamera/src/widgets/format.dart';
import 'package:librecamera/src/widgets/resolution.dart';
import 'package:package_info_plus/package_info_plus.dart';
import 'package:provider/provider.dart';
import 'package:screen_brightness/screen_brightness.dart';
import 'package:url_launcher/url_launcher.dart';

class SettingsButton extends StatelessWidget {
  const SettingsButton({
    required this.onPressed,
    required this.controller,
    super.key,
  });

  final void Function()? onPressed;
  final CameraController? controller;

  @override
  Widget build(BuildContext context) {
    return IconButton(
      disabledColor: Colors.white24,
      onPressed: onPressed,
      icon: const Icon(Icons.settings),
      tooltip: AppLocalizations.of(context)!.settings,
      iconSize: 35,
      color: Colors.white,
    );
  }
}

class SettingsPage extends StatefulWidget {
  const SettingsPage({
    required this.controller,
    super.key,
  });

  final CameraController? controller;

  @override
  State<SettingsPage> createState() => _SettingsPageState();
}

class _SettingsPageState extends State<SettingsPage> {
  String currentSavePath = Preferences.getSavePath();
  bool isMoreOptions = false;

  ScrollController listScrollController = ScrollController();

  //Compress quality slider
  double value = Preferences.getCompressQuality().toDouble();

  TextStyle style = const TextStyle(
    fontSize: 16,
    fontWeight: .bold,
  );

  Widget _moreTile() {
    return Padding(
      padding: const .only(left: 32),
      child: Column(
        children: [
          _useMaterial3Tile(),
          const Divider(),
          _captureOrientationLockedTile(),
          const Divider(),
          _onboardingScreenTile(),
          const Divider(),
          _aboutTile(),
        ],
      ),
    );
  }

  Widget _headingTile(String text) {
    return ListTile(title: Text(text, style: style));
  }

  Widget _aboutListTile({String? version}) {
    Future<void> launchGitHubURL() async {
      final url = Uri.parse('https://github.com/iakdis/librecamera');
      if (await canLaunchUrl(url)) {
        await launchUrl(url, mode: LaunchMode.externalApplication);
      }
    }

    return AboutListTile(
      icon: const Icon(Icons.info),
      applicationName: 'Libre Camera',
      applicationVersion: version != null
          ? AppLocalizations.of(context)!.version(version)
          : null,
      applicationIcon: const Image(
        image: AssetImage('assets/images/icon.png'),
        width: 50,
        height: 50,
      ),
      applicationLegalese: 'GNU Public License v3',
      aboutBoxChildren: [
        Text(AppLocalizations.of(context)!.license),
        const Divider(),
        TextButton.icon(
          icon: const Icon(Icons.open_in_new),
          onPressed: launchGitHubURL,
          label: SelectableText(
            'https://github.com/iakdis/librecamera',
            onTap: launchGitHubURL,
          ),
        ),
      ],
    );
  }

  Widget _aboutTile() {
    return FutureBuilder(
      future: PackageInfo.fromPlatform(),
      builder: (context, snapshot) {
        return _aboutListTile(
          version: snapshot.hasData ? (snapshot.data!).version : null,
        );
      },
    );
  }

  Widget _onboardingScreenTile() {
    return ListTile(
      title: Text(AppLocalizations.of(context)!.onboardingScreen),
      subtitle: Text(
        AppLocalizations.of(context)!.onboardingScreen_description,
      ),
      trailing: const Icon(Icons.logout),
      onTap: () async {
        await Preferences.setOnBoardingComplete(complete: false);

        await Navigator.of(context).pushReplacement(
          MaterialPageRoute<void>(builder: (context) => const OnboardingPage()),
        );
      },
    );
  }

  Widget _captureOrientationLockedTile() {
    return SwitchListTile(
      title: Text(AppLocalizations.of(context)!.lockCaptureOrientation),
      subtitle: Text(
        AppLocalizations.of(context)!.lockCaptureOrientation_description,
      ),
      value: Preferences.getIsCaptureOrientationLocked(),
      onChanged: (value) async {
        await Preferences.setIsCaptureOrientationLocked(
          isCaptureOrientationLocked: value,
        );
        setState(() {});
      },
    );
  }

  Widget _showMoreTile() {
    return InkWell(
      onTap: () => setState(() {
        isMoreOptions = !isMoreOptions;

        WidgetsBinding.instance.addPostFrameCallback((_) async {
          await listScrollController.animateTo(
            listScrollController.position.maxScrollExtent,
            duration: const Duration(milliseconds: 200),
            curve: Curves.easeInOut,
          );
        });
      }),
      child: ListTile(
        title: Text(
          isMoreOptions
              ? AppLocalizations.of(context)!.less
              : AppLocalizations.of(context)!.more,
          style: style,
        ),
        trailing: Padding(
          padding: const .all(8),
          child: Icon(
            isMoreOptions ? Icons.expand_less : Icons.expand_more,
            size: 35,
          ),
        ),
      ),
    );
  }

  Widget _savePathTile() {
    return ListTile(
      title: Text(AppLocalizations.of(context)!.savePath),
      subtitle: Text(
        AppLocalizations.of(context)!.savePath_description(currentSavePath),
      ),
      trailing: ElevatedButton(
        onPressed: () async {
          final selectedDirectory = await FilePicker.platform
              .getDirectoryPath();

          if (selectedDirectory == null) {
            // User canceled the picker
          }

          await Preferences.setSavePath(
            selectedDirectory ?? Preferences.getSavePath(),
          );

          currentSavePath = Preferences.getSavePath();
          setState(() {});
        },
        child: Text(AppLocalizations.of(context)!.choosePath),
      ),
    );
  }

  Widget _keepEXIFMetadataTile() {
    return SwitchListTile(
      title: Text(AppLocalizations.of(context)!.keepEXIFMetadata),
      subtitle: Text(
        AppLocalizations.of(context)!.keepEXIFMetadata_description,
      ),
      value: Preferences.getKeepEXIFMetadata(),
      onChanged: (value) async {
        await Preferences.setKeepEXIFMetadata(keepEXIFMetadata: value);
        setState(() {});
      },
    );
  }

  Widget _imageCompressionFormat() {
    return ListTile(
      title: Text(AppLocalizations.of(context)!.format),
      subtitle: Text(AppLocalizations.of(context)!.format_description),
      trailing: const FormatButton(),
    );
  }

  Widget _imageCompressionTile() {
    return SwitchListTile(
      value: Preferences.getEnableCompression(),
      onChanged: (value) async {
        await Preferences.setEnableCompression(value: value);
        setState(() {});

        WidgetsBinding.instance.addPostFrameCallback((_) async {
          await listScrollController.animateTo(
            listScrollController.position.maxScrollExtent,
            duration: const Duration(milliseconds: 200),
            curve: Curves.easeInOut,
          );
        });
      },
      title: Text(AppLocalizations.of(context)!.enableCompression),
      subtitle: Text(
        AppLocalizations.of(context)!.enableCompression_description,
      ),
    );
  }

  Widget _imageCompressionQualityTile() {
    return ListTile(
      title: Text(
        AppLocalizations.of(context)!.imageCompressionQuality,
      ),
      subtitle: Column(
        crossAxisAlignment: .start,
        children: [
          Text(
            AppLocalizations.of(context)!.imageCompressionQuality_description,
          ),
          Padding(
            padding: const .fromLTRB(16, 0, 16, 0),
            child: Row(
              children: [
                const Text(
                  '10',
                  style: TextStyle(fontWeight: .bold),
                ),
                Flexible(
                  child: Slider(
                    value: value,
                    onChanged: (value) => setState(() => this.value = value),
                    onChangeEnd: (value) async {
                      await Preferences.setCompressQuality(value.toInt());
                    },
                    min: 10,
                    max: 100,
                    label: value.round().toString(),
                    //label: Preferences.getCompressQuality().round().toString(),
                    divisions: 90,
                  ),
                ),
                const Text(
                  '100',
                  style: TextStyle(fontWeight: .bold),
                ),
              ],
            ),
          ),
        ],
      ),
    );
  }

  Widget _disableShutterSoundTile() {
    return SwitchListTile(
      title: Text(
        AppLocalizations.of(context)!.shutterSound,
      ),
      subtitle: Text(AppLocalizations.of(context)!.shutterSound_description),
      value: !Preferences.getDisableShutterSound(),
      onChanged: (value) async {
        await Preferences.setDisableShutterSound(disable: !value);
        setState(() {});
      },
    );
  }

  Widget _captureAtVolumePressTile() {
    return SwitchListTile(
      title: Text(AppLocalizations.of(context)!.captureAtVolumePress),
      subtitle: Text(
        AppLocalizations.of(context)!.captureAtVolumePress_description,
      ),
      value: Preferences.getCaptureAtVolumePress(),
      onChanged: (value) async {
        await Preferences.setCaptureAtVolumePress(enable: value);
        setState(() {});
      },
    );
  }

  Widget _disableAudioTile() {
    return SwitchListTile(
      title: Text(AppLocalizations.of(context)!.disableAudio),
      subtitle: Text(AppLocalizations.of(context)!.disableAudio_description),
      value: !Preferences.getEnableAudio(),
      onChanged: (value) async {
        await Preferences.setEnableAudio(enableAudio: !value);
        setState(() {});
      },
    );
  }

  Widget _enableModeRow() {
    return SwitchListTile(
      title: Text(AppLocalizations.of(context)!.enableModeRow),
      subtitle: Text(AppLocalizations.of(context)!.enableModeRow_description),
      value: Preferences.getEnableModeRow(),
      onChanged: (value) async {
        await Preferences.setEnableModeRow(enable: value);
        setState(() {});
      },
    );
  }

  Widget _startWithFrontCameraTile() {
    return SwitchListTile(
      title: Text(AppLocalizations.of(context)!.startWithFrontCamera),
      subtitle: Text(
        AppLocalizations.of(context)!.startWithFrontCamera_description,
      ),
      value: !Preferences.getStartWithRearCamera(),
      onChanged: (value) async {
        await Preferences.setStartWithRearCamera(rear: !value);
        setState(() {});
      },
    );
  }

  Widget _flipPhotosFrontCameraTile() {
    return SwitchListTile(
      title: Text(AppLocalizations.of(context)!.flipPhotosFrontCamera),
      subtitle: Text(
        AppLocalizations.of(context)!.flipPhotosFrontCamera_description,
      ),
      value: !Preferences.getFlipFrontCameraPhoto(),
      onChanged: (value) async {
        await Preferences.setFlipFrontCameraPhoto(flip: !value);
        setState(() {});
      },
    );
  }

  Widget _resolutionTile() {
    return ListTile(
      title: Text(AppLocalizations.of(context)!.resolution),
      subtitle: Text(AppLocalizations.of(context)!.resolution_description),
      trailing: const ResolutionButton(enabled: true),
    );
  }

  Widget _videoFpsTile() {
    final options = <int>[24, 25, 30, 60, 120];

    return ListTile(
      title: Text(AppLocalizations.of(context)!.fps),
      subtitle: Text(AppLocalizations.of(context)!.fps_description),
      trailing: DropdownButton<int>(
        value: Preferences.getVideoFps(),
        items: options
            .map(
              (option) => DropdownMenuItem(
                value: option,
                child: Padding(
                  padding: const EdgeInsets.only(right: 8),
                  child: Text('$option fps'),
                ),
              ),
            )
            .toList(),
        onChanged: (v) async {
          if (v == null) return;
          await Preferences.setVideoFps(v);
          setState(() {});
        },
      ),
    );
  }

  Widget _enableExposureSliderTile() {
    return SwitchListTile(
      title: Text(AppLocalizations.of(context)!.enableExposureSlider),
      subtitle: Text(
        AppLocalizations.of(context)!.enableExposureSlider_description,
      ),
      value: Preferences.getEnableExposureSlider(),
      onChanged: (value) async {
        await Preferences.setEnableExposureSlider(enable: value);
        setState(() {});
      },
    );
  }

  Widget _enableZoomSliderTile() {
    return SwitchListTile(
      title: Text(AppLocalizations.of(context)!.enableZoomSlider),
      subtitle: Text(
        AppLocalizations.of(context)!.enableZoomSlider_description,
      ),
      value: Preferences.getEnableZoomSlider(),
      onChanged: (value) async {
        await Preferences.setEnableZoomSlider(enable: value);
        setState(() {});
      },
    );
  }

  Widget _themeTile() {
    final themeProvider = Provider.of<ThemeProvider>(context, listen: false);

    return ListTile(
      title: Text(AppLocalizations.of(context)!.theme),
      subtitle: Text(AppLocalizations.of(context)!.theme_description),
      trailing: DropdownButton(
        icon: switch (CustomThemeMode.values.byName(
          Preferences.getThemeMode(),
        )) {
          .system => const Icon(Icons.settings_display),
          .light => const Icon(Icons.light_mode),
          .dark => const Icon(Icons.dark_mode),
          .black => const Icon(Icons.dark_mode),
        },
        value: CustomThemeMode.values.byName(Preferences.getThemeMode()),
        items: [
          ...CustomThemeMode.values.map(
            (customThemeMode) => DropdownMenuItem(
              value: customThemeMode,
              onTap: () => themeProvider.setTheme(customThemeMode),
              child: Padding(
                padding: const .only(right: 8),
                child: Text(_nameByCustomThemeMode(customThemeMode)),
              ),
            ),
          ),
        ],
        onChanged: (_) {},
      ),
    );
  }

  Widget _maximumScreenBrightnessTile() {
    return SwitchListTile(
      title: Text(AppLocalizations.of(context)!.enableMaximumScreenBrightness),
      subtitle: Text(
        AppLocalizations.of(context)!.enableMaximumScreenBrightness_description,
      ),
      value: Preferences.getMaximumScreenBrightness(),
      onChanged: (value) async {
        await Preferences.setMaximumScreenBrightness(enable: value);
        Preferences.getMaximumScreenBrightness()
            ? await ScreenBrightness().setApplicationScreenBrightness(1)
            : await ScreenBrightness().resetApplicationScreenBrightness();
        setState(() {});
      },
    );
  }

  Widget _leftHandedModeTile() {
    return SwitchListTile(
      title: Text(AppLocalizations.of(context)!.leftHandedMode),
      subtitle: Text(AppLocalizations.of(context)!.leftHandedMode_description),
      value: Preferences.getLeftHandedMode(),
      onChanged: (value) async {
        await Preferences.setLeftHandedMode(enable: value);
        setState(() {});
      },
    );
  }

  Widget _languageTile() {
    final localeProvider = Provider.of<LocaleProvider>(context, listen: false);

    return ListTile(
      title: Text(AppLocalizations.of(context)!.language),
      subtitle: Text(AppLocalizations.of(context)!.language_description),
      trailing: DropdownButton<String>(
        icon: const Icon(Icons.language),
        value: Preferences.getLanguage().isNotEmpty
            ? Preferences.getLanguage()
            : null,
        items:
            Localization.supportedLocales.map((locale) {
              final name = Localization.getName(locale);

              return DropdownMenuItem(
                value: locale.toLanguageTag(),
                onTap: () => localeProvider.setLocale(locale),
                child: Text(name),
              );
            }).toList()..insert(
              0,
              DropdownMenuItem<String>(
                onTap: localeProvider.clearLocale,
                child: Padding(
                  padding: const .only(right: 8),
                  child: Text(AppLocalizations.of(context)!.systemLanguage),
                ),
              ),
            ),
        onChanged: (_) {},
      ),
    );
  }

  Widget _useMaterial3Tile() {
    final themeProvider = Provider.of<ThemeProvider>(context, listen: false);

    return SwitchListTile(
      title: Text(AppLocalizations.of(context)!.useMaterialYou),
      subtitle: Text(AppLocalizations.of(context)!.useMaterialYou_description),
      value: Preferences.getUseMaterial3(),
      onChanged: (value) async {
        await Preferences.setUseMaterial3(useMaterial3: value);
        await themeProvider.setTheme(themeProvider.themeMode);
      },
    );
  }

  @override
  Widget build(BuildContext context) {
    return Scaffold(
      appBar: AppBar(
        leading: IconButton(
          icon: const Icon(Icons.arrow_back),
          onPressed: () async {
            Navigator.pop(context);
          },
          tooltip: AppLocalizations.of(context)!.back,
        ),
        title: Text(AppLocalizations.of(context)!.settings),
      ),
      body: SafeArea(
        child: ListView(
          controller: listScrollController,
          padding: const .fromLTRB(0, 8, 0, 8),
          children: <Widget>[
            _headingTile(AppLocalizations.of(context)!.appSettings),
            _languageTile(),
            const Divider(),
            _themeTile(),
            const Divider(),
            _maximumScreenBrightnessTile(),
            const Divider(),
            _leftHandedModeTile(),
            const Divider(),
            _enableModeRow(),
            const Divider(),
            _enableZoomSliderTile(),
            const Divider(),
            _enableExposureSliderTile(),
            const Divider(),
            _headingTile(AppLocalizations.of(context)!.cameraBehaviour),
            _resolutionTile(),
            const Divider(),
            _captureAtVolumePressTile(),
            const Divider(),
            _disableShutterSoundTile(),
            const Divider(),
            _startWithFrontCameraTile(),
            const Divider(),
            _videoFpsTile(),
            const Divider(),
            _disableAudioTile(),
            const Divider(),
            _headingTile(AppLocalizations.of(context)!.saving),
            _flipPhotosFrontCameraTile(),
            const Divider(),
            _keepEXIFMetadataTile(),
            const Divider(),
            _savePathTile(),
            const Divider(),
            _imageCompressionTile(),
            if (Preferences.getEnableCompression()) ...[
              const Divider(),
              _imageCompressionQualityTile(),
              const Divider(),
              _imageCompressionFormat(),
            ],
            const Divider(),
            _showMoreTile(),
            if (isMoreOptions) _moreTile(),
          ],
        ),
      ),
    );
  }

  String _nameByCustomThemeMode(CustomThemeMode customThemeMode) =>
      switch (customThemeMode) {
        .system => AppLocalizations.of(context)!.themeSystem,
        .light => AppLocalizations.of(context)!.themeLight,
        .dark => AppLocalizations.of(context)!.themeDark,
        .black => AppLocalizations.of(context)!.themeBlack,
      };
}
