import 'package:device_info_plus/device_info_plus.dart';
import 'package:file_picker/file_picker.dart';
import 'package:flutter/foundation.dart';
import 'package:flutter/material.dart';
import 'package:flutter/services.dart';
import 'package:librecamera/l10n/app_localizations.dart';
import 'package:librecamera/src/pages/camera_page.dart';
import 'package:librecamera/src/provider/theme_provider.dart';
import 'package:librecamera/src/utils/preferences.dart';
import 'package:permission_handler/permission_handler.dart';
import 'package:provider/provider.dart';
import 'package:smooth_page_indicator/smooth_page_indicator.dart';

class OnboardingPage extends StatefulWidget {
  const OnboardingPage({super.key});

  @override
  State<OnboardingPage> createState() => _OnboardingPageState();
}

class _OnboardingPageState extends State<OnboardingPage> {
  final controller = PageController();
  bool isLastPage = false;
  bool cameraPermissionGranted = false;
  bool microphonePermissionGranted = false;
  bool storagePermissionGranted = false;
  String currentSavePath = Preferences.getSavePath();
  int currentPage = 0;
  TextEditingController textController = TextEditingController();
  late FocusNode focusNode;

  @override
  void initState() {
    textController.text = currentSavePath;
    textController
      ..addListener(() async {
        currentSavePath = textController.text;
        await Preferences.setSavePath(currentSavePath);
      })
      ..selection = TextSelection(
        baseOffset: textController.text.length,
        extentOffset: textController.text.length,
      );
    focusNode = FocusNode();
    super.initState();
  }

  bool checkPermissions() {
    return cameraPermissionGranted;
  }

  Future<void> cameraPermission() async {
    await Permission.camera.request();
    final status = await Permission.camera.status;

    if (status.isGranted) {
      if (kDebugMode) {
        print('Camera Permission: GRANTED');
      }
      setState(() {
        cameraPermissionGranted = true;
      });
    } else {
      if (kDebugMode) {
        print('Camera Permission: DENIED');
      }
    }
  }

  Future<void> storagePermission() async {
    final androidInfo = await DeviceInfoPlugin().androidInfo;
    PermissionStatus? status;

    if (androidInfo.version.sdkInt <= 32) {
      status = await Permission.storage.request();
    } else {
      status = await Permission.photos.request();
    }

    if (status.isGranted) {
      if (kDebugMode) {
        print('Storage Permission: GRANTED');
      }
      setState(() {
        storagePermissionGranted = true;
      });
    } else {
      if (kDebugMode) {
        print('Storage: DENIED');
      }
    }
  }

  Future<void> savePath() async {
    final selectedDirectory = await FilePicker.platform.getDirectoryPath();

    if (selectedDirectory == null) {
      // User canceled the picker
    }

    await Preferences.setSavePath(
      selectedDirectory ?? Preferences.getSavePath(),
    );

    setState(() {
      currentSavePath = Preferences.getSavePath();
      textController.text = currentSavePath;
      textController.selection = TextSelection(
        baseOffset: textController.text.length,
        extentOffset: textController.text.length,
      );
    });
  }

  Future<void> previousPage() async {
    focusNode.unfocus();
    await controller.previousPage(
      duration: const Duration(milliseconds: 200),
      curve: Curves.easeInOut,
    );
    await unfocusAndRestore();
  }

  Future<void> nextPage() async {
    focusNode.unfocus();
    await controller.nextPage(
      duration: const Duration(milliseconds: 200),
      curve: Curves.easeInOut,
    );
    await unfocusAndRestore();
  }

  bool nextEnabled() {
    if (currentPage == 0) {
      if (checkPermissions()) {
        return true;
      }
    } else if (currentPage == 1) {
      if (currentSavePath.isNotEmpty) {
        return true;
      }
    } else if (currentPage == 2) {
      return true;
    }

    return false;
  }

  @override
  void dispose() {
    controller.dispose();
    textController.dispose();
    super.dispose();
  }

  Widget _permissionsPage() {
    return ColoredBox(
      color: Colors.orange.shade100,
      child: MediaQuery.orientationOf(context) == .portrait
          ? Column(
              mainAxisAlignment: .center,
              children: [_permissionsPageInfo(), _permissionsPageButtons()],
            )
          : Row(
              mainAxisAlignment: .center,
              children: [
                Expanded(child: _permissionsPageInfo()),
                Expanded(child: _permissionsPageButtons()),
              ],
            ),
    );
  }

  Widget _permissionsPageInfo() {
    return Column(
      mainAxisAlignment: .center,
      children: [
        Text(
          AppLocalizations.of(context)!.permissionsTitle,
          style: TextStyle(
            color: Colors.teal.shade700,
            fontSize: 24,
            fontWeight: .bold,
          ),
        ),
        const SizedBox(height: 24),
        if (MediaQuery.orientationOf(context) == .portrait)
          const Icon(Icons.settings, size: 100, color: Colors.white)
        else
          Container(),
        if (MediaQuery.orientationOf(context) == .portrait)
          const SizedBox(height: 24)
        else
          Container(),
        Container(
          padding: const .symmetric(horizontal: 64),
          child: Text(
            AppLocalizations.of(context)!.permissionsTitle_description,
            style: TextStyle(
              color: Colors.teal.shade700,
              fontSize: 17,
              fontWeight: .w500,
            ),
          ),
        ),
      ],
    );
  }

  Widget _permissionsPageButtons() {
    return Column(
      mainAxisAlignment: .center,
      children: [
        const SizedBox(height: 24),
        ElevatedButton(
          onPressed: cameraPermissionGranted ? null : cameraPermission,
          child: Text(AppLocalizations.of(context)!.giveCameraPermission),
        ),
        const SizedBox(height: 16),
        ElevatedButton(
          onPressed: storagePermissionGranted ? null : storagePermission,
          child: Text(AppLocalizations.of(context)!.giveStoragePermission),
        ),
      ],
    );
  }

  Widget _savePathPage() {
    return ColoredBox(
      color: Colors.blue.shade100,
      child: MediaQuery.orientationOf(context) == .portrait
          ? Center(
              child: SingleChildScrollView(
                padding: const .symmetric(vertical: 64),
                child: Column(
                  mainAxisAlignment: .center,
                  children: [_savePathPageInfo(), _savePathPageButtons()],
                ),
              ),
            )
          : Row(
              mainAxisAlignment: .center,
              children: [
                Expanded(child: _savePathPageInfo()),
                Expanded(child: _savePathPageButtons()),
              ],
            ),
    );
  }

  Widget _savePathPageInfo() {
    return Column(
      mainAxisAlignment: .center,
      children: [
        Text(
          AppLocalizations.of(context)!.savePathTitle,
          style: TextStyle(
            color: Colors.teal.shade700,
            fontSize: 24,
            fontWeight: .bold,
          ),
        ),
        const SizedBox(height: 24),
        if (MediaQuery.orientationOf(context) == .portrait)
          const Icon(Icons.save_as, size: 100, color: Colors.white)
        else
          Container(),
        if (MediaQuery.orientationOf(context) == .portrait)
          const SizedBox(height: 24)
        else
          Container(),
        Container(
          padding: const .symmetric(horizontal: 64),
          child: Text(
            AppLocalizations.of(context)!.savePathTitle_description,
            style: TextStyle(
              color: Colors.teal.shade700,
              fontSize: 17,
              fontWeight: .w500,
            ),
          ),
        ),
      ],
    );
  }

  Widget _savePathPageButtons() {
    return Column(
      mainAxisAlignment: .center,
      children: [
        const SizedBox(height: 24),
        ElevatedButton(
          onPressed: savePath,
          child: Text(AppLocalizations.of(context)!.choosePath),
        ),
        const SizedBox(height: 48),
        SizedBox(
          width: 256,
          child: TextField(
            controller: textController,
            focusNode: focusNode,
            onSubmitted: (value) async {
              currentSavePath = value;
              await Preferences.setSavePath(currentSavePath);
              focusNode.unfocus();
              await SystemChrome.restoreSystemUIOverlays();
            },
            style: TextStyle(color: Colors.grey[600], fontSize: 17),
            decoration: InputDecoration(
              border: const OutlineInputBorder(),
              labelText: AppLocalizations.of(context)!.savePath,
              isDense: true,
              hintText: 'storage/emulated/0/DCIM',
              hintStyle: TextStyle(
                color: Colors.grey[500],
                fontStyle: .italic,
              ),
            ),
          ),
        ),
      ],
    );
  }

  Widget _welcomePageInfo() {
    return ColoredBox(
      color: Colors.green.shade100,
      child: Column(
        mainAxisAlignment: .center,
        children: [
          Text(
            AppLocalizations.of(context)!.welcomeTitle,
            style: TextStyle(
              color: Colors.teal.shade700,
              fontSize: 36,
              fontWeight: .bold,
            ),
          ),
          const SizedBox(height: 24),
          if (MediaQuery.orientationOf(context) == .portrait)
            const Icon(Icons.handshake, size: 100, color: Colors.white)
          else
            Container(),
          if (MediaQuery.orientationOf(context) == .portrait)
            const SizedBox(height: 24)
          else
            Container(),
          Container(
            padding: const .symmetric(horizontal: 64),
            child: Text(
              AppLocalizations.of(context)!.welcomeTitle_description,
              style: TextStyle(
                color: Colors.teal.shade700,
                fontSize: 17,
                fontWeight: .w500,
              ),
            ),
          ),
        ],
      ),
    );
  }

  Widget _welcomePageBottomButton() {
    return TextButton(
      style: TextButton.styleFrom(
        foregroundColor: Colors.white,
        shape: RoundedRectangleBorder(borderRadius: .circular(0)),
        backgroundColor: Colors.teal.shade700,
        minimumSize: const Size.fromHeight(80),
      ),
      child: Text(
        AppLocalizations.of(context)!.getStarted,
        style: const TextStyle(fontSize: 24),
      ),
      onPressed: () async {
        await Preferences.setOnBoardingComplete(complete: true);

        await Navigator.of(context).pushReplacement(
          MaterialPageRoute<void>(builder: (context) => const CameraPage()),
        );
      },
    );
  }

  Widget _bottomPageIndicator() {
    return Container(
      padding: const .symmetric(horizontal: 16),
      height: 80,
      child: Row(
        mainAxisAlignment: .spaceBetween,
        children: [
          TextButton(
            onPressed: previousPage,
            child: Text(AppLocalizations.of(context)!.back.toUpperCase()),
          ),
          Center(
            child: SmoothPageIndicator(
              controller: controller,
              count: 3,
              effect: WormEffect(
                spacing: 16,
                dotColor: Colors.black26,
                activeDotColor: Colors.teal.shade700,
              ),
            ),
          ),
          TextButton(
            onPressed: nextEnabled() ? nextPage : null,
            child: Text(AppLocalizations.of(context)!.next.toUpperCase()),
          ),
        ],
      ),
    );
  }

  Future<void> unfocusAndRestore() async {
    final currentFocus = FocusScope.of(context);

    if (!currentFocus.hasPrimaryFocus) {
      currentFocus.unfocus();
      await SystemChrome.restoreSystemUIOverlays();
    }
  }

  @override
  Widget build(BuildContext context) {
    return GestureDetector(
      onTapDown: (details) => unfocusAndRestore(),
      child: Theme(
        data: context.watch<ThemeProvider>().theme(
          colorScheme: ColorScheme.fromSeed(seedColor: seedColor),
        ),
        child: Scaffold(
          body: PageView(
            physics: nextEnabled()
                ? const ScrollPhysics()
                : const NeverScrollableScrollPhysics(),
            controller: controller,
            onPageChanged: (index) async {
              setState(() {
                isLastPage = index == 2;
                currentPage = index;
              });
              await unfocusAndRestore();
            },
            children: [
              _permissionsPage(),
              _savePathPage(),
              _welcomePageInfo(),
            ],
          ),
          bottomNavigationBar: SafeArea(
            left: false,
            right: false,
            child: isLastPage
                ? _welcomePageBottomButton()
                : _bottomPageIndicator(),
          ),
        ),
      ),
    );
  }
}
