package com.hfut.schedule.ui.screen.home.calendar.timetable.ui

import androidx.compose.foundation.clickable
import androidx.compose.foundation.layout.Arrangement
import androidx.compose.foundation.layout.Box
import androidx.compose.foundation.layout.Column
import androidx.compose.foundation.layout.PaddingValues
import androidx.compose.foundation.layout.fillMaxSize
import androidx.compose.foundation.layout.fillMaxWidth
import androidx.compose.material3.MaterialTheme
import androidx.compose.material3.Surface
import androidx.compose.material3.Text
import androidx.compose.runtime.Composable
import androidx.compose.runtime.collectAsState
import androidx.compose.runtime.getValue
import androidx.compose.ui.Alignment
import androidx.compose.ui.Modifier
import androidx.compose.ui.draw.clip
import androidx.compose.ui.geometry.Offset
import androidx.compose.ui.graphics.Color
import androidx.compose.ui.text.style.TextAlign
import androidx.compose.ui.text.style.TextOverflow
import androidx.compose.ui.unit.dp
import androidx.compose.ui.unit.sp
import com.hfut.schedule.application.MyApplication
import com.hfut.schedule.logic.util.other.AppVersion
import com.hfut.schedule.logic.util.storage.kv.DataStoreManager
import com.hfut.schedule.ui.screen.AppNavRoute
import com.hfut.schedule.ui.screen.home.calendar.common.calendarSquareGlass
import com.hfut.schedule.ui.screen.home.calendar.jxglstu.next.CourseDetailOrigin
import com.hfut.schedule.ui.screen.home.calendar.timetable.logic.DEFAULT_END_TIME
import com.hfut.schedule.ui.screen.home.calendar.timetable.logic.DEFAULT_START_TIME
import com.hfut.schedule.ui.screen.home.calendar.timetable.logic.TimeTableItem
import com.hfut.schedule.ui.screen.home.calendar.timetable.logic.TimeTableType
import com.hfut.schedule.ui.screen.home.calendar.timetable.logic.parseTimeToFloat
import com.xah.mirror.util.ShaderState
import com.xah.transition.component.containerShare
import com.xah.uicommon.style.ClickScale
import com.xah.uicommon.style.clickableWithScale

private const val timeTextFactor = 0.85
private const val placeTextFactor = 0.9

@Composable
fun TimeTable(
    items: List<List<TimeTableItem>>,
    week : Int,
    showAll: Boolean,
    innerPadding : PaddingValues,
    modifier: Modifier = Modifier,
    squareModifier : Modifier = Modifier,
    shaderState : ShaderState? = null,
    onTapBlankRegion : ((Offset) -> Unit)? = null,
    onLongTapBlankRegion : ((Offset) -> Unit)? = null,
    onDoubleTapBlankRegion : ((Offset) -> Unit)? = null,
    onSquareClick : (List<TimeTableItem>) -> Unit,
) {
    val enableLiquidGlass by DataStoreManager.enableLiquidGlass.collectAsState(initial = AppVersion.CAN_SHADER)
    val customBackgroundAlpha by DataStoreManager.customCalendarSquareAlpha.collectAsState(initial = MyApplication.CALENDAR_SQUARE_ALPHA)
    val calendarSquareHeight by DataStoreManager.calendarSquareHeightNew.collectAsState(initial = MyApplication.CALENDAR_SQUARE_HEIGHT_NEW)
    val enableMergeSquare by DataStoreManager.enableMergeSquare.collectAsState(initial = false)
    val calendarSquareTextSize by DataStoreManager.calendarSquareTextSize.collectAsState(initial = 1f)
    val calendarSquareTextPadding by DataStoreManager.calendarSquareTextPadding.collectAsState(initial = MyApplication.CALENDAR_SQUARE_TEXT_PADDING)

    val list = if(week > items.size || week > MyApplication.MAX_WEEK) {
        Exception("NewTimeTableUI received week out of bounds for length ${items.size} of items[${week-1}]").printStackTrace()
        emptyList()
    }  else {
        items[week-1]
    }

    val textSize = (if(!showAll) 12.5.sp else 11.sp) * calendarSquareTextSize
    val lineHeight = textSize * calendarSquareTextPadding
    val timeTextLineHeight = lineHeight*timeTextFactor
    val timeTextSize = textSize*timeTextFactor
    val placeTextLineHeight = lineHeight*placeTextFactor
    val placeTextSize = textSize*placeTextFactor

    val hasBackground = shaderState != null

    val round = MaterialTheme.shapes.extraSmall

    val earliestTime = list.minOfOrNull { it.startTime }
    val latestTime = list.maxOfOrNull { it.endTime }
    val startTime =  parseTimeToFloat(DEFAULT_START_TIME).let { defaultTime ->
        earliestTime?.let {
            minOf(parseTimeToFloat(it), defaultTime)
        } ?: defaultTime
    }
    val endTime =  parseTimeToFloat(DEFAULT_END_TIME).let { defaultTime ->
        latestTime?.let {
            maxOf(parseTimeToFloat(it), defaultTime)
        } ?: defaultTime
    }

    if(enableMergeSquare) {
        TimetableCommonSquare(
            items = list,
            modifier = modifier,
            showAll = showAll,
            showLine = !hasBackground,
            innerPadding = innerPadding,
            hourHeight = calendarSquareHeight.dp,
            startTime = startTime,
            endTime = endTime,
            onDoubleTapBlankRegion = onDoubleTapBlankRegion,
            onLongTapBlankRegion = onLongTapBlankRegion,
            onTapBlankRegion = onTapBlankRegion
        ) { list ->
            val color: Pair<Color, Color> = if (!hasBackground) {
                when {
                    list.size > 1 -> Pair(
                        MaterialTheme.colorScheme.errorContainer,
                        MaterialTheme.colorScheme.onErrorContainer.copy(.6f)
                    )

                    else -> Pair(
                        MaterialTheme.colorScheme.primaryContainer,
                        MaterialTheme.colorScheme.onPrimaryContainer.copy(.6f)
                    )
                }
            } else {
                Pair(
                    MaterialTheme.colorScheme.surfaceContainer,
                    MaterialTheme.colorScheme.onSurface.copy(.6f)
                )
            }
            Surface(
                color = if (!hasBackground) color.first else Color.Transparent,
                shape = round,
                modifier = squareModifier
                    .let {
                        if (hasBackground) {
                            it
                                .clip(round)
                                .let {
                                    if (AppVersion.CAN_SHADER) {
                                        it.calendarSquareGlass(
                                            shaderState,
                                            MaterialTheme.colorScheme.surface.copy(
                                                customBackgroundAlpha
                                            ),
                                            enableLiquidGlass,
                                        )
                                    } else {
                                        it
                                    }
                                }
                        } else {
                            it
                        }
                    }
                    .let {
                        if (!hasBackground) {
                            it.clickableWithScale(ClickScale.SMALL.scale) {
                                onSquareClick(list)
                            }
                        } else {
                            it.clickable {
                                onSquareClick(list)
                            }
                        }
                    }
                    .let {
                        if (!hasBackground && list.size == 1) {
                            val item = list[0]
                            val origin = CourseDetailOrigin.CALENDAR_JXGLSTU.t + "@${item.hashCode()}"
                            when (item.type) {
                                TimeTableType.COURSE -> {
                                    it.containerShare(
                                        AppNavRoute.CourseDetail.withArgs(item.name, origin), MaterialTheme.shapes.extraSmall
                                    )
                                }
                                TimeTableType.FOCUS -> {
                                    item.id?.let { id ->
                                        it.containerShare(AppNavRoute.AddEvent.withArgs(id, origin), MaterialTheme.shapes.extraSmall)
                                    } ?: it
                                }
                                TimeTableType.EXAM -> {
                                    it.containerShare(AppNavRoute.Exam.withArgs(origin), MaterialTheme.shapes.extraSmall)
                                }
                            }
                        } else {
                            it
                        }
                    }
            ) {
                Column(
                    modifier = Modifier.fillMaxSize(),
                    verticalArrangement = Arrangement.SpaceBetween,
                    horizontalAlignment = Alignment.CenterHorizontally,
                ) {
                    if (list.size == 1) {
                        val item = list[0]

                        Text(
                            text = item.startTime,
                            fontSize = timeTextSize,
                            textAlign = TextAlign.Center,
                            lineHeight = timeTextLineHeight,
                            overflow = TextOverflow.Clip,
                            maxLines = 1,
                            color = color.second,
                            modifier = Modifier.fillMaxWidth()
                        )

                        Box(
                            modifier = Modifier
                                .weight(1f) // 占据中间剩余的全部空间
                                .fillMaxWidth(),
                            contentAlignment = Alignment.Center
                        ) {
                            Text(
                                text = item.name + (item.teacher?.let { "@$it" } ?: ""),
                                fontSize = textSize,
                                textAlign = TextAlign.Center,
                                lineHeight = lineHeight,
                                overflow = TextOverflow.Ellipsis,
                                modifier = Modifier.fillMaxWidth()
                            )
                        }
                        item.place?.let {
                            Text(
                                text = it,
                                fontSize = placeTextSize,
                                textAlign = TextAlign.Center,
                                lineHeight = placeTextLineHeight,
                                modifier = Modifier.fillMaxWidth(),
                            )
                        }
                        Text(
                            text = item.endTime,
                            fontSize = timeTextSize,
                            textAlign = TextAlign.Center,
                            overflow = TextOverflow.Clip,
                            maxLines = 1,
                            lineHeight = timeTextLineHeight,
                            color = color.second,
                            modifier = Modifier.fillMaxWidth()
                        )
                    } else if (list.size > 1) {
                        val startTime = list.minOf { it.startTime }
                        val endTime = list.maxOf { it.endTime }
                        val courses = list.joinToString(",") { it.name.substring(0, 1) }

                        Text(
                            text = startTime,
                            fontSize = timeTextSize,
                            textAlign = TextAlign.Center,
                            overflow = TextOverflow.Clip,
                            maxLines = 1,
                            lineHeight = timeTextLineHeight,
                            color = color.second,
                            modifier = Modifier.fillMaxWidth()
                        )
                        Box(
                            modifier = Modifier
                                .weight(1f) // 占据中间剩余的全部空间
                                .fillMaxWidth(),
                            contentAlignment = Alignment.Center
                        ) {
                            Text(
                                text = "冲突${list.size}项",
                                fontSize = textSize,
                                lineHeight = lineHeight,
                                textAlign = TextAlign.Center,
                                overflow = TextOverflow.Ellipsis,
                                modifier = Modifier.fillMaxWidth()
                            )
                        }
                        Text(
                            text = courses,
                            fontSize = placeTextSize,
                            textAlign = TextAlign.Center,
                            lineHeight = placeTextLineHeight,
                            modifier = Modifier.fillMaxWidth()
                        )
                        Text(
                            text = endTime,
                            fontSize = timeTextSize,
                            textAlign = TextAlign.Center,
                            overflow = TextOverflow.Clip,
                            maxLines = 1,
                            lineHeight = timeTextLineHeight,
                            color = color.second,
                            modifier = Modifier.fillMaxWidth()
                        )
                    }
                }
            }
        }
    } else {
        TimetableSingleSquare(
            items = list,
            modifier = modifier,
            showAll = showAll,
            showLine = !hasBackground,
            innerPadding = innerPadding,
            hourHeight = calendarSquareHeight.dp,
            startTime = startTime,
            endTime = endTime,
            onDoubleTapBlankRegion = onDoubleTapBlankRegion,
            onLongTapBlankRegion = onLongTapBlankRegion,
            onTapBlankRegion = onTapBlankRegion
        ) { item ->
            val color: Pair<Color, Color> = if (!hasBackground) {
                when (item.type) {
                    TimeTableType.FOCUS -> Pair(
                        MaterialTheme.colorScheme.primary,
                        MaterialTheme.colorScheme.onPrimary.copy(.6f)
                    )

                    TimeTableType.COURSE -> Pair(
                        MaterialTheme.colorScheme.primaryContainer,
                        MaterialTheme.colorScheme.onPrimaryContainer.copy(.6f)
                    )

                    TimeTableType.EXAM -> Pair(
                        MaterialTheme.colorScheme.errorContainer,
                        MaterialTheme.colorScheme.onErrorContainer.copy(.6f)
                    )
                }
            } else {
                Pair(
                    MaterialTheme.colorScheme.surfaceContainer,
                    MaterialTheme.colorScheme.onSurface.copy(.6f)
                )
            }
            Surface(
                color = if (!hasBackground) color.first else Color.Transparent,
                shape = round,
                modifier = squareModifier
                    .let {
                        if (hasBackground) {
                            it
                                .clip(round)
                                .let {
                                    if (AppVersion.CAN_SHADER) {
                                        it.calendarSquareGlass(
                                            shaderState,
                                            MaterialTheme.colorScheme.surface.copy(
                                                customBackgroundAlpha
                                            ),
                                            enableLiquidGlass,
                                        )
                                    } else {
                                        it
                                    }
                                }
                        } else {
                            it
                        }
                    }
                    .let {
                        if (!hasBackground) {
                            it.clickableWithScale(ClickScale.SMALL.scale) {
                                onSquareClick(listOf(item))
                            }
                        } else {
                            it.clickable {
                                onSquareClick(listOf(item))
                            }
                        }
                    }
                    .let {
                        if (!hasBackground) {
                            val origin = CourseDetailOrigin.CALENDAR_JXGLSTU.t + "@${item.hashCode()}"
                            when (item.type) {
                                TimeTableType.COURSE -> {
                                    it.containerShare(
                                        AppNavRoute.CourseDetail.withArgs(item.name, origin), MaterialTheme.shapes.extraSmall
                                    )
                                }
                                TimeTableType.FOCUS -> {
                                    item.id?.let { id ->
                                        it.containerShare(AppNavRoute.AddEvent.withArgs(id, origin), MaterialTheme.shapes.extraSmall)
                                    } ?: it
                                }
                                TimeTableType.EXAM -> {
                                    it.containerShare(AppNavRoute.Exam.withArgs(origin), MaterialTheme.shapes.extraSmall)
                                }
                            }
                        } else {
                            it
                        }
                    }
            ) {
                Column(
                    modifier = Modifier.fillMaxSize(),
                    verticalArrangement = Arrangement.SpaceBetween,
                    horizontalAlignment = Alignment.CenterHorizontally,
                ) {
                    Text(
                        text = item.startTime,
                        fontSize = timeTextSize,
                        lineHeight = timeTextLineHeight,
                        textAlign = TextAlign.Center,
                        overflow = TextOverflow.Clip,
                        maxLines = 1,
                        color = color.second,
                        modifier = Modifier.fillMaxWidth()
                    )
                    Box(
                        modifier = Modifier
                            .weight(1f) // 占据中间剩余的全部空间
                            .fillMaxWidth(),
                        contentAlignment = Alignment.Center
                    ) {
                        Text(
                            text = item.name + (item.teacher?.let { "@$it" } ?: ""),
                            lineHeight = lineHeight,
                            fontSize = textSize,
                            textAlign = TextAlign.Center,
                            overflow = TextOverflow.Ellipsis,
                            modifier = Modifier.fillMaxWidth()
                        )
                    }
                    Column(
                        modifier = Modifier.fillMaxWidth(),
                        horizontalAlignment = Alignment.CenterHorizontally
                    ) {
                        item.place?.let {
                            Text(
                                text = it,
                                fontSize = placeTextSize,
                                lineHeight = placeTextLineHeight,
                                textAlign = TextAlign.Center,
                                modifier = Modifier.fillMaxWidth(),
                            )
                        }
                        Text(
                            text = item.endTime,
                            fontSize = timeTextSize,
                            textAlign = TextAlign.Center,
                            lineHeight = timeTextLineHeight,
                            overflow = TextOverflow.Clip,
                            maxLines = 1,
                            color = color.second,
                            modifier = Modifier.fillMaxWidth()
                        )
                    }
                }
            }
        }
    }
}

