/*
 * openScale
 * Copyright (C) 2025 olie.xdev <olie.xdeveloper@googlemail.com>
 *
 * This program is free software: you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation, either version 3 of the License, or
 * (at your option) any later version.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with this program.  If not, see <https://www.gnu.org/licenses/>.
 */
package com.health.openscale.core.bluetooth.libs

class OneByoneLib(// male = 1; female = 0
    private val sex: Int,
    private val age: Int,
    private val height: Float, // low activity = 0; medium activity = 1; high activity = 2
    private val peopleType: Int
) {
    fun getBMI(weight: Float): Float {
        return weight / (((height * height) / 100.0f) / 100.0f)
    }

    fun getLBM(weight: Float, bodyFat: Float): Float {
        return weight - (bodyFat / 100.0f * weight)
    }

    fun getMuscle(weight: Float, impedanceValue: Float): Float {
        return ((height * height / impedanceValue * 0.401) + (sex * 3.825) - (age * 0.071) + 5.102).toFloat() / weight * 100.0f
    }

    fun getWater(bodyFat: Float): Float {
        val coeff: Float
        val water = (100.0f - bodyFat) * 0.7f

        if (water < 50) {
            coeff = 1.02f
        } else {
            coeff = 0.98f
        }

        return coeff * water
    }

    fun getBoneMass(weight: Float, impedanceValue: Float): Float {
        var boneMass: Float
        val sexConst: Float
        var peopleCoeff = 0.0f

        when (peopleType) {
            0 -> peopleCoeff = 1.0f
            1 -> peopleCoeff = 1.0427f
            2 -> peopleCoeff = 1.0958f
        }

        boneMass =
            (9.058f * (height / 100.0f) * (height / 100.0f) + 12.226f + (0.32f * weight)) - (0.0068f * impedanceValue)

        if (sex == 1) { // male
            sexConst = 3.49305f
        } else {
            sexConst = 4.76325f
        }

        boneMass = boneMass - sexConst - (age * 0.0542f) * peopleCoeff

        if (boneMass <= 2.2f) {
            boneMass = boneMass - 0.1f
        } else {
            boneMass = boneMass + 0.1f
        }

        boneMass = boneMass * 0.05158f

        if (0.5f > boneMass) {
            return 0.5f
        } else if (boneMass > 8.0f) {
            return 8.0f
        }

        return boneMass
    }

    fun getVisceralFat(weight: Float): Float {
        val visceralFat: Float

        if (sex == 1) {
            if (height < ((1.6f * weight) + 63.0f)) {
                visceralFat =
                    (((weight * 305.0f) / (0.0826f * height * height - (0.4f * height) + 48.0f)) - 2.9f) + (age.toFloat() * 0.15f)

                if (peopleType == 0) {
                    return visceralFat
                } else {
                    return subVisceralFat_A(visceralFat)
                }
            } else {
                visceralFat =
                    ((age.toFloat() * 0.15f) + ((weight * (-0.0015f * height + 0.765f)) - height * 0.143f)) - 5.0f

                if (peopleType == 0) {
                    return visceralFat
                } else {
                    return subVisceralFat_A(visceralFat)
                }
            }
        } else {
            if (((0.5f * height) - 13.0f) > weight) {
                visceralFat =
                    ((age.toFloat() * 0.07f) + ((weight * (-0.0024f * height + 0.691f)) - (height * 0.027f))) - 10.5f

                if (peopleType != 0) {
                    return subVisceralFat_A(visceralFat)
                } else {
                    return visceralFat
                }
            } else {
                visceralFat =
                    (weight * 500.0f) / (((1.45f * height) + 0.1158f * height * height) - 120.0f) - 6.0f + (age.toFloat() * 0.07f)

                if (peopleType == 0) {
                    return visceralFat
                } else {
                    return subVisceralFat_A(visceralFat)
                }
            }
        }
    }

    private fun subVisceralFat_A(visceralFat: Float): Float {
        var visceralFat = visceralFat
        if (peopleType != 0) {
            if (10.0f <= visceralFat) {
                return subVisceralFat_B(visceralFat)
            } else {
                visceralFat = visceralFat - 4.0f
                return visceralFat
            }
        } else {
            if (10.0f > visceralFat) {
                visceralFat = visceralFat - 2.0f
                return visceralFat
            } else {
                return subVisceralFat_B(visceralFat)
            }
        }
    }

    private fun subVisceralFat_B(visceralFat: Float): Float {
        var visceralFat = visceralFat
        if (visceralFat < 10.0f) {
            visceralFat = visceralFat * 0.85f
            return visceralFat
        } else {
            if (20.0f < visceralFat) {
                visceralFat = visceralFat * 0.85f
                return visceralFat
            } else {
                visceralFat = visceralFat * 0.8f
                return visceralFat
            }
        }
    }

    fun getBodyFat(weight: Float, impedanceValue: Float): Float {
        var bodyFatConst = 0f

        if (impedanceValue >= 1200.0f) bodyFatConst = 8.16f
        else if (impedanceValue >= 200.0f) bodyFatConst = 0.0068f * impedanceValue
        else if (impedanceValue >= 50.0f) bodyFatConst = 1.36f

        val peopleTypeCoeff: Float
        var bodyVar: Float
        val bodyFat: Float

        if (peopleType == 0) {
            peopleTypeCoeff = 1.0f
        } else {
            if (peopleType == 1) {
                peopleTypeCoeff = 1.0427f
            } else {
                peopleTypeCoeff = 1.0958f
            }
        }

        bodyVar = (9.058f * height) / 100.0f
        bodyVar = bodyVar * height
        bodyVar = bodyVar / 100.0f + 12.226f
        bodyVar = bodyVar + 0.32f * weight
        bodyVar = bodyVar - bodyFatConst

        if (age > 0x31) {
            bodyFatConst = 7.25f

            if (sex == 1) {
                bodyFatConst = 0.8f
            }
        } else {
            bodyFatConst = 9.25f

            if (sex == 1) {
                bodyFatConst = 0.8f
            }
        }

        bodyVar = bodyVar - bodyFatConst
        bodyVar = bodyVar - (age * 0.0542f)
        bodyVar = bodyVar * peopleTypeCoeff

        if (sex != 0) {
            if (61.0f > weight) {
                bodyVar *= 0.98f
            }
        } else {
            if (50.0f > weight) {
                bodyVar *= 1.02f
            }

            if (weight > 60.0f) {
                bodyVar *= 0.96f
            }

            if (height > 160.0f) {
                bodyVar *= 1.03f
            }
        }

        bodyVar = bodyVar / weight
        bodyFat = 100.0f * (1.0f - bodyVar)

        if (1.0f > bodyFat) {
            return 1.0f
        } else {
            if (bodyFat > 45.0f) {
                return 45.0f
            } else {
                return bodyFat
            }
        }
    }
}
