import 'dart:async';
import 'dart:convert';
import 'dart:typed_data';

import 'package:http/http.dart';
import 'package:http_client_helper/src/cancellation_token.dart';
import 'package:http_client_helper/src/retry_helper.dart';

class HttpClientHelper {
  HttpClientHelper._();
  static Client _httpClient = Client();
  static Client get httpClient => _httpClient;
  static void set(Client value) {
    _httpClient = value;
  }

  static void setClient(Client client) {
    _httpClient = client;
  }

  //http get with cancel, delay try again
  static Future<Response?> get(
    Uri url, {
    Map<String, String>? headers,
    CancellationToken? cancelToken,
    int retries = 3,
    Duration? timeLimit,
    Duration timeRetry = const Duration(milliseconds: 100),
    FutureOr<Response> Function()? onTimeout,
  }) async {
    cancelToken?.throwIfCancellationRequested();
    return await RetryHelper.tryRun<Response>(
      () {
        return CancellationTokenSource.register(
          cancelToken,
          timeLimit == null
              ? _httpClient.get(
                  url,
                  headers: headers,
                )
              : _httpClient
                  .get(
                    url,
                    headers: headers,
                  )
                  .timeout(
                    timeLimit,
                    onTimeout: onTimeout,
                  ),
        );
      },
      cancelToken: cancelToken,
      timeRetry: timeRetry,
      retries: retries,
    );
  }

  //http post with cancel, delay try again
  static Future<Response?> post(
    Uri url, {
    Map<String, String>? headers,
    Object? body,
    Encoding? encoding,
    CancellationToken? cancelToken,
    int retries = 3,
    Duration? timeLimit,
    Duration timeRetry = const Duration(milliseconds: 100),
    FutureOr<Response> Function()? onTimeout,
  }) async {
    cancelToken?.throwIfCancellationRequested();
    return await RetryHelper.tryRun<Response>(
      () {
        return CancellationTokenSource.register(
          cancelToken,
          timeLimit == null
              ? _httpClient.post(
                  url,
                  headers: headers,
                  body: body,
                  encoding: encoding,
                )
              : _httpClient
                  .post(
                    url,
                    headers: headers,
                    body: body,
                    encoding: encoding,
                  )
                  .timeout(
                    timeLimit,
                    onTimeout: onTimeout,
                  ),
        );
      },
      cancelToken: cancelToken,
      timeRetry: timeRetry,
      retries: retries,
    );
  }

  //http head with cancel, delay try again
  static Future<Response?> head(
    Uri url, {
    Map<String, String>? headers,
    CancellationToken? cancelToken,
    int retries = 3,
    Duration? timeLimit,
    Duration timeRetry = const Duration(milliseconds: 100),
    FutureOr<Response> Function()? onTimeout,
  }) async {
    cancelToken?.throwIfCancellationRequested();
    return await RetryHelper.tryRun<Response>(
      () {
        return CancellationTokenSource.register(
          cancelToken,
          timeLimit == null
              ? _httpClient.head(
                  url,
                  headers: headers,
                )
              : _httpClient
                  .head(
                    url,
                    headers: headers,
                  )
                  .timeout(
                    timeLimit,
                    onTimeout: onTimeout,
                  ),
        );
      },
      cancelToken: cancelToken,
      timeRetry: timeRetry,
      retries: retries,
    );
  }

  //http put with cancel, delay try again
  static Future<Response?> put(
    Uri url, {
    Map<String, String>? headers,
    Object? body,
    Encoding? encoding,
    CancellationToken? cancelToken,
    int retries = 3,
    Duration? timeLimit,
    Duration timeRetry = const Duration(milliseconds: 100),
    FutureOr<Response> Function()? onTimeout,
  }) async {
    cancelToken?.throwIfCancellationRequested();
    return await RetryHelper.tryRun<Response>(() {
      return CancellationTokenSource.register(
        cancelToken,
        timeLimit == null
            ? _httpClient.put(
                url,
                headers: headers,
                body: body,
                encoding: encoding,
              )
            : _httpClient
                .put(
                  url,
                  headers: headers,
                  body: body,
                  encoding: encoding,
                )
                .timeout(
                  timeLimit,
                  onTimeout: onTimeout,
                ),
      );
    }, cancelToken: cancelToken, timeRetry: timeRetry, retries: retries);
  }

  //http patch with cancel, delay try again
  static Future<Response?> patch(
    Uri url, {
    Map<String, String>? headers,
    Object? body,
    Encoding? encoding,
    CancellationToken? cancelToken,
    int retries = 3,
    Duration? timeLimit,
    Duration timeRetry = const Duration(milliseconds: 100),
    FutureOr<Response> Function()? onTimeout,
  }) async {
    cancelToken?.throwIfCancellationRequested();
    return await RetryHelper.tryRun<Response>(
      () {
        return CancellationTokenSource.register(
          cancelToken,
          timeLimit == null
              ? _httpClient.patch(
                  url,
                  headers: headers,
                  body: body,
                  encoding: encoding,
                )
              : _httpClient
                  .patch(
                    url,
                    headers: headers,
                    body: body,
                    encoding: encoding,
                  )
                  .timeout(
                    timeLimit,
                    onTimeout: onTimeout,
                  ),
        );
      },
      cancelToken: cancelToken,
      timeRetry: timeRetry,
      retries: retries,
    );
  }

  //http delete with cancel, delay try again
  static Future<Response?> delete(
    Uri url, {
    Map<String, String>? headers,
    CancellationToken? cancelToken,
    int retries = 3,
    Duration? timeLimit,
    Duration timeRetry = const Duration(milliseconds: 100),
    FutureOr<Response> Function()? onTimeout,
  }) async {
    cancelToken?.throwIfCancellationRequested();
    return await RetryHelper.tryRun<Response>(
      () {
        return CancellationTokenSource.register(
          cancelToken,
          timeLimit == null
              ? _httpClient.delete(
                  url,
                  headers: headers,
                )
              : _httpClient
                  .delete(
                    url,
                    headers: headers,
                  )
                  .timeout(
                    timeLimit,
                    onTimeout: onTimeout,
                  ),
        );
      },
      cancelToken: cancelToken,
      timeRetry: timeRetry,
      retries: retries,
    );
  }

  //http read with cancel, delay try again
  static Future<String?> read(
    Uri url, {
    Map<String, String>? headers,
    CancellationToken? cancelToken,
    int retries = 3,
    Duration? timeLimit,
    Duration timeRetry = const Duration(milliseconds: 100),
    FutureOr<String> Function()? onTimeout,
  }) async {
    cancelToken?.throwIfCancellationRequested();
    return await RetryHelper.tryRun<String>(
      () {
        return CancellationTokenSource.register(
          cancelToken,
          timeLimit == null
              ? _httpClient.read(
                  url,
                  headers: headers,
                )
              : _httpClient
                  .read(
                    url,
                    headers: headers,
                  )
                  .timeout(
                    timeLimit,
                    onTimeout: onTimeout,
                  ),
        );
      },
      cancelToken: cancelToken,
      timeRetry: timeRetry,
      retries: retries,
    );
  }

  //http readBytes with cancel, delay try again
  static Future<Uint8List?> readBytes(Uri url,
      {Map<String, String>? headers,
      CancellationToken? cancelToken,
      int retries = 3,
      Duration? timeLimit,
      Duration timeRetry = const Duration(milliseconds: 100),
      FutureOr<Uint8List> Function()? onTimeout}) async {
    cancelToken?.throwIfCancellationRequested();
    return await RetryHelper.tryRun<Uint8List>(
      () {
        return CancellationTokenSource.register(
          cancelToken,
          timeLimit == null
              ? _httpClient.readBytes(
                  url,
                  headers: headers,
                )
              : _httpClient
                  .readBytes(
                    url,
                    headers: headers,
                  )
                  .timeout(
                    timeLimit,
                    onTimeout: onTimeout,
                  ),
        );
      },
      cancelToken: cancelToken,
      timeRetry: timeRetry,
      retries: retries,
    );
  }

  //http send with cancel, delay try again
  static Future<StreamedResponse?> send(
    BaseRequest request, {
    CancellationToken? cancelToken,
    int retries = 3,
    Duration? timeLimit,
    Duration timeRetry = const Duration(milliseconds: 100),
    FutureOr<StreamedResponse> Function()? onTimeout,
  }) async {
    cancelToken?.throwIfCancellationRequested();
    return await RetryHelper.tryRun<StreamedResponse>(
      () {
        return CancellationTokenSource.register(
          cancelToken,
          timeLimit == null
              ? _httpClient.send(request)
              : _httpClient.send(request).timeout(
                    timeLimit,
                    onTimeout: onTimeout,
                  ),
        );
      },
      cancelToken: cancelToken,
      timeRetry: timeRetry,
      retries: retries,
    );
  }

  //http close
  static void close() {
    _httpClient.close();
  }
}
