#ifndef GDK_GA_SESSION_HPP
#define GDK_GA_SESSION_HPP
#pragma once

#include <array>
#include <chrono>
#include <map>
#include <optional>
#include <string>
#include <vector>

#include "amount.hpp"
#include "ga_wally.hpp"
#include "session_impl.hpp"

namespace green {

    struct cache;
    class green_user_pubkeys;

    class ga_session final : public session_impl {
    public:
        using nlocktime_t = std::map<std::string, nlohmann::json>; // txhash:pt_idx -> lock info

        explicit ga_session(network_parameters&& net_params);
        ~ga_session();

        void reconnect_hint_session(const nlohmann::json& hint, const nlohmann::json& proxy);

        void emit_notification(nlohmann::json details, bool async);

        nlohmann::json register_user(std::shared_ptr<signer> signer);

        std::string get_challenge(const pub_key_t& public_key);
        nlohmann::json authenticate(const std::string& sig_der_hex, std::shared_ptr<signer> signer);

        void register_subaccount_xpubs(
            const std::vector<uint32_t>& pointers, const std::vector<std::string>& bip32_xpubs);

        nlohmann::json credentials_from_pin_data(const nlohmann::json& pin_data);
        nlohmann::json login_wo(std::shared_ptr<signer> signer);

        nlohmann::json set_wo_credentials(const nlohmann::json& credentials);
        std::string get_watch_only_username();
        bool remove_account(const nlohmann::json& twofactor_data);

        void change_settings_limits(const nlohmann::json& details, const nlohmann::json& twofactor_data);

        nlohmann::json get_subaccounts_impl(locker_t& locker);
        std::vector<uint32_t> get_subaccount_pointers();
        void update_subaccount(uint32_t subaccount, const nlohmann::json& details);
        uint32_t get_next_subaccount(const std::string& sa_type);
        nlohmann::json create_subaccount(nlohmann::json details, uint32_t subaccount, const std::string& xpub);
        nlohmann::json get_receive_address(const nlohmann::json& details);
        nlohmann::json get_previous_addresses(const nlohmann::json& details);
        void set_local_encryption_keys(locker_t& locker, const pub_key_t& public_key, std::shared_ptr<signer> signer);
        nlohmann::json get_available_currencies() const;
        bool is_rbf_enabled() const;

        nlohmann::json get_twofactor_config(bool reset_cached);
        nlohmann::json get_twofactor_config(locker_t& locker, bool reset_cached = false);
        std::vector<std::string> get_enabled_twofactor_methods();

        nlohmann::json get_settings() const;
        void change_settings(const nlohmann::json& settings);

        void set_email(const std::string& email, const nlohmann::json& twofactor_data);
        void activate_email(const std::string& code);
        nlohmann::json init_enable_twofactor(
            const std::string& method, const std::string& data, const nlohmann::json& twofactor_data);
        void enable_twofactor(const std::string& method, const std::string& code);
        void enable_gauth(const std::string& code, const nlohmann::json& twofactor_data);
        void disable_twofactor(const std::string& method, const nlohmann::json& twofactor_data);
        nlohmann::json auth_handler_request_code(
            const std::string& method, const std::string& action, const nlohmann::json& twofactor_data);
        std::string auth_handler_request_proxy_code(const std::string& action, const nlohmann::json& twofactor_data);

        nlohmann::json request_twofactor_reset(const std::string& email);
        nlohmann::json confirm_twofactor_reset(
            const std::string& email, bool is_dispute, const nlohmann::json& twofactor_data);

        nlohmann::json request_undo_twofactor_reset(const std::string& email);
        nlohmann::json confirm_undo_twofactor_reset(const std::string& email, const nlohmann::json& twofactor_data);

        nlohmann::json cancel_twofactor_reset(const nlohmann::json& twofactor_data);

        nlohmann::json encrypt_with_pin(const nlohmann::json& details);
        nlohmann::json decrypt_with_pin(const nlohmann::json& details);
        void disable_all_pin_logins();

        nlohmann::json get_unspent_outputs(const nlohmann::json& details, unique_pubkeys_and_scripts_t& missing);
        void process_unspent_outputs(nlohmann::json& utxos);
        nlohmann::json set_unspent_outputs_status(const nlohmann::json& details, const nlohmann::json& twofactor_data);
        Tx get_raw_transaction_details(const std::string& txhash_hex) const;

        nlohmann::json service_sign_transaction(const nlohmann::json& details, const nlohmann::json& twofactor_data,
            std::vector<std::vector<unsigned char>>& old_scripts);
        nlohmann::json send_transaction(const nlohmann::json& details, const nlohmann::json& twofactor_data);
        nlohmann::json broadcast_transaction(const nlohmann::json& details);

        void send_nlocktimes();
        void set_csvtime(const nlohmann::json& locktime_details, const nlohmann::json& twofactor_data);
        void set_nlocktime(const nlohmann::json& locktime_details, const nlohmann::json& twofactor_data);

        void upload_confidential_addresses(uint32_t subaccount, const std::vector<std::string>& addresses);

        nlohmann::json get_fee_estimates();

        std::string get_system_message();
        std::pair<std::string, std::vector<uint32_t>> get_system_message_info(const std::string& message);
        void ack_system_message(const std::string& message_hash_hex, const std::string& sig_der_hex);

        nlohmann::json convert_amount(const nlohmann::json& amount_json) const;

        bool encache_blinding_data(const std::string& pubkey_hex, const std::string& script_hex,
            const std::string& nonce_hex, const std::string& blinding_pubkey_hex);
        void encache_new_scriptpubkeys(uint32_t subaccount);
        nlohmann::json get_scriptpubkey_data(byte_span_t scriptpubkey);

        amount get_min_fee_rate() const;
        amount get_default_fee_rate() const;
        uint32_t get_block_height() const;
        nlohmann::json get_spending_limits() const;
        bool is_spending_limits_decrease(const nlohmann::json& details);

        std::pair<std::string, bool> get_cached_master_blinding_key();
        void set_cached_master_blinding_key_impl(locker_t& locker, const std::string& master_blinding_key_hex);

        void encache_signer_xpubs(std::shared_ptr<signer> signer);

        nlohmann::json sync_transactions(uint32_t subaccount, unique_pubkeys_and_scripts_t& missing);
        void store_transactions(uint32_t subaccount, nlohmann::json& txs);
        void postprocess_transactions(nlohmann::json& tx_list);
        nlohmann::json get_transactions(const nlohmann::json& details);

    private:
        void reset_cached_session_data(locker_t& locker);
        void delete_reorg_block_txs(locker_t& locker, bool from_latest_cached);
        void reset_all_session_data(bool in_dtor);

        void derive_wallet_identifiers(
            locker_t& locker, nlohmann::json& login_data, const std::vector<unsigned char>& entropy, bool is_relogin);
        nlohmann::json load_client_blob_impl(locker_t& locker);
        nlohmann::json save_client_blob_impl(
            locker_t& locker, const std::string& old_hmac, const std::string& blob_b64, const std::string& hmac);
        void get_cached_local_client_blob(locker_t& locker, const std::string& server_hmac);
        void encache_local_client_blob(
            locker_t& locker, std::string data_b64, byte_span_t data, const std::string& hmac);

        void load_local_signer_xpubs(locker_t& locker, std::shared_ptr<signer> signer);

        void ack_system_message(locker_t& locker, const std::string& message_hash_hex, const std::string& sig_der_hex);

        nlohmann::json sign_or_send_tx(const nlohmann::json& details, const nlohmann::json& twofactor_data,
            bool is_send, std::vector<std::vector<unsigned char>>& old_scripts);
        nlohmann::json get_appearance() const;
        bool subaccount_allows_csv(uint32_t subaccount) const;
        const std::string& get_default_address_type(uint32_t) const;
        void set_twofactor_config(locker_t& locker, const nlohmann::json& config);
        nlohmann::json set_twofactor_reset_config(const nlohmann::json& config);
        void set_enabled_twofactor_methods(locker_t& locker);
        nlohmann::json authenticate_wo(locker_t& locker, const std::string& username, const std::string& password,
            const std::string& user_agent, bool with_blob);
        nlohmann::json on_post_login(locker_t& locker, nlohmann::json& login_data, const std::string& root_bip32_xpub,
            bool watch_only, bool is_relogin);
        void update_fiat_rate(locker_t& locker, const std::string& rate_str);
        void update_spending_limits(locker_t& locker, const nlohmann::json& limits);
        nlohmann::json get_spending_limits(locker_t& locker) const;
        nlohmann::json convert_amount(locker_t& locker, const nlohmann::json& amount_json) const;
        nlohmann::json convert_fiat_cents(locker_t& locker, amount::value_type fiat_cents) const;
        nlohmann::json get_settings(locker_t& locker) const;
        bool unblind_utxo(locker_t& locker, nlohmann::json& utxo, const std::string& for_txhash,
            unique_pubkeys_and_scripts_t& missing);
        std::vector<unsigned char> get_alternate_blinding_nonce(
            locker_t& locker, nlohmann::json& utxo, const std::vector<unsigned char>& nonce_commitment);
        bool cleanup_utxos(session_impl::locker_t& locker, nlohmann::json& utxos, const std::string& for_txhash,
            unique_pubkeys_and_scripts_t& missing);

        std::unique_ptr<locker_t> get_multi_call_locker(uint32_t category_flags, bool wait_for_lock);
        void on_new_transaction(const std::vector<uint32_t>& subaccounts, nlohmann::json details);
        void purge_tx_notification(const std::string& txhash_hex);
        void on_new_block(nlohmann::json details, bool is_relogin);
        void on_new_block(locker_t& locker, nlohmann::json details, bool is_relogin);
        void on_new_tickers(nlohmann::json details);
        void set_pricing_source(
            locker_t& locker, const std::string& currency, const std::string& exchange, bool is_login);

        void remap_appearance_settings(session_impl::locker_t& locker, const nlohmann::json& src_json,
            nlohmann::json& dst_json, bool from_settings) const;

        nlohmann::json insert_subaccount(locker_t& locker, uint32_t subaccount, const std::string& sa_type,
            const std::string& name, const std::string& receiving_id, const std::optional<xpub_hdkey>& recovery_key,
            uint32_t required_ca);

        void set_fee_estimates(locker_t& locker, const nlohmann::json& fee_estimates);

        nlohmann::json refresh_http_data(const std::string& page, const std::string& key, bool refresh);

        void update_address_info(nlohmann::json& address, bool is_historic);
        std::shared_ptr<nlocktime_t> update_nlocktime_info(session_impl::locker_t& locker);

        void save_cache();

        void subscribe_all(locker_t& locker);

        std::vector<unsigned char> get_pin_password(const std::string& pin, const std::string& pin_identifier);
        nlohmann::json decrypt_with_pin_impl(const nlohmann::json& details, bool is_login);

        std::optional<pbkdf2_hmac512_t> m_local_encryption_key;
        std::array<uint32_t, 32> m_gait_path;
        nlohmann::json m_limits_data;
        nlohmann::json m_twofactor_config;
        amount::value_type m_min_fee_rate;
        std::string m_fiat_source;
        std::string m_fiat_rate;
        std::string m_fiat_currency;
        uint64_t m_earliest_block_time;
        uint64_t m_nlocktime;
        uint32_t m_csv_blocks;

        nlohmann::json m_assets;

        std::map<uint32_t, nlohmann::json> m_subaccounts; // Includes 0 for main
        uint32_t m_next_subaccount;
        std::vector<uint32_t> m_fee_estimates;
        std::chrono::system_clock::time_point m_fee_estimates_ts;

        uint32_t m_system_message_id; // Next system message
        uint32_t m_system_message_ack_id; // Currently returned message id to ack
        std::string m_system_message_ack; // Currently returned message to ack
        std::vector<std::string> m_tx_notifications;
        std::chrono::system_clock::time_point m_tx_last_notification;
        nlohmann::json m_last_block_notification;

        uint32_t m_multi_call_category;
        std::shared_ptr<nlocktime_t> m_nlocktimes;

        std::shared_ptr<cache> m_cache;
        std::set<uint32_t> m_synced_subaccounts;
        const std::string m_user_agent;
        std::shared_ptr<wamp_transport> m_wamp;
    };

} // namespace green

#endif
