package com.blockstream.compose.components

import androidx.compose.foundation.Image
import androidx.compose.foundation.layout.*
import androidx.compose.material3.*
import androidx.compose.runtime.Composable
import androidx.compose.ui.Alignment
import androidx.compose.ui.Modifier
import androidx.compose.ui.layout.ContentScale
import androidx.compose.ui.text.style.TextOverflow
import androidx.compose.ui.unit.dp
import androidx.compose.ui.unit.sp
import blockstream_green.common.generated.resources.Res
import blockstream_green.common.generated.resources.id_select_account
import blockstream_green.common.generated.resources.id_select_account__asset
import blockstream_green.common.generated.resources.pencil_simple_line
import com.adamglin.PhosphorIcons
import com.adamglin.phosphoricons.Regular
import com.adamglin.phosphoricons.regular.CaretRight
import com.blockstream.common.extensions.previewAccountAsset
import com.blockstream.common.gdk.GdkSession
import com.blockstream.common.gdk.data.AccountAssetBalance
import com.blockstream.common.utils.StringHolder
import com.blockstream.compose.GreenPreview
import com.blockstream.compose.extensions.assetIcon
import com.blockstream.compose.extensions.policyIcon
import com.blockstream.compose.theme.*
import com.blockstream.ui.components.GreenColumn
import com.blockstream.ui.utils.ifTrue
import org.jetbrains.compose.resources.painterResource
import org.jetbrains.compose.resources.stringResource
import org.jetbrains.compose.ui.tooling.preview.Preview

@Composable
fun GreenAccountAsset(
    modifier: Modifier = Modifier,
    accountAssetBalance: AccountAssetBalance? = null,
    session: GdkSession? = null,
    title: String? = null,
    message: String? = null,
    selectText: String? = null,
    withAsset: Boolean = true,
    withAssetIcon: Boolean = true,
    withEditIcon: Boolean = false,
    withArrow: Boolean = false,
    selectable: Boolean = false,
    isSelected: Boolean = false,
    helperText: String? = null,
    isError: Boolean = false,
    onClick: (() -> Unit)? = null,
) {
    GreenDataLayout(
        modifier = modifier,
        title = title,
        onClick = onClick,
        withPadding = false,
        helperText = helperText,
        helperContainerColor = if(isError) md_theme_errorContainer else null,
    ) {

        Row(
            modifier = Modifier.padding(start = if(selectable) 8.dp else 16.dp),
            verticalAlignment = Alignment.CenterVertically
        ) {
            if (selectable) {
                Checkbox(
                    checked = isSelected,
                    onCheckedChange = { onClick?.invoke() },
                    modifier = Modifier.padding(end = 8.dp)
                )
            }

            if (withAssetIcon) {
                Box {
                    Image(
                        painter = (accountAssetBalance?.asset?.assetId).assetIcon(
                            session = session,
                            isLightning = accountAssetBalance?.account?.isLightning == true
                        ),
                        contentDescription = null,
                        contentScale = ContentScale.Fit,
                        modifier = Modifier
                            .padding(vertical = 16.dp)
                            .padding(end = 8.dp)
                            .size(24.dp)
                    )

                    if (accountAssetBalance != null) {
                        Image(
                            painter = painterResource(accountAssetBalance.account.policyIcon()),
                            contentDescription = "Policy",
                            modifier = Modifier
                                .align(Alignment.BottomEnd)
                                .padding(bottom = 7.dp)
                                .size(18.dp)
                        )
                    }
                }
            }

            Box(
                modifier = Modifier
                    .padding(start = 8.dp)
                    .padding(end = if (withEditIcon && onClick != null) 0.dp else 10.dp)
                    .padding(vertical = 16.dp)
                    .weight(1f),
                contentAlignment = Alignment.CenterStart
            ) {
                if (accountAssetBalance == null) {
                    Text(
                        text = selectText
                            ?: stringResource(if (withAsset) Res.string.id_select_account__asset else Res.string.id_select_account),
                        style = labelLarge,
                        overflow = TextOverflow.Ellipsis
                    )
                } else {
                    Row(
                        modifier = Modifier.fillMaxWidth(),
                        verticalAlignment = Alignment.CenterVertically,
                        horizontalArrangement = Arrangement.spacedBy(8.dp)
                    ) {
                        Column(
                            modifier = Modifier
                                .weight(1f)
                                .ifTrue(message != null) {
                                    it.padding(vertical = 16.dp)
                                }) {
                            val primary =
                                if (withAsset) accountAssetBalance.asset.name(session) else StringHolder.create(accountAssetBalance.account.name)
                            val secondary = if (withAsset) accountAssetBalance.account.name else null

                            // Asset or Account
                            Text(
                                text = primary.string(),
                                style = titleSmall,
                                maxLines = 1,
                                overflow = TextOverflow.Ellipsis,
                                color = whiteHigh
                            )

                            if (secondary != null) {
                                // Account Name
                                Text(
                                    text = secondary,
                                    style = labelLarge,
                                    color = whiteMedium,
                                    maxLines = 1,
                                    overflow = TextOverflow.Ellipsis
                                )
                            }

                            // Account Policy
                            Text(
                                text = accountAssetBalance.account.type.toString().uppercase(),
                                style = labelMedium.copy(fontSize = 8.sp, lineHeight = 12.sp),
                                maxLines = 1,
                                overflow = TextOverflow.Ellipsis,
                                color = whiteLow
                            )

                            message?.also {
                                Text(
                                    text = it,
                                    style = bodyMedium,
                                    color = whiteMedium,
                                    overflow = TextOverflow.Ellipsis,
                                    modifier = Modifier.padding(top = 8.dp)
                                )
                            }
                        }

                        if (accountAssetBalance.balance != null) {
                            Column(horizontalAlignment = Alignment.End) {
                                // Amount
                                Text(
                                    text = accountAssetBalance.balance ?: "",
                                    style = labelLarge,
                                    maxLines = 1,
                                    overflow = TextOverflow.Ellipsis,
                                    color = whiteMedium
                                )

                                accountAssetBalance.balanceExchange?.also {
                                    // Fiat
                                    Text(
                                        text = it,
                                        style = bodyMedium,
                                        maxLines = 1,
                                        overflow = TextOverflow.Ellipsis,
                                        color = whiteMedium,
                                    )
                                }
                            }
                        }
                    }
                }
            }

            if (withEditIcon && onClick != null) {
                IconButton(onClick = onClick) {
                    Icon(
                        painter = painterResource(Res.drawable.pencil_simple_line),
                        contentDescription = "Edit",
                        modifier = Modifier.minimumInteractiveComponentSize()
                    )
                }
            }

            if (withArrow) {
                Icon(
                    PhosphorIcons.Regular.CaretRight,
                    contentDescription = null,
                    tint = whiteMedium,
                    modifier = Modifier.padding(end = 12.dp)
                )
            }
        }
    }
}


@Preview
@Composable
fun GreenAccountAssetPreview() {
    GreenPreview {
        GreenColumn {
            GreenAccountAsset(
                accountAssetBalance = previewAccountAsset().accountAssetBalance,
                title = "Account Asset",
                withAssetIcon = false,
            )
            GreenAccountAsset(
                accountAssetBalance = previewAccountAsset().accountAssetBalance,
                title = "Account",
                withAsset = false,
                withAssetIcon = false,
            )
            GreenAccountAsset(accountAssetBalance = previewAccountAsset().let {
                AccountAssetBalance(
                    account = it.account,
                    asset = it.asset,
                    balance = "123 BTC",
                    balanceExchange = "45,000 USD"
                )
            })
            GreenAccountAsset(accountAssetBalance = previewAccountAsset().let {
                AccountAssetBalance(
                    account = it.account,
                    asset = it.asset,
                    balance = "123 BTC",
                    balanceExchange = "45,000 USD"
                )
            }, withAsset = false)
            GreenAccountAsset(accountAssetBalance = previewAccountAsset().let {
                AccountAssetBalance(
                    account = it.account,
                    asset = it.asset,
                    balance = "123 BTC",
                    balanceExchange = "45,000 USD"
                )
            }, withEditIcon = true)

            GreenAccountAsset(accountAssetBalance = previewAccountAsset().let {
                AccountAssetBalance(
                    account = it.account.copy(
                        gdkName = "Account Name Quite Large"
                    ),
                    asset = it.asset.copy(
                        name = "Bitcoin with Quite Large Name"
                    ),
                    balance = "1,123,1231 BTC",
                    balanceExchange = "23,432,425,445 USD"
                )
            }, withEditIcon = true)
            GreenAccountAsset(
                accountAssetBalance = previewAccountAsset().accountAssetBalance,
                withEditIcon = true,
                onClick = {})
            GreenAccountAsset(withEditIcon = true, onClick = {})
            GreenAccountAsset(
                accountAssetBalance = previewAccountAsset().accountAssetBalance,
                message = "Redeposit Expired 2FA coins",
                withAsset = false,
                withArrow = true,
                onClick = {})
        }
    }
}
