use crate::error::*;
use crate::headers::compute_merkle_root;
use crate::spv::calc_difficulty_retarget;
use electrum_client::GetMerkleRes;
use gdk_common::bitcoin::blockdata::constants::{
    genesis_block, DIFFCHANGE_INTERVAL, TARGET_BLOCK_SPACING,
};
use gdk_common::bitcoin::consensus::{deserialize, serialize};
use gdk_common::bitcoin::hash_types::TxMerkleNode;
use gdk_common::bitcoin::{block, CompactTarget, Network};
use gdk_common::bitcoin::{BlockHash, Txid};
use gdk_common::electrum_client;
use gdk_common::elements::hashes::Hash;
use gdk_common::log::{info, warn};
use gdk_common::once_cell::sync::Lazy;
use std::collections::HashMap;
use std::fs::{File, OpenOptions};
use std::io::{Read, Seek, SeekFrom, Write};
use std::iter::FromIterator;
use std::path::{Path, PathBuf};
use std::str::FromStr;
use std::sync::Mutex;

pub static HEADERS_FILE_MUTEX: Lazy<HashMap<Network, Mutex<()>>> = Lazy::new(|| {
    HashMap::from_iter([
        (Network::Bitcoin, Mutex::new(())),
        (Network::Testnet, Mutex::new(())),
        (Network::Regtest, Mutex::new(())),
        (Network::Signet, Mutex::new(())), // unused
    ])
});

#[derive(Debug)]
pub struct HeadersChain {
    path: PathBuf,
    height: u32,
    last: block::Header,
    checkpoints: HashMap<u32, BlockHash>,
    pub network: Network,
}

impl HeadersChain {
    /// Create a chain of headers which is persisted inside given `path` parameter using a file name
    /// dependent on the given `network`
    ///
    /// if the file doesn't exist, a chain with only the genesis block (relative to `network`) is returned
    pub fn new<P: AsRef<Path>>(path: P, network: Network) -> Result<HeadersChain, Error> {
        std::fs::create_dir_all(path.as_ref())?;
        let mut filepath: PathBuf = path.as_ref().into();
        filepath.push(format!("headers_chain_{}", network));
        let checkpoints = get_checkpoints(network);
        if !filepath.exists() {
            info!("{:?} chain file doesn't exist, creating", filepath);
            let last = genesis_block(network).header;
            let mut file = File::create(&filepath)?;
            file.write_all(&serialize(&last))?;
            let height = 0;

            Ok(HeadersChain {
                path: filepath,
                height,
                last,
                checkpoints,
                network,
            })
        } else {
            info!("{:?} chain file exists, reading", filepath);
            let mut file = File::open(&filepath)?;
            let file_size = file.metadata()?.len();
            if file_size % 80 != 0 || file_size < 80 {
                return Err(Error::InvalidHeaders);
            }
            let wanted_seek = file_size - 80;
            let effective_seek = file.seek(SeekFrom::Start(wanted_seek))?;
            if wanted_seek != effective_seek {
                warn!("Seek failed wanted:{} effective:{}", wanted_seek, effective_seek);
                return Err(Error::Generic("failed seek".into()));
            }
            let mut buf = [0u8; 80];
            file.read_exact(&mut buf)?;
            let height = (file_size as u32 / 80) - 1;
            let last: block::Header = deserialize(&buf)?;

            Ok(HeadersChain {
                path: filepath,
                height,
                last,
                checkpoints,
                network,
            })
        }
    }

    pub fn height(&self) -> u32 {
        self.height
    }

    fn pow_allow_min_difficulty_blocks(&self) -> bool {
        // Special difficulty rule for testnet and regtest:
        // If the next block's timestamp is more than 2* 10 minutes
        // then allow mining a min-difficulty block.
        // Source: https://github.com/bitcoin/bitcoin/blob/master/src/pow.cpp
        match self.network {
            Network::Testnet | Network::Regtest => true,
            _ => false,
        }
    }

    fn curr_bits(&self) -> Result<u32, Error> {
        if self.pow_allow_min_difficulty_blocks() {
            let mut height = self.height();
            // loop at most DIFFCHANGE_INTERVAL times
            let bits = loop {
                let header = self.get(height)?;
                if height == 0
                    || height % DIFFCHANGE_INTERVAL == 0
                    || header.difficulty(&self.network) != 1
                {
                    break header.bits;
                }
                height -= 1;
            };
            Ok(bits.to_consensus())
        } else {
            Ok(self.tip().bits.to_consensus())
        }
    }

    pub fn get(&self, height: u32) -> Result<block::Header, Error> {
        let mut file = File::open(&self.path)?;
        let wanted_seek = height as u64 * 80;
        let effective_seek = file.seek(SeekFrom::Start(wanted_seek))?;
        if wanted_seek != effective_seek {
            warn!("Seek failed wanted:{} effective:{}", wanted_seek, effective_seek);
            return Err(Error::Generic("failed seek".into()));
        }
        let mut buf = [0u8; 80];
        file.read_exact(&mut buf)?;
        let header: block::Header = deserialize(&buf)?;
        Ok(header)
    }

    /// to handle reorgs, it's necessary to remove some of the last headers
    pub fn remove(&mut self, headers_to_remove: u32) -> Result<(), Error> {
        let headers_to_remove = headers_to_remove.min(self.height);
        let new_height = self.height - headers_to_remove;
        let new_size = (new_height + 1) as u64 * 80;
        let file = OpenOptions::new().write(true).open(&self.path)?;
        self.last = self.get(new_height)?;
        self.height = new_height;
        file.set_len(new_size)?;
        Ok(())
    }

    pub fn tip(&self) -> block::Header {
        self.last
    }

    /// write new headers to the file if checks are passed
    pub fn push(&mut self, new_headers: Vec<block::Header>) -> Result<(), Error> {
        let mut curr_bits = self.curr_bits()?;
        let mut serialized = Vec::with_capacity(new_headers.len() * 80);
        let mut cache = HashMap::new();
        for new_header in new_headers {
            let new_height = self.height + 1;
            if self.last.block_hash() != new_header.prev_blockhash
                || new_header.validate_pow(new_header.target()).is_err()
            {
                return Err(Error::InvalidHeaders);
            }

            if new_height % DIFFCHANGE_INTERVAL == 0 {
                if let Network::Regtest = self.network {
                    // regtest doesn't retarget https://github.com/bitcoin/bitcoin/blob/7fcf53f7b4524572d1d0c9a5fdc388e87eb02416/src/pow.cpp#L51
                } else {
                    let first_height = new_height - DIFFCHANGE_INTERVAL;
                    let first = match cache.remove(&first_height) {
                        Some(header) => header,
                        None => self.get(first_height)?,
                    };
                    let new_target = calc_difficulty_retarget(&first, &self.last);
                    if new_header.bits.to_consensus()
                        != bitcoin_29::BlockHeader::compact_target_from_u256(&new_target)
                    {
                        return Err(Error::InvalidHeaders);
                    }
                    curr_bits = new_header.bits.to_consensus();
                }
            } else {
                if new_header.bits != CompactTarget::from_consensus(curr_bits) {
                    if !self.pow_allow_min_difficulty_blocks()
                        || new_header.difficulty(&self.network) != 1
                        || new_header.time.checked_sub(self.last.time).unwrap_or(0)
                            <= 2 * TARGET_BLOCK_SPACING
                    {
                        return Err(Error::InvalidHeaders);
                    }
                }
            }
            if let Some(hash) = self.checkpoints.get(&new_height) {
                if hash != &new_header.block_hash() {
                    return Err(Error::InvalidHeaders);
                }
                info!("checkpoint {} {} is ok", new_height, hash);
            }
            cache.insert(new_height, new_header.clone());
            serialized.extend(serialize(&new_header));
            self.last = new_header;
            self.height = new_height;
        }
        self.flush(&mut serialized)?;
        info!(
            "chain tip height {} hash {} file {:?}",
            self.height,
            self.tip().block_hash(),
            self.path
        );
        Ok(())
    }

    /// verify the given txid and the proof against our chain of headers
    pub fn verify_tx_proof(
        &self,
        txid: &Txid,
        height: u32,
        merkle: GetMerkleRes,
    ) -> Result<(), Error> {
        let calculated_merkle_root =
            TxMerkleNode::from_byte_array(compute_merkle_root(txid.to_byte_array(), merkle)?);

        let header = self.get(height)?;
        if header.merkle_root == calculated_merkle_root {
            info!("proof for txid {}, block height {}, merkle root matches", txid, height);
            Ok(())
        } else {
            Err(Error::InvalidHeaders)
        }
    }

    /// write `serialized` bytes to the file, forcing flush so we are sure next `get()` will have
    /// also this data if requested
    fn flush(&mut self, serialized: &mut Vec<u8>) -> Result<(), Error> {
        if !serialized.is_empty() {
            let mut file = OpenOptions::new().append(true).open(&self.path)?;
            file.write_all(&serialized)?;
            file.flush()?;
            serialized.clear();
        }
        Ok(())
    }
}

fn get_checkpoints(network: Network) -> HashMap<u32, BlockHash> {
    let mut checkpoints = HashMap::new();
    let mut i = |n, s| checkpoints.insert(n, BlockHash::from_str(s).unwrap());
    match network {
        Network::Bitcoin => {
            i(100_000, "000000000003ba27aa200b1cecaad478d2b00432346c3f1f3986da1afd33e506");
            i(200_000, "000000000000034a7dedef4a161fa058a2d67a173a90155f3a2fe6fc132e0ebf");
            i(300_000, "000000000000000082ccf8f1557c5d40b21edabb18d2d691cfbf87118bac7254");
            i(400_000, "000000000000000004ec466ce4732fe6f1ed1cddc2ed4b328fff5224276e3f6f");
            i(500_000, "00000000000000000024fb37364cbf81fd49cc2d51c09c75c35433c3a1945d04");
            i(600_000, "00000000000000000007316856900e76b4f7a9139cfbfba89842c8d196cd5f91");
            i(700_000, "0000000000000000000590fc0f3eba193a278534220b2b37e9849e1a770ca959");
            i(720_000, "00000000000000000000664d48a530c8a9047ae31f6ba81ff5c49c22072d4536");
            i(800_000, "00000000000000000002a7c4c1e48d76c5a37902165a270156b7a8d72728a054");
        }
        Network::Testnet => {
            i(1_000_000, "0000000000478e259a3eda2fafbeeb0106626f946347955e99278fe6cc848414");
            i(2_000_000, "000000000000010dd0863ec3d7a0bae17c1957ae1de9cbcdae8e77aad33e3b8c");
            i(2_100_000, "000000000000002befeeec5aaa3b675ef421896c870e28669f00b0932e277eef");
            i(3_000_000, "0000000000003c46fc60e56b9c2ae202b1efec83fcc7899d21de16757dea40a4");
        }
        _ => (),
    };
    checkpoints
}

#[cfg(test)]
mod test {
    use crate::headers::bitcoin::HeadersChain;
    use gdk_common::bitcoin::block;
    use gdk_common::bitcoin::consensus::encode::Decodable;
    use gdk_common::bitcoin::hash_types::BlockHash;
    use gdk_common::bitcoin::hashes::hex::FromHex;
    use gdk_common::bitcoin::{Network, Txid};
    use gdk_common::electrum_client::GetMerkleRes;
    use std::str::FromStr;
    use tempfile::TempDir;

    #[test]
    fn test_headers() {
        // first 199 bitcoin block headers, excluding the genesis_block
        let bitcoin_headers = "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";
        let bitcoin_headers = Vec::<u8>::from_hex(bitcoin_headers).unwrap();
        let mut parsed_headers = vec![];
        let mut i = 0;
        while let Ok(header) = block::Header::consensus_decode(&mut &bitcoin_headers[i..]) {
            parsed_headers.push(header);
            i += 80;
        }
        assert_eq!(parsed_headers.len(), 199);

        let temp = TempDir::new().unwrap();
        let mut chain = HeadersChain::new(&temp, Network::Bitcoin).unwrap();
        chain.push(parsed_headers).unwrap();
        assert_eq!(chain.height(), 199);

        assert_eq!(
            BlockHash::from_str("000000007bc154e0fa7ea32218a72fe2c1bb9f86cf8c9ebf9a715ed27fdb229a")
                .unwrap(),
            chain.get(100).unwrap().block_hash()
        );

        // first non-coinbase tx
        let txid =
            Txid::from_str("f4184fc596403b9d638783cf57adfe4c75c605f6356fbc91338530e9831e9e16")
                .unwrap();
        let block_height = 170;
        let merkle_tree = GetMerkleRes {
            block_height,
            pos: 1,
            merkle: vec![[
                177, 254, 165, 36, 134, 206, 12, 98, 187, 68, 43, 83, 10, 63, 1, 50, 184, 38, 199,
                78, 71, 61, 31, 44, 34, 11, 250, 120, 17, 28, 80, 130,
            ]],
        };
        chain.verify_tx_proof(&txid, block_height as u32, merkle_tree).unwrap();

        // test a coinbase,
        let txid =
            Txid::from_str("4a5e1e4baab89f3a32518a88c31bc87f618f76673e2cc77ab2127b7afdeda33b")
                .unwrap();
        let block_height = 0;
        let merkle_tree = GetMerkleRes {
            block_height: 0,
            pos: 0,
            merkle: vec![],
        };
        chain.verify_tx_proof(&txid, block_height as u32, merkle_tree).unwrap();

        let merkle_tree = GetMerkleRes {
            block_height: 0,
            pos: 0,
            merkle: vec![],
        };
        assert!(
            chain.verify_tx_proof(&txid, 1, merkle_tree).is_err(),
            "wrong block height should error"
        );

        // first non-coinbase tx, changed first byte of merkle proof
        let txid =
            Txid::from_str("f4184fc596403b9d638783cf57adfe4c75c605f6356fbc91338530e9831e9e16")
                .unwrap();
        let block_height = 170;
        let merkle_tree = GetMerkleRes {
            block_height,
            pos: 1,
            merkle: vec![[
                176, 254, 165, 36, 134, 206, 12, 98, 187, 68, 43, 83, 10, 63, 1, 50, 184, 38, 199,
                78, 71, 61, 31, 44, 34, 11, 250, 120, 17, 28, 80, 130,
            ]],
        };
        assert!(chain.verify_tx_proof(&txid, block_height as u32, merkle_tree).is_err());

        assert!(
            chain.push(vec![chain.get(100).unwrap()]).is_err(),
            "pushing a previous block should err"
        );

        let old_tip = chain.tip();
        chain.remove(1).unwrap();
        assert_eq!(chain.height, 198);
        assert!(chain.get(199).is_err());
        chain.push(vec![old_tip]).unwrap();
        assert_eq!(chain.height, 199);
        assert_eq!(
            BlockHash::from_str("00000000e85458c1467176b04a65d5efaccfecaaab717b17a587b4069276e143")
                .unwrap(),
            chain.get(198).unwrap().block_hash()
        );
        assert_eq!(
            BlockHash::from_str("00000000b7691ccc084542565697eca256e56bb7f67e560b48789db27f0468eb")
                .unwrap(),
            chain.get(199).unwrap().block_hash()
        );
        assert!(chain.get(200).is_err());
    }
}
