import { select, selectAll, pointer } from 'd3-selection'
import { scaleOrdinal, scaleLinear } from 'd3-scale'
import { min as d3Min, max as d3Max, bisector } from 'd3-array'
import { drag } from 'd3-drag'
import { axisLeft, axisBottom } from 'd3-axis'
import { curveBasis, curveLinear, line, area as d3Area, symbol, symbolTriangle } from 'd3-shape'
import {
    schemeAccent,
    schemeDark2,
    schemeSet2,
    schemeCategory10,
    schemeSet3,
    schemePaired
} from 'd3-scale-chromatic'

const defaultOptions = {
    width: 800,
    height: 280,
    margins: {
        top: 10,
        right: 30,
        bottom: 55,
        left: 50
    },
    imperial: false,
    expand: true,
    expandControls: true,
    translation: {},
}

/**
 * @param options
 *   - pointSelectedCallback: receives a point ({lng, lat}), an elevation (number) and a description (string)
 *                            the point can be null which means no point was selected
 *   - areaSelectedCallback: receives the bounds of a selected area rectangle ({sw, ne})
 *   - routeSegmentsSelectedCallback: receives an array of points ([{lng, lat}])
 */
export class HeightGraph {
    constructor(container, options = {}, callbacks = {}) {
        this._container = container;
        options = Object.assign(defaultOptions, options);
        this._margin = options.margins;
        this._width = options.width;
        this._height = options.height;
        this._graphStyle = options.graphStyle || {}
        this._xTicks = options.xTicks;
        this._yTicks = options.yTicks;
        this._translation = options.translation;
        this._imperial = options.imperial;
        this._expand = options.expand;
        this._expandControls = options.expandControls;
        this._expandCallback = options.expandCallback;
        this._pointSelected = callbacks.pointSelectedCallback || Function();
        this._areaSelected = callbacks.areaSelectedCallback || Function();
        this._routeSegmentsSelected = callbacks.routeSegmentsSelectedCallback || Function();
        this._chooseSelectionCallback = options.chooseSelectionCallback;

        this._defaultTranslation = {
            distance: "Distance",
            elevation: "Elevation",
            segment_length: "Segment length",
            type: "Type",
            legend: "Legend"
        }

        this._d3ColorCategorical = [
            schemeAccent,
            schemeDark2,
            schemeSet2,
            schemeCategory10,
            schemeSet3,
            schemePaired
        ]

        this._containerMouseUpHandler = () => this._mouseUpHandler();

        this._svgWidth = this._width - this._margin.left - this._margin.right;
        this._svgHeight = this._height - this._margin.top - this._margin.bottom;
        if (this._expandControls) {
            this._button = this._createDOMElement('div', "heightgraph-toggle", this._container);
            this._createDOMElement("a", "heightgraph-toggle-icon", this._button);
            this._button.addEventListener('click', () => this._expandContainer());

            this._closeButton = this._createDOMElement("a", "heightgraph-close-icon", this._container)
            this._closeButton.addEventListener('click', () => this._expandContainer());
        }
        this._showState = false;
        this._stopPropagation();

        // Note: this._svg really contains the <g> inside the <svg>
        this._svg = select(this._container).append("svg").attr("class", "heightgraph-container")
            .attr("width", this._width)
            .attr("height", this._height).append("g")
            .attr("transform", "translate(" + this._margin.left + "," + this._margin.top + ")")

        // legend
        this._legendMain = document.createElement('div');
        this._legendMain.id = 'heightgraph-legend';
        this._legend = document.createElement('ul');
        this._legendMain.append(this._legend)
        const legendText = document.createElement('span');
        legendText.id = 'heightgraph-legend-text';
        legendText.innerHTML = this._getTranslation("legend");
        this._legendMain.append(legendText)
        legendText.addEventListener("click", e => {
            this._legend.classList.toggle("heightgraph-show");
        });
        this._container.append(this._legendMain);

        // switch between categories
        this._selectorMain = document.createElement('div');
        this._selectorMain.id = 'heightgraph-selector'
        this._container.append(this._selectorMain);
        this._optionsSelect = document.createElement('ul');
        this._selectorMain.append(this._optionsSelect);
        this._optionsSelectInput = document.createElement('input');
        this._selectorMain.append(this._optionsSelectInput);
        const arrowIconForInput = document.createElement('span');
        arrowIconForInput.innerHTML = '&#9662;'
        arrowIconForInput.id = 'arrow-icon-for-input'
        this._selectorMain.append(arrowIconForInput);
        this._optionsSelectInput.addEventListener("click", e => {
            this._optionsSelectInput.select();
            for (let i = 0; i < this._optionsSelect.children.length; i++)
                this._optionsSelect.children[i].style.display = ''
            this._optionsSelect.classList.toggle("heightgraph-show");
        });

        this._optionsSelectInput.addEventListener('input', e => {
            const filterValue = this._optionsSelectInput.value.toLowerCase();
            for (let i = 0; i < this._optionsSelect.children.length; i++) {
                const item = this._optionsSelect.children[i];
                const text = item.textContent.toLowerCase();
                item.style.display = text.indexOf(filterValue) > -1 ? '' : 'none';
            }
        });

        this._optionsSelect.addEventListener('click', e => {
            this._optionsSelectInput.value = e.target.textContent;
            this._optionsSelect.classList.remove("heightgraph-show");

            const idx = this._currentSelection = e.target.getAttribute('value');
            if (this._categories.length === 0) return;
            if (typeof this._chooseSelectionCallback === "function") {
                this._chooseSelectionCallback(idx, this._categories[idx].info);
            }

            // todo: no real reason to clear this, but if we keep it we should make sure the horizontal line is not removed
            this._routeSegmentsSelected([]);
            this._removeChart();
            this._createChart(idx);
            this._removeDragRectangle();
            if (this._rectangle)
                this._drawDragRectangle(this._rectangle[0], this._rectangle[1]);
        })

        if (this._expand) this._expandContainer();
    }

    resize(size) {
        if (size.width)
            this._width = size.width;
        if (size.height)
            this._height = size.height;

        // Resize the <svg> along with its container
        select(this._container).selectAll("svg")
            .attr("width", this._width)
            .attr("height", this._height);
        this._svgWidth = this._width - this._margin.left - this._margin.right;
        this._svgHeight = this._height - this._margin.top - this._margin.bottom;

        // Re-add the data to redraw the chart.
        this.redraw()
    }

    redraw() {
        this.setData(this._data, this._mappings, this._currentSelection);
    }

    setData(data, mappings, selection) {
        if (typeof selection !== 'undefined')
            this._currentSelection = selection;
        if (typeof this._currentSelection === 'undefined' || this._currentSelection >= data.length)
            this._currentSelection = 0;
        this._mappings = mappings;
        this._routeSegmentsSelected([]);
        if (this._svg !== undefined) {
            this._svg.selectAll("*")
                .remove();
        }
        this._removeDragRectangle();
        this._rectangle = null;

        this._data = data;
        this._prepareData();
        this._calculateElevationBounds();
        this._appendScales();
        this._appendGrid();
        if (Object.keys(data).length !== 0) {
            this._createChart(this._currentSelection);
        }
        this._createOptions();
    }

    _createOptions() {
        // remove previously added options
        while(this._optionsSelect.firstChild) this._optionsSelect.removeChild(this._optionsSelect.lastChild);

        this._data.forEach((k, idx) => {
            const label = k.properties.label ? k.properties.label : k.properties.summary
            if (idx === this._currentSelection) this._optionsSelectInput.value = label
            let option = document.createElement('li');
            this._optionsSelect.append(option);
            option.setAttribute('value', idx);
            option.innerHTML = label;
        })
    }

    _stopPropagation() {
        this._container.addEventListener('click mousedown touchstart dblclick', e => {
            if (e.stopPropagation) {
                e.stopPropagation();
            } else if (e.originalEvent) {
                e.originalEvent._stopped = true;
            } else {
                e.cancelBubble = true;
            }
        });
    }

    _dragHandler(event) {
        //we don´t want map events to occur here
        if (typeof event !== 'undefined') {
            event.preventDefault();
            event.stopPropagation();
        }
        this._gotDragged = true;
        if (this._dragStartCoords) {
            const dragEndCoords = this._dragCurrentCoords = pointer(event, this._background.node())
            const x1 = Math.min(this._dragStartCoords[0], dragEndCoords[0]),
                x2 = Math.max(this._dragStartCoords[0], dragEndCoords[0])
            this._drawDragRectangle(x1, x2);
        }
    }

    /**
     * Draws a rectangle between the given x-values over the chart.
     */
    _drawDragRectangle(x1, x2) {
        this._rectangle = [x1, x2]
        // todonow: sometimes (depending on which path detail we select) the rectangle is shown *behind* the graph?!
        if (!this._dragRectangle && !this._dragRectangleG) {
            const g = select(this._container).select("svg").select("g")
            this._dragRectangleG = g.append("g");
            this._dragRectangle = this._dragRectangleG.append("rect")
                .attr("width", x2 - x1)
                .attr("height", this._svgHeight)
                .attr("x", x1)
                .attr('class', 'mouse-drag')
                .style("fill", "grey")
                .style("opacity", 0.5)
                .style("pointer-events", "none");
        } else {
            this._dragRectangle.attr("width", x2 - x1)
                .attr("x", x1);
        }
    }

    _removeDragRectangle() {
        if (this._dragRectangleG) {
            this._dragRectangleG.remove();
            this._dragRectangleG = null;
            this._dragRectangle = null;
        }
    }

    getBounds() {
        return this._calculateFullExtent(this._areasFlattended);
    }

    isImperial() {
        return this._imperial;
    }

    setImperial(imperial) {
        this._imperial = imperial;
    }

    _mouseUpHandler() {
        if (this._arrowClick) {
            this._arrowClick = false;
            return;
        }
        if (!this._dragStartCoords || !this._gotDragged) {
            // this was just a click on the graph
            this._dragStartCoords = null;
            this._gotDragged = false;
            // if there was a rectangle we remove it now
            if (this._dragRectangleG) {
                this._removeDragRectangle()
                this._rectangle = null;
                this._areaSelected(null)
            }
            return;
        }
        // this was a drag
        const item1 = this._findItemForX(this._dragStartCoords[0]),
            item2 = this._findItemForX(this._dragCurrentCoords[0])
        this._fitSection(item1, item2);
        this._dragStartCoords = null;
        this._gotDragged = false;
    }

    _dragStartHandler(event) {
        event.preventDefault();
        event.stopPropagation();
        this._gotDragged = false;
        this._dragStartCoords = pointer(event, this._background.node());
    }

    /*
     * Calculates the full extent of the data array
     */
    _calculateFullExtent(data) {
        if (!data || data.length < 1) {
            return null;
        }
        let minLat = data[0].latlng.lat;
        let minLng = data[0].latlng.lng;
        let maxLat = data[0].latlng.lat;
        let maxLng = data[0].latlng.lng;
        data.forEach(c => {
            minLng = Math.min(minLng, c.latlng.lng);
            minLat = Math.min(minLat, c.latlng.lat);
            maxLng = Math.max(maxLng, c.latlng.lng);
            maxLat = Math.max(maxLat, c.latlng.lat);
        });
        return {
            sw: this._createCoordinate(minLng, minLat),
            ne: this._createCoordinate(maxLng, maxLat)
        };
    }

    /**
     * Make the map fit the route section between given indexes.
     */
    _fitSection(index1, index2) {
        const start = Math.min(index1, index2), end = Math.max(index1, index2)
        let ext
        if (start !== end) {
            ext = this._calculateFullExtent(this._areasFlattended.slice(start, end + 1));
        } else if (this._areasFlattended.length > 0) {
            ext = [this._areasFlattended[start].latlng, this._areasFlattended[end].latlng];
        }
        if (ext) this._areaSelected(ext);
    }

    /**
     * Expand container when button clicked and shrink when close-Button clicked
     */
    _expandContainer() {
        if (this._expandControls !== true) {
            // always expand, never collapse
            this._showState = false;
        }
        if (!this._showState) {
            select(this._button).style("display", "none");
            select(this._container).selectAll('svg').style("display", "block");
            select(this._closeButton).style("display", "block");
            select(this._legendMain).style("display", "block");
            select(this._selectorMain).style("display", "block");
        } else {
            select(this._button).style("display", "block");
            select(this._container).selectAll('svg').style("display", "none");
            select(this._closeButton).style("display", "none");
            select(this._legendMain).style("display", "none");
            select(this._selectorMain).style("display", "none");
        }
        this._showState = !this._showState;
        if (typeof this._expandCallback === "function") {
            this._expandCallback(this._showState);
        }
    }

    /**
     * Removes the svg elements from the d3 chart
     */
    _removeChart() {
        if (this._svg !== undefined) {
            // remove areas
            this._svg.selectAll("path.area")
                .remove();
            // remove top border
            this._svg.selectAll("path.border-top")
                .remove();
            // remove legend
            this._svg.selectAll(".legend")
                .remove();
            // remove horizontal Line
            this._svg.selectAll(".lineSelection")
                .remove();
            this._svg.selectAll(".horizontalLine")
                .remove();
            this._svg.selectAll(".horizontalLineText")
                .remove();
        }
    }

    /**
     * Creates a random int between 0 and max
     */
    _randomNumber(max) {
        return Math.round((Math.random() * (max - 0)));
    }

    /**
     * Prepares the data needed for the height graph
     */
    _prepareData() {
        this._coordinates = [];
        this._elevations = [];
        this._cumulatedDistances = [];
        this._cumulatedDistances.push(0);
        this._categories = [];
        const data = this._data
        let colorScale
        if (this._mappings === undefined) {
            const randomNumber = this._randomNumber(this._d3ColorCategorical.length - 1)
            colorScale = scaleOrdinal(this._d3ColorCategorical[randomNumber]);
        }
        for (let y = 0; y < data.length; y++) {
            let cumDistance = 0
            this._categories[y] = {
                info: {
                    id: y,
                    text: data[y].properties.label || data[y].properties.summary
                },
                distances: [],
                attributes: [],
                geometries: [],
                legend: {}
            };
            let i, cnt = 0
            const usedColors = {}
            const isMappingFunction = this._mappings !== undefined && typeof this._mappings[data[y].properties.summary] === 'function';
            for (i = 0; i < data[y].features.length; i++) {
                // data is redundant in every element of data which is why we collect it once
                let altitude, ptA, ptB, ptDistance
                const geometry = []
                const coordsLength = data[y].features[i].geometry.coordinates.length
                // save attribute types related to blocks
                const attributeType = data[y].features[i].properties.attributeType
                // check if mappings are defined, otherwise random colors
                let text, color
                if (this._mappings === undefined) {
                    if (attributeType in usedColors) {
                        text = attributeType;
                        color = usedColors[attributeType];
                    } else {
                        text = attributeType;
                        color = colorScale(i);
                        usedColors[attributeType] = color;
                    }
                } else {
                    if (isMappingFunction) {
                        const result = this._mappings[data[y].properties.summary](attributeType);
                        text = result.text;
                        color = result.color;
                    } else {
                        text = this._mappings[data[y].properties.summary][attributeType].text;
                        color = this._mappings[data[y].properties.summary][attributeType].color;
                    }
                }
                const attribute = {
                    type: attributeType, text: text, color: color
                }
                this._categories[y].attributes.push(attribute);
                // add to legend
                if (!(attributeType in this._categories[y].legend)) {
                    this._categories[y].legend[attributeType] = attribute;
                }
                for (let j = 0; j < coordsLength; j++) {
                    ptA = this._createCoordinate(data[y].features[i].geometry.coordinates[j][0], data[y].features[i].geometry.coordinates[j][1]);
                    altitude = data[y].features[i].geometry.coordinates[j][2];
                    // add elevations, coordinates and point distances only once
                    // last point in feature is first of next which is why we have to juggle with indices
                    if (j < coordsLength - 1) {
                        ptB = this._createCoordinate(data[y].features[i].geometry.coordinates[j + 1][0], data[y].features[i].geometry.coordinates[j + 1][1]);
                        ptDistance = this._calcDistance(ptA, ptB) / 1000;
                        // calculate distances of specific block
                        cumDistance += ptDistance;
                        if (y === 0) {
                            this._elevations.push(altitude);
                            this._coordinates.push(ptA);
                            this._cumulatedDistances.push(cumDistance);
                        }
                        cnt += 1;
                    } else if (j === coordsLength - 1 && i === data[y].features.length - 1) {
                        if (y === 0) {
                            this._elevations.push(altitude);
                            this._coordinates.push(ptB);
                        }
                        cnt += 1;
                    }
                    // save the position which corresponds to the distance along the route.
                    let position
                    if (j === coordsLength - 1 && i < data[y].features.length - 1) {
                        position = this._cumulatedDistances[cnt];
                    } else {
                        position = this._cumulatedDistances[cnt - 1];
                    }
                    geometry.push({
                        altitude: altitude,
                        position: position,
                        x: ptA.lng,
                        y: ptA.lat,
                        latlng: ptA,
                        type: text,
                        areaIdx: i
                    });
                }
                this._categories[y].distances.push(cumDistance);
                this._categories[y].geometries.push(geometry);
            }
            if (y === data.length - 1) {
                this._totalDistance = cumDistance;
            }
        }
    }

    /**
     * calculates minimum and maximum values for the elevation scale drawn with d3
     */
    _calculateElevationBounds() {
        if (this._elevations.length === 0) {
            this._elevationBounds = {min: 0, max: 0};
            return;
        }
        const max = d3Max(this._elevations)
        const min = d3Min(this._elevations)
        const range = max - min
        this._elevationBounds = {
            min: range < 10 ? min - 10 : min - 0.1 * range,
            max: range < 10 ? max + 10 : max + 0.1 * range
        }
    }

    /**
     * Creates the elevation profile
     */
    _createChart(idx) {
        let areas = this._categories.length === 0
            ? []
            : this._categories[idx].geometries;
        this._areasFlattended = [].concat.apply([], areas);
        for (let i = 0; i < areas.length; i++) {
            this._appendAreas(areas[i], idx, i);
        }
        this._createFocus();
        this._appendBackground();
        this._createBorderTopLine();
        this._createLegend();
        this._createHorizontalLine();
    }

    /**
     *  Creates focus Line and focus box while hovering
     */
    _createFocus() {
        const boxPosition = this._elevationBounds.min
        const textDistance = 15
        if (this._focus) {
            this._focus.remove();
            this._focusLineGroup.remove();
        }
        this._focus = this._svg.append("g")
            .attr("class", "focusbox");
        // background box
        this._focusRect = this._focus.append("rect")
            .attr("x", 3)
            .attr("y", -this._y(boxPosition))
            .attr("display", "none");
        // text line 1
        this._focusDistance = this._focus.append("text")
            .attr("x", 7)
            .attr("y", -this._y(boxPosition) + textDistance)
            .attr("id", "heightgraph.distance")
            .text(this._getTranslation('distance') + ':');
        // text line 2
        this._focusHeight = this._focus.append("text")
            .attr("x", 7)
            .attr("y", -this._y(boxPosition) + 2 * textDistance)
            .attr("id", "heightgraph.height")
            .text(this._getTranslation('elevation') + ':');
        // text line 3
        this._focusBlockDistance = this._focus.append("text")
            .attr("x", 7)
            .attr("y", -this._y(boxPosition) + 3 * textDistance)
            .attr("id", "heightgraph.blockdistance")
            .text(this._getTranslation('segment_length') + ':');
        // text line 4
        this._focusType = this._focus.append("text")
            .attr("x", 7)
            .attr("y", -this._y(boxPosition) + 4 * textDistance)
            .attr("id", "heightgraph.type")
            .text(this._getTranslation('type') + ':');
        this._areaTspan = this._focusBlockDistance.append('tspan')
            .attr("class", "tspan");
        this._typeTspan = this._focusType.append('tspan')
            .attr("class", "tspan");
        const height = selectAll(".focusbox text").size();
        selectAll('.focusbox rect')
            .attr("height", height * textDistance + (textDistance / 2))
            .attr("display", "block");
        this._focusLineGroup = this._svg.append("g")
            .attr("class", "focusLine");
        this._focusLine = this._focusLineGroup.append("line")
            .attr("y1", 0)
            .attr("y2", this._y(this._elevationBounds.min));
        this._distTspan = this._focusDistance.append('tspan')
            .attr("class", "tspan");
        this._altTspan = this._focusHeight.append('tspan')
            .attr("class", "tspan");
    }

    /**
     *  Creates horizontal Line for dragging
     */
    _createHorizontalLine() {
        this._horizontalLine = this._svg.append("line")
            .attr("class", "horizontalLine")
            .attr("x1", 0)
            .attr("x2", this._width - this._margin.left - this._margin.right)
            .attr("y1", this._y(this._elevationBounds.min))
            .attr("y2", this._y(this._elevationBounds.min))
            .style("stroke", "black");
        this._elevationValueText = this._svg.append("text")
            .attr("class", "horizontalLineText")
            .attr("x", this._width - this._margin.left - this._margin.right - 20)
            .attr("y", this._y(this._elevationBounds.min) - 10)
            .attr("fill", "black");
        //triangle symbol as controller
        const jsonTriangle = [
            {
                "x": this._width - this._margin.left - this._margin.right + 7,
                "y": this._y(this._elevationBounds.min),
                "color": "black",
                "type": symbolTriangle,
                "angle": -90,
                "size": 100
            }
        ]
        const dragstart = (element) => {
            select(element).raise().classed("active", true)
            select(".horizontalLine").raise().classed("active", true)
        }

        const dragged = (element, event) => {
            const maxY = this._svgHeight
            const eventY = pointer(event, this._container)[1] - 10
            const newY = Math.max(0, Math.min(eventY, maxY));
            select(element)
                .attr("transform", d => "translate(" + d.x + "," + newY + ") rotate(" + d.angle + ")");
            select(".horizontalLine")
                .attr("y1", newY)
                .attr("y2", newY);
            if (eventY >= maxY) {
                this._highlightedCoords = [];
            } else {
                this._highlightedCoords = this._findCoordsForY(eventY);
            }
            select(".horizontalLineText")
                .attr("y", (newY < 10 ? 0 : newY - 10))
                .text(elevationToString(this._y.invert(newY), this._imperial, true));
            this._routeSegmentsSelected(this._highlightedCoords);
        }

        const dragend = (element) => {
            select(element)
                .classed("active", false);
            select(".horizontalLine")
                .classed("active", false);
            this._routeSegmentsSelected(this._highlightedCoords);
        }

        const horizontalDrag = this._svg.selectAll(".horizontal-symbol").data(jsonTriangle).enter().append("path").
        attr("class", "lineSelection")
            .attr("d", symbol().type(d => d.type).size(d => d.size))
            .attr("transform", d => "translate(" + d.x + "," + d.y + ") rotate(" + d.angle + ")")
            .attr("id", d => d.id)
            .style("fill", d => d.color)
            .call(drag()
                .on("start", function(event, d) { dragstart(this); })
                .on("drag", function(event, d) { dragged(this, event); })
                .on("end", function(event, d) { dragend(this); })
            )
    }

    /**
     * Defines the ranges and format of x- and y- scales and appends them
     */
    _appendScales() {
        this._x = scaleLinear()
            .range([0, this._svgWidth]);
        this._y = scaleLinear()
            .range([this._svgHeight, 0]);
        this._x.domain([0, this._totalDistance]);
        this._y.domain([this._elevationBounds.min, this._elevationBounds.max]);
        this._xAxis = axisBottom()
            .scale(this._x)
        this._xAxis.tickFormat(d => distanceToString(d * 1000, this._totalDistance, this._imperial, false))
        this._xAxis.ticks(this._xTicks ? Math.pow(2, this._xTicks) : Math.round(this._svgWidth / 75), "s");
        this._yAxis = axisLeft()
            .scale(this._y)
            .tickFormat(d => elevationToString(d, this._imperial, false));
        this._yAxis.ticks(this._yTicks ? Math.pow(2, this._yTicks) : Math.round(this._svgHeight / 30), "s");
    }

    /**
     * Appends a background and adds mouse handlers
     */
    _appendBackground() {
        const background = this._background = select(this._container)
            .select("svg")
            .select("g")
            .append("rect")
            .attr("width", this._svgWidth)
            .attr("height", this._svgHeight)
            .style("fill", "none")
            .style("stroke", "none")
            .style("pointer-events", "all")
            .on("mousemove.focusbox", (event, d) => this._mousemoveHandler(event))
            .on("mouseout.focusbox", (event, d) => this._mouseoutHandler())
        if (this._isMobile()) {
            background
                .on("touchstart.drag", (event, d) => this._dragHandler(event))
                .on("touchstart.drag", (event, d) => this._dragStartHandler(event))
                .on("touchstart.focusbox", (event, d) => this._mousemoveHandler(event, d));
            // todonow: not working on mobile??
            this._container.addEventListener('touchend', this._containerMouseUpHandler);
        } else {
            background
                .on("mousemove.focusbox", (event, d) => this._mousemoveHandler(event))
                .on("mouseout.focusbox", (event, d) => this._mouseoutHandler())
                .on("mousedown.drag", (event, d) => this._dragStartHandler(event))
                .on("mousemove.drag", (event, d) => this._dragHandler(event));
            // we need the _containerDragEndlistener reference to make sure we do not add multiple listeners
            // when we call appendBackground with different this contexts
            this._container.addEventListener('mouseup', this._containerMouseUpHandler);
        }
    }

    /**
     * Appends a grid to the graph
     */
    _appendGrid() {
        this._svg.append("g")
            .attr("class", "grid")
            .attr("transform", "translate(0," + this._svgHeight + ")")
            .call(this._make_x_axis()
                .tickSize(-this._svgHeight, 0, 0)
                .ticks(Math.round(this._svgWidth / 75))
                .tickFormat(""));
        this._svg.append("g")
            .attr("class", "grid")
            .call(this._make_y_axis()
                .tickSize(-this._svgWidth, 0, 0)
                .ticks(Math.round(this._svgHeight / 30))
                .tickFormat(""));
        this._svg.append('g')
            .attr("transform", "translate(0," + this._svgHeight + ")")
            .attr('class', 'x axis')
            .call(this._xAxis);
        this._svg.append('g')
            .attr("transform", "translate(-2,0)")
            .attr('class', 'y axis')
            .call(this._yAxis);
    }

    /**
     * Appends the areas to the graph
     */
    _appendAreas(block, idx, eleIdx) {
        const c = this._categories[idx].attributes[eleIdx].color
        this._area = d3Area().x(d => {
            const xDiagonalCoordinate = this._x(d.position)
            d.xDiagonalCoordinate = xDiagonalCoordinate
            return xDiagonalCoordinate
        }).y0(this._svgHeight).y1(d => this._y(d.altitude)).curve(curveLinear)
        this._areapath = this._svg.append("path")
            .attr("class", "area");
        this._areapath.datum(block)
            .attr("d", this._area)
            .attr("stroke", c)
        Object.entries(this._graphStyle).forEach(([prop, val]) => {
            this._areapath.style(prop, val)
        })
        this._areapath
            .style("fill", c)
            .style("pointer-events", "none");
    }

    // grid lines in x axis function
    _make_x_axis() {
        return axisBottom()
            .scale(this._x);
    }

    // grid lines in y axis function
    _make_y_axis() {
        return axisLeft()
            .scale(this._y);
    }

    /**
     * Creates and appends legend to chart
     */
    _createLegend() {
        // remove previous data
        while(this._legend.firstChild) this._legend.removeChild(this._legend.lastChild);

        if (this._categories.length > 0)
            for (let item in this._categories[this._currentSelection].legend) {
                const data = this._categories[this._currentSelection].legend[item]
                const el = document.createElement('li')
                this._legend.append(el)

                const iconEl = document.createElement('span')
                iconEl.style.backgroundColor = data.color
                iconEl.id = 'heightgraph-legend-icon'
                el.append(iconEl)

                const textEl = document.createElement('span')
                textEl.innerHTML = data.text; // data.type
                el.append(textEl)
            }
    }

    /**
     * Creates top border line on graph
     */
    _createBorderTopLine() {
        const data = this._areasFlattended
        const borderTopLine = line()
            .x(d => {
                const x = this._x
                return x(d.position)
            })
            .y(d => {
                const y = this._y
                return y(d.altitude)
            })
            .curve(curveBasis)
        this._svg.append("svg:path")
            .attr("d", borderTopLine(data))
            .attr('class', 'border-top');
    }

    /**
     * Handles the mouseout event when the mouse leaves the background
     */
    _mouseoutHandler() {
        for (let param of ['_focusLine', '_focus', '_pointG', '_mouseHeightFocus', '_mouseHeightFocusLabel'])
            if (this[param]) {
                this[param].style('display', 'none');
            }
        this._pointSelected(null);
    }

    /**
     * Handles the mouseover the chart and displays distance and altitude level
     */
    _mousemoveHandler(event) {
        const coords = pointer(event, this._svg.node())
        const item = this._areasFlattended[this._findItemForX(coords[0])];
        if (item) this._internalMousemoveHandler(item);
    }

    /**
     * Handles the mouseover, given the current item the mouse is over
     */
    _internalMousemoveHandler(item, showMapMarker = true) {
        let areaLength
        const alt = item.altitude, dist = item.position,
            ll = item.latlng, areaIdx = item.areaIdx, type = item.type
        const boxWidth = dynamicBoxSize(".focusbox text").width + 10
        if (areaIdx === 0) {
            areaLength = this._categories[this._currentSelection].distances[areaIdx];
        } else {
            areaLength = this._categories[this._currentSelection].distances[areaIdx] - this._categories[this._currentSelection].distances[areaIdx - 1];
        }
        if (showMapMarker) {
            this._pointSelected(ll, alt, type);
        }
        this._distTspan.text(" " + distanceToString(dist * 1000, this._totalDistance, this._imperial, true));
        this._altTspan.text(" " + elevationToString(alt, this._imperial, true));
        this._areaTspan.text(" " + distanceToString(areaLength * 1000, this._totalDistance, this._imperial, true));
        this._typeTspan.text(" " + type);
        this._focusRect.attr("width", boxWidth);
        this._focusLine.style("display", "block")
            .attr('x1', this._x(dist))
            .attr('x2', this._x(dist));
        const xPositionBox = this._x(dist) - (boxWidth + 5)
        const totalWidth = this._width - this._margin.left - this._margin.right
        if (this._x(dist) + boxWidth < totalWidth) {
            this._focus.style("display", "initial")
                .attr("transform", "translate(" + this._x(dist) + "," + this._y(this._elevationBounds.min) + ")");
        }
        if (this._x(dist) + boxWidth > totalWidth) {
            this._focus.style("display", "initial")
                .attr("transform", "translate(" + xPositionBox + "," + this._y(this._elevationBounds.min) + ")");
        }
    }

    /**
     * Finds a data entry for a given x-coordinate of the diagram
     */
    _findItemForX(x) {
        const bisect = bisector(d => d.position).left
        const xInvert = this._x.invert(x)
        return bisect(this._areasFlattended, xInvert);
    }

    /**
     * Finds data entries above a given y-elevation value and returns geo-coordinates
     */
    _findCoordsForY(y) {
        let bisect = (b, yInvert) => {
            //save indexes of elevation values above the horizontal line
            const list = []
            for (let i = 0; i < b.length; i++) {
                if (b[i].altitude >= yInvert) {
                    list.push(i);
                }
            }
            //split index list into coherent blocks of coordinates
            const newList = []
            let start = 0
            for (let j = 0; j < list.length - 1; j++) {
                if (list[j + 1] !== list[j] + 1) {
                    newList.push(list.slice(start, j + 1));
                    start = j + 1;
                }
            }
            newList.push(list.slice(start, list.length));
            //get lat lon coordinates based on indexes
            for (let k = 0; k < newList.length; k++) {
                for (let l = 0; l < newList[k].length; l++) {
                    newList[k][l] = b[newList[k][l]].latlng;
                }
            }
            return newList;
        }

        const yInvert = this._y.invert(y)
        return bisect(this._areasFlattended, yInvert);
    }

    /**
     * Checks the user passed translations, if they don't exist, fallback to the default translations
     */
    _getTranslation(key) {
        if (this._translation[key])
            return this._translation[key];
        if (this._defaultTranslation[key])
            return this._defaultTranslation[key];
        console.error("Unexpected error when looking up the translation for " + key);
        return 'No translation found';
    }

    _createCoordinate(lng, lat) {
        return {lat: lat, lng: lng};
    }
    /**
     * calculates the distance between two (lat,lng) coordinates in meters
     */
    _calcDistance(a, b) {
        const deg2rad = Math.PI / 180;
        const sinDLat = Math.sin((b.lat - a.lat) * deg2rad / 2);
        const sinDLon = Math.sin((b.lng - a.lng) * deg2rad / 2);
        const f = sinDLat * sinDLat + Math.cos(a.lat * deg2rad) * Math.cos(b.lat * deg2rad) * sinDLon * sinDLon;
        return 6371000 * 2 * Math.atan2(Math.sqrt(f), Math.sqrt(1 - f));
    }

    _createDOMElement(tagName, className, container) {
        const el = document.createElement(tagName);
        el.className = className || '';
        if (container)
            container.appendChild(el);
        return el;
    }

    _isMobile() {
        return /Android|webOS|iPhone|iPad|Mac|Macintosh|iPod|BlackBerry|IEMobile|Opera Mini/i.test(navigator.userAgent);
    }
}

/**
 * Creates an svg element displaying the given elevation and description
 * that can be used as map marker
 */
export const createMapMarker = (elevation, description, imperial) => {
    // we append the svg to body so we can calculate sizes, but later remove it again (not sure if this is really needed)
    // otherwise we could just use d3.create('svg')
    const res = select('body').append('svg')
    const offset = {
        x: 5,
        y: 5
    }
    const fontSize = 12;
    const height = 80;
    // the base circle that should be located at the marker position
    res.append('circle')
        .attr('class', 'height-focus circle')
        .attr('r', offset.x)
        .attr('cx', offset.x)
        .attr('cy', height - offset.y)
        .style('display', 'block');
    // vertical line / 'flagpole'
    res.append('line')
        .attr('class', 'height-focus line')
        .attr('x1', offset.x)
        .attr('x2', offset.x)
        .attr('y1', 0)
        .attr('y2', 80)
        .style('display', 'block');

    // content box / 'flag'
    const labelBox = res.append('g')
        .attr('class', 'height-focus label')
        .style('display', 'block');
    labelBox.append('rect')
        .attr('x', offset.x + 3)
        .attr('y', 0)
        .attr('class', 'bBox');
    labelBox.append('text')
        .attr('x', offset.x + 5)
        .attr('y', fontSize)
        .text(elevationToString(elevation, imperial, true))
        .attr('class', 'tspan mouse-height-box-text');
    labelBox.append('text')
        .attr('x', offset.x + 5)
        .attr('y', 2*fontSize)
        .text(description)
        .attr('class', 'tspan mouse-height-box-text');

    const textSize = dynamicBoxSize('text.tspan')
    // flag height should be smaller change if description is empty
    const flagHeight = (description === '') ? fontSize + 10 : 2 * fontSize + 10
    selectAll('.bBox')
        .attr('width', textSize.width + 10)
        .attr('height', flagHeight);
    res.attr('width', textSize.width + 40).attr('height', height);
    res.remove();
    return res.node();
}

function distanceToString(meters, totalKm, imperial, precise) {
    if (imperial) {
        if (metersToMiles(totalKm * 1000) < 0.1) return metersToFeet(meters).toFixed(precise ? 1 : 0) + " ft"
        else if (metersToMiles(totalKm * 1000) < 10) return metersToMiles(meters).toFixed(precise ? 3 : 1) + " mi"
        else return metersToMiles(meters).toFixed(precise ? 3 : 0) + " mi"
    } else {
        if (totalKm < 1) return meters.toFixed(precise ? 2 : 0) + " m";
        else if (totalKm < 10) return (meters / 1000).toFixed(precise ? 3 : 1) + " km"
        else return (meters / 1000).toFixed(precise ? 3 : 0) + " km"
    }
}

function elevationToString(meters, imperial, precise) {
    if (imperial) {
        return metersToFeet(meters).toFixed(precise ? 1 : 0) + " ft";
    } else {
        return meters.toFixed(precise ? 2 : 0) + " m";
    }
}

function metersToMiles(meters) {
    return meters / 1609.34
}

function metersToFeet(meters) {
    return meters / 0.3048
}

/**
 * calculates the maximum size of all boxes for the given class name
 */
const dynamicBoxSize = (className) => {
    const widths = [];
    const heights = [];
    selectAll(className).nodes()
        .map(s => s.getBBox())
        .forEach(b => {
            widths.push(b.width);
            heights.push(b.height)
        });
    return {width: d3Max(widths), height: d3Max(heights)};
}

