import java.time.LocalDateTime
import java.time.format.DateTimeFormatter
import java.time.Instant
import java.time.ZoneOffset
import java.time.OffsetDateTime
import java.util.Properties
import java.io.FileInputStream

plugins {
    id("com.android.application")
    id("org.jetbrains.kotlin.android")
    id("org.jetbrains.kotlin.plugin.compose")
    id("com.google.devtools.ksp")
    id("com.google.dagger.hilt.android")
    id("pl.allegro.tech.build.axion-release")
    kotlin("plugin.serialization")
}

// Global excludes to remove non-deterministic baseline profile installer library (transitively pulled by lifecycle/activity)
configurations.configureEach {
    exclude(group = "androidx.profileinstaller", module = "profileinstaller")
}

scmVersion {
    tag {
        prefix.set("v")
        versionSeparator.set("")
    }
    versionCreator { version, _ ->
        // Remove 'v' prefix if present for clean version names
        version.replace("^v".toRegex(), "")
    }
    
    // Configure semantic versioning - since we have feat: commits, use minor increment
    branchVersionIncrementer = mapOf(
        "main" to "incrementMinor"
    )
    repository {
        pushTagsOnly.set(false)
    }
    localOnly.set(true)  // Don't auto-push, create local release only
    checks {
        aheadOfRemote.set(false)
        uncommittedChanges.set(false)
    }

}

val keystoreProperties = Properties()
val keystorePropertiesFile = rootProject.file("keystore.properties")
if (keystorePropertiesFile.exists()) {
    keystoreProperties.load(keystorePropertiesFile.inputStream())
}

android {
    namespace = "com.glosdalen.app"
    compileSdk = 34

    defaultConfig {
        applicationId = "com.glosdalen.app"
        minSdk = 26
        targetSdk = 34
        
        // Static version code and name - F-Droid parses these, increment manually for each release
        versionCode = 22
        versionName = "2.5.0"
        
        // Check if repository is dirty (has uncommitted changes)
        // Skip dirty check for F-Droid builds (indicated by SOURCE_DATE_EPOCH)
        val isFDroidBuild = System.getenv("SOURCE_DATE_EPOCH") != null
        val isRepoDirty = if (isFDroidBuild) {
            false
        } else {
            providers.exec {
                commandLine("git", "status", "--porcelain")
            }.standardOutput.asText.get().trim().isNotEmpty()
        }
        
        // Adjust version name based on repository state
        versionName = versionName + if (isRepoDirty) "-DIRTY" else ""

        testInstrumentationRunner = "androidx.test.runner.AndroidJUnitRunner"
        vectorDrawables {
            useSupportLibrary = true
        }
        
        // Build configuration fields
        val gitHash = providers.exec {
            commandLine("git", "rev-parse", "--short", "HEAD")
        }.standardOutput.asText.get().trim()
        
        // Use current time for dirty builds, git commit date for clean builds
        val buildDate = when {
            // SOURCE_DATE_EPOCH takes precedence (for reproducible builds)
            System.getenv("SOURCE_DATE_EPOCH") != null -> {
                val epoch = System.getenv("SOURCE_DATE_EPOCH").toLongOrNull()
                epoch?.let { 
                    Instant.ofEpochSecond(it)
                        .atZone(ZoneOffset.UTC)
                        .format(DateTimeFormatter.ofPattern("yyyy-MM-dd HH:mm"))
                }
            }
            // Dirty repository: use current build time
            isRepoDirty -> {
                LocalDateTime.now()
                    .format(DateTimeFormatter.ofPattern("yyyy-MM-dd HH:mm"))
            }
            // Clean repository: use last commit date
            else -> {
                providers.exec {
                    commandLine("git", "log", "-1", "--format=%cI")
                }.standardOutput.asText.get().trim().let { gitDate ->
                    // Parse ISO 8601 git date and format consistently
                    OffsetDateTime.parse(gitDate)
                        .atZoneSameInstant(ZoneOffset.UTC)
                        .format(DateTimeFormatter.ofPattern("yyyy-MM-dd HH:mm"))
                }
            }
        } ?: "unknown"
        
        buildConfigField("String", "GIT_HASH", "\"$gitHash\"")
        buildConfigField("String", "BUILD_DATE", "\"$buildDate\"")
    }

    buildTypes {
        release {
            isMinifyEnabled = true
            isShrinkResources = true
            proguardFiles(
                getDefaultProguardFile("proguard-android-optimize.txt"),
                "proguard-rules.pro"
            )
        }
        // Disable dependency metadata for F-Droid compatibility
        all {
            enableAndroidTestCoverage = false
        }
    }
    
    dependenciesInfo {
        // Disable dependency metadata block for F-Droid
        includeInApk = false
        includeInBundle = false
    }
    compileOptions {
        sourceCompatibility = JavaVersion.VERSION_17
        targetCompatibility = JavaVersion.VERSION_17
    }
    kotlinOptions {
        jvmTarget = "17"
    }
    buildFeatures {
        compose = true
        buildConfig = true
    }
    packaging {
        resources {
            excludes += "/META-INF/{AL2.0,LGPL2.1}"
            // Drop any remaining baseline profile artifacts just in case (should be gone after exclusion above)
            excludes += "META-INF/**.prof"
            excludes += "META-INF/**baseline-prof.txt"
        }
    }
    
    testOptions {
        unitTests.all {
            it.useJUnitPlatform()
        }
    }
}

composeCompiler {
    includeSourceInformation = false
}

dependencies {
    // AnkiDroid API
    implementation("com.github.ankidroid:Anki-Android:api-v1.1.0")
    
    implementation("androidx.core:core-ktx:1.13.1")
    implementation("androidx.lifecycle:lifecycle-runtime-ktx:2.8.4")
    implementation("androidx.activity:activity-compose:1.9.1")
    implementation(platform("androidx.compose:compose-bom:2024.06.00"))
    implementation("androidx.compose.ui:ui")
    implementation("androidx.compose.ui:ui-graphics")
    implementation("androidx.compose.ui:ui-tooling-preview")
    implementation("androidx.compose.material3:material3")
    implementation("androidx.compose.material:material-icons-extended:1.6.8")
    
    // Navigation
    implementation("androidx.navigation:navigation-compose:2.8.0")
    
    // ViewModel
    implementation("androidx.lifecycle:lifecycle-viewmodel-compose:2.8.4")
    
    // Networking
    implementation("com.squareup.retrofit2:retrofit:2.11.0")
    implementation("com.squareup.okhttp3:logging-interceptor:4.12.0")
    
    // Kotlinx Serialization (for all API communication)
    implementation("org.jetbrains.kotlinx:kotlinx-serialization-json:1.6.0")
    implementation("com.jakewharton.retrofit:retrofit2-kotlinx-serialization-converter:1.0.0")
    
    // Security (for Copilot library - EncryptedSharedPreferences)
    implementation("androidx.security:security-crypto:1.1.0-alpha06")
    
    // Dependency Injection
    implementation("com.google.dagger:hilt-android:2.48")
    ksp("com.google.dagger:hilt-compiler:2.48")
    implementation("androidx.hilt:hilt-navigation-compose:1.1.0")
    
    // Coroutines
    implementation("org.jetbrains.kotlinx:kotlinx-coroutines-android:1.8.1")
    
    // Preferences DataStore
    implementation("androidx.datastore:datastore-preferences:1.1.1")
    
    // Testing
    testImplementation("org.junit.jupiter:junit-jupiter:5.10.3")
    testImplementation("org.junit.jupiter:junit-jupiter-engine:5.10.3")
    androidTestImplementation("androidx.test.ext:junit:1.2.1")
    androidTestImplementation("androidx.test.espresso:espresso-core:3.6.1")
    androidTestImplementation(platform("androidx.compose:compose-bom:2024.06.00"))
    androidTestImplementation("androidx.compose.ui:ui-test-junit4")
    debugImplementation("androidx.compose.ui:ui-tooling")
    debugImplementation("androidx.compose.ui:ui-test-manifest")
}
