package com.github.umer0586.droidpad.daotests

import androidx.room.Room
import androidx.test.core.app.ApplicationProvider
import com.github.umer0586.droidpad.MainDispatcherRule
import com.github.umer0586.droidpad.data.database.AppDatabase
import com.github.umer0586.droidpad.data.database.dao.ConnectionConfigurationDao
import com.github.umer0586.droidpad.data.database.dao.ControlPadDao
import com.github.umer0586.droidpad.data.database.entities.ConnectionConfig
import com.github.umer0586.droidpad.data.database.entities.ConnectionType
import com.github.umer0586.droidpad.data.database.entities.ControlPad
import com.github.umer0586.droidpad.data.database.entities.Orientation
import kotlinx.coroutines.test.runTest
import org.junit.After
import org.junit.Assert.assertEquals
import org.junit.Assert.assertNull
import org.junit.Before
import org.junit.Rule
import org.junit.Test
import org.junit.runner.RunWith
import org.robolectric.RobolectricTestRunner
import java.io.IOException

@RunWith(RobolectricTestRunner::class)
class ConnectionConfigDaoTest {

    @get:Rule
    val mainDispatcherRule = MainDispatcherRule()

    private lateinit var connectionConfigurationDao: ConnectionConfigurationDao
    // We need a controlPadDao to test the foreign key constraint
    private lateinit var controlPadDao: ControlPadDao
    private lateinit var db: AppDatabase

    @Before
    fun setup() {
        db = Room.inMemoryDatabaseBuilder(
            ApplicationProvider.getApplicationContext(),
            AppDatabase::class.java
        ).build()

        connectionConfigurationDao = db.connectionConfigurationDao()
        controlPadDao = db.controlPadDao()
    }

    @After
    @Throws(IOException::class)
    fun teardown() {
        db.close()
    }

    @Test
    fun `test getAll`() = runTest {

    }

    @Test
    fun `test getById`() = runTest {

        val controlPad = ControlPad(
            name = "my control pad",
            orientation = Orientation.LANDSCAPE
        )
        val newControlPadId = controlPadDao.insert(controlPad)

        val config = ConnectionConfig(
            controlPadId = newControlPadId,
            connectionType = ConnectionType.TCP,
            configJson = "{}"
        )

        val newConfigId = connectionConfigurationDao.insert(config)

        connectionConfigurationDao.getById(newConfigId)?.let { retrievedConfig ->
            // We cannot test the id because it is autogenerated
            // Therefore object equality fails
            assertEquals(config.controlPadId, retrievedConfig.controlPadId)
            assertEquals(config.connectionType, retrievedConfig.connectionType)
            assertEquals(config.configJson, retrievedConfig.configJson)
        }

    }

    @Test
    fun `test getById_notFound`() = runTest {

        val controlPad = ControlPad(
            name = "my control pad",
            orientation = Orientation.LANDSCAPE
        )
        val newControlPadId = controlPadDao.insert(controlPad)

        val config = ConnectionConfig(
            controlPadId = newControlPadId,
            connectionType = ConnectionType.TCP,
            configJson = "{}"
        )

        val newConfigId = connectionConfigurationDao.insert(config)

        connectionConfigurationDao.getById(newConfigId)?.let { retrievedConfig ->
            connectionConfigurationDao.delete(retrievedConfig)
            assertNull(connectionConfigurationDao.getById(newConfigId))
        }

    }

    @Test
    fun `test insert`() = runTest {
        val controlPad = ControlPad(
            name = "my control pad",
            orientation = Orientation.LANDSCAPE
        )
        val newControlPadId = controlPadDao.insert(controlPad)

        val config = ConnectionConfig(
            controlPadId = newControlPadId,
            connectionType = ConnectionType.TCP,
            configJson = "{}"
        )

        val newConfigId = connectionConfigurationDao.insert(config)

        connectionConfigurationDao.getById(newConfigId)?.let { retrievedConfig ->
            // We cannot test the id because it is autogenerated
            assertEquals(config.controlPadId, retrievedConfig.controlPadId)
            assertEquals(config.connectionType, retrievedConfig.connectionType)
            assertEquals(config.configJson, retrievedConfig.configJson)
        }

    }

    @Test
    fun `test update`() = runTest {

        val controlPad = ControlPad(
            name = "my control pad",
            orientation = Orientation.LANDSCAPE
        )
        val newControlPadId = controlPadDao.insert(controlPad)

        val config = ConnectionConfig(
            controlPadId = newControlPadId,
            connectionType = ConnectionType.TCP,
            configJson = "{}"
        )
        val newConfigId = connectionConfigurationDao.insert(config)

        connectionConfigurationDao.getById(newConfigId)?.let { retrievedConfig ->
            val updatedConfig = retrievedConfig.copy(connectionType = ConnectionType.UDP)
            connectionConfigurationDao.update(updatedConfig)

            connectionConfigurationDao.getById(newConfigId)?.let { updatedRetrievedConfig ->
                assertEquals(updatedConfig, updatedRetrievedConfig)
            }
        }



    }

    @Test
    fun `test deleteById`() = runTest {

        val controlPad = ControlPad(
            name = "my control pad",
            orientation = Orientation.LANDSCAPE
        )
        val newControlPadId = controlPadDao.insert(controlPad)

        val config = ConnectionConfig(
            controlPadId = newControlPadId,
            connectionType = ConnectionType.TCP,
            configJson = "{}"
        )
        val newConfigId = connectionConfigurationDao.insert(config)

        connectionConfigurationDao.deleteById(newConfigId)
        assertNull(connectionConfigurationDao.getById(newConfigId))

    }

    @Test
    fun `test delete`() = runTest {

        val controlPad = ControlPad(
            name = "my control pad",
            orientation = Orientation.LANDSCAPE
        )
        val controlPad2 = ControlPad(
            name = "my other control pad",
            orientation = Orientation.PORTRAIT
        )
        val newControlPadId = controlPadDao.insert(controlPad)
        val newControlPad2Id = controlPadDao.insert(controlPad2)

        val config = ConnectionConfig(
            controlPadId = newControlPadId,
            connectionType = ConnectionType.TCP,
            configJson = "{}"
        )
        val newConfigId = connectionConfigurationDao.insert(config)

        val config2 = ConnectionConfig(
            controlPadId = newControlPad2Id,
            connectionType = ConnectionType.WEBSOCKET,
            configJson = "{}"
        )
        val newConfig2Id = connectionConfigurationDao.insert(config2)

        connectionConfigurationDao.getById(newConfigId)?.let { retrievedConfig ->
            connectionConfigurationDao.delete(retrievedConfig)
            assertNull(connectionConfigurationDao.getById(newConfigId))
        }

        // Check that when controlPad is deleted,
        // the connectionConfiguration is also deleted (Enforce by foreign key constraint)
        controlPadDao.getById(newControlPad2Id)?.let { retrievedControlPad ->
            controlPadDao.delete(retrievedControlPad)
            assertNull(connectionConfigurationDao.getById(newConfig2Id))

        }

    }
}