#! /usr/bin/python3

# Copyright (c) 2018, 2023 Calvin Loncaric
# Copyright (c) 2023 Thomas More
#
# Permission is hereby granted, free of charge, to any person obtaining a copy
# of this software and associated documentation files (the "Software"), to deal
# in the Software without restriction, including without limitation the rights
# to use, copy, modify, merge, publish, distribute, sublicense, and/or sell
# copies of the Software, and to permit persons to whom the Software is
# furnished to do so, subject to the following conditions:
#
# The above copyright notice and this permission notice shall be included in all
# copies or substantial portions of the Software.
#
# THE SOFTWARE IS PROVIDED "AS IS", WITHOUT WARRANTY OF ANY KIND, EXPRESS OR
# IMPLIED, INCLUDING BUT NOT LIMITED TO THE WARRANTIES OF MERCHANTABILITY,
# FITNESS FOR A PARTICULAR PURPOSE AND NONINFRINGEMENT. IN NO EVENT SHALL THE
# AUTHORS OR COPYRIGHT HOLDERS BE LIABLE FOR ANY CLAIM, DAMAGES OR OTHER
# LIABILITY, WHETHER IN AN ACTION OF CONTRACT, TORT OR OTHERWISE, ARISING FROM,
# OUT OF OR IN CONNECTION WITH THE SOFTWARE OR THE USE OR OTHER DEALINGS IN THE
# SOFTWARE.

# From: https://gist.github.com/Calvin-L/5232f876b8acf48a216941b8904632bb
# See:  https://git.silence.dev/Silence/Silence-Android/-/issues/317

# This utility fixes invalid XML produced by the Silence app. It is basically just Calvin Loncaric's fix_codepoints()
# function, with code at the end (based on some of the comments in the Gist) to call it on each line of STDIN and
# print the fixed XML to STDOUT.

# Usage: 'silence-xml-fixer.py < silence-xxx.xml > silence-xxx-fixed.xml'

import io
import re
import struct
import sys

# Per "struct" module docs
UNSIGNED_SHORT = "H"


def shorts_as_utf16(short_sequence):
    if not isinstance(short_sequence, list):
        short_sequence = list(short_sequence)
    format = UNSIGNED_SHORT * (len(short_sequence) + 1)
    # 0xFEFF is a byte-order marker---however it gets encoded by pack(), the
    # UTF-16 decoder is supposed to understand it and use that interpretation
    # for the endianness of the remaining bytes.  We probably don't need it
    # here, but it can't hurt!
    try:
        bits = struct.pack(format, 0xFEFF, *short_sequence)
    except:
        print("failed sequence: {!r}".format(short_sequence))
        raise
    return bits.decode("UTF-16")


# Numeric XML entities, e.g. "&#55357;&#56860;".
rgx1 = re.compile(r"(?:&#\d+;)+")

# Capture one of the numbers inside an entity
rgx2 = re.compile(r"&#(\d+);")


def fix_codepoints(s, raw=False):
    """Fix malformed XML entities generated by "SMS Backup & Restore".

    Note: this function may break well-formed numeric entities, so be sure that
    the input string does not mix the two.

    Input:
        s -- a string
        raw -- whether to do "raw" conversion (see "Output" below)

    Output:
        The string s, but with bad entities fixed.  If "raw" is True, the bad
        entities are replaced with their actual unicode characters.  If "raw"
        is False, the bad entities are replaced with correct XML entities.

    Details:
        SMS Backup app encodes complicated unicode characters as
          &#XXXXX;&#XXXXX;
        where the two XXX numbers are two unsigned shorts that form a UTF-16
        character.  (Makes sense---it's probably implemented in Java, which
        uses UTF-16 encoding for in-memory strings.)  Proper XML uses unicode
        codepoints (Python's ord()) for the XXX digits, so we need to do
        conversion.
    """

    matches = list(rgx1.finditer(s))
    if not matches:
        return s
    with io.StringIO() as out:
        i = 0
        for m in matches:
            out.write(s[i:m.start()])
            i = m.end()
            shorts = [int(i) for i in rgx2.findall(m.group(0))]
            assert len(shorts) > 0
            if all(s < 32767 * 2 + 1 for s in shorts):
                repl = shorts_as_utf16(shorts)
            else:
                repl = "".join(chr(s) for s in shorts)
            if raw:
                out.write(repl)
            else:
                for c in repl:
                    out.write("&#{};".format(ord(c)))
        out.write(s[i:])
        return out.getvalue()


# Added by Thomas More:
sys.stdin.reconfigure(encoding='utf-8')
for line in sys.stdin:
    print(fix_codepoints(line.rstrip()))
