// Copyright 2013 The Flutter Authors. All rights reserved.
// Use of this source code is governed by a BSD-style license that can be
// found in the LICENSE file.

// This file specifically tests the test PigeonInstanceManager generated by core_tests.

import 'package:flutter_test/flutter_test.dart';
import 'package:leak_tracker/leak_tracker.dart';
import 'package:shared_test_plugin_code/src/generated/proxy_api_tests.gen.dart';

void main() {
  group('InstanceManager', () {
    test('addHostCreatedInstance', () {
      final PigeonInstanceManager instanceManager =
          PigeonInstanceManager(onWeakReferenceRemoved: (_) {});

      final CopyableObject object = CopyableObject(
        pigeon_instanceManager: instanceManager,
      );

      instanceManager.addHostCreatedInstance(object, 0);

      expect(instanceManager.getIdentifier(object), 0);
      expect(
        instanceManager.getInstanceWithWeakReference(0),
        isA<CopyableObject>(),
      );
    });

    test('addHostCreatedInstance prevents already used objects and ids', () {
      final PigeonInstanceManager instanceManager =
          PigeonInstanceManager(onWeakReferenceRemoved: (_) {});

      final CopyableObject object = CopyableObject(
        pigeon_instanceManager: instanceManager,
      );

      instanceManager.addHostCreatedInstance(object, 0);

      expect(
        () => instanceManager.addHostCreatedInstance(object, 0),
        throwsAssertionError,
      );

      expect(
        () => instanceManager.addHostCreatedInstance(
          CopyableObject(pigeon_instanceManager: instanceManager),
          0,
        ),
        throwsAssertionError,
      );
    });

    test('addFlutterCreatedInstance', () {
      final PigeonInstanceManager instanceManager =
          PigeonInstanceManager(onWeakReferenceRemoved: (_) {});

      final CopyableObject object = CopyableObject(
        pigeon_instanceManager: instanceManager,
      );

      instanceManager.addDartCreatedInstance(object);

      final int? instanceId = instanceManager.getIdentifier(object);
      expect(instanceId, isNotNull);
      expect(
        instanceManager.getInstanceWithWeakReference(instanceId!),
        object,
      );
    });

    test('removeWeakReference', () {
      int? weakInstanceId;
      final PigeonInstanceManager instanceManager =
          PigeonInstanceManager(onWeakReferenceRemoved: (int instanceId) {
        weakInstanceId = instanceId;
      });

      final CopyableObject object = CopyableObject(
        pigeon_instanceManager: instanceManager,
      );

      instanceManager.addHostCreatedInstance(object, 0);

      expect(instanceManager.removeWeakReference(object), 0);
      expect(
        instanceManager.getInstanceWithWeakReference(0),
        isA<CopyableObject>(),
      );
      expect(weakInstanceId, 0);
    });

    test('removeWeakReference removes only weak reference', () {
      final PigeonInstanceManager instanceManager =
          PigeonInstanceManager(onWeakReferenceRemoved: (_) {});

      final CopyableObject object = CopyableObject(
        pigeon_instanceManager: instanceManager,
      );

      instanceManager.addHostCreatedInstance(object, 0);

      expect(instanceManager.removeWeakReference(object), 0);
      final CopyableObject copy = instanceManager.getInstanceWithWeakReference(
        0,
      )!;
      expect(identical(object, copy), isFalse);
    });

    test('remove', () {
      final PigeonInstanceManager instanceManager =
          PigeonInstanceManager(onWeakReferenceRemoved: (_) {});

      final CopyableObject object = CopyableObject(
        pigeon_instanceManager: instanceManager,
      );

      instanceManager.addHostCreatedInstance(object, 0);
      instanceManager.removeWeakReference(object);
      expect(instanceManager.remove(0), isA<CopyableObject>());
      expect(instanceManager.containsIdentifier(0), isFalse);
    });

    test('remove throws AssertionError if weak reference still exists', () {
      final PigeonInstanceManager instanceManager =
          PigeonInstanceManager(onWeakReferenceRemoved: (_) {});

      final CopyableObject object = CopyableObject(
        pigeon_instanceManager: instanceManager,
      );

      instanceManager.addDartCreatedInstance(object);
      expect(() => instanceManager.remove(0), throwsAssertionError);
    });

    test('getInstance can add a new weak reference', () {
      final PigeonInstanceManager instanceManager =
          PigeonInstanceManager(onWeakReferenceRemoved: (_) {});

      final CopyableObject object = CopyableObject(
        pigeon_instanceManager: instanceManager,
      );

      instanceManager.addHostCreatedInstance(object, 0);
      instanceManager.removeWeakReference(object);

      final CopyableObject newWeakCopy =
          instanceManager.getInstanceWithWeakReference(
        0,
      )!;
      expect(identical(object, newWeakCopy), isFalse);
    });

    test('addDartCreatedInstance should add finalizer to original object',
        () async {
      bool weakReferencedRemovedCalled = false;
      final PigeonInstanceManager instanceManager = PigeonInstanceManager(
        onWeakReferenceRemoved: (_) {
          weakReferencedRemovedCalled = true;
        },
      );

      CopyableObject? object = CopyableObject(
        pigeon_instanceManager: instanceManager,
      );

      instanceManager.addDartCreatedInstance(object);

      object = null;
      await forceGC();

      expect(weakReferencedRemovedCalled, isTrue);
    });

    test('addHostCreatedInstance should not add finalizer to original object',
        () async {
      bool weakReferencedRemovedCalled = false;
      final PigeonInstanceManager instanceManager = PigeonInstanceManager(
        onWeakReferenceRemoved: (_) {
          weakReferencedRemovedCalled = true;
        },
      );

      CopyableObject? object = CopyableObject(
        pigeon_instanceManager: instanceManager,
      );

      instanceManager.addHostCreatedInstance(object, 0);

      object = null;
      await forceGC();

      expect(weakReferencedRemovedCalled, isFalse);
    });
  });
}

class CopyableObject extends PigeonInternalProxyApiBaseClass {
  // ignore: non_constant_identifier_names
  CopyableObject({super.pigeon_instanceManager});

  @override
  // ignore: non_constant_identifier_names
  CopyableObject pigeon_copy() {
    return CopyableObject(pigeon_instanceManager: pigeon_instanceManager);
  }
}
