// Copyright 2013 The Flutter Authors. All rights reserved.
// Use of this source code is governed by a BSD-style license that can be
// found in the LICENSE file.

// This file specifically tests the test code generated by dartTestOut.

import 'package:flutter/services.dart';
import 'package:flutter_test/flutter_test.dart';
import 'package:shared_test_plugin_code/src/generated/core_tests.gen.dart'
    show AllNullableTypes;
import 'package:shared_test_plugin_code/src/generated/message.gen.dart';
import 'package:shared_test_plugin_code/test_types.dart';

import 'test_message.gen.dart';

class Mock implements TestHostApi {
  List<String> log = <String>[];

  @override
  void initialize() {
    log.add('initialize');
  }

  @override
  MessageSearchReply search(MessageSearchRequest arg) {
    log.add('search');
    return MessageSearchReply()..result = arg.query;
  }
}

class MockNested implements TestNestedApi {
  bool didCall = false;
  @override
  MessageSearchReply search(MessageNested arg) {
    didCall = true;
    if (arg.request == null) {
      return MessageSearchReply();
    } else {
      return MessageSearchReply()..result = arg.request?.query;
    }
  }
}

void main() {
  TestWidgetsFlutterBinding.ensureInitialized();

  group('equality method', () {
    final List<Object?> correctList = <Object?>['a', 2, 'three'];
    final List<Object?> matchingList = correctList.toList();
    final List<Object?> differentList = <Object?>['a', 2, 'three', 4.0];
    final Map<String, Object?> correctMap = <String, Object?>{
      'a': 1,
      'b': 2,
      'c': 'three'
    };
    final Map<String, Object?> matchingMap = <String, Object?>{...correctMap};
    final Map<String, Object?> differentKeyMap = <String, Object?>{
      'a': 1,
      'b': 2,
      'd': 'three'
    };
    final Map<String, Object?> differentValueMap = <String, Object?>{
      'a': 1,
      'b': 2,
      'c': 'five'
    };
    final Map<String, Object?> correctListInMap = <String, Object?>{
      'a': 1,
      'b': 2,
      'c': correctList
    };
    final Map<String, Object?> matchingListInMap = <String, Object?>{
      'a': 1,
      'b': 2,
      'c': matchingList
    };
    final Map<String, Object?> differentListInMap = <String, Object?>{
      'a': 1,
      'b': 2,
      'c': differentList
    };
    final List<Object?> correctMapInList = <Object?>['a', 2, correctMap];
    final List<Object?> matchingMapInList = <Object?>['a', 2, matchingMap];
    final List<Object?> differentKeyMapInList = <Object?>[
      'a',
      2,
      differentKeyMap
    ];
    final List<Object?> differentValueMapInList = <Object?>[
      'a',
      2,
      differentValueMap
    ];

    test('equality method correctly checks deep equality', () {
      final AllNullableTypes generic = genericAllNullableTypes;
      final AllNullableTypes identical =
          AllNullableTypes.decode(generic.encode());
      expect(identical, generic);
    });

    test('equality method correctly identifies non-matching classes', () {
      final AllNullableTypes generic = genericAllNullableTypes;
      final AllNullableTypes allNull = AllNullableTypes();
      expect(allNull == generic, false);
    });

    test('equality method correctly identifies non-matching lists in classes',
        () {
      final AllNullableTypes withList = AllNullableTypes(list: correctList);
      final AllNullableTypes withDifferentList =
          AllNullableTypes(list: differentList);
      expect(withList == withDifferentList, false);
    });

    test(
        'equality method correctly identifies matching -but unique- lists in classes',
        () {
      final AllNullableTypes withList = AllNullableTypes(list: correctList);
      final AllNullableTypes withDifferentList =
          AllNullableTypes(list: matchingList);
      expect(withList, withDifferentList);
    });

    test(
        'equality method correctly identifies non-matching keys in maps in classes',
        () {
      final AllNullableTypes withMap = AllNullableTypes(map: correctMap);
      final AllNullableTypes withDifferentMap =
          AllNullableTypes(map: differentKeyMap);
      expect(withMap == withDifferentMap, false);
    });

    test(
        'equality method correctly identifies non-matching values in maps in classes',
        () {
      final AllNullableTypes withMap = AllNullableTypes(map: correctMap);
      final AllNullableTypes withDifferentMap =
          AllNullableTypes(map: differentValueMap);
      expect(withMap == withDifferentMap, false);
    });

    test(
        'equality method correctly identifies matching -but unique- maps in classes',
        () {
      final AllNullableTypes withMap = AllNullableTypes(map: correctMap);
      final AllNullableTypes withDifferentMap =
          AllNullableTypes(map: matchingMap);
      expect(withMap, withDifferentMap);
    });

    test(
        'equality method correctly identifies non-matching lists nested in maps in classes',
        () {
      final AllNullableTypes withListInMap =
          AllNullableTypes(map: correctListInMap);
      final AllNullableTypes withDifferentListInMap =
          AllNullableTypes(map: differentListInMap);
      expect(withListInMap == withDifferentListInMap, false);
    });

    test(
        'equality method correctly identifies matching -but unique- lists nested in maps in classes',
        () {
      final AllNullableTypes withListInMap =
          AllNullableTypes(map: correctListInMap);
      final AllNullableTypes withDifferentListInMap =
          AllNullableTypes(map: matchingListInMap);
      expect(withListInMap, withDifferentListInMap);
    });

    test(
        'equality method correctly identifies non-matching keys in maps nested in lists in classes',
        () {
      final AllNullableTypes withMapInList =
          AllNullableTypes(list: correctMapInList);
      final AllNullableTypes withDifferentMapInList =
          AllNullableTypes(list: differentKeyMapInList);
      expect(withMapInList == withDifferentMapInList, false);
    });

    test(
        'equality method correctly identifies non-matching values in maps nested in lists in classes',
        () {
      final AllNullableTypes withMapInList =
          AllNullableTypes(list: correctMapInList);
      final AllNullableTypes withDifferentMapInList =
          AllNullableTypes(list: differentValueMapInList);
      expect(withMapInList == withDifferentMapInList, false);
    });

    test(
        'equality method correctly identifies matching -but unique- maps nested in lists in classes',
        () {
      final AllNullableTypes withMapInList =
          AllNullableTypes(list: correctMapInList);
      final AllNullableTypes withDifferentMapInList =
          AllNullableTypes(list: matchingMapInList);
      expect(withMapInList, withDifferentMapInList);
    });
  });
  test('simple', () async {
    final MessageNestedApi api = MessageNestedApi();
    final MockNested mock = MockNested();
    TestNestedApi.setUp(mock);
    final MessageSearchReply reply =
        await api.search(MessageNested()..request = null);
    expect(mock.didCall, true);
    expect(reply.result, null);
  });

  test('nested', () async {
    final MessageApi api = MessageApi();
    final Mock mock = Mock();
    TestHostApi.setUp(mock);
    final MessageSearchReply reply =
        await api.search(MessageSearchRequest()..query = 'foo');
    expect(mock.log, <String>['search']);
    expect(reply.result, 'foo');
  });

  test('no-arg calls', () async {
    final MessageApi api = MessageApi();
    final Mock mock = Mock();
    TestHostApi.setUp(mock);
    await api.initialize();
    expect(mock.log, <String>['initialize']);
  });

  test(
    'calling methods with null',
    () async {
      final Mock mock = Mock();
      TestHostApi.setUp(mock);
      expect(
        await const BasicMessageChannel<Object?>(
          'dev.flutter.pigeon.pigeon_integration_tests.MessageApi.initialize',
          StandardMessageCodec(),
        ).send(<Object?>[null]),
        isEmpty,
      );
      try {
        await const BasicMessageChannel<Object?>(
          'dev.flutter.pigeon.pigeon_integration_tests.MessageApi.search',
          StandardMessageCodec(),
        ).send(<Object?>[null]) as List<Object?>?;
        expect(true, isFalse); // should not reach here
      } catch (error) {
        expect(error, isAssertionError);
        expect(
          error.toString(),
          contains(
            'Argument for dev.flutter.pigeon.pigeon_integration_tests.MessageApi.search was null, expected non-null MessageSearchRequest.',
          ),
        );
      }
      expect(mock.log, <String>['initialize']);
    },
  );
}
