// Copyright 2025 The Chromium Authors
// Use of this source code is governed by a BSD-style license that can be
// found in the LICENSE file.

#include "crypto/test_support.h"

#include <array>

namespace crypto::test {

// To generate these keys:
// $ openssl genrsa -out test.key $BITS
// $ openssl rsa -in test.key -outform DER | xxd -i > private.key
// $ openssl rsa -in test.key -outform DER -pubout | xxd -i > public.key
//
// That produces the big arrays of hex given below. These are DER-encoded
// private and public keys respectively.

static constexpr auto kRsa2048Private = std::to_array<uint8_t>(
    {0x30, 0x82, 0x04, 0xbd, 0x02, 0x01, 0x00, 0x30, 0x0d, 0x06, 0x09, 0x2a,
     0x86, 0x48, 0x86, 0xf7, 0x0d, 0x01, 0x01, 0x01, 0x05, 0x00, 0x04, 0x82,
     0x04, 0xa7, 0x30, 0x82, 0x04, 0xa3, 0x02, 0x01, 0x00, 0x02, 0x82, 0x01,
     0x01, 0x00, 0xb7, 0x73, 0x11, 0xa1, 0x47, 0x56, 0xb8, 0xdf, 0x45, 0xab,
     0x81, 0xc1, 0x17, 0x24, 0x00, 0x6d, 0x70, 0xc2, 0x3f, 0x54, 0xac, 0x71,
     0xda, 0xf5, 0x5d, 0xd8, 0x7c, 0x77, 0x36, 0xa6, 0x2e, 0x30, 0xbd, 0x07,
     0x27, 0x34, 0x20, 0xdf, 0xc5, 0xf7, 0xdb, 0x9d, 0x93, 0xdc, 0x68, 0xa1,
     0xc4, 0x02, 0x4f, 0xcb, 0xb1, 0x1e, 0xb9, 0xd0, 0x12, 0x92, 0xa2, 0xf0,
     0x0d, 0xfb, 0xe6, 0xa0, 0x62, 0x36, 0xd2, 0x6e, 0xae, 0x03, 0xd5, 0xa8,
     0x53, 0x46, 0x77, 0x7a, 0xe3, 0x83, 0xfe, 0x8d, 0x54, 0x29, 0x03, 0xc1,
     0x19, 0x6d, 0x22, 0x44, 0x0f, 0x1a, 0x58, 0x57, 0x01, 0xef, 0x81, 0x3f,
     0x0b, 0xf3, 0x96, 0x8e, 0x51, 0x40, 0xd3, 0x67, 0xd8, 0x85, 0x9f, 0x85,
     0x80, 0x8e, 0x84, 0xc6, 0xaa, 0x35, 0xae, 0x35, 0xff, 0x38, 0x20, 0xbb,
     0xe3, 0x95, 0xc5, 0xe3, 0xd6, 0xdd, 0xe9, 0x28, 0x87, 0xb9, 0x38, 0xce,
     0xbd, 0x53, 0x17, 0x97, 0x90, 0xed, 0x87, 0xd5, 0x08, 0x6c, 0x84, 0xef,
     0xb1, 0x9e, 0x71, 0x5e, 0x2c, 0x0b, 0xce, 0xc1, 0x76, 0xbb, 0xfb, 0xd6,
     0x5b, 0x55, 0x85, 0xe0, 0x4d, 0xf7, 0xaf, 0xcf, 0x34, 0xe1, 0x40, 0x0a,
     0x76, 0x1f, 0xb1, 0xdc, 0xd7, 0xb0, 0x63, 0xa2, 0xc7, 0x23, 0x85, 0x56,
     0xfd, 0x8c, 0xec, 0x6b, 0x0f, 0xfb, 0x6e, 0xcb, 0xba, 0x3b, 0x12, 0x04,
     0xbb, 0xf7, 0x5d, 0x28, 0xe2, 0x57, 0x9c, 0x38, 0x1d, 0x8f, 0xc9, 0x51,
     0x3f, 0x43, 0x8a, 0x6e, 0xb0, 0xa1, 0xbb, 0x6f, 0x19, 0x79, 0x8f, 0x81,
     0x5c, 0xd9, 0x89, 0x91, 0xde, 0xfa, 0xce, 0x09, 0x53, 0xb7, 0x37, 0x36,
     0x04, 0x48, 0x5f, 0x44, 0x7c, 0x19, 0x09, 0xb8, 0x1b, 0x8f, 0xf1, 0xf5,
     0x86, 0x3a, 0x81, 0xf8, 0x44, 0x76, 0x72, 0xc8, 0x4b, 0xbf, 0x70, 0x75,
     0xb7, 0xc1, 0x5d, 0xff, 0x91, 0xdd, 0x02, 0x03, 0x01, 0x00, 0x01, 0x02,
     0x82, 0x01, 0x00, 0x4a, 0x50, 0x5a, 0x69, 0x77, 0x81, 0x39, 0x85, 0xf4,
     0xee, 0xab, 0xb3, 0x01, 0x41, 0x4b, 0xcd, 0x09, 0xbb, 0x1e, 0xb5, 0x99,
     0xdf, 0xcb, 0x7f, 0xcf, 0x42, 0xf5, 0xf4, 0xc3, 0x16, 0x21, 0xab, 0x0b,
     0xc0, 0x1c, 0x91, 0x00, 0xea, 0x35, 0x83, 0x99, 0x1b, 0x25, 0xfd, 0x49,
     0x03, 0x92, 0xe8, 0x0e, 0xd4, 0x28, 0x8d, 0x96, 0x5f, 0x24, 0x4e, 0xf3,
     0xa3, 0x84, 0x3e, 0xb1, 0xa7, 0xf1, 0xf1, 0x5c, 0x60, 0x8a, 0xac, 0xb1,
     0xfe, 0x2c, 0xb1, 0xe4, 0x8f, 0xcb, 0x1f, 0xba, 0xdc, 0x1c, 0xa6, 0x3a,
     0xc7, 0x9c, 0x19, 0xba, 0x22, 0x50, 0xed, 0xee, 0xb0, 0x5c, 0x6f, 0xdd,
     0xef, 0x20, 0xd3, 0xcd, 0xff, 0x1c, 0x0a, 0x43, 0x39, 0x93, 0x9c, 0x59,
     0xc9, 0x56, 0x91, 0x25, 0x57, 0x67, 0x37, 0x34, 0xb9, 0xda, 0x08, 0x2e,
     0x36, 0x7e, 0xd4, 0xba, 0xe3, 0xf7, 0xb4, 0x50, 0x91, 0xe3, 0x1c, 0xa3,
     0x96, 0x0e, 0xec, 0x37, 0x52, 0xe7, 0xd9, 0x64, 0xb8, 0xa7, 0x54, 0x81,
     0xe6, 0x08, 0x87, 0xda, 0x71, 0x6a, 0xb7, 0x86, 0x00, 0xce, 0xe8, 0x22,
     0xcf, 0xed, 0x6d, 0x5f, 0x72, 0xa0, 0x16, 0xe4, 0x3b, 0x0f, 0xc0, 0xc5,
     0xf9, 0x57, 0x21, 0xee, 0x00, 0x2b, 0xf5, 0x0c, 0x37, 0x3d, 0x49, 0x00,
     0x91, 0x00, 0xc4, 0x0d, 0x1f, 0xa6, 0x3a, 0x76, 0x59, 0x6e, 0xd4, 0xeb,
     0x0f, 0x8b, 0x82, 0x0c, 0x98, 0x61, 0xaf, 0x4c, 0x98, 0x6a, 0xb4, 0x09,
     0xe0, 0xde, 0xb8, 0x12, 0x1a, 0xcc, 0xf9, 0xdd, 0xfb, 0x6d, 0x3d, 0x3f,
     0xa6, 0xe2, 0x43, 0x31, 0x01, 0x77, 0xcf, 0xfa, 0xf1, 0xd5, 0x32, 0xe3,
     0xd6, 0xe6, 0x1a, 0x92, 0x88, 0xbd, 0x78, 0xbb, 0x1e, 0xdc, 0x13, 0xbb,
     0xff, 0x1c, 0x12, 0x2f, 0x93, 0x28, 0x82, 0xe8, 0x03, 0xf6, 0xec, 0x56,
     0x4c, 0x08, 0xa2, 0xe7, 0xc3, 0xc9, 0x3f, 0x02, 0x81, 0x81, 0x00, 0xf8,
     0xcc, 0xfa, 0xca, 0xdf, 0x58, 0x59, 0x8e, 0x1d, 0x70, 0x99, 0xce, 0x78,
     0x9e, 0xdc, 0x56, 0x08, 0x91, 0x5f, 0x63, 0xb0, 0x92, 0x69, 0xbb, 0x17,
     0xc4, 0xd1, 0x69, 0xc4, 0xdb, 0xc3, 0xf0, 0x36, 0xe8, 0x0e, 0x76, 0x42,
     0x21, 0xd2, 0x9a, 0x2c, 0xe5, 0x38, 0x5d, 0x89, 0xe7, 0xe0, 0x9f, 0x4c,
     0xde, 0xb4, 0x9c, 0x7a, 0x7c, 0x00, 0x5c, 0x09, 0xed, 0x41, 0x98, 0x7f,
     0x3c, 0x90, 0xfd, 0x35, 0xac, 0x77, 0x3d, 0xa2, 0x66, 0x4a, 0x9d, 0x0c,
     0x9b, 0xdf, 0xab, 0xeb, 0x5e, 0xa9, 0xb3, 0x84, 0x71, 0x9a, 0xb9, 0x2a,
     0x9e, 0x1c, 0x38, 0xe4, 0xea, 0xaf, 0xf7, 0xa3, 0x46, 0x5b, 0x14, 0xdf,
     0x0f, 0xd9, 0xc4, 0x8c, 0x18, 0x70, 0x21, 0xfd, 0x2f, 0xd6, 0x1c, 0xb9,
     0x83, 0x36, 0x56, 0xe6, 0x15, 0xd2, 0x3a, 0xf9, 0xbf, 0x50, 0x5a, 0x8a,
     0x82, 0xd7, 0xc8, 0x23, 0xd7, 0x2e, 0x0b, 0x02, 0x81, 0x81, 0x00, 0xbc,
     0xc1, 0xfe, 0x10, 0x4c, 0xe4, 0x0e, 0x9d, 0x65, 0xfc, 0x31, 0x96, 0x44,
     0xdf, 0xf2, 0x46, 0x59, 0xd4, 0xf5, 0xe1, 0x44, 0xab, 0x67, 0x80, 0x43,
     0xb0, 0xa5, 0x73, 0xec, 0xcb, 0xb5, 0x9a, 0xc1, 0xce, 0xa7, 0x22, 0xb0,
     0x3e, 0xbf, 0x4f, 0x2e, 0x6c, 0x3c, 0xb3, 0x54, 0x44, 0x61, 0xb4, 0x70,
     0xc3, 0x99, 0xdf, 0xfd, 0xa5, 0x39, 0x27, 0x1d, 0x5c, 0x0b, 0xf7, 0xbe,
     0x6a, 0x65, 0x80, 0xb7, 0xb0, 0x59, 0xf5, 0xb3, 0xb8, 0x3f, 0xa5, 0x73,
     0x2b, 0x49, 0xe0, 0x9e, 0x55, 0xf8, 0x46, 0xdb, 0x22, 0xb4, 0x4b, 0xc1,
     0xb6, 0x50, 0x36, 0xd9, 0xa7, 0x1c, 0xf2, 0x7c, 0xca, 0x0e, 0xa7, 0xef,
     0xce, 0xa1, 0x42, 0xb7, 0x42, 0x17, 0x07, 0xc4, 0xc0, 0x3e, 0xd4, 0x56,
     0x27, 0x07, 0xd7, 0x31, 0x47, 0xe6, 0xbf, 0xe6, 0xa5, 0x2e, 0x2b, 0x7a,
     0x7d, 0xcc, 0x60, 0x12, 0x99, 0xf8, 0xb7, 0x02, 0x81, 0x80, 0x5a, 0xd5,
     0xc3, 0x8e, 0x83, 0xe2, 0x66, 0xb7, 0xdb, 0x09, 0xbc, 0x2d, 0xc4, 0x9e,
     0x03, 0x45, 0xa9, 0xd5, 0x21, 0x65, 0x6d, 0x16, 0xd7, 0x61, 0x46, 0x39,
     0x46, 0x57, 0x7e, 0x56, 0xd9, 0xff, 0x7e, 0x9c, 0x54, 0x83, 0x5a, 0x7b,
     0xac, 0xbf, 0x3b, 0x3a, 0xe8, 0xcc, 0x45, 0xc8, 0x11, 0x9b, 0x37, 0x5e,
     0x6b, 0xc4, 0x61, 0x77, 0x9a, 0x4e, 0x00, 0x15, 0xce, 0x08, 0x16, 0x14,
     0x0f, 0xbf, 0x52, 0x74, 0x48, 0x08, 0x89, 0x9d, 0x1d, 0x0a, 0x9f, 0x8a,
     0xdd, 0x2b, 0x90, 0x40, 0x3c, 0x66, 0xdd, 0x28, 0xf8, 0xdb, 0x37, 0xb3,
     0x08, 0x0c, 0xc1, 0x8e, 0xe9, 0x75, 0xd8, 0xf7, 0x9b, 0xd3, 0x4f, 0xe9,
     0x22, 0x91, 0x7e, 0xb0, 0x81, 0x67, 0xf7, 0x5f, 0x1a, 0xa5, 0xdc, 0x19,
     0x0a, 0xa2, 0xc9, 0x58, 0x18, 0x2c, 0x0d, 0xf8, 0x8a, 0x26, 0xb4, 0x41,
     0x36, 0xf4, 0xcc, 0x19, 0x08, 0xa7, 0x02, 0x81, 0x80, 0x62, 0x13, 0x59,
     0xf3, 0x16, 0x40, 0x98, 0xe7, 0x67, 0x8a, 0x36, 0x29, 0xa1, 0xf7, 0xca,
     0x66, 0x8b, 0x5e, 0x7f, 0xb3, 0x60, 0x7e, 0xbe, 0xf4, 0x82, 0x37, 0x52,
     0x80, 0x7d, 0x55, 0x0b, 0x33, 0x31, 0xe8, 0x32, 0x27, 0x6f, 0xf3, 0xea,
     0x6b, 0x35, 0xef, 0xbf, 0x4a, 0x5e, 0x4a, 0x79, 0x89, 0xcb, 0xdd, 0x96,
     0x22, 0x30, 0x24, 0x9d, 0x21, 0x99, 0xbb, 0xad, 0xec, 0x37, 0xe0, 0x08,
     0x85, 0x6c, 0xec, 0x10, 0x91, 0xfd, 0xa3, 0x8a, 0x4e, 0x69, 0x1c, 0xe0,
     0xf1, 0xf8, 0xd3, 0x2a, 0x81, 0x86, 0x72, 0xed, 0xc3, 0x3f, 0x0f, 0x7f,
     0x76, 0x40, 0x78, 0xf8, 0x2d, 0x76, 0x71, 0x76, 0x54, 0x03, 0xe2, 0x15,
     0x20, 0x19, 0x20, 0x19, 0xdf, 0x4b, 0x77, 0xa8, 0x2d, 0xa3, 0xe5, 0xfb,
     0xc8, 0xf0, 0x2e, 0x2f, 0xd3, 0x1e, 0x00, 0x4d, 0x91, 0x01, 0xc0, 0x43,
     0x64, 0xd7, 0xc5, 0x70, 0xd9, 0x02, 0x81, 0x81, 0x00, 0xe9, 0x63, 0xbf,
     0xee, 0xc7, 0x30, 0x95, 0x4e, 0x89, 0x41, 0xdb, 0x6f, 0x58, 0x10, 0x56,
     0x50, 0x4b, 0x37, 0xf6, 0x90, 0xa9, 0x20, 0x12, 0x95, 0xdd, 0xa1, 0x61,
     0x88, 0x3a, 0xe1, 0x85, 0x32, 0x2d, 0x4d, 0x41, 0x44, 0x8e, 0xae, 0x33,
     0x2e, 0x30, 0xc7, 0xf7, 0x2e, 0x14, 0xde, 0xb0, 0x38, 0x9d, 0x1e, 0x56,
     0x85, 0xb3, 0x31, 0x8e, 0x12, 0x55, 0x7a, 0x88, 0x53, 0xe8, 0x75, 0x3c,
     0x5b, 0x38, 0x98, 0xf3, 0x9b, 0x65, 0x70, 0x33, 0xf4, 0x62, 0x58, 0x04,
     0x6f, 0x55, 0x1d, 0xa5, 0xff, 0xb1, 0xa4, 0x48, 0x34, 0x4d, 0xf3, 0xbc,
     0xa9, 0xbc, 0xa5, 0x8d, 0x66, 0x18, 0xad, 0x60, 0x45, 0x20, 0x9b, 0xd7,
     0x5b, 0xd6, 0x53, 0x11, 0x59, 0x9d, 0x68, 0xf9, 0x02, 0x5a, 0x25, 0x77,
     0x41, 0x09, 0x6d, 0x9d, 0x6a, 0x99, 0x02, 0xc5, 0x25, 0x7a, 0x0b, 0xbc,
     0x0d, 0xcb, 0x2e, 0x84, 0xad});

static constexpr auto kRsa2048Public = std::to_array<uint8_t>(
    {0x30, 0x82, 0x01, 0x22, 0x30, 0x0d, 0x06, 0x09, 0x2a, 0x86, 0x48, 0x86,
     0xf7, 0x0d, 0x01, 0x01, 0x01, 0x05, 0x00, 0x03, 0x82, 0x01, 0x0f, 0x00,
     0x30, 0x82, 0x01, 0x0a, 0x02, 0x82, 0x01, 0x01, 0x00, 0xb7, 0x73, 0x11,
     0xa1, 0x47, 0x56, 0xb8, 0xdf, 0x45, 0xab, 0x81, 0xc1, 0x17, 0x24, 0x00,
     0x6d, 0x70, 0xc2, 0x3f, 0x54, 0xac, 0x71, 0xda, 0xf5, 0x5d, 0xd8, 0x7c,
     0x77, 0x36, 0xa6, 0x2e, 0x30, 0xbd, 0x07, 0x27, 0x34, 0x20, 0xdf, 0xc5,
     0xf7, 0xdb, 0x9d, 0x93, 0xdc, 0x68, 0xa1, 0xc4, 0x02, 0x4f, 0xcb, 0xb1,
     0x1e, 0xb9, 0xd0, 0x12, 0x92, 0xa2, 0xf0, 0x0d, 0xfb, 0xe6, 0xa0, 0x62,
     0x36, 0xd2, 0x6e, 0xae, 0x03, 0xd5, 0xa8, 0x53, 0x46, 0x77, 0x7a, 0xe3,
     0x83, 0xfe, 0x8d, 0x54, 0x29, 0x03, 0xc1, 0x19, 0x6d, 0x22, 0x44, 0x0f,
     0x1a, 0x58, 0x57, 0x01, 0xef, 0x81, 0x3f, 0x0b, 0xf3, 0x96, 0x8e, 0x51,
     0x40, 0xd3, 0x67, 0xd8, 0x85, 0x9f, 0x85, 0x80, 0x8e, 0x84, 0xc6, 0xaa,
     0x35, 0xae, 0x35, 0xff, 0x38, 0x20, 0xbb, 0xe3, 0x95, 0xc5, 0xe3, 0xd6,
     0xdd, 0xe9, 0x28, 0x87, 0xb9, 0x38, 0xce, 0xbd, 0x53, 0x17, 0x97, 0x90,
     0xed, 0x87, 0xd5, 0x08, 0x6c, 0x84, 0xef, 0xb1, 0x9e, 0x71, 0x5e, 0x2c,
     0x0b, 0xce, 0xc1, 0x76, 0xbb, 0xfb, 0xd6, 0x5b, 0x55, 0x85, 0xe0, 0x4d,
     0xf7, 0xaf, 0xcf, 0x34, 0xe1, 0x40, 0x0a, 0x76, 0x1f, 0xb1, 0xdc, 0xd7,
     0xb0, 0x63, 0xa2, 0xc7, 0x23, 0x85, 0x56, 0xfd, 0x8c, 0xec, 0x6b, 0x0f,
     0xfb, 0x6e, 0xcb, 0xba, 0x3b, 0x12, 0x04, 0xbb, 0xf7, 0x5d, 0x28, 0xe2,
     0x57, 0x9c, 0x38, 0x1d, 0x8f, 0xc9, 0x51, 0x3f, 0x43, 0x8a, 0x6e, 0xb0,
     0xa1, 0xbb, 0x6f, 0x19, 0x79, 0x8f, 0x81, 0x5c, 0xd9, 0x89, 0x91, 0xde,
     0xfa, 0xce, 0x09, 0x53, 0xb7, 0x37, 0x36, 0x04, 0x48, 0x5f, 0x44, 0x7c,
     0x19, 0x09, 0xb8, 0x1b, 0x8f, 0xf1, 0xf5, 0x86, 0x3a, 0x81, 0xf8, 0x44,
     0x76, 0x72, 0xc8, 0x4b, 0xbf, 0x70, 0x75, 0xb7, 0xc1, 0x5d, 0xff, 0x91,
     0xdd, 0x02, 0x03, 0x01, 0x00, 0x01});

static constexpr auto kRsa4096Private = std::to_array<uint8_t>(
    {0x30, 0x82, 0x09, 0x43, 0x02, 0x01, 0x00, 0x30, 0x0d, 0x06, 0x09, 0x2a,
     0x86, 0x48, 0x86, 0xf7, 0x0d, 0x01, 0x01, 0x01, 0x05, 0x00, 0x04, 0x82,
     0x09, 0x2d, 0x30, 0x82, 0x09, 0x29, 0x02, 0x01, 0x00, 0x02, 0x82, 0x02,
     0x01, 0x00, 0xc5, 0xca, 0x53, 0x4b, 0x6e, 0x08, 0x5c, 0xa2, 0x96, 0x53,
     0xe3, 0x3e, 0xfd, 0xf8, 0x31, 0x93, 0x56, 0x50, 0x0b, 0xec, 0x46, 0x5c,
     0x21, 0x78, 0x66, 0x1a, 0x94, 0x2f, 0xcb, 0xec, 0x52, 0xba, 0xa7, 0xe5,
     0xeb, 0x39, 0x8a, 0x34, 0xd6, 0x94, 0x37, 0xd9, 0x4e, 0xd5, 0xab, 0xaa,
     0xb8, 0xde, 0xc4, 0xe4, 0x05, 0xa5, 0x51, 0xeb, 0x74, 0x15, 0x9a, 0x99,
     0x79, 0xe2, 0x51, 0xb2, 0x4d, 0xc9, 0x90, 0xd8, 0x47, 0x5b, 0x78, 0xee,
     0x15, 0xa4, 0xb7, 0x72, 0x94, 0x39, 0x09, 0x77, 0xf9, 0x6e, 0x48, 0xae,
     0x7c, 0xa5, 0x65, 0xf3, 0x92, 0xb2, 0xad, 0x3f, 0x43, 0x4a, 0xfb, 0x9f,
     0x33, 0x46, 0x18, 0x72, 0x08, 0xe2, 0x68, 0xf7, 0xa6, 0x65, 0x32, 0x0b,
     0xa4, 0xa7, 0xc4, 0x64, 0xbb, 0xd4, 0x6e, 0x60, 0x9f, 0x38, 0x3a, 0x5b,
     0x26, 0x82, 0x1c, 0x20, 0x66, 0x8f, 0x4b, 0x34, 0x44, 0x45, 0xdf, 0xd6,
     0xe2, 0x32, 0x1e, 0xa8, 0x4e, 0xb2, 0x3d, 0xa5, 0x5d, 0xe7, 0x22, 0x11,
     0x3f, 0x4a, 0xc0, 0xba, 0xea, 0x58, 0xf5, 0x22, 0xa4, 0x53, 0x49, 0xfa,
     0x83, 0xd1, 0xa1, 0x90, 0xd8, 0xfc, 0x5c, 0x02, 0x74, 0xa1, 0xd8, 0xaf,
     0xc7, 0xec, 0x97, 0x2f, 0x1f, 0xf9, 0xda, 0xb6, 0xd0, 0x4e, 0x7e, 0x48,
     0xd0, 0x22, 0xf7, 0x4d, 0x8a, 0x6d, 0xef, 0xe4, 0xc0, 0xf9, 0x04, 0xbb,
     0x1e, 0x84, 0x96, 0xa8, 0x78, 0xd0, 0x21, 0xb6, 0x24, 0x0c, 0x39, 0x2f,
     0xd0, 0x47, 0x3f, 0x1d, 0x6e, 0xcc, 0xef, 0x59, 0xf1, 0xd8, 0xba, 0x11,
     0xab, 0x2c, 0x72, 0x6b, 0xef, 0x81, 0xcd, 0xf7, 0x1b, 0x3d, 0xcf, 0x39,
     0x66, 0x7e, 0xf2, 0x11, 0x41, 0x8b, 0x19, 0x7c, 0x77, 0xdf, 0x89, 0x35,
     0x04, 0xd8, 0x55, 0xfa, 0xb1, 0xe5, 0xca, 0xdd, 0xca, 0x85, 0x03, 0xcb,
     0xb1, 0xca, 0xa2, 0x76, 0x70, 0xa8, 0x4a, 0x6e, 0x20, 0x0e, 0x25, 0xd4,
     0xb4, 0xda, 0xe6, 0x10, 0x04, 0x82, 0xd3, 0x70, 0x0c, 0xd3, 0x01, 0x82,
     0x46, 0x63, 0x1f, 0x96, 0x4c, 0x9a, 0x9b, 0x24, 0x9e, 0xab, 0xc3, 0x0b,
     0x9a, 0x3c, 0x94, 0x9e, 0xf0, 0xef, 0x1a, 0xac, 0xa0, 0x74, 0x3b, 0x0d,
     0x10, 0xdd, 0xbe, 0xca, 0x7a, 0x69, 0x5e, 0x70, 0x90, 0xaa, 0x96, 0xe2,
     0xa8, 0x46, 0x9b, 0x9c, 0xa2, 0x19, 0xc2, 0x55, 0x07, 0xbf, 0x6c, 0x1a,
     0x8b, 0x7f, 0x34, 0xf2, 0x3c, 0x6f, 0x21, 0x60, 0x6b, 0x7e, 0xc0, 0x79,
     0x35, 0x5b, 0x8e, 0x5a, 0x2e, 0xec, 0xfb, 0x92, 0xfb, 0xa7, 0x73, 0x6b,
     0xde, 0xe5, 0xd3, 0xbb, 0xa7, 0x83, 0xbc, 0xdb, 0x73, 0x1b, 0xdc, 0xb8,
     0xe1, 0x9c, 0x1c, 0xec, 0x39, 0x2c, 0xaf, 0xcd, 0x25, 0x45, 0xa6, 0x56,
     0x70, 0x9b, 0x31, 0x88, 0x66, 0x01, 0x09, 0x7c, 0x61, 0xfb, 0xca, 0x05,
     0x93, 0x7a, 0x16, 0x2a, 0xdb, 0xbd, 0xf2, 0xe1, 0x46, 0x01, 0x5e, 0x8e,
     0x82, 0x68, 0x8f, 0xf1, 0xd8, 0x9a, 0x30, 0xda, 0xa3, 0xf1, 0x9a, 0x67,
     0xd8, 0x59, 0xe8, 0x4c, 0x89, 0x70, 0x97, 0xdc, 0x6d, 0x83, 0x1a, 0x01,
     0xe8, 0x00, 0x74, 0xfd, 0x44, 0x73, 0x0f, 0x86, 0x72, 0x03, 0x71, 0x48,
     0x90, 0x01, 0x12, 0xd3, 0xd1, 0x85, 0x2e, 0x67, 0xf1, 0x07, 0x78, 0x8d,
     0x9f, 0xc8, 0x44, 0xd1, 0xa3, 0x41, 0x9f, 0xd7, 0xc6, 0x50, 0x44, 0x9d,
     0x09, 0xf4, 0xab, 0x7c, 0x12, 0x65, 0x59, 0xc4, 0x43, 0x51, 0x94, 0x25,
     0x9d, 0x0a, 0xb1, 0x1b, 0xd2, 0x3d, 0xeb, 0xd4, 0xef, 0xbf, 0xd7, 0x66,
     0xa3, 0x51, 0x67, 0xc1, 0x42, 0x88, 0xe2, 0x7a, 0x28, 0x9a, 0x40, 0x1a,
     0x60, 0xce, 0xa2, 0x42, 0x72, 0x45, 0x8a, 0x4c, 0x08, 0xcc, 0x06, 0xa0,
     0x6d, 0xab, 0xa3, 0xea, 0xee, 0x63, 0x65, 0x9e, 0x40, 0x19, 0x02, 0x03,
     0x01, 0x00, 0x01, 0x02, 0x82, 0x02, 0x00, 0x0d, 0xec, 0x84, 0x80, 0xdd,
     0xc4, 0x03, 0xae, 0x73, 0xf5, 0xff, 0x9c, 0x4c, 0x81, 0x7b, 0x8e, 0xf7,
     0x5f, 0x6e, 0xf9, 0x71, 0x0a, 0x15, 0x16, 0x9e, 0x5c, 0x7e, 0x64, 0x5f,
     0x7f, 0x19, 0x63, 0x57, 0xc4, 0xc3, 0x32, 0x93, 0xa8, 0xcc, 0xd6, 0xd6,
     0x18, 0x2b, 0x99, 0x35, 0xea, 0xb0, 0xb5, 0x49, 0x07, 0xd2, 0xe0, 0x6c,
     0xa4, 0x0e, 0x51, 0xe3, 0x86, 0x6c, 0xaa, 0xca, 0xac, 0xca, 0x56, 0x57,
     0x66, 0xac, 0x5e, 0x53, 0x84, 0xf2, 0x3b, 0xdc, 0x68, 0xb9, 0xe7, 0xca,
     0x83, 0x7a, 0x49, 0x21, 0xdf, 0x7b, 0xb9, 0xa2, 0x93, 0xdb, 0x36, 0xce,
     0x24, 0xb3, 0x1a, 0x6d, 0x1c, 0x8b, 0xdd, 0xe2, 0x44, 0x6b, 0xdc, 0xf1,
     0x7e, 0x06, 0xa6, 0x1e, 0xd7, 0xec, 0x3d, 0x5d, 0xc1, 0x68, 0x2a, 0x28,
     0x23, 0x8f, 0xed, 0xe3, 0xc8, 0xcd, 0x2d, 0x11, 0x69, 0x14, 0x03, 0x40,
     0x2c, 0x73, 0xa6, 0x76, 0x1f, 0xb0, 0xe0, 0x8d, 0xa1, 0x34, 0x5a, 0x0e,
     0xa6, 0x2c, 0xb6, 0x43, 0xe3, 0x12, 0x31, 0x33, 0x60, 0x6d, 0x35, 0x3a,
     0x3b, 0x98, 0x35, 0x1f, 0x52, 0xfe, 0x12, 0xe4, 0xc4, 0x77, 0x4c, 0x0b,
     0xb7, 0xa6, 0x09, 0x8b, 0x41, 0x33, 0xaa, 0x33, 0x89, 0xef, 0x1e, 0xb6,
     0x1f, 0x92, 0xca, 0x45, 0xd0, 0x5f, 0x2b, 0xa8, 0xfc, 0x79, 0xd3, 0x6b,
     0xb5, 0x0f, 0x43, 0x30, 0x8a, 0xb2, 0x37, 0x2c, 0x0a, 0x7b, 0xe7, 0xce,
     0x09, 0xbe, 0x1f, 0xa8, 0xaa, 0x75, 0xa7, 0x80, 0x59, 0x54, 0x76, 0xa3,
     0x9e, 0x71, 0xbc, 0xbb, 0xbb, 0x98, 0xf0, 0xe9, 0xf4, 0xb0, 0x81, 0x93,
     0x33, 0xf3, 0x44, 0x5e, 0x89, 0x58, 0x4e, 0x75, 0x40, 0xd2, 0xfc, 0x05,
     0x7d, 0x2a, 0x4c, 0xaf, 0xa2, 0x6e, 0x7f, 0x02, 0xcd, 0x81, 0x56, 0xcf,
     0x99, 0x5d, 0xc3, 0x8f, 0xac, 0xf8, 0x2d, 0x0f, 0x53, 0x16, 0x72, 0xd8,
     0xb5, 0xe6, 0x05, 0x2b, 0x1d, 0xc6, 0x4c, 0x2e, 0x71, 0xbf, 0x20, 0xfa,
     0xda, 0xef, 0x24, 0x9c, 0xba, 0x06, 0x47, 0x5b, 0xab, 0xc1, 0xf7, 0x5c,
     0x8f, 0x49, 0x6c, 0xfd, 0xb6, 0x10, 0x55, 0x88, 0x65, 0xd1, 0xed, 0x91,
     0x56, 0xa2, 0xc0, 0xd1, 0x76, 0x76, 0x3b, 0x77, 0x93, 0x18, 0x83, 0xcd,
     0xa4, 0xc7, 0x7c, 0xe2, 0xfb, 0x37, 0x88, 0xcc, 0x87, 0x84, 0xb1, 0x30,
     0x2e, 0x9a, 0x12, 0xca, 0xd8, 0xe5, 0xff, 0xe9, 0x90, 0x89, 0x9d, 0x6e,
     0xf8, 0xae, 0x8f, 0x4b, 0x33, 0x66, 0x81, 0xbb, 0xe4, 0xef, 0xe1, 0xb4,
     0xbe, 0x05, 0x7e, 0x40, 0x72, 0x7c, 0x42, 0x7f, 0xb8, 0x02, 0xd7, 0xd0,
     0x94, 0x2e, 0x47, 0xa5, 0x01, 0xe6, 0x0c, 0x60, 0xd2, 0xe7, 0x8e, 0x38,
     0x5a, 0x21, 0xf7, 0x4e, 0xf8, 0xa7, 0x7f, 0x9f, 0xaf, 0x2a, 0x8e, 0x6d,
     0x82, 0x89, 0x87, 0x76, 0xad, 0x29, 0xfe, 0x9d, 0xa4, 0x24, 0xe8, 0x68,
     0x12, 0x94, 0x80, 0x7f, 0x70, 0xe7, 0x7c, 0x8e, 0xfd, 0x8c, 0xb4, 0x5c,
     0xc0, 0x3a, 0xb0, 0xca, 0xbb, 0x6f, 0x31, 0x50, 0x71, 0x16, 0xa7, 0x71,
     0x00, 0xfa, 0x19, 0x8b, 0x45, 0xf2, 0xb4, 0x97, 0xee, 0x45, 0x4c, 0xb3,
     0xf4, 0x8c, 0x35, 0x5b, 0x29, 0x60, 0xc1, 0xb1, 0xb4, 0x98, 0x2f, 0x89,
     0xbe, 0xc5, 0xb5, 0x6d, 0xda, 0x1a, 0xc0, 0xa4, 0x24, 0x9e, 0xa3, 0x6c,
     0xa9, 0x02, 0x00, 0xd8, 0x7d, 0x9d, 0x04, 0xea, 0xe4, 0xbb, 0x85, 0xb1,
     0x7f, 0xf8, 0xd3, 0x51, 0x17, 0x86, 0xf1, 0x69, 0x68, 0x17, 0x2c, 0x27,
     0x64, 0x77, 0xf9, 0x62, 0x7c, 0xbb, 0x10, 0x3c, 0xd4, 0x2e, 0x9e, 0x5c,
     0xb3, 0xaa, 0xec, 0x97, 0x18, 0x50, 0x0d, 0xbc, 0x20, 0x2d, 0x7b, 0xd0,
     0x69, 0xe8, 0xae, 0x72, 0x3c, 0xdd, 0xdc, 0x40, 0xff, 0x57, 0xb4, 0xfe,
     0x08, 0x05, 0x81, 0x02, 0x82, 0x01, 0x01, 0x00, 0xf2, 0x9d, 0xdf, 0xd3,
     0xef, 0xa3, 0x95, 0xf9, 0xae, 0xc6, 0x73, 0xcb, 0xcb, 0x26, 0x65, 0x99,
     0x54, 0xd6, 0x3c, 0x3f, 0xfd, 0x86, 0xec, 0x61, 0x9d, 0x6a, 0xcf, 0x64,
     0x64, 0x3d, 0xe5, 0xeb, 0x9d, 0x49, 0xcb, 0x07, 0x3e, 0xf0, 0x17, 0xb3,
     0xcd, 0x00, 0x9b, 0xf6, 0x53, 0x13, 0x30, 0x23, 0x47, 0xe5, 0xec, 0xaa,
     0x43, 0xfb, 0x7a, 0x1b, 0x4b, 0xfb, 0x9c, 0xa6, 0xdd, 0x05, 0x70, 0xd2,
     0xab, 0xaa, 0xee, 0x10, 0x32, 0xdd, 0xcd, 0x81, 0x00, 0xe1, 0x52, 0x71,
     0x43, 0xa8, 0x64, 0x64, 0xb1, 0x8c, 0x2a, 0xa7, 0xbf, 0x05, 0x26, 0x99,
     0x6e, 0x65, 0xe1, 0x8c, 0x5d, 0x1c, 0x97, 0xc6, 0xa0, 0x47, 0xcc, 0xd1,
     0x06, 0xb8, 0xe8, 0x06, 0x97, 0x1c, 0x9c, 0x7c, 0xde, 0xd5, 0x28, 0xc1,
     0x9f, 0xa3, 0xce, 0x3e, 0x75, 0xe0, 0x8c, 0x6f, 0x06, 0xb0, 0xff, 0xc9,
     0x96, 0x34, 0x68, 0x56, 0xe1, 0xca, 0x21, 0x89, 0xc7, 0x66, 0xb5, 0x87,
     0x8b, 0xe2, 0xf4, 0x55, 0x6f, 0x85, 0x7d, 0x6e, 0x57, 0xbd, 0x0f, 0x4a,
     0x92, 0xfd, 0xa8, 0x8e, 0x6e, 0x87, 0x82, 0xb0, 0x4b, 0x50, 0x80, 0xeb,
     0x2c, 0x8b, 0xf4, 0xf1, 0x35, 0x23, 0xaf, 0x42, 0xfd, 0xf6, 0x36, 0xce,
     0xcb, 0x5e, 0xe8, 0x4f, 0xcc, 0x67, 0xa3, 0x43, 0x74, 0x10, 0x61, 0x88,
     0xc2, 0x1c, 0x61, 0xde, 0xc0, 0xfc, 0xd1, 0x60, 0xd1, 0xed, 0x0e, 0x07,
     0xd8, 0x05, 0xdf, 0xee, 0xcb, 0xcb, 0x26, 0x90, 0x90, 0xa1, 0xf9, 0x80,
     0x86, 0x84, 0xcb, 0x38, 0x0c, 0x30, 0xc3, 0x2d, 0x8b, 0x18, 0xc8, 0x4e,
     0xfc, 0x26, 0xad, 0x20, 0x8c, 0x0c, 0x2b, 0x08, 0x78, 0x21, 0x28, 0xcb,
     0xe4, 0x09, 0x70, 0x64, 0xbf, 0xdd, 0xfa, 0x3d, 0xff, 0xa7, 0xf4, 0xf5,
     0x0c, 0x51, 0x84, 0x6c, 0xda, 0x93, 0xc4, 0x0b, 0x75, 0x18, 0xdd, 0x99,
     0x02, 0x82, 0x01, 0x01, 0x00, 0xd0, 0xb3, 0x6e, 0xd7, 0x23, 0x36, 0x15,
     0xaa, 0x0d, 0xf9, 0x46, 0xfc, 0xa0, 0xfc, 0xef, 0x39, 0xa0, 0x51, 0x82,
     0xcb, 0x8a, 0x97, 0xba, 0xc5, 0xaa, 0x48, 0xe2, 0xc6, 0x36, 0xfe, 0xb2,
     0x95, 0x77, 0x84, 0x84, 0x3f, 0x87, 0x37, 0x9b, 0xa7, 0x9f, 0x2c, 0xad,
     0x65, 0x00, 0x1b, 0x94, 0x53, 0xef, 0xf5, 0x86, 0x10, 0x35, 0x7d, 0x1f,
     0x16, 0xe8, 0xa7, 0x5a, 0xfc, 0x56, 0xd0, 0xe1, 0xff, 0xaf, 0x34, 0x10,
     0x71, 0x50, 0x71, 0xfd, 0x5c, 0xb2, 0x0c, 0x6a, 0xab, 0x70, 0xa3, 0xa3,
     0x3d, 0xe8, 0x02, 0xbf, 0xdd, 0xe9, 0x4d, 0x80, 0x79, 0xe5, 0xef, 0xe3,
     0x17, 0x39, 0xd9, 0x05, 0xa2, 0xbe, 0xd1, 0x03, 0x59, 0x21, 0x00, 0xcf,
     0xa4, 0x06, 0xa6, 0xe2, 0x3d, 0xa4, 0x2f, 0x2f, 0xa1, 0x55, 0x8a, 0x86,
     0x6f, 0xba, 0x3b, 0x48, 0x13, 0x79, 0x65, 0x0a, 0x82, 0xed, 0x00, 0x26,
     0x77, 0xcb, 0x91, 0x56, 0xc2, 0x86, 0x61, 0xe7, 0x1f, 0x6a, 0x66, 0x6b,
     0xd1, 0xea, 0x76, 0x58, 0xd4, 0xf1, 0x5a, 0x09, 0xb3, 0xed, 0x5f, 0x1b,
     0x70, 0x14, 0x6b, 0x6a, 0x47, 0xe1, 0xb5, 0x95, 0x36, 0x7d, 0xb7, 0x34,
     0x4d, 0xd3, 0x3f, 0x3b, 0x09, 0x0f, 0x76, 0x36, 0x57, 0x89, 0x4d, 0x7d,
     0x36, 0x69, 0xcd, 0x05, 0x19, 0x26, 0x3a, 0x52, 0x2d, 0x3b, 0x7c, 0x85,
     0x58, 0x96, 0x29, 0x9a, 0x7d, 0xed, 0xe7, 0x05, 0x55, 0xfb, 0xfe, 0x7d,
     0x26, 0x1f, 0xbe, 0x53, 0x81, 0xe2, 0xb9, 0xb7, 0xa5, 0x4a, 0xb3, 0x82,
     0x98, 0xb7, 0x61, 0xb3, 0x87, 0xba, 0x83, 0xb0, 0x51, 0x0e, 0xa0, 0x0b,
     0x00, 0x23, 0x93, 0x54, 0x93, 0xcf, 0xba, 0xab, 0xef, 0x91, 0x6a, 0x89,
     0x69, 0x65, 0x53, 0xe8, 0x18, 0x31, 0x90, 0x4c, 0x39, 0x32, 0x19, 0x06,
     0xe9, 0x22, 0x92, 0xcb, 0xe9, 0x93, 0x2d, 0x06, 0x81, 0x02, 0x82, 0x01,
     0x01, 0x00, 0x85, 0x2a, 0xea, 0xa4, 0x65, 0xb0, 0xa0, 0xad, 0x3f, 0xa5,
     0x66, 0x01, 0xc2, 0x2b, 0xfd, 0x30, 0x40, 0x44, 0xa7, 0x25, 0x68, 0x7f,
     0x1a, 0x58, 0x2a, 0x13, 0x6a, 0x6a, 0x6c, 0x2b, 0x1d, 0x7c, 0x4a, 0x05,
     0x2b, 0x0f, 0x7b, 0x18, 0x45, 0xaa, 0x47, 0x27, 0xc2, 0x73, 0x5d, 0xfd,
     0xf6, 0x3a, 0x27, 0x48, 0xa8, 0xb2, 0x7c, 0x46, 0x18, 0x2b, 0xcb, 0x74,
     0xfb, 0xf3, 0x0a, 0xf5, 0xe5, 0x8b, 0x7c, 0xf6, 0x81, 0x5d, 0x1a, 0xce,
     0x6b, 0xf2, 0x17, 0x0d, 0x96, 0x36, 0xdd, 0x30, 0x1c, 0x8a, 0xb8, 0x79,
     0x7b, 0x20, 0x3f, 0xd5, 0x0f, 0xee, 0xbf, 0x8d, 0xe4, 0x53, 0x2a, 0xf8,
     0x7a, 0xc6, 0x67, 0x4b, 0x81, 0xbc, 0x69, 0xb8, 0x2c, 0x83, 0x11, 0x86,
     0x7e, 0xce, 0x7b, 0x70, 0xfc, 0xc1, 0xea, 0x61, 0xfa, 0xde, 0x85, 0xcd,
     0x0e, 0xc8, 0x12, 0x20, 0x2d, 0x05, 0xf5, 0x10, 0x27, 0x05, 0x29, 0x41,
     0xcd, 0x4e, 0xd6, 0xc8, 0x25, 0x73, 0x94, 0xf0, 0xa9, 0xb4, 0x0b, 0x56,
     0x76, 0x16, 0x8e, 0xca, 0x13, 0x0e, 0x97, 0xc7, 0xeb, 0x30, 0xe2, 0xb2,
     0x36, 0xa5, 0x95, 0x71, 0x46, 0x91, 0xcf, 0x0d, 0xb3, 0x10, 0x82, 0x5e,
     0x67, 0xe0, 0x99, 0x8c, 0xff, 0x60, 0x5a, 0x78, 0x69, 0x83, 0x03, 0x6b,
     0x0f, 0x4e, 0x02, 0xf2, 0xca, 0xd6, 0x49, 0xd5, 0x52, 0x52, 0xf6, 0x12,
     0xce, 0xca, 0x19, 0xef, 0xc7, 0xf7, 0xbe, 0x36, 0xdb, 0x47, 0x33, 0x4c,
     0xfa, 0x89, 0xf7, 0x19, 0x30, 0xdd, 0xbe, 0xf8, 0x3c, 0xa3, 0x32, 0xed,
     0xc0, 0xf3, 0xca, 0x99, 0x7b, 0xb8, 0xfe, 0xe6, 0x2b, 0xb8, 0xe5, 0xa5,
     0xf8, 0x28, 0xd9, 0xe7, 0x39, 0x81, 0x50, 0x55, 0x6d, 0xff, 0x9e, 0xe0,
     0xb4, 0x6e, 0x3b, 0x59, 0x71, 0x36, 0xdf, 0xe0, 0x0e, 0x93, 0xc1, 0x15,
     0xc2, 0x51, 0x97, 0xa0, 0x62, 0x61, 0x02, 0x82, 0x01, 0x00, 0x02, 0x1e,
     0xf0, 0xaf, 0x6b, 0x02, 0x2f, 0xb2, 0x2c, 0xb6, 0x2d, 0xcc, 0x7f, 0x6e,
     0x52, 0x98, 0x09, 0x53, 0x0a, 0xbb, 0x3a, 0xcb, 0x53, 0xf0, 0x92, 0x4c,
     0x6f, 0x51, 0x88, 0x59, 0x8a, 0x43, 0x0e, 0x95, 0xe0, 0x2a, 0x2d, 0x1b,
     0x99, 0x8f, 0x58, 0x84, 0xc1, 0xb6, 0x57, 0x0b, 0xf0, 0xb3, 0xf1, 0xaa,
     0x53, 0x14, 0x73, 0x16, 0xb4, 0x6c, 0x2d, 0x2d, 0x16, 0x35, 0x9e, 0x44,
     0x3d, 0x27, 0xb6, 0x06, 0x17, 0x6c, 0xaf, 0x5e, 0x99, 0x2e, 0x89, 0xf8,
     0xaa, 0x54, 0xd7, 0xae, 0x32, 0x08, 0x7d, 0x05, 0x1a, 0x22, 0x0d, 0x2e,
     0xe6, 0x71, 0x56, 0xae, 0xdb, 0x65, 0xef, 0x06, 0x8f, 0x92, 0x19, 0xd2,
     0x51, 0xf0, 0x63, 0xef, 0x78, 0x2c, 0xb8, 0x8c, 0x95, 0x9b, 0xfe, 0xc1,
     0x24, 0x00, 0xc5, 0xb7, 0xc4, 0xa9, 0xfa, 0x00, 0x84, 0x38, 0xfe, 0x70,
     0xd2, 0x6c, 0x86, 0x30, 0x0c, 0x34, 0x07, 0x73, 0x90, 0xa8, 0x25, 0x69,
     0x75, 0x49, 0xd2, 0x70, 0xfc, 0x03, 0x84, 0x18, 0x73, 0xca, 0xa6, 0x31,
     0x3e, 0x0c, 0x00, 0x0b, 0x89, 0x61, 0xd8, 0x33, 0x47, 0x3f, 0x37, 0xc1,
     0xfa, 0xa4, 0x35, 0x1e, 0xd9, 0x7f, 0x38, 0xf8, 0x59, 0x87, 0x3c, 0x0a,
     0xfd, 0x7e, 0x62, 0x3c, 0xd1, 0x5a, 0xee, 0x34, 0x51, 0x2a, 0xf2, 0x42,
     0x81, 0x77, 0x48, 0x35, 0x79, 0xbd, 0x6e, 0xb9, 0x39, 0x82, 0xb2, 0x1a,
     0x38, 0xe9, 0xa8, 0xc7, 0xeb, 0x49, 0xa9, 0xe4, 0xeb, 0x40, 0x54, 0xa7,
     0x82, 0x80, 0x41, 0x84, 0x15, 0x7d, 0xab, 0xcf, 0x68, 0x5d, 0xa6, 0xbd,
     0x93, 0xdb, 0x1f, 0x04, 0xed, 0x57, 0xb1, 0x04, 0xdc, 0x45, 0x2c, 0x45,
     0x3f, 0x5e, 0x0d, 0xe2, 0x41, 0x47, 0x3a, 0xea, 0x61, 0x5f, 0x6d, 0x91,
     0x83, 0xd4, 0xc8, 0xf7, 0x8c, 0x24, 0x6e, 0x5f, 0x83, 0x86, 0xfa, 0x21,
     0xe9, 0x01, 0x02, 0x82, 0x01, 0x01, 0x00, 0x9c, 0xe1, 0x80, 0x78, 0xc7,
     0x12, 0xb2, 0x59, 0xe6, 0x3a, 0xe4, 0x3b, 0x0e, 0xf3, 0x28, 0xec, 0x01,
     0xbf, 0x5a, 0xde, 0xa9, 0x40, 0x3a, 0x44, 0x44, 0x73, 0xaf, 0xe0, 0x2a,
     0xe7, 0x75, 0xec, 0x38, 0x9b, 0x25, 0x4f, 0x2d, 0xf6, 0x2f, 0x29, 0x90,
     0x13, 0x7b, 0xa3, 0x5a, 0xab, 0x55, 0xde, 0x13, 0xbd, 0x01, 0x94, 0x5f,
     0x8b, 0xfc, 0x31, 0x43, 0x7d, 0xb2, 0x25, 0xe0, 0xbe, 0xc4, 0xbb, 0x3f,
     0xf2, 0x9f, 0x05, 0x6a, 0x75, 0xe1, 0xf5, 0x49, 0xb2, 0xb3, 0xd2, 0x19,
     0x1b, 0x32, 0x58, 0x2b, 0x6b, 0x48, 0xc2, 0x70, 0x64, 0x0d, 0x25, 0x0b,
     0x44, 0x2e, 0x31, 0xf6, 0x50, 0xcd, 0x5b, 0xb5, 0xc4, 0xd9, 0xc3, 0x68,
     0x24, 0x2a, 0x2b, 0x01, 0x31, 0xf5, 0xb8, 0x13, 0x0d, 0x08, 0xd9, 0x6d,
     0x74, 0x69, 0xf4, 0x86, 0x65, 0x17, 0x0c, 0xe9, 0x36, 0x39, 0x98, 0xb0,
     0x41, 0x8c, 0xf3, 0xcc, 0x5b, 0x5e, 0x19, 0x58, 0x98, 0x49, 0xb6, 0xcf,
     0x81, 0xc2, 0x5f, 0x8e, 0x39, 0xf3, 0x00, 0x99, 0x4a, 0xca, 0x1a, 0xe5,
     0x29, 0x3a, 0xe0, 0x9e, 0xfc, 0x7c, 0xbb, 0x74, 0x55, 0x12, 0xa8, 0x58,
     0xe0, 0x60, 0x52, 0xbf, 0xd6, 0xcd, 0x75, 0x81, 0x1e, 0x62, 0xdb, 0x30,
     0x71, 0x5f, 0xb0, 0xf6, 0x1a, 0xa8, 0x45, 0xe2, 0x26, 0xc7, 0xa2, 0x73,
     0xfd, 0xd8, 0xeb, 0x14, 0xee, 0x43, 0x5f, 0x7d, 0x19, 0xa6, 0x9d, 0xed,
     0x12, 0x31, 0x91, 0x36, 0x1c, 0x87, 0x17, 0x97, 0xbb, 0x98, 0x74, 0x6c,
     0xce, 0x5a, 0xe4, 0x9a, 0x26, 0xa0, 0x6f, 0x67, 0x99, 0x70, 0x78, 0xe6,
     0x90, 0x01, 0x36, 0x8d, 0x38, 0x0e, 0x85, 0xeb, 0x04, 0x2c, 0xcb, 0xe0,
     0xd5, 0xca, 0xe1, 0x8b, 0xff, 0xea, 0xe7, 0xf1, 0x20, 0xed, 0x56, 0x6d,
     0x6a, 0x4c, 0xf6, 0xa2, 0xc7, 0x54, 0xf5, 0xbf, 0xc5, 0x48, 0x19});

static constexpr auto kRsa4096Public = std::to_array<uint8_t>(
    {0x30, 0x82, 0x02, 0x22, 0x30, 0x0d, 0x06, 0x09, 0x2a, 0x86, 0x48, 0x86,
     0xf7, 0x0d, 0x01, 0x01, 0x01, 0x05, 0x00, 0x03, 0x82, 0x02, 0x0f, 0x00,
     0x30, 0x82, 0x02, 0x0a, 0x02, 0x82, 0x02, 0x01, 0x00, 0xc5, 0xca, 0x53,
     0x4b, 0x6e, 0x08, 0x5c, 0xa2, 0x96, 0x53, 0xe3, 0x3e, 0xfd, 0xf8, 0x31,
     0x93, 0x56, 0x50, 0x0b, 0xec, 0x46, 0x5c, 0x21, 0x78, 0x66, 0x1a, 0x94,
     0x2f, 0xcb, 0xec, 0x52, 0xba, 0xa7, 0xe5, 0xeb, 0x39, 0x8a, 0x34, 0xd6,
     0x94, 0x37, 0xd9, 0x4e, 0xd5, 0xab, 0xaa, 0xb8, 0xde, 0xc4, 0xe4, 0x05,
     0xa5, 0x51, 0xeb, 0x74, 0x15, 0x9a, 0x99, 0x79, 0xe2, 0x51, 0xb2, 0x4d,
     0xc9, 0x90, 0xd8, 0x47, 0x5b, 0x78, 0xee, 0x15, 0xa4, 0xb7, 0x72, 0x94,
     0x39, 0x09, 0x77, 0xf9, 0x6e, 0x48, 0xae, 0x7c, 0xa5, 0x65, 0xf3, 0x92,
     0xb2, 0xad, 0x3f, 0x43, 0x4a, 0xfb, 0x9f, 0x33, 0x46, 0x18, 0x72, 0x08,
     0xe2, 0x68, 0xf7, 0xa6, 0x65, 0x32, 0x0b, 0xa4, 0xa7, 0xc4, 0x64, 0xbb,
     0xd4, 0x6e, 0x60, 0x9f, 0x38, 0x3a, 0x5b, 0x26, 0x82, 0x1c, 0x20, 0x66,
     0x8f, 0x4b, 0x34, 0x44, 0x45, 0xdf, 0xd6, 0xe2, 0x32, 0x1e, 0xa8, 0x4e,
     0xb2, 0x3d, 0xa5, 0x5d, 0xe7, 0x22, 0x11, 0x3f, 0x4a, 0xc0, 0xba, 0xea,
     0x58, 0xf5, 0x22, 0xa4, 0x53, 0x49, 0xfa, 0x83, 0xd1, 0xa1, 0x90, 0xd8,
     0xfc, 0x5c, 0x02, 0x74, 0xa1, 0xd8, 0xaf, 0xc7, 0xec, 0x97, 0x2f, 0x1f,
     0xf9, 0xda, 0xb6, 0xd0, 0x4e, 0x7e, 0x48, 0xd0, 0x22, 0xf7, 0x4d, 0x8a,
     0x6d, 0xef, 0xe4, 0xc0, 0xf9, 0x04, 0xbb, 0x1e, 0x84, 0x96, 0xa8, 0x78,
     0xd0, 0x21, 0xb6, 0x24, 0x0c, 0x39, 0x2f, 0xd0, 0x47, 0x3f, 0x1d, 0x6e,
     0xcc, 0xef, 0x59, 0xf1, 0xd8, 0xba, 0x11, 0xab, 0x2c, 0x72, 0x6b, 0xef,
     0x81, 0xcd, 0xf7, 0x1b, 0x3d, 0xcf, 0x39, 0x66, 0x7e, 0xf2, 0x11, 0x41,
     0x8b, 0x19, 0x7c, 0x77, 0xdf, 0x89, 0x35, 0x04, 0xd8, 0x55, 0xfa, 0xb1,
     0xe5, 0xca, 0xdd, 0xca, 0x85, 0x03, 0xcb, 0xb1, 0xca, 0xa2, 0x76, 0x70,
     0xa8, 0x4a, 0x6e, 0x20, 0x0e, 0x25, 0xd4, 0xb4, 0xda, 0xe6, 0x10, 0x04,
     0x82, 0xd3, 0x70, 0x0c, 0xd3, 0x01, 0x82, 0x46, 0x63, 0x1f, 0x96, 0x4c,
     0x9a, 0x9b, 0x24, 0x9e, 0xab, 0xc3, 0x0b, 0x9a, 0x3c, 0x94, 0x9e, 0xf0,
     0xef, 0x1a, 0xac, 0xa0, 0x74, 0x3b, 0x0d, 0x10, 0xdd, 0xbe, 0xca, 0x7a,
     0x69, 0x5e, 0x70, 0x90, 0xaa, 0x96, 0xe2, 0xa8, 0x46, 0x9b, 0x9c, 0xa2,
     0x19, 0xc2, 0x55, 0x07, 0xbf, 0x6c, 0x1a, 0x8b, 0x7f, 0x34, 0xf2, 0x3c,
     0x6f, 0x21, 0x60, 0x6b, 0x7e, 0xc0, 0x79, 0x35, 0x5b, 0x8e, 0x5a, 0x2e,
     0xec, 0xfb, 0x92, 0xfb, 0xa7, 0x73, 0x6b, 0xde, 0xe5, 0xd3, 0xbb, 0xa7,
     0x83, 0xbc, 0xdb, 0x73, 0x1b, 0xdc, 0xb8, 0xe1, 0x9c, 0x1c, 0xec, 0x39,
     0x2c, 0xaf, 0xcd, 0x25, 0x45, 0xa6, 0x56, 0x70, 0x9b, 0x31, 0x88, 0x66,
     0x01, 0x09, 0x7c, 0x61, 0xfb, 0xca, 0x05, 0x93, 0x7a, 0x16, 0x2a, 0xdb,
     0xbd, 0xf2, 0xe1, 0x46, 0x01, 0x5e, 0x8e, 0x82, 0x68, 0x8f, 0xf1, 0xd8,
     0x9a, 0x30, 0xda, 0xa3, 0xf1, 0x9a, 0x67, 0xd8, 0x59, 0xe8, 0x4c, 0x89,
     0x70, 0x97, 0xdc, 0x6d, 0x83, 0x1a, 0x01, 0xe8, 0x00, 0x74, 0xfd, 0x44,
     0x73, 0x0f, 0x86, 0x72, 0x03, 0x71, 0x48, 0x90, 0x01, 0x12, 0xd3, 0xd1,
     0x85, 0x2e, 0x67, 0xf1, 0x07, 0x78, 0x8d, 0x9f, 0xc8, 0x44, 0xd1, 0xa3,
     0x41, 0x9f, 0xd7, 0xc6, 0x50, 0x44, 0x9d, 0x09, 0xf4, 0xab, 0x7c, 0x12,
     0x65, 0x59, 0xc4, 0x43, 0x51, 0x94, 0x25, 0x9d, 0x0a, 0xb1, 0x1b, 0xd2,
     0x3d, 0xeb, 0xd4, 0xef, 0xbf, 0xd7, 0x66, 0xa3, 0x51, 0x67, 0xc1, 0x42,
     0x88, 0xe2, 0x7a, 0x28, 0x9a, 0x40, 0x1a, 0x60, 0xce, 0xa2, 0x42, 0x72,
     0x45, 0x8a, 0x4c, 0x08, 0xcc, 0x06, 0xa0, 0x6d, 0xab, 0xa3, 0xea, 0xee,
     0x63, 0x65, 0x9e, 0x40, 0x19, 0x02, 0x03, 0x01, 0x00, 0x01});

PrivateKey FixedRsa2048PrivateKeyForTesting() {
  return *PrivateKey::FromPrivateKeyInfo(kRsa2048Private);
}

PublicKey FixedRsa2048PublicKeyForTesting() {
  return *PublicKey::FromSubjectPublicKeyInfo(kRsa2048Public);
}

PrivateKey FixedRsa4096PrivateKeyForTesting() {
  return *PrivateKey::FromPrivateKeyInfo(kRsa4096Private);
}

PublicKey FixedRsa4096PublicKeyForTesting() {
  return *PublicKey::FromSubjectPublicKeyInfo(kRsa4096Public);
}

}  // namespace crypto::test
