/******************************************************************************
 *                                                                            *
 * Copyright (C) 2021 by nekohasekai <contact-sagernet@sekai.icu>             *
 *                                                                            *
 * This program is free software: you can redistribute it and/or modify       *
 * it under the terms of the GNU General Public License as published by       *
 * the Free Software Foundation, either version 3 of the License, or          *
 *  (at your option) any later version.                                       *
 *                                                                            *
 * This program is distributed in the hope that it will be useful,            *
 * but WITHOUT ANY WARRANTY; without even the implied warranty of             *
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the              *
 * GNU General Public License for more details.                               *
 *                                                                            *
 * You should have received a copy of the GNU General Public License          *
 * along with this program. If not, see <http://www.gnu.org/licenses/>.       *
 *                                                                            *
 ******************************************************************************/

package io.nekohasekai.sagernet.fmt

import android.Manifest
import android.content.pm.PackageManager
import android.os.Build
import android.provider.Settings
import com.github.shadowsocks.plugin.PluginConfiguration
import com.github.shadowsocks.plugin.PluginManager
import com.google.gson.JsonArray
import com.google.gson.JsonSyntaxException
import io.nekohasekai.sagernet.Key
import io.nekohasekai.sagernet.LogLevel
import io.nekohasekai.sagernet.RouteMode
import io.nekohasekai.sagernet.SagerNet
import io.nekohasekai.sagernet.Shadowsocks2022Implementation
import io.nekohasekai.sagernet.TLS_FRAGMENTATION_METHOD
import io.nekohasekai.sagernet.TunImplementation
import io.nekohasekai.sagernet.bg.VpnService
import io.nekohasekai.sagernet.database.DataStore
import io.nekohasekai.sagernet.database.ProxyEntity
import io.nekohasekai.sagernet.database.SagerDatabase
import io.nekohasekai.sagernet.fmt.V2rayBuildResult.IndexEntity
import io.nekohasekai.sagernet.fmt.anytls.AnyTLSBean
import io.nekohasekai.sagernet.fmt.gson.gson
import io.nekohasekai.sagernet.fmt.http.HttpBean
import io.nekohasekai.sagernet.fmt.http3.Http3Bean
import io.nekohasekai.sagernet.fmt.hysteria2.Hysteria2Bean
import io.nekohasekai.sagernet.fmt.internal.BalancerBean
import io.nekohasekai.sagernet.fmt.internal.ChainBean
import io.nekohasekai.sagernet.fmt.internal.ConfigBean
import io.nekohasekai.sagernet.fmt.juicity.JuicityBean
import io.nekohasekai.sagernet.fmt.mieru.MieruBean
import io.nekohasekai.sagernet.fmt.shadowsocks.ShadowsocksBean
import io.nekohasekai.sagernet.fmt.shadowsocksr.ShadowsocksRBean
import io.nekohasekai.sagernet.fmt.shadowtls.ShadowTLSBean
import io.nekohasekai.sagernet.fmt.socks.SOCKSBean
import io.nekohasekai.sagernet.fmt.ssh.SSHBean
import io.nekohasekai.sagernet.fmt.trojan.TrojanBean
import io.nekohasekai.sagernet.fmt.tuic5.Tuic5Bean
import io.nekohasekai.sagernet.fmt.v2ray.StandardV2RayBean
import io.nekohasekai.sagernet.fmt.v2ray.V2RayConfig
import io.nekohasekai.sagernet.fmt.v2ray.V2RayConfig.BrowserDialerObject
import io.nekohasekai.sagernet.fmt.v2ray.V2RayConfig.BrowserForwarderObject
import io.nekohasekai.sagernet.fmt.v2ray.V2RayConfig.DNSOutboundConfigurationObject
import io.nekohasekai.sagernet.fmt.v2ray.V2RayConfig.DnsObject
import io.nekohasekai.sagernet.fmt.v2ray.V2RayConfig.DokodemoDoorInboundConfigurationObject
import io.nekohasekai.sagernet.fmt.v2ray.V2RayConfig.FakeDnsObject
import io.nekohasekai.sagernet.fmt.v2ray.V2RayConfig.FreedomOutboundConfigurationObject
import io.nekohasekai.sagernet.fmt.v2ray.V2RayConfig.GrpcObject
import io.nekohasekai.sagernet.fmt.v2ray.V2RayConfig.HTTPInboundConfigurationObject
import io.nekohasekai.sagernet.fmt.v2ray.V2RayConfig.HTTPOutboundConfigurationObject
import io.nekohasekai.sagernet.fmt.v2ray.V2RayConfig.HTTPUpgradeObject
import io.nekohasekai.sagernet.fmt.v2ray.V2RayConfig.HttpObject
import io.nekohasekai.sagernet.fmt.v2ray.V2RayConfig.Hysteria2Object
import io.nekohasekai.sagernet.fmt.v2ray.V2RayConfig.Hysteria2OutboundConfigurationObject
import io.nekohasekai.sagernet.fmt.v2ray.V2RayConfig.InboundObject
import io.nekohasekai.sagernet.fmt.v2ray.V2RayConfig.KcpObject
import io.nekohasekai.sagernet.fmt.v2ray.V2RayConfig.LazyInboundConfigurationObject
import io.nekohasekai.sagernet.fmt.v2ray.V2RayConfig.LazyOutboundConfigurationObject
import io.nekohasekai.sagernet.fmt.v2ray.V2RayConfig.LogObject
import io.nekohasekai.sagernet.fmt.v2ray.V2RayConfig.MeekObject
import io.nekohasekai.sagernet.fmt.v2ray.V2RayConfig.MekyaObject
import io.nekohasekai.sagernet.fmt.v2ray.V2RayConfig.MultiObservatoryObject
import io.nekohasekai.sagernet.fmt.v2ray.V2RayConfig.ObservatoryObject
import io.nekohasekai.sagernet.fmt.v2ray.V2RayConfig.OutboundObject
import io.nekohasekai.sagernet.fmt.v2ray.V2RayConfig.PolicyObject
import io.nekohasekai.sagernet.fmt.v2ray.V2RayConfig.QuicObject
import io.nekohasekai.sagernet.fmt.v2ray.V2RayConfig.RealityObject
import io.nekohasekai.sagernet.fmt.v2ray.V2RayConfig.ReverseObject
import io.nekohasekai.sagernet.fmt.v2ray.V2RayConfig.RoutingObject
import io.nekohasekai.sagernet.fmt.v2ray.V2RayConfig.RoutingObject.BalancerObject.StrategyObject
import io.nekohasekai.sagernet.fmt.v2ray.V2RayConfig.SSHOutbountConfigurationObject
import io.nekohasekai.sagernet.fmt.v2ray.V2RayConfig.ShadowsocksOutboundConfigurationObject
import io.nekohasekai.sagernet.fmt.v2ray.V2RayConfig.Shadowsocks_2022OutboundConfigurationObject
import io.nekohasekai.sagernet.fmt.v2ray.V2RayConfig.SocksInboundConfigurationObject
import io.nekohasekai.sagernet.fmt.v2ray.V2RayConfig.SocksOutboundConfigurationObject
import io.nekohasekai.sagernet.fmt.v2ray.V2RayConfig.SplitHTTPObject
import io.nekohasekai.sagernet.fmt.v2ray.V2RayConfig.StreamSettingsObject
import io.nekohasekai.sagernet.fmt.v2ray.V2RayConfig.TLSObject
import io.nekohasekai.sagernet.fmt.v2ray.V2RayConfig.TcpObject
import io.nekohasekai.sagernet.fmt.v2ray.V2RayConfig.TrojanOutboundConfigurationObject
import io.nekohasekai.sagernet.fmt.v2ray.V2RayConfig.VLESSOutboundConfigurationObject
import io.nekohasekai.sagernet.fmt.v2ray.V2RayConfig.VMessOutboundConfigurationObject
import io.nekohasekai.sagernet.fmt.v2ray.V2RayConfig.WebSocketObject
import io.nekohasekai.sagernet.fmt.v2ray.V2RayConfig.WireGuardOutboundConfigurationObject
import io.nekohasekai.sagernet.fmt.v2ray.VLESSBean
import io.nekohasekai.sagernet.fmt.v2ray.VMessBean
import io.nekohasekai.sagernet.fmt.wireguard.WireGuardBean
import io.nekohasekai.sagernet.ktx.app
import io.nekohasekai.sagernet.ktx.getArray
import io.nekohasekai.sagernet.ktx.getBoolean
import io.nekohasekai.sagernet.ktx.getEnabled
import io.nekohasekai.sagernet.ktx.getInt
import io.nekohasekai.sagernet.ktx.getObject
import io.nekohasekai.sagernet.ktx.getString
import io.nekohasekai.sagernet.ktx.getStringValue
import io.nekohasekai.sagernet.ktx.isValidHysteriaMultiPort
import io.nekohasekai.sagernet.ktx.joinHostPort
import io.nekohasekai.sagernet.ktx.listByLine
import io.nekohasekai.sagernet.ktx.listByLineOrComma
import io.nekohasekai.sagernet.ktx.mkPort
import io.nekohasekai.sagernet.ktx.parseJson
import io.nekohasekai.sagernet.ktx.toHysteriaPort
import io.nekohasekai.sagernet.ktx.unescapeLineFeed
import io.nekohasekai.sagernet.ktx.uuidOrGenerate
import io.nekohasekai.sagernet.utils.PackageCache
import kotlin.io.encoding.Base64
import libcore.Libcore

const val TAG_SOCKS = "socks"
const val TAG_HTTP = "http"
const val TAG_TRANS = "trans"
const val TAG_TRANS6 = "trans6"

const val TAG_AGENT = "proxy"
const val TAG_DIRECT = "direct"
const val TAG_BYPASS = "bypass"
const val TAG_BLOCK = "block"

const val TAG_DNS_IN = "dns-in"
const val TAG_DNS_OUT = "dns-out"

const val TAG_DNS_DIRECT = "dns-direct"

const val LOCALHOST = "127.0.0.1"
const val LOCALHOST6 = "::1"

class V2rayBuildResult(
    var config: String,
    var index: List<IndexEntity>,
    var requireWs: Boolean,
    var wsPort: Int,
    var requireSh: Boolean,
    var shPort: Int,
    var outboundTags: List<String>,
    var outboundTagsCurrent: List<String>,
    var outboundTagsAll: Map<String, ProxyEntity>,
    var bypassTag: String,
    var observerTag: String,
    var observatoryTags: Set<String>,
    val dumpUID: Boolean,
    val alerts: List<Pair<Int, String>>,
) {
    data class IndexEntity(var isBalancer: Boolean, var chain: LinkedHashMap<Int, ProxyEntity>)
}

fun buildV2RayConfig(
    proxy: ProxyEntity, forTest: Boolean = false, forExport: Boolean = false
): V2rayBuildResult {

    val outboundTags = ArrayList<String>()
    val outboundTagsCurrent = ArrayList<String>()
    val outboundTagsAll = HashMap<String, ProxyEntity>()
    val globalOutbounds = ArrayList<String>()

    fun ProxyEntity.resolveChainRecursively(): MutableList<ProxyEntity> {
        val bean = requireBean()
        if (bean is BalancerBean) {
            error("Chain is incompatible with balancer")
        }
        if (bean is ChainBean) {
            val beans = SagerDatabase.proxyDao.getEntities(bean.proxies)
            val beansMap = beans.associateBy { it.id }
            val beanList = ArrayList<ProxyEntity>()
            for ((index, proxyId) in bean.proxies.withIndex()) {
                val item = beansMap[proxyId] ?: continue
                if (!item.requireBean().canMapping() && index != 0) {
                    error("Configuration ${item.displayName()} can be the front proxy only.")
                }
                if (item.type == ProxyEntity.TYPE_WG && index != bean.proxies.size - 1) {
                    error("Configuration ${item.displayName()} can be the landing proxy only.")
                }
                beanList.addAll(item.resolveChainRecursively())
            }
            return beanList
        }
        return mutableListOf(this)
    }

    fun ProxyEntity.resolveChain(): MutableList<ProxyEntity> {
        val bean = requireBean()
        if (bean is BalancerBean) {
            val beans = if (bean.type == BalancerBean.TYPE_LIST) {
                SagerDatabase.proxyDao.getEntities(bean.proxies)
            } else {
                SagerDatabase.proxyDao.getByGroup(bean.groupId).filter {
                    if (bean.nameFilter.isEmpty()) {
                        true
                    } else {
                        !Regex(bean.nameFilter).containsMatchIn(it.requireBean().name)
                    }
                }
            }
            val beansMap = beans.associateBy { it.id }
            val beanList = ArrayList<ProxyEntity>()
            for (proxyId in beansMap.keys) {
                val item = beansMap[proxyId] ?: continue
                if (item.id == id) continue
                when (item.type) {
                    ProxyEntity.TYPE_BALANCER -> error("Nested balancers are not supported")
                    ProxyEntity.TYPE_CHAIN -> error("Chain is incompatible with balancer")
                }
                beanList.add(item)
            }
            return beanList
        }
        val list = resolveChainRecursively().asReversed()
        SagerDatabase.groupDao.getById(groupId)?.let { group ->
            group.frontProxy.takeIf { it > 0L }?.let { id ->
                SagerDatabase.proxyDao.getById(id)?.let {
                    when (it.requireBean()) {
                        is BalancerBean -> error("Front proxy is incompatible with balancer")
                        is ChainBean -> error("Front proxy is incompatible with chain")
                        else -> list.add(it)
                    }
                } ?: error("front proxy set but not found for group ${group.displayName()}")
            }
            group.landingProxy.takeIf { it > 0L }?.let { id ->
                SagerDatabase.proxyDao.getById(id)?.let {
                    when (it.requireBean()) {
                        is BalancerBean -> error("Landing proxy is incompatible with balancer")
                        is ChainBean -> error("Landing proxy is incompatible with chain")
                        else -> list.add(0, it)
                    }
                } ?: error("landing proxy set but not found for group ${group.displayName()}")
            }
        }
        return list
    }

    val routeMode = DataStore.routeMode
    val proxies = proxy.resolveChain()
    val extraRules = if (forTest || routeMode != RouteMode.RULE) listOf() else SagerDatabase.rulesDao.enabledRules()
    val extraProxies = if (forTest) mapOf() else SagerDatabase.proxyDao.getEntities(extraRules.mapNotNull { rule ->
        rule.outbound.takeIf { it > 0 && it != proxy.id }
    }.toHashSet().toList()).associate {
        (it.id to ((it.type == ProxyEntity.TYPE_BALANCER) to lazy {
            it.balancerBean
        })) to it.resolveChain()
    }

    val allowAccess = DataStore.allowAccess
    val bind = if (!forTest && allowAccess) "0.0.0.0" else LOCALHOST

    var directDNS = DataStore.directDns.listByLineOrComma().filter { !it.startsWith("#") }
    if (DataStore.useLocalDnsAsDirectDns) directDNS = listOf("localhost")
    val remoteDNS = if (routeMode == RouteMode.DIRECT) {
        directDNS
    } else {
        DataStore.remoteDns.listByLineOrComma().filter { !it.startsWith("#") }
    }
    var bootstrapDNS = DataStore.bootstrapDns.listByLineOrComma().filter { !it.startsWith("#") }
    if (DataStore.useLocalDnsAsBootstrapDns) bootstrapDNS = listOf("localhost")
    val useFakeDns = DataStore.enableFakeDns
    val remoteDnsQueryStrategy = DataStore.remoteDnsQueryStrategy
    val directDnsQueryStrategy = DataStore.directDnsQueryStrategy
    val trafficSniffing = DataStore.trafficSniffing
    val indexMap = ArrayList<IndexEntity>()
    var requireWs = false
    var requireSh = false
    val requireHttp = !forTest && DataStore.requireHttp
    val requireTransproxy = if (forTest) false else DataStore.requireTransproxy
    val destinationOverride = DataStore.destinationOverride
    val trafficStatistics = !forTest && DataStore.profileTrafficStatistics

    val shouldDumpUID = extraRules.any { it.packages.isNotEmpty() }
    val alerts = mutableListOf<Pair<Int, String>>()

    lateinit var result: V2rayBuildResult
    V2RayConfig().apply {

        dns = DnsObject().apply {
            if (DataStore.hosts.isNotEmpty()) {
                hosts = mutableMapOf()
                for (singleLine in DataStore.hosts.listByLine()) {
                    val key = singleLine.substringBefore(" ")
                    val values = singleLine.substringAfter(" ").split("\\s+".toRegex()).toMutableList()
                    if (hosts.contains(key)) {
                        if (!hosts[key]!!.valueX.isNullOrEmpty()) {
                            values.add(hosts[key]!!.valueX)
                        } else if (!hosts[key]!!.valueY.isNullOrEmpty()) {
                            values.addAll(hosts[key]!!.valueY)
                        }
                    }
                    if (values.size > 1) {
                        hosts[key] = DnsObject.StringOrListObject().apply {
                            valueX = null
                            valueY = values
                        }
                    } else if (values.size == 1) {
                        hosts[key] = DnsObject.StringOrListObject().apply {
                            valueX = values[0]
                            valueY = null
                        }
                    }
                }
            }
            servers = mutableListOf()
            fallbackStrategy = "disabledIfAnyMatch"
        }

        log = LogObject().apply {
            loglevel = when (DataStore.logLevel) {
                LogLevel.DEBUG -> "debug"
                LogLevel.INFO -> "info"
                LogLevel.WARNING -> "warning"
                LogLevel.ERROR -> "error"
                else -> "none"
            }
            if (DataStore.logLevel == LogLevel.NONE) {
                access = "none"
            }
        }

        policy = PolicyObject().apply {
            levels = mapOf(
                // dns
                "1" to PolicyObject.LevelPolicyObject().apply {
                    connIdle = 30
                })

            if (trafficStatistics) {
                system = PolicyObject.SystemPolicyObject().apply {
                    statsOutboundDownlink = true
                    statsOutboundUplink = true
                }
            }
        }
        inbounds = mutableListOf()

        if (!forTest) inbounds.add(InboundObject().apply {
            tag = TAG_SOCKS
            listen = bind
            port = DataStore.socksPort
            protocol = "socks"
            settings = LazyInboundConfigurationObject(this,
                SocksInboundConfigurationObject().apply {
                    auth = "noauth"
                    udp = true
                })
            if (trafficSniffing || useFakeDns) {
                sniffing = InboundObject.SniffingObject().apply {
                    enabled = true
                    destOverride = when {
                        useFakeDns && !trafficSniffing -> listOf("fakedns")
                        useFakeDns -> listOf("fakedns", "http", "tls", "quic")
                        else -> listOf("http", "tls", "quic")
                    }
                    metadataOnly = useFakeDns && !trafficSniffing
                    routeOnly = !destinationOverride
                }
            }
            if (shouldDumpUID) dumpUID = true
        })

        if (requireHttp) {
            inbounds.add(InboundObject().apply {
                tag = TAG_HTTP
                listen = bind
                port = DataStore.httpPort
                protocol = "http"
                settings = LazyInboundConfigurationObject(this,
                    HTTPInboundConfigurationObject().apply {
                        allowTransparent = true
                    })
                if (trafficSniffing || useFakeDns) {
                    sniffing = InboundObject.SniffingObject().apply {
                        enabled = true
                        destOverride = when {
                            useFakeDns && !trafficSniffing -> listOf("fakedns")
                            useFakeDns -> listOf("fakedns", "http", "tls")
                            else -> listOf("http", "tls")
                        }
                        metadataOnly = useFakeDns && !trafficSniffing
                        routeOnly = !destinationOverride
                    }
                }
                if (shouldDumpUID) dumpUID = true
            })
        }

        if (requireTransproxy) {
            inbounds.add(InboundObject().apply {
                tag = TAG_TRANS
                listen = bind
                port = DataStore.transproxyPort
                protocol = "dokodemo-door"
                settings = LazyInboundConfigurationObject(this,
                    DokodemoDoorInboundConfigurationObject().apply {
                        // network = "tcp,udp"
                        network = "tcp"
                        followRedirect = true
                    })
                if (trafficSniffing || useFakeDns) {
                    sniffing = InboundObject.SniffingObject().apply {
                        enabled = true
                        destOverride = when {
                            useFakeDns && !trafficSniffing -> listOf("fakedns")
                            // useFakeDns -> listOf("fakedns", "http", "tls", "quic")
                            useFakeDns -> listOf("fakedns", "http", "tls")
                            // else -> listOf("http", "tls", "quic")
                            else -> listOf("http", "tls")
                        }
                        metadataOnly = useFakeDns && !trafficSniffing
                        routeOnly = !destinationOverride
                    }
                }
                /*when (DataStore.transproxyMode) {
                    1 -> streamSettings = StreamSettingsObject().apply {
                        sockopt = StreamSettingsObject.SockoptObject().apply {
                            tproxy = "tproxy"
                        }
                    }
                }*/
                if (shouldDumpUID) dumpUID = true
            })
            if (bind == LOCALHOST) {
                inbounds.add(InboundObject().apply {
                    tag = TAG_TRANS6
                    listen = LOCALHOST6
                    port = DataStore.transproxyPort
                    protocol = "dokodemo-door"
                    settings = LazyInboundConfigurationObject(this,
                        DokodemoDoorInboundConfigurationObject().apply {
                            // network = "tcp,udp"
                            network = "tcp"
                            followRedirect = true
                        })
                    if (trafficSniffing || useFakeDns) {
                        sniffing = InboundObject.SniffingObject().apply {
                            enabled = true
                            destOverride = when {
                                useFakeDns && !trafficSniffing -> listOf("fakedns")
                                // useFakeDns -> listOf("fakedns", "http", "tls", "quic")
                                useFakeDns -> listOf("fakedns", "http", "tls")
                                // else -> listOf("http", "tls", "quic")
                                else -> listOf("http", "tls")
                            }
                            metadataOnly = useFakeDns && !trafficSniffing
                            routeOnly = !destinationOverride
                        }
                        /*when (DataStore.transproxyMode) {
                            1 -> streamSettings = StreamSettingsObject().apply {
                                sockopt = StreamSettingsObject.SockoptObject().apply {
                                    tproxy = "tproxy"
                                }
                            }
                        }*/
                    }
                    if (shouldDumpUID) dumpUID = true
                })
            }
        }

        outbounds = mutableListOf()

        routing = RoutingObject().apply {
            domainStrategy = DataStore.domainStrategy

            rules = mutableListOf()

            val wsRules = HashMap<String, RoutingObject.RuleObject>()

            for (proxyEntity in proxies) {
                val bean = proxyEntity.requireBean()

                val needBrowserForwarder = when {
                    bean !is StandardV2RayBean -> false
                    bean.type == "ws" && bean.wsUseBrowserForwarder -> true
                    bean.type == "splithttp" && bean.shUseBrowserForwarder -> true
                    else -> false
                }
                if (needBrowserForwarder) {
                    bean as StandardV2RayBean
                    // dirty hack to exclude browser forwarder traffic from VpnService
                    // this will not work on all cases,
                    // but this is not the main function of this software, just keep it broken
                    if (bean.security == "none" && bean.host.isNotEmpty()) {
                        val host = try {
                            Libcore.parseURL("placeholder://" + bean.host).host
                        } catch (_: Exception) {
                            bean.host
                        }
                        wsRules[host] = RoutingObject.RuleObject().apply {
                            type = "field"
                            outboundTag = TAG_DIRECT
                            port = bean.serverPort.toString()
                            if (Libcore.isIP(host)) {
                                ip = listOf(host)
                                if (DataStore.domainStrategy != "AsIs") {
                                    skipDomain = true
                                }
                            } else {
                                domains = listOf(host)
                            }
                        }
                    }
                    if (bean.security != "none" && bean.sni.isNotEmpty()) {
                        wsRules[bean.sni] = RoutingObject.RuleObject().apply {
                            type = "field"
                            outboundTag = TAG_DIRECT
                            port = bean.serverPort.toString()
                            if (!Libcore.isIP(bean.sni)) {
                                domains = listOf(bean.sni)
                            }
                        }
                    }
                    if (bean.serverAddress.isNotEmpty()) {
                        wsRules[bean.serverAddress] = RoutingObject.RuleObject().apply {
                            type = "field"
                            outboundTag = TAG_DIRECT
                            port = bean.serverPort.toString()
                            if (Libcore.isIP(bean.serverAddress)) {
                                ip = listOf(bean.serverAddress)
                                if (DataStore.domainStrategy != "AsIs") {
                                    skipDomain = true
                                }
                            } else {
                                domains = listOf(bean.serverAddress)
                            }
                        }
                    }
                }

            }

            rules.addAll(wsRules.values)
        }

        var rootBalancer: RoutingObject.RuleObject? = null
        var rootObserver: MultiObservatoryObject.MultiObservatoryItem? = null

        fun buildChain(
            tagOutbound: String,
            profileList: List<ProxyEntity>,
            isBalancer: Boolean,
            balancer: () -> BalancerBean?,
        ): String {
            var pastExternal = false
            lateinit var pastOutbound: OutboundObject
            lateinit var currentOutbound: OutboundObject
            lateinit var pastInboundTag: String
            val chainMap = LinkedHashMap<Int, ProxyEntity>()
            indexMap.add(IndexEntity(isBalancer, chainMap))
            val chainOutbounds = ArrayList<OutboundObject>()
            var chainOutbound = ""

            profileList.forEachIndexed { index, proxyEntity ->
                val bean = proxyEntity.requireBean()
                currentOutbound = OutboundObject()

                val tagIn: String
                var needGlobal: Boolean

                if (isBalancer || index == profileList.lastIndex && !pastExternal) {
                    tagIn = "$TAG_AGENT-global-${proxyEntity.id}"
                    needGlobal = true
                } else {
                    tagIn = if (index == 0) tagOutbound else {
                        "$tagOutbound-${proxyEntity.id}"
                    }
                    needGlobal = false
                }

                if (index == 0) {
                    chainOutbound = tagIn
                }

                if (needGlobal) {
                    if (!globalOutbounds.contains(tagIn)) {
                        needGlobal = false
                        globalOutbounds.add(tagIn)
                    }
                }

                if (!needGlobal) {

                    outboundTagsAll[tagIn] = proxyEntity

                    if (isBalancer || index == 0) {
                        outboundTags.add(tagIn)
                        if (tagOutbound == TAG_AGENT) {
                            outboundTagsCurrent.add(tagIn)
                        }
                    }

                    var currentDomainStrategy = DataStore.outboundDomainStrategy

                    if (proxyEntity.needExternal()) {
                        val localPort = mkPort()
                        chainMap[localPort] = proxyEntity
                        currentOutbound.apply {
                            protocol = "socks"
                            settings = LazyOutboundConfigurationObject(this,
                                SocksOutboundConfigurationObject().apply {
                                    servers = listOf(SocksOutboundConfigurationObject.ServerObject()
                                        .apply {
                                            address = LOCALHOST
                                            port = localPort
                                        })
                                    if (getEnabled(DataStore.experimentalFlags, "singuot")) {
                                        proxyEntity.naiveBean?.singUoT?.takeIf { it }?.let {
                                            uot = true
                                        }
                                    }
                                })
                        }
                    } else {
                        currentOutbound.apply {
                            if (bean is StandardV2RayBean) {
                                if (bean is VMessBean) {
                                    protocol = "vmess"
                                    settings = LazyOutboundConfigurationObject(this,
                                        VMessOutboundConfigurationObject().apply {
                                            vnext = listOf(VMessOutboundConfigurationObject.ServerObject()
                                                .apply {
                                                    address = bean.serverAddress
                                                    port = bean.serverPort
                                                    users = listOf(VMessOutboundConfigurationObject.ServerObject.UserObject()
                                                        .apply {
                                                            id = uuidOrGenerate(bean.uuid)
                                                            if (bean.alterId > 0) {
                                                                alterId = bean.alterId
                                                            }
                                                            security = bean.encryption.takeIf { it.isNotEmpty() }
                                                                ?: "auto"
                                                            experiments = ""
                                                            if (bean.experimentalAuthenticatedLength) {
                                                                experiments += "AuthenticatedLength"
                                                            }
                                                            if (bean.experimentalNoTerminationSignal) {
                                                                if (experiments != "") {
                                                                    experiments += "|"
                                                                }
                                                                experiments += "NoTerminationSignal"
                                                            }
                                                            if (experiments.isEmpty()) experiments = null
                                                        })
                                                })
                                            when (bean.packetEncoding) {
                                                "packet" -> {
                                                    packetEncoding = "packet"
                                                }
                                                "xudp" -> {
                                                    packetEncoding = "xudp"
                                                }
                                            }
                                        })
                                } else if (bean is VLESSBean) {
                                    protocol = "vless"
                                    settings = LazyOutboundConfigurationObject(this,
                                        VLESSOutboundConfigurationObject().apply {
                                            vnext = listOf(VLESSOutboundConfigurationObject.ServerObject()
                                                .apply {
                                                    address = bean.serverAddress
                                                    port = bean.serverPort
                                                    users = listOf(VLESSOutboundConfigurationObject.ServerObject.UserObject()
                                                        .apply {
                                                            id = uuidOrGenerate(bean.uuid)
                                                            encryption = bean.encryption
                                                            if (bean.flow.isNotEmpty()) {
                                                                flow = bean.flow
                                                            }
                                                        })
                                                })
                                            when (bean.packetEncoding) {
                                                "packet" -> {
                                                    packetEncoding = "packet"
                                                }
                                                "xudp" -> {
                                                    packetEncoding = "xudp"
                                                }
                                            }
                                        })
                                } else if (bean is TrojanBean) {
                                    protocol = "trojan"
                                    settings = LazyOutboundConfigurationObject(this,
                                        TrojanOutboundConfigurationObject().apply {
                                            servers = listOf(TrojanOutboundConfigurationObject.ServerObject()
                                                .apply {
                                                    address = bean.serverAddress
                                                    port = bean.serverPort
                                                    password = bean.password
                                                })
                                        })
                                } else if (bean is ShadowsocksBean) {
                                    if (bean.method.startsWith("2022-blake3-") && DataStore.shadowsocks2022Implementation == Shadowsocks2022Implementation.V2FLY_V2RAY_CORE) {
                                        protocol = "shadowsocks2022"
                                        settings = LazyOutboundConfigurationObject(this,
                                            Shadowsocks_2022OutboundConfigurationObject().apply {
                                                address = bean.serverAddress
                                                port = bean.serverPort
                                                method = bean.method
                                                val keys = bean.password.split(":")
                                                if (keys.size == 1) {
                                                    psk = keys[0]
                                                }
                                                if (keys.size > 1) {
                                                    ipsk = mutableListOf()
                                                    for (i in 0..(keys.size - 2)) {
                                                        ipsk.add(keys[i])
                                                    }
                                                    psk = keys[keys.size - 1]
                                                }
                                                if (bean.plugin.isNotEmpty()) {
                                                    val pluginConfiguration = PluginConfiguration(bean.plugin)
                                                    if (forExport) {
                                                        plugin = pluginConfiguration.selected
                                                        pluginOpts = pluginConfiguration.getOptions().toString()
                                                    } else {
                                                        try {
                                                            PluginManager.init(pluginConfiguration)?.let { (path, opts, isV2) ->
                                                                plugin = path
                                                                if (!forTest && DataStore.serviceMode == Key.MODE_VPN && DataStore.tunImplementation == TunImplementation.SYSTEM) {
                                                                    pluginWorkingDir = SagerNet.deviceStorage.noBackupFilesDir.toString()
                                                                    if (isV2) {
                                                                        opts["__android_vpn"] = ""
                                                                    } else {
                                                                        pluginArgs = listOf("-V")
                                                                    }
                                                                }
                                                                pluginOpts = opts.toString()
                                                            }
                                                        } catch (e: PluginManager.PluginNotFoundException) {
                                                            if (e.plugin in arrayOf("v2ray-plugin", "obfs-local")) {
                                                                plugin = e.plugin
                                                                pluginOpts = pluginConfiguration.getOptions().toString()
                                                            } else {
                                                                throw e
                                                            }
                                                        }
                                                    }
                                                }
                                                if (bean.singUoT && getEnabled(DataStore.experimentalFlags, "singuot")) {
                                                    uot = bean.singUoT
                                                }
                                            }
                                        )
                                    } else {
                                        protocol = "shadowsocks"
                                        settings = LazyOutboundConfigurationObject(this,
                                            ShadowsocksOutboundConfigurationObject().apply {
                                                servers = listOf(ShadowsocksOutboundConfigurationObject.ServerObject().apply {
                                                    address = bean.serverAddress
                                                    port = bean.serverPort
                                                    password = bean.password
                                                        method = bean.method
                                                    if (!bean.method.startsWith("2022-blake3-") && bean.experimentReducedIvHeadEntropy) {
                                                        experimentReducedIvHeadEntropy = bean.experimentReducedIvHeadEntropy
                                                    }
                                                    if (bean.plugin.isNotEmpty()) {
                                                        val pluginConfiguration = PluginConfiguration(bean.plugin)
                                                        if (forExport) {
                                                            plugin = pluginConfiguration.selected
                                                            pluginOpts = pluginConfiguration.getOptions().toString()
                                                        } else {
                                                            try {
                                                                PluginManager.init(pluginConfiguration)?.let { (path, opts, isV2) ->
                                                                    plugin = path
                                                                    if (!forTest && DataStore.serviceMode == Key.MODE_VPN && DataStore.tunImplementation == TunImplementation.SYSTEM) {
                                                                        pluginWorkingDir = SagerNet.deviceStorage.noBackupFilesDir.toString()
                                                                        if (isV2) {
                                                                            opts["__android_vpn"] = ""
                                                                        } else {
                                                                            pluginArgs = listOf("-V")
                                                                        }
                                                                    }
                                                                    pluginOpts = opts.toString()
                                                                }
                                                            } catch (e: PluginManager.PluginNotFoundException) {
                                                                if (e.plugin in arrayOf("v2ray-plugin", "obfs-local")) {
                                                                    plugin = e.plugin
                                                                    pluginOpts = pluginConfiguration.getOptions().toString()
                                                                } else {
                                                                    throw e
                                                                }
                                                            }
                                                        }
                                                    }
                                                    if (bean.singUoT && getEnabled(DataStore.experimentalFlags, "singuot")) {
                                                        uot = bean.singUoT
                                                    }
                                                })
                                            }
                                        )
                                    }
                                } else if (bean is SOCKSBean) {
                                    protocol = "socks"
                                    settings = LazyOutboundConfigurationObject(this,
                                        SocksOutboundConfigurationObject().apply {
                                            servers = listOf(SocksOutboundConfigurationObject.ServerObject().apply {
                                                address = bean.serverAddress
                                                port = bean.serverPort
                                                if (bean.username.isNotEmpty() || bean.password.isNotEmpty() && bean.protocol == SOCKSBean.PROTOCOL_SOCKS5) {
                                                    users = listOf(SocksOutboundConfigurationObject.ServerObject.UserObject().apply {
                                                        user = bean.username
                                                        pass = bean.password
                                                    })
                                                }
                                                if (bean.protocol == SOCKSBean.PROTOCOL_SOCKS4 || bean.protocol == SOCKSBean.PROTOCOL_SOCKS4A && bean.username.isNotEmpty()) {
                                                    users = listOf(SocksOutboundConfigurationObject.ServerObject.UserObject().apply {
                                                        user = bean.username
                                                    })
                                                }
                                            })
                                            version = bean.protocolVersionName()
                                            if (bean.singUoT && getEnabled(DataStore.experimentalFlags, "singuot")) {
                                                uot = bean.singUoT
                                            }
                                        }
                                    )
                                } else if (bean is HttpBean) {
                                    protocol = "http"
                                    settings = LazyOutboundConfigurationObject(this,
                                        HTTPOutboundConfigurationObject().apply {
                                            servers = listOf(HTTPOutboundConfigurationObject.ServerObject().apply {
                                                address = bean.serverAddress
                                                port = bean.serverPort
                                                if (bean.username.isNotEmpty() || bean.password.isNotEmpty()) {
                                                    users = listOf(HTTPInboundConfigurationObject.AccountObject().apply {
                                                        user = bean.username
                                                        pass = bean.password
                                                    })
                                                }
                                            })
                                        }
                                    )
                                }
                                streamSettings = StreamSettingsObject().apply {
                                    network = bean.type
                                    if (bean.security.isNotEmpty()) {
                                        security = bean.security
                                    }
                                    when (security) {
                                        "tls" -> {
                                            tlsSettings = TLSObject().apply {
                                                if (bean.sni.isNotEmpty()) {
                                                    serverName = bean.sni
                                                }
                                                if (bean.alpn.isNotEmpty()) {
                                                    alpn = bean.alpn.listByLineOrComma()
                                                }
                                                if (bean.certificates.isNotEmpty() || bean.mtlsCertificate.isNotEmpty() || bean.mtlsCertificatePrivateKey.isNotEmpty()) {
                                                    certificates = mutableListOf()
                                                    if (bean.certificates.isNotEmpty()) {
                                                        disableSystemRoot = true
                                                        certificates.add(TLSObject.CertificateObject().apply {
                                                            usage = "verify"
                                                            certificate = bean.certificates.lines()
                                                        })
                                                    }
                                                    if (bean.mtlsCertificate.isNotEmpty() || bean.mtlsCertificatePrivateKey.isNotEmpty()) {
                                                        certificates.add(TLSObject.CertificateObject().apply {
                                                            usage = "encipherment"
                                                            certificate = bean.mtlsCertificate.lines()
                                                            key = bean.mtlsCertificatePrivateKey.lines()
                                                        })
                                                    }
                                                }
                                                if (bean.pinnedPeerCertificateSha256.isNotEmpty()) {
                                                    pinnedPeerCertificateSha256 = mutableListOf()
                                                    bean.pinnedPeerCertificateSha256.listByLineOrComma().forEach {
                                                        pinnedPeerCertificateSha256.add(it.replace(":", ""))
                                                    }
                                                }
                                                if (bean.pinnedPeerCertificatePublicKeySha256.isNotEmpty()) {
                                                    pinnedPeerCertificatePublicKeySha256 = bean.pinnedPeerCertificatePublicKeySha256.listByLineOrComma()
                                                }
                                                if (bean.pinnedPeerCertificateChainSha256.isNotEmpty()) {
                                                    pinnedPeerCertificateChainSha256 = bean.pinnedPeerCertificateChainSha256.listByLineOrComma()
                                                }
                                                if (bean.allowInsecure) {
                                                    allowInsecure = true
                                                }
                                                if (bean.utlsFingerprint.isNotEmpty()) {
                                                    fingerprint = bean.utlsFingerprint
                                                }
                                                if (bean.echConfig.isNotEmpty()) {
                                                    echConfig = bean.echConfig
                                                }
                                            }
                                        }
                                        "reality" -> {
                                            realitySettings = RealityObject().apply {
                                                if (bean.sni.isNotEmpty()) {
                                                    serverName = bean.sni
                                                }
                                                if (bean.realityPublicKey.isNotEmpty()) {
                                                    publicKey = bean.realityPublicKey
                                                }
                                                if (bean.realityShortId.isNotEmpty()) {
                                                    shortId = bean.realityShortId
                                                }
                                                if (bean.realityFingerprint.isNotEmpty()) {
                                                    fingerprint = bean.realityFingerprint
                                                }
                                                if (DataStore.realityDisableX25519Mlkem768 || bean.realityDisableX25519Mlkem768 && !forExport) {
                                                    disableX25519MLKEM768 = true
                                                }
                                            }
                                        }
                                    }

                                    when (network) {
                                        "tcp" -> {
                                            tcpSettings = TcpObject().apply {
                                                if (bean.headerType == "http") {
                                                    header = TcpObject.HeaderObject().apply {
                                                        type = "http"
                                                        if (bean.host.isNotEmpty() || bean.path.isNotEmpty()) {
                                                            request = TcpObject.HeaderObject.HTTPRequestObject()
                                                                .apply {
                                                                    headers = mutableMapOf()
                                                                    if (bean.host.isNotEmpty()) {
                                                                        headers["Host"] = TcpObject.HeaderObject.StringOrListObject()
                                                                            .apply {
                                                                                valueY = bean.host.listByLineOrComma()
                                                                            }
                                                                    }
                                                                    if (bean.path.isNotEmpty()) {
                                                                        path = bean.path.listByLineOrComma()
                                                                    }
                                                                }
                                                        }
                                                    }
                                                }
                                            }
                                        }
                                        "kcp" -> {
                                            kcpSettings = KcpObject().apply {
                                                mtu = 1350
                                                tti = 50
                                                uplinkCapacity = 12
                                                downlinkCapacity = 100
                                                congestion = false
                                                readBufferSize = 1
                                                writeBufferSize = 1
                                                header = KcpObject.HeaderObject().apply {
                                                    type = bean.headerType
                                                }
                                                if (bean.mKcpSeed.isNotEmpty()) {
                                                    seed = bean.mKcpSeed
                                                }
                                            }
                                        }
                                        "ws" -> {
                                            wsSettings = WebSocketObject().apply {
                                                headers = mutableMapOf()

                                                if (bean.host.isNotEmpty()) {
                                                    headers["Host"] = bean.host
                                                }

                                                path = bean.path.takeIf { it.isNotEmpty() } ?: "/"

                                                if (bean.maxEarlyData > 0) {
                                                    maxEarlyData = bean.maxEarlyData
                                                }

                                                if (bean.earlyDataHeaderName.isNotEmpty()) {
                                                    earlyDataHeaderName = bean.earlyDataHeaderName
                                                }

                                                if (bean.wsUseBrowserForwarder) {
                                                    useBrowserForwarding = true
                                                    requireWs = true
                                                }
                                            }
                                        }
                                        "http" -> {
                                            network = "http"

                                            httpSettings = HttpObject().apply {
                                                if (bean.host.isNotEmpty()) {
                                                    host = bean.host.listByLineOrComma()
                                                }
                                                if (bean.path.isNotEmpty()) {
                                                    path = bean.path
                                                }
                                            }
                                        }
                                        "quic" -> {
                                            quicSettings = QuicObject().apply {
                                                security = bean.quicSecurity
                                                key = bean.quicKey
                                                header = QuicObject.HeaderObject().apply {
                                                    type = bean.headerType
                                                }
                                            }
                                        }
                                        "grpc" -> {
                                            grpcSettings = GrpcObject().apply {
                                                serviceName = bean.grpcServiceName
                                                if (DataStore.grpcServiceNameCompat || bean.grpcServiceNameCompat) {
                                                    serviceNameCompat = true
                                                }
                                                if (bean.grpcMultiMode) {
                                                    multiMode = true
                                                }
                                            }
                                        }
                                        "meek" -> {
                                            meekSettings = MeekObject().apply {
                                                if (bean.meekUrl.isNotEmpty()) {
                                                    url = bean.meekUrl
                                                }
                                            }
                                        }
                                        "httpupgrade" -> {
                                            httpupgradeSettings = HTTPUpgradeObject().apply {
                                                if (bean.host.isNotEmpty()) {
                                                    host = bean.host
                                                }
                                                if (bean.path.isNotEmpty()) {
                                                    path = bean.path
                                                }
                                                if (bean.maxEarlyData > 0) {
                                                    maxEarlyData = bean.maxEarlyData
                                                }
                                                if (bean.earlyDataHeaderName.isNotEmpty()) {
                                                    earlyDataHeaderName = bean.earlyDataHeaderName
                                                }
                                            }
                                        }
                                        "splithttp" -> {
                                            splithttpSettings = SplitHTTPObject().apply {
                                                if (bean.host.isNotEmpty()) {
                                                    host = bean.host
                                                }
                                                if (bean.path.isNotEmpty()) {
                                                    path = bean.path
                                                }
                                                if (bean.splithttpMode != "auto") {
                                                    mode = bean.splithttpMode
                                                }
                                                if (bean.splithttpExtra.isNotEmpty()) {
                                                    try {
                                                        parseJson(bean.splithttpExtra).asJsonObject?.also { extra ->
                                                            // fuck RPRX `extra`
                                                            extra.getInt("scMaxEachPostBytes", ignoreCase = true)?.also {
                                                                scMaxEachPostBytes = it.toString()
                                                            } ?: extra.getString("scMaxEachPostBytes", ignoreCase = true)?.also {
                                                                scMaxEachPostBytes = it
                                                            }
                                                            extra.getInt("scMinPostsIntervalMs", ignoreCase = true)?.also {
                                                                scMinPostsIntervalMs = it.toString()
                                                            } ?: extra.getString("scMinPostsIntervalMs", ignoreCase = true)?.also {
                                                                scMinPostsIntervalMs = it
                                                            }
                                                            extra.getInt("xPaddingBytes", ignoreCase = true)?.also {
                                                                xPaddingBytes = it.toString()
                                                            } ?: extra.getString("xPaddingBytes", ignoreCase = true)?.also {
                                                                xPaddingBytes = it
                                                            }
                                                            extra.getBoolean("noGRPCHeader", ignoreCase = true)?.also {
                                                                noGRPCHeader = it
                                                            }
                                                            extra.getObject("headers", ignoreCase = true)?.also {
                                                                headers = mutableMapOf<String, String>()
                                                                for (key in it.keySet()) {
                                                                    it.getString(key)?.also { value ->
                                                                        headers[key] = value
                                                                    }
                                                                }
                                                            }
                                                        }
                                                    } catch (e: Exception) {
                                                        error(e)
                                                    }
                                                }
                                                if (bean.shUseBrowserForwarder) {
                                                    useBrowserForwarding = true
                                                    requireSh = true
                                                }
                                            }
                                        }
                                        "hysteria2" -> {
                                            hy2Settings = Hysteria2Object().apply {
                                                if (bean.hy2Password.isNotEmpty()) {
                                                    password = bean.hy2Password
                                                }
                                                congestion = Hysteria2Object.CongestionObject().apply {
                                                    down_mbps = bean.hy2DownMbps
                                                    up_mbps = bean.hy2UpMbps
                                                }
                                            }
                                        }
                                        "mekya" -> {
                                            mekyaSettings = MekyaObject().apply {
                                                kcp = KcpObject().apply {
                                                    mtu = 1350
                                                    tti = 50
                                                    uplinkCapacity = 12
                                                    downlinkCapacity = 100
                                                    congestion = false
                                                    readBufferSize = 1
                                                    writeBufferSize = 1
                                                    header = KcpObject.HeaderObject().apply {
                                                        type = bean.mekyaKcpHeaderType
                                                    }
                                                    if (bean.mKcpSeed.isNotEmpty()) {
                                                        seed = bean.mekyaKcpSeed
                                                    }
                                                }
                                                if (bean.mekyaUrl.isNotEmpty()) {
                                                    url = bean.mekyaUrl
                                                }
                                                // magic values from https://github.com/v2fly/v2ray-core/pull/3120
                                                maxWriteDelay = 80
                                                maxRequestSize = 96000
                                                pollingIntervalInitial = 200
                                                h2PoolSize = 8
                                            }
                                        }
                                    }
                                    if (DataStore.enableFragment
                                        && (network != "kcp" && network != "quic" && network != "hysteria2")
                                        && (security == "tls" || security == "reality")
                                        && !(bean is ShadowsocksBean && bean.plugin.isNotEmpty() && PluginConfiguration(bean.plugin).selected.isNotEmpty())
                                        && !(network == "ws" && bean.wsUseBrowserForwarder)
                                        && !(network == "splithttp" && bean.shUseBrowserForwarder)
                                    ) {
                                        sockopt = StreamSettingsObject.SockoptObject().apply {
                                            tlsFragmentation = StreamSettingsObject.SockoptObject.TLSFragmentationObject().apply {
                                                when (DataStore.fragmentMethod) {
                                                    TLS_FRAGMENTATION_METHOD.TLS_RECORD_FRAGMENTATION -> {
                                                        tlsRecordFragmentation = true
                                                    }
                                                    TLS_FRAGMENTATION_METHOD.TCP_SEGMENTATION -> {
                                                        tcpSegmentation = true
                                                    }
                                                    TLS_FRAGMENTATION_METHOD.TLS_RECORD_FRAGMENTATION_AND_TCP_SEGMENTATION -> {
                                                        tlsRecordFragmentation = true
                                                        tcpSegmentation = true
                                                    }
                                                }
                                            }
                                        }
                                    }
                                }
                            } else if (bean is ShadowsocksRBean) {
                                protocol = "shadowsocks"
                                settings = LazyOutboundConfigurationObject(this,
                                    ShadowsocksOutboundConfigurationObject().apply {
                                        servers = listOf(ShadowsocksOutboundConfigurationObject.ServerObject().apply {
                                                address = bean.serverAddress
                                                port = bean.serverPort
                                                method = bean.method
                                                password = bean.password
                                            }
                                        )
                                        plugin = "shadowsocksr"
                                        pluginArgs = listOf(
                                            "--obfs=${bean.obfs}",
                                            "--obfs-param=${bean.obfsParam}",
                                            "--protocol=${bean.protocol}",
                                            "--protocol-param=${bean.protocolParam}"
                                        )
                                    }
                                )
                            } else if (bean is WireGuardBean) {
                                protocol = "wireguard"
                                settings = LazyOutboundConfigurationObject(this,
                                    WireGuardOutboundConfigurationObject().apply {
                                        address = bean.localAddress.listByLineOrComma()
                                        secretKey = bean.privateKey
                                        mtu = bean.mtu
                                        val values = bean.reserved.listByLineOrComma()
                                        if (values.size == 3) {
                                            val reserved0 = values[0].toUByteOrNull()
                                            val reserved1 = values[1].toUByteOrNull()
                                            val reserved2 = values[2].toUByteOrNull()
                                            if (reserved0 != null && reserved1 != null && reserved2 != null) {
                                                reserved = listOf(reserved0.toInt(), reserved1.toInt(), reserved2.toInt())
                                            }
                                        } else {
                                            val array = Base64.decode(bean.reserved)
                                            if (array.size == 3) {
                                                reserved = listOf(array[0].toUByte().toInt(), array[1].toUByte().toInt(), array[2].toUByte().toInt())
                                            }
                                        }
                                        peers = listOf(WireGuardOutboundConfigurationObject.WireGuardPeerObject().apply {
                                            publicKey = bean.peerPublicKey
                                            if (bean.peerPreSharedKey.isNotEmpty()) {
                                                preSharedKey = bean.peerPreSharedKey
                                            }
                                            if (bean.keepaliveInterval > 0) {
                                                keepAlive = bean.keepaliveInterval
                                            }
                                            endpoint = joinHostPort(bean.serverAddress, bean.serverPort)
                                        })
                                    })
                                if (currentDomainStrategy == "AsIs") {
                                    currentDomainStrategy = "UseIP"
                                }
                            } else if (bean is SSHBean) {
                                protocol = "ssh"
                                settings = LazyOutboundConfigurationObject(this,
                                    SSHOutbountConfigurationObject().apply {
                                        address = bean.finalAddress
                                        port = bean.finalPort
                                        user = bean.username
                                        when (bean.authType) {
                                            SSHBean.AUTH_TYPE_PUBLIC_KEY -> {
                                                privateKey = bean.privateKey
                                                if (bean.privateKeyPassphrase.isNotEmpty()) {
                                                    privateKeyPassphrase = bean.privateKeyPassphrase
                                                }
                                            }
                                            SSHBean.AUTH_TYPE_PASSWORD -> {
                                                password = bean.password
                                            }
                                        }
                                        if (bean.publicKey.isNotEmpty()) {
                                            publicKey = bean.publicKey
                                        }
                                    })
                            } else if (bean is Hysteria2Bean) {
                                protocol = "hysteria2"
                                settings = LazyOutboundConfigurationObject(this,
                                    Hysteria2OutboundConfigurationObject().apply {
                                        servers = listOf(Hysteria2OutboundConfigurationObject.ServerObject().apply {
                                            address = bean.serverAddress
                                            port = bean.serverPorts.toHysteriaPort()
                                        })
                                    }
                                )
                                streamSettings = StreamSettingsObject().apply {
                                    network = "hysteria2"
                                    security = "tls"
                                    hy2Settings = Hysteria2Object().apply {
                                        use_udp_extension = true
                                        if (bean.auth.isNotEmpty()) {
                                            password = bean.auth
                                        }
                                        congestion = Hysteria2Object.CongestionObject().apply {
                                            down_mbps = bean.downloadMbps
                                            up_mbps = bean.uploadMbps
                                        }
                                        if (bean.obfs.isNotEmpty()) {
                                            obfs = Hysteria2Object.OBFSObject().apply {
                                                type = "salamander"
                                                password = bean.obfs
                                            }
                                        }
                                        if (bean.serverPorts.isNotEmpty() && bean.serverPorts.isValidHysteriaMultiPort()) {
                                            hopPorts = bean.serverPorts
                                            if (bean.hopInterval > 0) {
                                                hopInterval = bean.hopInterval
                                            }
                                        }
                                    }
                                    tlsSettings = TLSObject().apply {
                                        if (bean.sni.isNotEmpty()) {
                                            serverName = bean.sni
                                        }
                                        if (bean.allowInsecure) {
                                            allowInsecure = true
                                        }
                                        if (bean.pinnedPeerCertificateSha256.isNotEmpty()) {
                                            pinnedPeerCertificateSha256 = mutableListOf()
                                            bean.pinnedPeerCertificateSha256.listByLineOrComma().forEach {
                                                pinnedPeerCertificateSha256.add(it.replace(":", "").replace("-", ""))
                                            }
                                        }
                                        if (bean.pinnedPeerCertificatePublicKeySha256.isNotEmpty()) {
                                            pinnedPeerCertificatePublicKeySha256 = bean.pinnedPeerCertificatePublicKeySha256.listByLineOrComma()
                                        }
                                        if (bean.pinnedPeerCertificateChainSha256.isNotEmpty()) {
                                            pinnedPeerCertificateChainSha256 = bean.pinnedPeerCertificateChainSha256.listByLineOrComma()
                                        }
                                        if (bean.certificates.isNotEmpty() || bean.mtlsCertificate.isNotEmpty() || bean.mtlsCertificatePrivateKey.isNotEmpty()) {
                                            certificates = mutableListOf()
                                            if (bean.certificates.isNotEmpty()) {
                                                disableSystemRoot = true
                                                certificates.add(TLSObject.CertificateObject().apply {
                                                    usage = "verify"
                                                    certificate = bean.certificates.lines()
                                                })
                                            }
                                            if (bean.mtlsCertificate.isNotEmpty() || bean.mtlsCertificatePrivateKey.isNotEmpty()) {
                                                certificates.add(TLSObject.CertificateObject().apply {
                                                    usage = "encipherment"
                                                    certificate = bean.mtlsCertificate.lines()
                                                    key = bean.mtlsCertificatePrivateKey.lines()
                                                })
                                            }
                                        }
                                        if (bean.echConfig.isNotEmpty()) {
                                            echConfig = bean.echConfig
                                        }
                                    }
                                }
                            } else if (bean is Tuic5Bean) {
                                protocol = "tuic"
                                settings = LazyOutboundConfigurationObject(this,
                                    V2RayConfig.TUICOutboundConfigurationObject().apply {
                                        address = bean.serverAddress
                                        port = bean.serverPort
                                        uuid = bean.uuid
                                        password = bean.password
                                        congestionControl = bean.congestionControl
                                        udpRelayMode = bean.udpRelayMode
                                        if (bean.zeroRTTHandshake) zeroRTTHandshake = bean.zeroRTTHandshake
                                        if (bean.disableSNI) disableSNI = bean.disableSNI
                                        if (bean.singUDPOverStream && getEnabled(DataStore.experimentalFlags, "singuot")) {
                                            udpOverStream = bean.singUDPOverStream
                                        }
                                        tlsSettings = TLSObject().apply {
                                            if (bean.sni.isNotEmpty()) {
                                                serverName = bean.sni
                                            }
                                            if (bean.alpn.isNotEmpty()) {
                                                alpn = bean.alpn.listByLineOrComma()
                                            }
                                            if (bean.allowInsecure) {
                                                allowInsecure = true
                                            }
                                            if (bean.certificates.isNotEmpty() || bean.mtlsCertificate.isNotEmpty() || bean.mtlsCertificatePrivateKey.isNotEmpty()) {
                                                certificates = mutableListOf()
                                                if (bean.certificates.isNotEmpty()) {
                                                    disableSystemRoot = true
                                                    certificates.add(TLSObject.CertificateObject().apply {
                                                        usage = "verify"
                                                        certificate = bean.certificates.lines()
                                                    })
                                                }
                                                if (bean.mtlsCertificate.isNotEmpty() || bean.mtlsCertificatePrivateKey.isNotEmpty()) {
                                                    certificates.add(TLSObject.CertificateObject().apply {
                                                        usage = "encipherment"
                                                        certificate = bean.mtlsCertificate.lines()
                                                        key = bean.mtlsCertificatePrivateKey.lines()
                                                    })
                                                }
                                            }
                                            if (bean.pinnedPeerCertificateSha256.isNotEmpty()) {
                                                pinnedPeerCertificateSha256 = mutableListOf()
                                                bean.pinnedPeerCertificateSha256.listByLineOrComma().forEach {
                                                    pinnedPeerCertificateSha256.add(it.replace(":", ""))
                                                }
                                            }
                                            if (bean.pinnedPeerCertificatePublicKeySha256.isNotEmpty()) {
                                                pinnedPeerCertificatePublicKeySha256 = bean.pinnedPeerCertificatePublicKeySha256.listByLineOrComma()
                                            }
                                            if (bean.pinnedPeerCertificateChainSha256.isNotEmpty()) {
                                                pinnedPeerCertificateChainSha256 = bean.pinnedPeerCertificateChainSha256.listByLineOrComma()
                                            }
                                            if (bean.echConfig.isNotEmpty()) {
                                                echConfig = bean.echConfig
                                            }
                                        }
                                    }
                                )
                            } else if (bean is Http3Bean) {
                                protocol = "http3"
                                settings = LazyOutboundConfigurationObject(this,
                                    V2RayConfig.HTTP3OutboundConfigurationObject().apply {
                                        address = bean.serverAddress
                                        port = bean.serverPort
                                        if (bean.username.isNotEmpty()) username = bean.username
                                        if (bean.password.isNotEmpty()) password = bean.password
                                        tlsSettings = TLSObject().apply {
                                            if (bean.sni.isNotEmpty()) {
                                                serverName = bean.sni
                                            }
                                            if (bean.certificates.isNotEmpty() || bean.mtlsCertificate.isNotEmpty() || bean.mtlsCertificatePrivateKey.isNotEmpty()) {
                                                certificates = mutableListOf()
                                                if (bean.certificates.isNotEmpty()) {
                                                    disableSystemRoot = true
                                                    certificates.add(TLSObject.CertificateObject().apply {
                                                        usage = "verify"
                                                        certificate = bean.certificates.lines()
                                                    })
                                                }
                                                if (bean.mtlsCertificate.isNotEmpty() || bean.mtlsCertificatePrivateKey.isNotEmpty()) {
                                                    certificates.add(TLSObject.CertificateObject().apply {
                                                        usage = "encipherment"
                                                        certificate = bean.mtlsCertificate.lines()
                                                        key = bean.mtlsCertificatePrivateKey.lines()
                                                    })
                                                }
                                            }
                                            if (bean.pinnedPeerCertificateSha256.isNotEmpty()) {
                                                pinnedPeerCertificateSha256 = mutableListOf()
                                                bean.pinnedPeerCertificateSha256.listByLineOrComma().forEach {
                                                    pinnedPeerCertificateSha256.add(it.replace(":", ""))
                                                }
                                            }
                                            if (bean.pinnedPeerCertificatePublicKeySha256.isNotEmpty()) {
                                                pinnedPeerCertificatePublicKeySha256 = bean.pinnedPeerCertificatePublicKeySha256.listByLineOrComma()
                                            }
                                            if (bean.pinnedPeerCertificateChainSha256.isNotEmpty()) {
                                                pinnedPeerCertificateChainSha256 = bean.pinnedPeerCertificateChainSha256.listByLineOrComma()
                                            }
                                            if (bean.echConfig.isNotEmpty()) {
                                                echConfig = bean.echConfig
                                            }
                                            if (bean.allowInsecure) {
                                                allowInsecure = true
                                            }
                                            if (bean.echConfig.isNotEmpty()) {
                                                echConfig = bean.echConfig
                                            }
                                        }
                                    }
                                )
                            } else if (bean is ShadowTLSBean) {
                                protocol = "shadowtls"
                                settings = LazyOutboundConfigurationObject(this,
                                    V2RayConfig.ShadowTLSOutboundConfigurationObject().apply {
                                        address = bean.serverAddress
                                        port = bean.serverPort
                                        if (bean.password.isNotEmpty()) password = bean.password
                                        version = bean.protocolVersion
                                        tlsSettings = TLSObject().apply {
                                            if (bean.sni.isNotEmpty()) {
                                                serverName = bean.sni
                                            }
                                            if (bean.alpn.isNotEmpty()) {
                                                alpn = bean.alpn.listByLineOrComma()
                                            }
                                            if (bean.allowInsecure) {
                                                allowInsecure = true
                                            }
                                            if (bean.certificates.isNotEmpty()) {
                                                disableSystemRoot = true
                                                certificates = listOf(TLSObject.CertificateObject().apply {
                                                    usage = "verify"
                                                    certificate = bean.certificates.lines()
                                                })
                                            }
                                        }
                                    }
                                )
                            } else if (bean is AnyTLSBean) {
                                protocol = "anytls"
                                settings = LazyOutboundConfigurationObject(this,
                                    V2RayConfig.AnyTLSOutboundConfigurationObject().apply {
                                        address = bean.serverAddress
                                        port = bean.serverPort
                                        if (bean.password.isNotEmpty()) password = bean.password
                                        idleSessionCheckInterval = bean.idleSessionCheckInterval
                                        idleSessionTimeout = bean.idleSessionTimeout
                                        minIdleSession = bean.minIdleSession
                                    }
                                )
                                streamSettings = StreamSettingsObject().apply {
                                    if (bean.security.isNotEmpty()) {
                                        security = bean.security
                                    }
                                    when (security) {
                                        "tls" -> {
                                            tlsSettings = TLSObject().apply {
                                                if (bean.sni.isNotEmpty()) {
                                                    serverName = bean.sni
                                                }
                                                if (bean.alpn.isNotEmpty()) {
                                                    alpn = bean.alpn.listByLineOrComma()
                                                }
                                                if (bean.certificates.isNotEmpty() || bean.mtlsCertificate.isNotEmpty() || bean.mtlsCertificatePrivateKey.isNotEmpty()) {
                                                    certificates = mutableListOf()
                                                    if (bean.certificates.isNotEmpty()) {
                                                        disableSystemRoot = true
                                                        certificates.add(TLSObject.CertificateObject().apply {
                                                            usage = "verify"
                                                            certificate = bean.certificates.lines()
                                                        })
                                                    }
                                                    if (bean.mtlsCertificate.isNotEmpty() || bean.mtlsCertificatePrivateKey.isNotEmpty()) {
                                                        certificates.add(TLSObject.CertificateObject().apply {
                                                            usage = "encipherment"
                                                            certificate = bean.mtlsCertificate.lines()
                                                            key = bean.mtlsCertificatePrivateKey.lines()
                                                        })
                                                    }
                                                }
                                                if (bean.pinnedPeerCertificateSha256.isNotEmpty()) {
                                                    pinnedPeerCertificateSha256 = mutableListOf()
                                                    bean.pinnedPeerCertificateSha256.listByLineOrComma().forEach {
                                                        pinnedPeerCertificateSha256.add(it.replace(":", ""))
                                                    }
                                                }
                                                if (bean.pinnedPeerCertificatePublicKeySha256.isNotEmpty()) {
                                                    pinnedPeerCertificatePublicKeySha256 = bean.pinnedPeerCertificatePublicKeySha256.listByLineOrComma()
                                                }
                                                if (bean.pinnedPeerCertificateChainSha256.isNotEmpty()) {
                                                    pinnedPeerCertificateChainSha256 = bean.pinnedPeerCertificateChainSha256.listByLineOrComma()
                                                }
                                                if (bean.allowInsecure) {
                                                    allowInsecure = true
                                                }
                                                if (bean.utlsFingerprint.isNotEmpty()) {
                                                    fingerprint = bean.utlsFingerprint
                                                }
                                                if (bean.echConfig.isNotEmpty()) {
                                                    echConfig = bean.echConfig
                                                }
                                            }
                                        }
                                        "reality" -> {
                                            realitySettings = RealityObject().apply {
                                                if (bean.sni.isNotEmpty()) {
                                                    serverName = bean.sni
                                                }
                                                if (bean.realityPublicKey.isNotEmpty()) {
                                                    publicKey = bean.realityPublicKey
                                                }
                                                if (bean.realityShortId.isNotEmpty()) {
                                                    shortId = bean.realityShortId
                                                }
                                                if (bean.realityFingerprint.isNotEmpty()) {
                                                    fingerprint = bean.realityFingerprint
                                                }
                                                if (DataStore.realityDisableX25519Mlkem768 || bean.realityDisableX25519Mlkem768 && !forExport) {
                                                    disableX25519MLKEM768 = true
                                                }
                                            }
                                        }
                                    }
                                    if (DataStore.enableFragment) {
                                        sockopt = StreamSettingsObject.SockoptObject().apply {
                                            tlsFragmentation = StreamSettingsObject.SockoptObject.TLSFragmentationObject().apply {
                                                when (DataStore.fragmentMethod) {
                                                    TLS_FRAGMENTATION_METHOD.TLS_RECORD_FRAGMENTATION -> {
                                                        tlsRecordFragmentation = true
                                                    }
                                                    TLS_FRAGMENTATION_METHOD.TCP_SEGMENTATION -> {
                                                        tcpSegmentation = true
                                                    }
                                                    TLS_FRAGMENTATION_METHOD.TLS_RECORD_FRAGMENTATION_AND_TCP_SEGMENTATION -> {
                                                        tlsRecordFragmentation = true
                                                        tcpSegmentation = true
                                                    }
                                                }
                                            }
                                        }
                                    }
                                }
                            } else if (bean is JuicityBean) {
                                protocol = "juicity"
                                settings = LazyOutboundConfigurationObject(this,
                                    V2RayConfig.JuicityOutboundConfigurationObject().apply {
                                        address = bean.serverAddress
                                        port = bean.serverPort
                                        uuid = bean.uuid
                                        password = bean.password
                                        tlsSettings = TLSObject().apply {
                                            if (bean.sni.isNotEmpty()) {
                                                serverName = bean.sni
                                            }
                                            if (bean.allowInsecure) {
                                                allowInsecure = true
                                            }
                                            if (bean.pinnedPeerCertificateSha256.isNotEmpty()) {
                                                pinnedPeerCertificateSha256 = mutableListOf()
                                                bean.pinnedPeerCertificateSha256.listByLineOrComma().forEach {
                                                    pinnedPeerCertificateSha256.add(it.replace(":", ""))
                                                }
                                            }
                                            if (bean.pinnedPeerCertificatePublicKeySha256.isNotEmpty()) {
                                                pinnedPeerCertificatePublicKeySha256 = bean.pinnedPeerCertificatePublicKeySha256.listByLineOrComma()
                                            }
                                            if (bean.pinnedPeerCertificateChainSha256.isNotEmpty()) {
                                                pinnedPeerCertificateChainSha256 = mutableListOf()
                                                bean.pinnedPeerCertificateChainSha256.listByLineOrComma().forEach {
                                                    pinnedPeerCertificateChainSha256.add(
                                                        when {
                                                            it.length == 64 -> {
                                                                Base64.encode(bean.pinnedPeerCertificateChainSha256.chunked(2).map { it.toInt(16).toByte() }.toByteArray())
                                                            }
                                                            else -> {
                                                                bean.pinnedPeerCertificateChainSha256.replace('_', '/').replace('-', '+')
                                                            }
                                                        }
                                                    )
                                                }
                                            }
                                            if (bean.certificates.isNotEmpty() || bean.mtlsCertificate.isNotEmpty() || bean.mtlsCertificatePrivateKey.isNotEmpty()) {
                                                certificates = mutableListOf()
                                                if (bean.certificates.isNotEmpty()) {
                                                    disableSystemRoot = true
                                                    certificates.add(TLSObject.CertificateObject().apply {
                                                        usage = "verify"
                                                        certificate = bean.certificates.lines()
                                                    })
                                                }
                                                if (bean.mtlsCertificate.isNotEmpty() || bean.mtlsCertificatePrivateKey.isNotEmpty()) {
                                                    certificates.add(TLSObject.CertificateObject().apply {
                                                        usage = "encipherment"
                                                        certificate = bean.mtlsCertificate.lines()
                                                        key = bean.mtlsCertificatePrivateKey.lines()
                                                    })
                                                }
                                            }
                                            if (bean.echConfig.isNotEmpty()) {
                                                echConfig = bean.echConfig
                                            }
                                        }
                                    }
                                )
                            } else if (bean is MieruBean) {
                                protocol = "mieru"
                                settings = LazyOutboundConfigurationObject(this,
                                    V2RayConfig.MieruOutboundConfigurationObject().apply {
                                        address = bean.serverAddress
                                        if (bean.portRange.isNotEmpty()) {
                                            portRange = bean.portRange.listByLineOrComma()
                                        } else {
                                            port = bean.serverPort
                                        }
                                        username = bean.username
                                        password = bean.password
                                        when (bean.protocol) {
                                            MieruBean.PROTOCOL_TCP -> protocol = "tcp"
                                            MieruBean.PROTOCOL_UDP -> protocol = "udp"
                                        }
                                        when (bean.multiplexingLevel) {
                                            MieruBean.MULTIPLEXING_DEFAULT -> multiplexing = "default"
                                            MieruBean.MULTIPLEXING_OFF -> multiplexing = "off"
                                            MieruBean.MULTIPLEXING_LOW -> multiplexing = "low"
                                            MieruBean.MULTIPLEXING_MIDDLE -> multiplexing = "middle"
                                            MieruBean.MULTIPLEXING_HIGH -> multiplexing = "high"
                                        }
                                        when (bean.handshakeMode) {
                                            MieruBean.HANDSHAKE_DEFAULT -> handshakeMode = "default"
                                            MieruBean.HANDSHAKE_STANDARD -> handshakeMode = "standard"
                                            MieruBean.HANDSHAKE_NO_WAIT -> handshakeMode = "nowait"
                                        }
                                    }
                                )
                            }
                            if (bean is StandardV2RayBean && bean.mux) {
                                mux = OutboundObject.MuxObject().apply {
                                    enabled = true
                                    concurrency = bean.muxConcurrency
                                    when (bean.muxPacketEncoding) {
                                        "packet" -> {
                                            packetEncoding = "packet"
                                        }
                                        "xudp" -> {
                                            packetEncoding = "xudp"
                                        }
                                    }
                                }
                            }
                            if ((bean is ShadowsocksBean || bean is TrojanBean || bean is VMessBean || bean is VLESSBean)
                                && bean.singMux && getEnabled(DataStore.experimentalFlags, "singmux")) {
                                smux = OutboundObject.SmuxObject().apply {
                                    enabled = bean.singMux
                                    protocol = bean.singMuxProtocol
                                    if (bean.singMuxMaxConnections > 0) {
                                        maxConnections = bean.singMuxMaxConnections
                                    }
                                    if (bean.singMuxMinStreams > 0) {
                                        minStreams = bean.singMuxMinStreams
                                    }
                                    if (bean.singMuxMaxStreams > 0) {
                                        maxStreams = bean.singMuxMaxStreams
                                    }
                                    if (bean.singMuxPadding) {
                                        padding = bean.singMuxPadding
                                    }
                                }
                            }
                        }
                    }

                    if (currentDomainStrategy != "AsIs") {
                        currentOutbound.domainStrategy = currentDomainStrategy
                    }

                    if (!(currentOutbound.domainStrategy == null && DataStore.outboundDomainStrategyForServer == "AsIs")
                        && !(currentOutbound.domainStrategy == "AsIs" && DataStore.outboundDomainStrategyForServer == "AsIs")
                        && currentOutbound.domainStrategy != DataStore.outboundDomainStrategyForServer) {
                        currentOutbound.dialDomainStrategy = DataStore.outboundDomainStrategyForServer
                    }

                    if (bean is ConfigBean && bean.type == "v2ray_outbound") {
                        currentOutbound = gson.fromJson(bean.content, OutboundObject::class.java).apply { init() }
                    }

                    currentOutbound.tag = tagIn

                }

                if (!isBalancer && index > 0) {
                    if (!pastExternal) {
                        pastOutbound.proxySettings = OutboundObject.ProxySettingsObject().apply {
                            tag = tagIn
                            transportLayer = true
                        }
                    } else {
                        routing.rules.add(RoutingObject.RuleObject().apply {
                            type = "field"
                            inboundTag = listOf(pastInboundTag)
                            outboundTag = tagIn
                        })
                    }
                }

                if (proxyEntity.needExternal() && !isBalancer && index != profileList.lastIndex) {
                    val mappingPort = mkPort()
                    bean.finalAddress = LOCALHOST
                    bean.finalPort = mappingPort
                    bean.isChain = true

                    inbounds.add(InboundObject().apply {
                        listen = LOCALHOST
                        port = mappingPort
                        tag = "$tagOutbound-mapping-${proxyEntity.id}"
                        protocol = "dokodemo-door"
                        settings = LazyInboundConfigurationObject(this,
                            DokodemoDoorInboundConfigurationObject().apply {
                                address = bean.serverAddress
                                network = bean.network()
                                port = bean.serverPort
                            })

                        pastInboundTag = tag
                    })
                } else if (bean.canMapping() && proxyEntity.needExternal()) {
                    val mappingPort = mkPort()
                    bean.finalAddress = LOCALHOST
                    bean.finalPort = mappingPort

                    inbounds.add(InboundObject().apply {
                        listen = LOCALHOST
                        port = mappingPort
                        tag = "$tagOutbound-mapping-${proxyEntity.id}"
                        protocol = "dokodemo-door"
                        settings = LazyInboundConfigurationObject(this,
                            DokodemoDoorInboundConfigurationObject().apply {
                                address = bean.serverAddress
                                network = bean.network()
                                port = bean.serverPort
                            })
                        routing.rules.add(RoutingObject.RuleObject().apply {
                            type = "field"
                            inboundTag = listOf(tag)
                            outboundTag = TAG_DIRECT
                        })
                    })

                }

                if (!needGlobal) {
                    outbounds.add(currentOutbound)
                    chainOutbounds.add(currentOutbound)
                    pastExternal = proxyEntity.needExternal()
                    pastOutbound = currentOutbound
                }

            }

            if (isBalancer) {
                val balancerBean = balancer()!!
                val observatory = ObservatoryObject().apply {
                    probeURL = balancerBean.probeUrl.ifEmpty {
                        DataStore.connectionTestURL
                    }
                    if (balancerBean.probeInterval > 0) {
                        probeInterval = "${balancerBean.probeInterval}s"
                    }
                    enableConcurrency = true
                    subjectSelector = HashSet(chainOutbounds.map { it.tag })
                }
                val observatoryItem = MultiObservatoryObject.MultiObservatoryItem().apply {
                    tag = "observer-$tagOutbound"
                    settings = mutableMapOf<String, Any>()
                    settings["probeURL"] = observatory.probeURL
                    settings["probeInterval"] = observatory.probeInterval
                    settings["enableConcurrency"] = observatory.enableConcurrency
                    settings["subjectSelector"] = observatory.subjectSelector
                }
                if (multiObservatory == null) multiObservatory = MultiObservatoryObject().apply {
                    observers = mutableListOf()
                }
                multiObservatory.observers.add(observatoryItem)

                if (routing.balancers == null) routing.balancers = ArrayList()
                routing.balancers.add(RoutingObject.BalancerObject().apply {
                    tag = "balancer-$tagOutbound"
                    selector = chainOutbounds.map { it.tag }
                    if (multiObservatory == null) {
                        multiObservatory = MultiObservatoryObject().apply {
                            observers = mutableListOf()
                        }
                    }
                    strategy = StrategyObject().apply {
                        type = balancerBean.strategy.takeIf { it.isNotEmpty() } ?: "random"
                        when (type) {
                            "leastPing", "leastLoad" -> {
                                settings = StrategyObject.strategyConfig().apply {
                                    observerTag = "observer-$tagOutbound"
                                }
                            }
                            else -> {
                                settings = StrategyObject.strategyConfig().apply {
                                    observerTag = "observer-$tagOutbound"
                                    aliveOnly = true
                                }
                            }
                        }
                    }
                })
                if (tagOutbound == TAG_AGENT) {
                    if (observatoryItem.settings["probeURL"] == DataStore.connectionTestURL) {
                        rootObserver = observatoryItem
                    }
                    // if all outbounds of a balancer are dead, the first (default) outbound will be used
                    rootBalancer = RoutingObject.RuleObject().apply {
                        type = "field"
                        network = "tcp,udp"
                        balancerTag = "balancer-$tagOutbound"
                    }
                }
            }

            return chainOutbound

        }

        val mainIsBalancer = proxy.balancerBean != null

        val tagProxy = buildChain(
            TAG_AGENT, proxies, mainIsBalancer
        ) { proxy.balancerBean }

        val balancerMap = mutableMapOf<Long, String>()
        val tagMap = mutableMapOf<Long, String>()
        extraProxies.forEach { (key, entities) ->
            val (id, balancer) = key
            val (isBalancer, balancerBean) = balancer
            tagMap[id] = buildChain("$TAG_AGENT-$id", entities, isBalancer, balancerBean::value)
            if (isBalancer) {
                balancerMap[id] = "balancer-$TAG_AGENT-$id"
            }
        }

        val isVpn = DataStore.serviceMode == Key.MODE_VPN

        for (rule in extraRules) {
            val uidList = mutableListOf<Int>()
            if (rule.packages.isNotEmpty()) {
                if (!isVpn) {
                    alerts.add(Alerts.ROUTE_ALERT_NOT_VPN to rule.displayName())
                    continue
                }
                PackageCache.awaitLoadSync()
                for (pkg in rule.packages) {
                    PackageCache[pkg]?.let {
                        uidList.add(it)
                    }
                }
                if (uidList.isEmpty()) {
                    alerts.add(Alerts.ROUTE_ALERT_ALL_PACKAGES_UNINSTALLED to rule.displayName())
                    continue
                }
            }
            routing.rules.add(RoutingObject.RuleObject().apply {
                type = "field"
                if (uidList.isNotEmpty()) {
                    uid = uidList
                }

                if (rule.ssid.isNotEmpty() && Build.VERSION.SDK_INT >= Build.VERSION_CODES.M) {
                    if (app.checkSelfPermission(Manifest.permission.ACCESS_FINE_LOCATION) != PackageManager.PERMISSION_GRANTED) {
                        throw Alerts.RouteAlertException(
                            Alerts.ROUTE_ALERT_NEED_FINE_LOCATION_ACCESS, rule.displayName()
                        )
                    }
                    if (Build.VERSION.SDK_INT >= Build.VERSION_CODES.Q && app.checkSelfPermission(Manifest.permission.ACCESS_BACKGROUND_LOCATION) != PackageManager.PERMISSION_GRANTED) {
                        throw Alerts.RouteAlertException(
                            Alerts.ROUTE_ALERT_NEED_BACKGROUND_LOCATION_ACCESS, rule.displayName()
                        )
                    }
                    val isLocationEnabled = if (Build.VERSION.SDK_INT >= Build.VERSION_CODES.P) {
                        SagerNet.location.isLocationEnabled
                    } else {
                        try {
                            Settings.Secure.getInt(
                                app.contentResolver, Settings.Secure.LOCATION_MODE
                            ) != Settings.Secure.LOCATION_MODE_OFF
                        } catch (e: Settings.SettingNotFoundException) {
                            e.printStackTrace()
                            false
                        }
                    }
                    if (!isLocationEnabled) {
                        throw Alerts.RouteAlertException(
                            Alerts.ROUTE_ALERT_LOCATION_DISABLED, rule.displayName()
                        )
                    }
                }

                if (rule.domains.isNotEmpty()) {
                    domains = rule.domains.listByLineOrComma()
                }
                if (rule.ip.isNotEmpty()) {
                    ip = rule.ip.listByLineOrComma()
                }
                if (rule.port.isNotEmpty()) {
                    port = rule.port
                }
                if (rule.sourcePort.isNotEmpty()) {
                    sourcePort = rule.sourcePort
                }
                if (rule.network.isNotEmpty()) {
                    network = rule.network
                }
                if (rule.source.isNotEmpty()) {
                    source = rule.source.listByLineOrComma()
                }
                if (rule.protocol.isNotEmpty()) {
                    protocol = rule.protocol.listByLineOrComma()
                }
                if (rule.attrs.isNotEmpty()) {
                    attrs = rule.attrs
                }
                if (rule.ssid.isNotEmpty()) {
                    // a hack for SSID containing `\n`
                    ssid = rule.ssid.split("\n").map { it.unescapeLineFeed() }
                }
                if (rule.networkType.isNotEmpty()) {
                    networkType = rule.networkType.toMutableList()
                }
                when {
                    rule.reverse -> {
                        inboundTag = listOf("reverse-${rule.id}")
                        val outId = rule.outbound
                        outboundTag = if (outId == proxy.id) tagProxy else {
                            tagMap[outId] ?: error("outbound not found in rule ${rule.displayName()}")
                        }
                    }
                    balancerMap.containsKey(rule.outbound) -> {
                        balancerTag = balancerMap[rule.outbound]
                    }
                    mainIsBalancer && rule.outbound == 0L -> balancerTag = "balancer-$TAG_AGENT"
                    else -> {
                        outboundTag = when (val outId = rule.outbound) {
                            0L -> tagProxy
                            -1L -> TAG_BYPASS
                            -2L -> TAG_BLOCK
                            else -> if (outId == proxy.id) tagProxy else {
                                tagMap[outId] ?: error("outbound not found in rule ${rule.displayName()}")
                            }
                        }
                    }
                }
            })

            if (rule.reverse) {
                outbounds.add(OutboundObject().apply {
                    tag = "reverse-out-${rule.id}"
                    protocol = "freedom"
                    settings = LazyOutboundConfigurationObject(this,
                        FreedomOutboundConfigurationObject().apply {
                            redirect = rule.redirect
                        })
                })
                if (reverse == null) {
                    reverse = ReverseObject().apply {
                        bridges = ArrayList()
                    }
                }
                reverse.bridges.add(ReverseObject.BridgeObject().apply {
                    tag = "reverse-${rule.id}"
                    domain = rule.domains.substringAfter("full:")
                })
                routing.rules.add(RoutingObject.RuleObject().apply {
                    type = "field"
                    inboundTag = listOf("reverse-${rule.id}")
                    outboundTag = "reverse-out-${rule.id}"
                })
            }

        }

        if (requireWs) {
            browserForwarder = BrowserForwarderObject().apply {
                listenAddr = LOCALHOST
                listenPort = mkPort()
            }
        }

        if (requireSh) {
            browserDialer = BrowserDialerObject().apply {
                listenAddr = LOCALHOST
                listenPort = mkPort()
            }
        }

        outbounds.add(OutboundObject().apply {
            tag = TAG_DIRECT
            protocol = "freedom"
        })
        outbounds.add(OutboundObject().apply {
            tag = TAG_BYPASS
            protocol = "freedom"
            if (DataStore.enableFragment && DataStore.enableFragmentForDirect) {
                streamSettings = StreamSettingsObject().apply {
                    sockopt = StreamSettingsObject.SockoptObject().apply {
                        tlsFragmentation = StreamSettingsObject.SockoptObject.TLSFragmentationObject().apply {
                            when (DataStore.fragmentMethod) {
                                TLS_FRAGMENTATION_METHOD.TLS_RECORD_FRAGMENTATION -> {
                                    tlsRecordFragmentation = true
                                }
                                TLS_FRAGMENTATION_METHOD.TCP_SEGMENTATION -> {
                                    tcpSegmentation = true
                                }
                                TLS_FRAGMENTATION_METHOD.TLS_RECORD_FRAGMENTATION_AND_TCP_SEGMENTATION -> {
                                    tlsRecordFragmentation = true
                                    tcpSegmentation = true
                                }
                            }
                        }
                    }
                }
            }
            if (DataStore.outboundDomainStrategyForDirect != "AsIs") {
                settings = LazyOutboundConfigurationObject(this,
                    FreedomOutboundConfigurationObject().apply {
                        domainStrategy = DataStore.outboundDomainStrategyForDirect
                    }
                )
            }
        })

        outbounds.add(OutboundObject().apply {
            tag = TAG_BLOCK
            protocol = "blackhole"
        })

        if (!forTest && DataStore.requireDnsInbound && DataStore.localDNSPort > 0) {
            inbounds.add(InboundObject().apply {
                tag = TAG_DNS_IN
                listen = bind
                port = DataStore.localDNSPort
                protocol = "dokodemo-door"
                settings = LazyInboundConfigurationObject(this,
                    DokodemoDoorInboundConfigurationObject().apply {
                        address = bind // placeholder, all queries are handled internally
                        network = "tcp,udp"
                        port = 53 // placeholder, all queries are handled internally
                    }
                )
                if (shouldDumpUID) dumpUID = true
            })
        }

        outbounds.add(OutboundObject().apply {
            protocol = "dns"
            tag = TAG_DNS_OUT
            settings = LazyOutboundConfigurationObject(this,
                DNSOutboundConfigurationObject().apply {
                    userLevel = 1
                    if (getEnabled(DataStore.experimentalFlags, "lookupAsExchange")) {
                        lookupAsExchange = true
                    }
                })
        })

        val bypassDomain = HashSet<String>()
        val bypassDomainSkipFakeDns = HashSet<String>()
        val proxyDomain = HashSet<String>()
        val bootstrapDomain = HashSet<String>()

        (proxies + extraProxies.values.flatten()).forEach { it ->
            val bean = it.requireBean()
            bean.apply {
                if (bean is ConfigBean && bean.type == "v2ray_outbound") {
                    // too dirty to read server addresses from a custom outbound config
                    // let users provide them manually
                    bean.serverAddresses.listByLineOrComma().forEach {
                        when {
                            it.isEmpty() -> {}
                            !Libcore.isIP(it) -> {
                                bypassDomainSkipFakeDns.add("full:$it")
                            }
                        }
                    }
                } else {
                    if (!Libcore.isIP(serverAddress)) {
                        bypassDomainSkipFakeDns.add("full:$serverAddress")
                    }
                }
            }
        }

        if (DataStore.enableDnsRouting) {
            val directDNSDomainList = getStringValue(DataStore.experimentalFlags, "directDNSDomainList")
            if (directDNSDomainList != null) {
                bypassDomain.addAll(directDNSDomainList.split(","))
            } else {
                for (bypassRule in extraRules.filter { it.isBypassRule() }) {
                    if (bypassRule.domains.isNotEmpty()) {
                        bypassDomain.addAll(bypassRule.domains.listByLineOrComma())
                    }
                }
            }
            val remoteDNSDomainList = getStringValue(DataStore.experimentalFlags, "remoteDNSDomainList")
            if (remoteDNSDomainList != null) {
                proxyDomain.addAll(remoteDNSDomainList.split(","))
            } else {
                for (proxyRule in extraRules.filter { it.isProxyRule() }) {
                    if (proxyRule.domains.isNotEmpty()) {
                        proxyDomain.addAll(proxyRule.domains.listByLineOrComma())
                    }
                }
            }
        }

        remoteDNS.forEach {
            try {
                if (it.lowercase() != "localhost" && it.lowercase() != "fakedns") {
                    if (it.contains("://")) {
                        val url = Libcore.parseURL(it)
                        if (!Libcore.isIP(url.host)) {
                            bypassDomainSkipFakeDns.add("full:${url.host}")
                        }
                    } else if (!Libcore.isIP(it)) {
                        bypassDomainSkipFakeDns.add("full:$it")
                    }
                }
            } catch (_: Exception) {}
        }

        directDNS.forEach {
            try {
                if (it.lowercase() != "localhost" && it.lowercase() != "fakedns") {
                    if (it.contains("://")) {
                        val url = Libcore.parseURL(it)
                        if (!Libcore.isIP(url.host)) {
                            bootstrapDomain.add("full:${url.host}")
                        }
                    } else if (!Libcore.isIP(it)) {
                        bootstrapDomain.add("full:$it")
                    }
                }
            } catch (_: Exception) {}
        }

        var hasDnsTagDirect = false
        if (bypassDomain.isNotEmpty() || bypassDomainSkipFakeDns.isNotEmpty() || bootstrapDomain.isNotEmpty()) {
            dns.servers.addAll(remoteDNS.map {
                DnsObject.StringOrServerObject().apply {
                    valueY = DnsObject.ServerObject().apply {
                        address = it
                        domains = proxyDomain.toList() // v2fly/v2ray-core#1558, v2fly/v2ray-core#1855
                        queryStrategy = remoteDnsQueryStrategy
                        if (DataStore.ednsClientIp.isNotEmpty()) {
                            clientIp = DataStore.ednsClientIp
                        }
                        if (useFakeDns) {
                            fakedns = mutableListOf()
                            if (queryStrategy != "UseIPv6") {
                                fakedns.add(DnsObject.ServerObject.StringOrFakeDnsObject().apply {
                                    valueY = FakeDnsObject().apply {
                                        ipPool = "${VpnService.FAKEDNS_VLAN4_CLIENT}/${VpnService.FAKEDNS_VLAN4_CLIENT_PREFIX}"
                                        poolSize = VpnService.FAKEDNS_VLAN4_CLIENT_POOL_SIZE
                                    }
                                })
                            }
                            if (queryStrategy != "UseIPv4") {
                                fakedns.add(DnsObject.ServerObject.StringOrFakeDnsObject().apply {
                                    valueY = FakeDnsObject().apply {
                                        ipPool = "${VpnService.FAKEDNS_VLAN6_CLIENT}/${VpnService.FAKEDNS_VLAN6_CLIENT_PREFIX}"
                                        poolSize = VpnService.FAKEDNS_VLAN6_CLIENT_POOL_SIZE
                                    }
                                })
                            }
                        }
                    }
                }
            })
            if (bypassDomainSkipFakeDns.isNotEmpty()) {
                dns.servers.addAll(directDNS.map {
                    DnsObject.StringOrServerObject().apply {
                        valueY = DnsObject.ServerObject().apply {
                            address = it
                            // skip fake DNS for server addresses and DNS server addresses
                            domains = bypassDomainSkipFakeDns.toList()
                            queryStrategy = directDnsQueryStrategy
                            if (!it.lowercase().contains("+local://") && it.lowercase() != "localhost") {
                                tag = TAG_DNS_DIRECT
                                hasDnsTagDirect = true
                            }
                            fallbackStrategy = "disabled"
                        }
                    }
                })
            }
            if (bypassDomain.isNotEmpty()) {
                dns.servers.addAll(directDNS.map {
                    DnsObject.StringOrServerObject().apply {
                        valueY = DnsObject.ServerObject().apply {
                            address = it
                            //FIXME: This relies on the behavior of a bug.
                            domains = bypassDomain.toList() // v2fly/v2ray-core#1558, v2fly/v2ray-core#1855
                            queryStrategy = directDnsQueryStrategy
                            if (!it.contains("+local://") && it != "localhost") {
                                tag = TAG_DNS_DIRECT
                                hasDnsTagDirect = true
                            }
                            if (useFakeDns) {
                                fakedns = mutableListOf()
                                if (queryStrategy != "UseIPv6") {
                                    fakedns.add(DnsObject.ServerObject.StringOrFakeDnsObject().apply {
                                        valueY = FakeDnsObject().apply {
                                            ipPool = "${VpnService.FAKEDNS_VLAN4_CLIENT}/${VpnService.FAKEDNS_VLAN4_CLIENT_PREFIX}"
                                            poolSize = VpnService.FAKEDNS_VLAN4_CLIENT_POOL_SIZE
                                        }
                                    })
                                }
                                if (queryStrategy != "UseIPv4") {
                                    fakedns.add(DnsObject.ServerObject.StringOrFakeDnsObject().apply {
                                        valueY = FakeDnsObject().apply {
                                            ipPool = "${VpnService.FAKEDNS_VLAN6_CLIENT}/${VpnService.FAKEDNS_VLAN6_CLIENT_PREFIX}"
                                            poolSize = VpnService.FAKEDNS_VLAN6_CLIENT_POOL_SIZE
                                        }
                                    })
                                }
                            }
                            fallbackStrategy = "disabled"
                        }
                    }
                })
            }
            if (bootstrapDomain.isNotEmpty()) {
                dns.servers.addAll(bootstrapDNS.map {
                    DnsObject.StringOrServerObject().apply {
                        valueY = DnsObject.ServerObject().apply {
                            address = it
                            domains = bootstrapDomain.toList() // v2fly/v2ray-core#1558, v2fly/v2ray-core#1855
                            queryStrategy = directDnsQueryStrategy
                            if (!it.lowercase().contains("+local://") && it.lowercase() != "localhost") {
                                tag = TAG_DNS_DIRECT
                                hasDnsTagDirect = true
                            }
                            fallbackStrategy = "disabled"
                        }
                    }
                })
            }
        } else {
            dns.servers.addAll(remoteDNS.map {
                DnsObject.StringOrServerObject().apply {
                    valueY = DnsObject.ServerObject().apply {
                        address = it
                        queryStrategy = remoteDnsQueryStrategy
                        if (DataStore.ednsClientIp.isNotEmpty()) {
                            clientIp = DataStore.ednsClientIp
                        }
                        if (useFakeDns) {
                            fakedns = mutableListOf()
                            if (queryStrategy != "UseIPv6") {
                                fakedns.add(DnsObject.ServerObject.StringOrFakeDnsObject().apply {
                                    valueY = FakeDnsObject().apply {
                                        ipPool = "${VpnService.FAKEDNS_VLAN4_CLIENT}/${VpnService.FAKEDNS_VLAN4_CLIENT_PREFIX}"
                                        poolSize = VpnService.FAKEDNS_VLAN4_CLIENT_POOL_SIZE
                                    }
                                })
                            }
                            if (queryStrategy != "UseIPv4") {
                                fakedns.add(DnsObject.ServerObject.StringOrFakeDnsObject().apply {
                                    valueY = FakeDnsObject().apply {
                                        ipPool = "${VpnService.FAKEDNS_VLAN6_CLIENT}/${VpnService.FAKEDNS_VLAN6_CLIENT_PREFIX}"
                                        poolSize = VpnService.FAKEDNS_VLAN6_CLIENT_POOL_SIZE
                                    }
                                })
                            }
                        }
                    }
                }
            })
        }

        if (routeMode == RouteMode.DIRECT) {
            routing.rules.add(0, RoutingObject.RuleObject().apply {
                type = "field"
                port = "0-65535"
                outboundTag = TAG_BYPASS
            })
        }

        if (hasDnsTagDirect) {
            routing.rules.add(0, RoutingObject.RuleObject().apply {
                type = "field"
                inboundTag = listOf(TAG_DNS_DIRECT)
                outboundTag = TAG_BYPASS
            })
        }

        if (!forTest && trafficSniffing && DataStore.hijackDns) {
            routing.rules.add(0, RoutingObject.RuleObject().apply {
                type = "field"
                protocol = listOf("dns")
                outboundTag = TAG_DNS_OUT
            })
        }

        if (!forTest) {
            routing.rules.add(0, RoutingObject.RuleObject().apply {
                type = "field"
                inboundTag = listOf(TAG_DNS_IN)
                outboundTag = TAG_DNS_OUT
            })
        }

        if (rootBalancer != null) routing.rules.add(rootBalancer)

        if (trafficStatistics) stats = emptyMap()

        @Suppress("UNCHECKED_CAST")
        result = V2rayBuildResult(
            gson.toJson(this),
            indexMap,
            requireWs,
            if (requireWs) browserForwarder.listenPort else 0,
            requireSh,
            if (requireSh) browserDialer.listenPort else 0,
            outboundTags,
            outboundTagsCurrent,
            outboundTagsAll,
            TAG_BYPASS,
            rootObserver?.tag ?: "",
            rootObserver?.settings?.get("subjectSelector") as? Set<String> ?: HashSet(),
            shouldDumpUID,
            alerts
        )
    }

    return result

}

fun buildCustomConfig(proxy: ProxyEntity, port: Int): V2rayBuildResult {

    val bind = LOCALHOST
    val trafficSniffing = DataStore.trafficSniffing

    val bean = proxy.configBean!!
    val config = parseJson(bean.content, lenient = true).asJsonObject
    val inbounds = config.getArray("inbounds")
        ?.map { gson.fromJson(it.toString(), InboundObject::class.java) }
        ?.toMutableList() ?: ArrayList()

    var socksInbound = inbounds.find { it.tag == TAG_SOCKS }?.apply {
        if (protocol != "socks") error("Inbound $tag with type $protocol, excepted socks.")
    }

    if (socksInbound == null) {
        val socksInbounds = inbounds.filter { it.protocol == "socks" }
        if (socksInbounds.size == 1) {
            socksInbound = socksInbounds[0]
        }
    }

    if (socksInbound != null) {
        socksInbound.apply {
            listen = bind
            this.port = port
        }
    } else {
        inbounds.add(InboundObject().apply {
            tag = TAG_SOCKS
            listen = bind
            this.port = port
            protocol = "socks"
            settings = LazyInboundConfigurationObject(this,
                SocksInboundConfigurationObject().apply {
                    auth = "noauth"
                    udp = true
                })
            if (trafficSniffing) {
                sniffing = InboundObject.SniffingObject().apply {
                    enabled = true
                    destOverride = listOf("http", "tls", "quic")
                    metadataOnly = false
                }
            }
        })
    }

    val outbounds = try {
        config.getArray("outbounds")?.map { it ->
            gson.fromJson(it.toString() .takeIf { it.isNotEmpty() } ?: "{}",
                OutboundObject::class.java)
        }?.toMutableList()
    } catch (_: JsonSyntaxException) {
        null
    }
    var flushOutbounds = false

    val outboundTags = ArrayList<String>()
    val firstOutbound = outbounds?.get(0)
    if (firstOutbound != null) {
        if (firstOutbound.tag == null) {
            firstOutbound.tag = TAG_AGENT
            outboundTags.add(TAG_AGENT)
            flushOutbounds = true
        } else {
            outboundTags.add(firstOutbound.tag)
        }
    }

    var directTag = ""
    val directOutbounds = outbounds?.filter { it.protocol == "freedom" }
    if (!directOutbounds.isNullOrEmpty()) {
        val directOutbound = if (directOutbounds.size == 1) {
            directOutbounds[0]
        } else {
            val directOutboundsWithTag = directOutbounds.filter { it.tag != null }
            if (directOutboundsWithTag.isNotEmpty()) {
                directOutboundsWithTag[0]
            } else {
                directOutbounds[0]
            }
        }
        if (directOutbound.tag.isNullOrEmpty()) {
            directOutbound.tag = TAG_DIRECT
            flushOutbounds = true
        }
        directTag = directOutbound.tag
    }

    inbounds.forEach { it.init() }
    val inboundArray = JsonArray(inbounds.size)
    for (inbound in inbounds) {
        inboundArray.add(parseJson(gson.toJson(inbound), lenient = true))
    }
    config.add("inbounds", inboundArray)
    if (flushOutbounds) {
        outbounds!!.forEach { it.init() }
        val outboundArray = JsonArray(inbounds.size)
        for (outbound in outbounds) {
            outboundArray.add(parseJson(gson.toJson(outbound), lenient = true))
        }
    }


    return V2rayBuildResult(
        config.toString(),
        emptyList(),
        false, // requireWs
        0, // wsPort
        false, // requireSh
        0, // shPort
        outboundTags,
        outboundTags,
        emptyMap(),
        directTag,
        "",
        emptySet(),
        false,
        emptyList()
    )

}
