// Copyright 2016 The Chromium Authors
// Use of this source code is governed by a BSD-style license that can be
// found in the LICENSE file.

#ifndef NET_FILTER_MOCK_SOURCE_STREAM_H_
#define NET_FILTER_MOCK_SOURCE_STREAM_H_

#include <stdint.h>

#include <string>
#include <string_view>

#include "base/containers/queue.h"
#include "base/containers/span.h"
#include "base/memory/raw_span.h"
#include "base/memory/scoped_refptr.h"
#include "net/base/completion_once_callback.h"
#include "net/base/net_errors.h"
#include "net/filter/source_stream.h"

namespace net {

class IOBuffer;

// A SourceStream implementation used in tests. This allows tests to specify
// what data to return for each Read() call.
class MockSourceStream : public SourceStream {
 public:
  enum Mode {
    SYNC,
    ASYNC,
  };
  MockSourceStream();

  MockSourceStream(const MockSourceStream&) = delete;
  MockSourceStream& operator=(const MockSourceStream&) = delete;

  // The destructor will crash in debug build if there is any pending read.
  ~MockSourceStream() override;

  // SourceStream implementation
  int Read(IOBuffer* dest_buffer,
           int buffer_size,
           CompletionOnceCallback callback) override;
  std::string Description() const override;
  bool MayHaveMoreBytes() const override;

  // Enqueues a result to be returned by `Read`. This method does not make a
  // copy of `data`, so `data` must outlive this object. If `mode` is SYNC,
  // `Read` will return the supplied data synchronously; otherwise, consumer
  // needs to call `CompleteNextRead`
  void AddReadResult(base::span<const uint8_t> data, Error error, Mode mode);

  void AddReadResult(std::string_view data, Error error, Mode mode);

  // Completes a pending Read() call. Crash in debug build if there is no
  // pending read.
  void CompleteNextRead();

  // Affects behavior or AddReadResult.  When set to true, each character in
  // |data| passed to AddReadResult will be read as an individual byte, instead
  // of all at once. Default to false.
  // Note that setting it only affects future calls to AddReadResult, not
  // previous ones.
  void set_read_one_byte_at_a_time(bool read_one_byte_at_a_time) {
    read_one_byte_at_a_time_ = read_one_byte_at_a_time;
  }

  void set_always_report_has_more_bytes(bool always_report_has_more_bytes) {
    always_report_has_more_bytes_ = always_report_has_more_bytes;
  }

  // Returns true if a read is waiting to be completed.
  bool awaiting_completion() const { return awaiting_completion_; }

  void set_expect_all_input_consumed(bool expect_all_input_consumed) {
    expect_all_input_consumed_ = expect_all_input_consumed;
  }

 private:
  struct QueuedResult {
    QueuedResult(base::span<const uint8_t> data, Error error, Mode mode);

    const base::raw_span<const uint8_t> data;
    const Error error;
    const Mode mode;
  };

  bool read_one_byte_at_a_time_ = false;
  bool always_report_has_more_bytes_ = true;
  base::queue<QueuedResult> results_;
  bool awaiting_completion_ = false;
  scoped_refptr<IOBuffer> dest_buffer_;
  CompletionOnceCallback callback_;
  size_t dest_buffer_size_ = 0;
  bool expect_all_input_consumed_ = true;
};

}  // namespace net

#endif  // NET_FILTER_MOCK_SOURCE_STREAM_H_
