package com.github.andreyasadchy.xtra.util.chat

import com.github.andreyasadchy.xtra.util.WebSocket
import kotlinx.coroutines.CoroutineScope
import kotlinx.coroutines.launch
import org.json.JSONObject
import java.util.Timer
import javax.net.ssl.X509TrustManager
import kotlin.concurrent.schedule

class EventSubWebSocket(
    private val onConnect: () -> Unit,
    private val onDisconnect: ((String, String) -> Unit),
    private val onWelcomeMessage: (String) -> Unit,
    private val onChatMessage: (JSONObject, String?) -> Unit,
    private val onUserNotice: (JSONObject, String?) -> Unit,
    private val onClearChat: (JSONObject, String?) -> Unit,
    private val onRoomState: (JSONObject, String?) -> Unit,
    private val trustManager: X509TrustManager?,
    private val coroutineScope: CoroutineScope,
) {
    private var webSocket: WebSocket? = null
    private var pongTimer: Timer? = null
    val isActive: Boolean?
        get() = webSocket?.isActive
    private var timeout = 10000L
    private val handledMessageIds = mutableListOf<String>()

    fun connect() {
        webSocket = WebSocket("wss://eventsub.wss.twitch.tv/ws", trustManager, EventSubWebSocketListener())
        coroutineScope.launch {
            webSocket?.start()
        }
    }

    suspend fun disconnect() {
        pongTimer?.cancel()
        webSocket?.stop()
    }

    private fun startPongTimer() {
        pongTimer = Timer().apply {
            schedule(timeout) {
                coroutineScope.launch {
                    webSocket?.disconnect()
                }
            }
        }
    }

    private inner class EventSubWebSocketListener : WebSocket.Listener {
        override fun onOpen(webSocket: WebSocket) {
            onConnect()
        }

        override fun onMessage(webSocket: WebSocket, message: String) {
            try {
                val json = if (message.isNotBlank()) JSONObject(message) else null
                json?.let {
                    val metadata = json.optJSONObject("metadata")
                    val messageId = if (metadata?.isNull("message_id") == false) metadata.optString("message_id").takeIf { it.isNotBlank() } else null
                    val timestamp = if (metadata?.isNull("message_timestamp") == false) metadata.optString("message_timestamp").takeIf { it.isNotBlank() } else null
                    if (!messageId.isNullOrBlank()) {
                        if (handledMessageIds.contains(messageId)) {
                            return
                        } else {
                            if (handledMessageIds.size > 200) {
                                handledMessageIds.removeAt(0)
                            }
                            handledMessageIds.add(messageId)
                        }
                    }
                    when (metadata?.optString("message_type")) {
                        "notification" -> {
                            pongTimer?.cancel()
                            startPongTimer()
                            val payload = json.optJSONObject("payload")
                            val event = payload?.optJSONObject("event")
                            if (event != null) {
                                when (metadata.optString("subscription_type")) {
                                    "channel.chat.message" -> onChatMessage(event, timestamp)
                                    "channel.chat.notification" -> onUserNotice(event, timestamp)
                                    "channel.chat.clear" -> onClearChat(event, timestamp)
                                    "channel.chat_settings.update" -> onRoomState(event, timestamp)
                                }
                            }
                        }
                        "session_keepalive" -> {
                            pongTimer?.cancel()
                            startPongTimer()
                        }
                        "session_reconnect" -> {
                            val payload = json.optJSONObject("payload")
                            val session = payload?.optJSONObject("session")
                            val reconnectUrl = if (session?.isNull("reconnect_url") == false) session.optString("reconnect_url").takeIf { it.isNotBlank() } else null
                            pongTimer?.cancel()
                            coroutineScope.launch {
                                webSocket.disconnect()
                            }
                        }
                        "session_welcome" -> {
                            val payload = json.optJSONObject("payload")
                            val session = payload?.optJSONObject("session")
                            if (session?.isNull("keepalive_timeout_seconds") == false) {
                                session.optInt("keepalive_timeout_seconds").takeIf { it > 0 }?.let { timeout = it * 1000L }
                            }
                            pongTimer?.cancel()
                            startPongTimer()
                            val sessionId = session?.optString("id")
                            if (!sessionId.isNullOrBlank()) {
                                onWelcomeMessage(sessionId)
                            }
                        }
                    }
                }
            } catch (e: Exception) {

            }
        }

        override fun onFailure(webSocket: WebSocket, throwable: Throwable) {
            onDisconnect(throwable.toString(), throwable.stackTraceToString())
        }
    }
}
