/*
 * Geekttrss is a RSS feed reader application on the Android Platform.
 *
 * Copyright (C) 2017-2025 by Frederic-Charles Barthelery.
 *
 * This file is part of Geekttrss.
 *
 * Geekttrss is free software: you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation, either version 3 of the License, or
 * (at your option) any later version.
 *
 * Geekttrss is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with Geekttrss.  If not, see <http://www.gnu.org/licenses/>.
 */

import com.geekorum.build.configureVersionChangeset
import com.geekorum.build.dualTestImplementation
import com.google.protobuf.gradle.GenerateProtoTask
import com.google.protobuf.gradle.id
import org.jetbrains.kotlin.gradle.tasks.AbstractKotlinCompileTool

plugins {
    id("com.geekorum.build.conventions.android-application")
    alias(libs.plugins.kotlin.ksp)
    alias(libs.plugins.com.geekorum.gms.oss.license)
    id("com.geekorum.build.android-tests")
    id("com.geekorum.build.android-avdl")
    alias(libs.plugins.compose.compiler)
    alias(libs.plugins.androidx.navigation.safeargs.kotlin)
    alias(libs.plugins.dagger.hilt.android)
    alias(libs.plugins.androidx.room)
    alias(libs.plugins.google.protobuf)
    alias(libs.plugins.kotlinx.serialization)
}

// workaround bug https://issuetracker.google.com/issues/275534543
// try to remove it on next navigation-safe-args plugin release
buildscript {
    dependencies {
        classpath(libs.android.gradle.plugin)
    }
}

androidComponents {
    val major = 1
    val minor = 6
    val patch = 9
    configureVersionChangeset(project, major, minor, patch)
}

android {
    namespace = "com.geekorum.ttrss"
    defaultConfig {
        applicationId = "com.geekorum.ttrss"
        targetSdk = 36

        sourceSets {
            named("androidTest") {
                assets.srcDir(files("$projectDir/schemas"))
            }
        }
    }

    compileOptions {
        isCoreLibraryDesugaringEnabled = true
    }

    lint {
        abortOnError = false
        checkReleaseBuilds = false
        disable += listOf("MissingTranslation")
    }

    buildTypes {
        named("release") {
            isMinifyEnabled = true
            proguardFiles(getDefaultProguardFile("proguard-android-optimize.txt"),
                "proguard-rules.pro")
        }
    }

    buildFeatures {
        compose = true
        buildConfig = true
    }

    flavorDimensions += "distribution"
    productFlavors {
        register("free") {
            dimension = "distribution"
            applicationIdSuffix = ".free"
            // disable mapping file upload for free flavor as it doesn't work and the free flavor
            // doesn't use crashlytics anyway
        }

        register("google") {
            dimension = "distribution"
            versionNameSuffix = "-google"
        }
    }

    dynamicFeatures += setOf(":manage_feeds")

    packaging {
        // Fix: https://github.com/Kotlin/kotlinx.coroutines/issues/2023
        resources {
            excludes += listOf(
                "META-INF/AL2.0",
                "META-INF/LGPL2.1",
                "META-INF/LICENSE.md",
                "META-INF/LICENSE-notice.md",
                "DebugProbesKt.bin"
            )
        }
    }
}

room {
    schemaDirectory("$projectDir/schemas/")
}

ksp {
    arg("room.incremental", "true")
    arg("room.generateKotlin", "true")
}

protobuf {
    protoc {
        artifact = libs.protobuf.protoc.get().toString()
    }

    generateProtoTasks {
        all().forEach { task ->
            task.builtins {
                id("java") {
                    option("lite")
                }
                id("kotlin") {
                    option("lite")
                }
            }
        }
    }
}

// TODO remove when fixed. Fix ksp/protobuf dependendencies
// https://github.com/google/ksp/issues/1590
androidComponents {
    onVariants(selector().all()) { variant ->
        afterEvaluate {
            val protoTask =
                project.tasks.named<GenerateProtoTask>("generate" + variant.name.replaceFirstChar { it.uppercaseChar() } + "Proto")

            project.tasks.named<AbstractKotlinCompileTool<*>>("ksp" + variant.name.replaceFirstChar { it.uppercaseChar() } + "Kotlin") {
                setSource(protoTask)
            }
        }
    }
}

dependencies {

    implementation(libs.androidx.core)
    implementation(libs.androidx.fragment)
    implementation(libs.androidx.activity)
    implementation(libs.kotlinx.datetime)

    // androidx ui
    implementation(libs.androidx.constraintlayout)
    implementation(libs.androidx.coordinatorlayout)
    implementation(libs.androidx.preferences)

    // compose
    api(platform(libs.androidx.compose.bom))
    api(libs.androidx.compose.ui)
    api(libs.androidx.compose.ui.util)
    api(libs.androidx.compose.foundation)
    api(libs.androidx.compose.material3)
    api(libs.androidx.compose.material.icons.core)
    api(libs.androidx.compose.material.icons.extended)
    api(libs.androidx.compose.material3.window.sizes)
    api(libs.androidx.compose.ui.viewbinding)
    api(libs.androidx.activity.compose)
    api(libs.androidx.compose.runtime.livedata)
    api(libs.androidx.compose.animation.graphics)
    api(libs.androidx.paging.compose)
    api(libs.accompanist.drawablepainter)
    api(libs.androidx.compose.ui.tooling)
    api(libs.coil.compose)
    implementation(libs.materialKolor)
    implementation(libs.androidx.fragment.compose)


    // for layout inspector
    debugImplementation(libs.androidx.compose.ui.tooling)
    implementation(libs.androidx.compose.ui.tooling.preview)

    // androidx others
    implementation(libs.androidx.browser)
    implementation(libs.androidx.startup)

    implementation(enforcedPlatform(libs.kotlinx.serialization.bom))
    implementation(libs.kotlinx.serialization.json)
    implementation(libs.retrofit.kotlinx.serialization.converter)

    //geekdroid
    implementation(libs.geekdroid)
    implementation(libs.aboutoss.ui.material3)

    implementation(project(":htmlparsers"))
    implementation(project(":webapi"))
    implementation(project(":faviKonSnoop"))

    implementation(libs.android.material)

    implementation(enforcedPlatform(libs.okhttp.bom))
    implementation(libs.okhttp.logging.interceptor)
    testImplementation(libs.okhttp.mockwebserver)
    implementation(libs.coil)

    implementation(libs.ksoup)
    implementation(libs.androidx.datastore)
    api(libs.protobuf.kotlin.lite)

    implementation(libs.androidx.lifecycle.livedata.core)
    implementation(libs.androidx.lifecycle.livedata)
    implementation(libs.androidx.lifecycle.viewmodel)
    implementation(libs.androidx.lifecycle.runtime)
    implementation(libs.androidx.lifecycle.runtime.compose)
    implementation(libs.androidx.lifecycle.viewmodel.savedstate)
    dualTestImplementation(libs.androidx.arch.core.testing)
    implementation(libs.androidx.lifecycle.viewmodel.compose)


    implementation(libs.dagger)
    ksp(libs.dagger.compiler)
    kspTest(libs.dagger.compiler)
    implementation(libs.dagger.hilt.android)
    implementation(libs.androidx.hilt.work)
    ksp(libs.androidx.hilt.compiler)
    ksp(libs.dagger.hilt.compiler)
    testImplementation(libs.dagger.hilt.android.testing)
    kspTest(libs.dagger.hilt.compiler)
    androidTestImplementation(libs.dagger.hilt.android.testing)
    kspAndroidTest(libs.dagger.hilt.compiler)
    implementation(libs.androidx.hilt.navigation.compose)


    ksp(libs.androidx.room.compiler)
    androidTestImplementation(libs.androidx.room.testing)

    androidTestImplementation(libs.androidx.work.testing)

    implementation(enforcedPlatform(kotlin("bom")))
    implementation(kotlin("stdlib-jdk8"))

    implementation(enforcedPlatform(libs.kotlinx.coroutines.bom))
    testImplementation(enforcedPlatform(libs.kotlinx.coroutines.bom))
    androidTestImplementation(enforcedPlatform(libs.kotlinx.coroutines.bom))
    implementation(libs.kotlinx.coroutines.core)
    implementation(libs.kotlinx.coroutines.android)
    implementation(libs.kotlinx.coroutines.jdk8)
    testImplementation(libs.kotlinx.coroutines.test)
    testImplementation(libs.turbine)
    androidTestImplementation(libs.kotlinx.coroutines.test)


    add("googleImplementation", libs.google.play.feature.delivery)
    add("googleImplementation", libs.google.play.app.update)
    add("googleImplementation", libs.google.play.review)
    add("googleImplementation", libs.gms.play.services.base)
    add("googleImplementation", libs.androidx.navigation.dynamic.features.fragment)

    // api dependencies for features modules
    api(libs.androidx.appcompat)
    api(libs.androidx.work.runtime)
    api(libs.androidx.room.runtime)
    api(libs.androidx.room.paging)
    api(libs.androidx.room)
    api(libs.androidx.paging)
    api(libs.retrofit)
    api(enforcedPlatform(libs.okhttp.bom))
    api(libs.okhttp)
    api(libs.timber)

    api(libs.androidx.navigation.fragment)
    api(libs.androidx.navigation.ui)
    api(libs.androidx.navigation.compose)

    debugImplementation(libs.androidx.fragment.testing.manifest)

    debugImplementation(libs.leakcanary.android)

    testImplementation(libs.androidx.compose.ui.test.junit4)
    debugImplementation(libs.androidx.compose.ui.test.manifest)

    coreLibraryDesugaring(libs.android.desugar.jdk)
}

//TODO remove. Force room 2.6.1 for now has 2.7.x has critical bug for us
// see  https://issuetracker.google.com/issues/413924560
configurations.all {
    resolutionStrategy {
        force("androidx.room:room-ktx:2.6.1")
        force("androidx.room:room-runtime:2.6.1")
        force("androidx.room:room-paging:2.6.1")
        force("androidx.room:room-testing:2.6.1")
        force("androidx.room:room-compiler:2.6.1")
    }
}

