/*
 * Copyright (c) 2023  Gaurav Ujjwal.
 *
 * SPDX-License-Identifier:  GPL-3.0-or-later
 *
 * See COPYING.txt for more details.
 */

package com.gaurav.avnc.vnc

/**
 * From [1]:
 * "An XT keycode is an XT make scancode sequence encoded to fit in a single U32 quantity.
 * Single byte XT scancodes with a byte value less than 0x7f are encoded as is.
 * 2-byte XT scancodes whose first byte is 0xe0 and second byte is less than 0x7f
 * are encoded with the high bit of the first byte set."
 *
 * [1] https://github.com/rfbproto/rfbproto/blob/master/rfbproto.rst#qemu-extended-key-event-message
 */
object XTKeyCode {

    /**
     * Maps KeyEvent scancodes to XT keycode.
     *
     * KeyEvent scancodes are simply Linux kernel keycodes. Although Android doesn't
     * explicitly states this, it can confirmed by looking through Android source code.
     * In any case, all devices tested so far returns Linux keycodes.
     *
     * Returns 0 if no mapping exist.
     */
    fun fromAndroidScancode(scancode: Int) = LinuxToQnum.getOrNull(scancode) ?: 0

    /**
     * This mapping table is generated using output from the following command:
     *
     *     keymap-gen code-map --lang stdc++ keymaps.csv linux qnum
     *
     * qnum refers to XT keycode described above.
     * Source: https://github.com/qemu/keycodemapdb
     */
    private val LinuxToQnum = intArrayOf(
            0, /* linux:0 (KEY_RESERVED) -> linux:0 (KEY_RESERVED) -> qnum:None */
            0x1, /* linux:1 (KEY_ESC) -> linux:1 (KEY_ESC) -> qnum:1 */
            0x2, /* linux:2 (KEY_1) -> linux:2 (KEY_1) -> qnum:2 */
            0x3, /* linux:3 (KEY_2) -> linux:3 (KEY_2) -> qnum:3 */
            0x4, /* linux:4 (KEY_3) -> linux:4 (KEY_3) -> qnum:4 */
            0x5, /* linux:5 (KEY_4) -> linux:5 (KEY_4) -> qnum:5 */
            0x6, /* linux:6 (KEY_5) -> linux:6 (KEY_5) -> qnum:6 */
            0x7, /* linux:7 (KEY_6) -> linux:7 (KEY_6) -> qnum:7 */
            0x8, /* linux:8 (KEY_7) -> linux:8 (KEY_7) -> qnum:8 */
            0x9, /* linux:9 (KEY_8) -> linux:9 (KEY_8) -> qnum:9 */
            0xa, /* linux:10 (KEY_9) -> linux:10 (KEY_9) -> qnum:10 */
            0xb, /* linux:11 (KEY_0) -> linux:11 (KEY_0) -> qnum:11 */
            0xc, /* linux:12 (KEY_MINUS) -> linux:12 (KEY_MINUS) -> qnum:12 */
            0xd, /* linux:13 (KEY_EQUAL) -> linux:13 (KEY_EQUAL) -> qnum:13 */
            0xe, /* linux:14 (KEY_BACKSPACE) -> linux:14 (KEY_BACKSPACE) -> qnum:14 */
            0xf, /* linux:15 (KEY_TAB) -> linux:15 (KEY_TAB) -> qnum:15 */
            0x10, /* linux:16 (KEY_Q) -> linux:16 (KEY_Q) -> qnum:16 */
            0x11, /* linux:17 (KEY_W) -> linux:17 (KEY_W) -> qnum:17 */
            0x12, /* linux:18 (KEY_E) -> linux:18 (KEY_E) -> qnum:18 */
            0x13, /* linux:19 (KEY_R) -> linux:19 (KEY_R) -> qnum:19 */
            0x14, /* linux:20 (KEY_T) -> linux:20 (KEY_T) -> qnum:20 */
            0x15, /* linux:21 (KEY_Y) -> linux:21 (KEY_Y) -> qnum:21 */
            0x16, /* linux:22 (KEY_U) -> linux:22 (KEY_U) -> qnum:22 */
            0x17, /* linux:23 (KEY_I) -> linux:23 (KEY_I) -> qnum:23 */
            0x18, /* linux:24 (KEY_O) -> linux:24 (KEY_O) -> qnum:24 */
            0x19, /* linux:25 (KEY_P) -> linux:25 (KEY_P) -> qnum:25 */
            0x1a, /* linux:26 (KEY_LEFTBRACE) -> linux:26 (KEY_LEFTBRACE) -> qnum:26 */
            0x1b, /* linux:27 (KEY_RIGHTBRACE) -> linux:27 (KEY_RIGHTBRACE) -> qnum:27 */
            0x1c, /* linux:28 (KEY_ENTER) -> linux:28 (KEY_ENTER) -> qnum:28 */
            0x1d, /* linux:29 (KEY_LEFTCTRL) -> linux:29 (KEY_LEFTCTRL) -> qnum:29 */
            0x1e, /* linux:30 (KEY_A) -> linux:30 (KEY_A) -> qnum:30 */
            0x1f, /* linux:31 (KEY_S) -> linux:31 (KEY_S) -> qnum:31 */
            0x20, /* linux:32 (KEY_D) -> linux:32 (KEY_D) -> qnum:32 */
            0x21, /* linux:33 (KEY_F) -> linux:33 (KEY_F) -> qnum:33 */
            0x22, /* linux:34 (KEY_G) -> linux:34 (KEY_G) -> qnum:34 */
            0x23, /* linux:35 (KEY_H) -> linux:35 (KEY_H) -> qnum:35 */
            0x24, /* linux:36 (KEY_J) -> linux:36 (KEY_J) -> qnum:36 */
            0x25, /* linux:37 (KEY_K) -> linux:37 (KEY_K) -> qnum:37 */
            0x26, /* linux:38 (KEY_L) -> linux:38 (KEY_L) -> qnum:38 */
            0x27, /* linux:39 (KEY_SEMICOLON) -> linux:39 (KEY_SEMICOLON) -> qnum:39 */
            0x28, /* linux:40 (KEY_APOSTROPHE) -> linux:40 (KEY_APOSTROPHE) -> qnum:40 */
            0x29, /* linux:41 (KEY_GRAVE) -> linux:41 (KEY_GRAVE) -> qnum:41 */
            0x2a, /* linux:42 (KEY_LEFTSHIFT) -> linux:42 (KEY_LEFTSHIFT) -> qnum:42 */
            0x2b, /* linux:43 (KEY_BACKSLASH) -> linux:43 (KEY_BACKSLASH) -> qnum:43 */
            0x2c, /* linux:44 (KEY_Z) -> linux:44 (KEY_Z) -> qnum:44 */
            0x2d, /* linux:45 (KEY_X) -> linux:45 (KEY_X) -> qnum:45 */
            0x2e, /* linux:46 (KEY_C) -> linux:46 (KEY_C) -> qnum:46 */
            0x2f, /* linux:47 (KEY_V) -> linux:47 (KEY_V) -> qnum:47 */
            0x30, /* linux:48 (KEY_B) -> linux:48 (KEY_B) -> qnum:48 */
            0x31, /* linux:49 (KEY_N) -> linux:49 (KEY_N) -> qnum:49 */
            0x32, /* linux:50 (KEY_M) -> linux:50 (KEY_M) -> qnum:50 */
            0x33, /* linux:51 (KEY_COMMA) -> linux:51 (KEY_COMMA) -> qnum:51 */
            0x34, /* linux:52 (KEY_DOT) -> linux:52 (KEY_DOT) -> qnum:52 */
            0x35, /* linux:53 (KEY_SLASH) -> linux:53 (KEY_SLASH) -> qnum:53 */
            0x36, /* linux:54 (KEY_RIGHTSHIFT) -> linux:54 (KEY_RIGHTSHIFT) -> qnum:54 */
            0x37, /* linux:55 (KEY_KPASTERISK) -> linux:55 (KEY_KPASTERISK) -> qnum:55 */
            0x38, /* linux:56 (KEY_LEFTALT) -> linux:56 (KEY_LEFTALT) -> qnum:56 */
            0x39, /* linux:57 (KEY_SPACE) -> linux:57 (KEY_SPACE) -> qnum:57 */
            0x3a, /* linux:58 (KEY_CAPSLOCK) -> linux:58 (KEY_CAPSLOCK) -> qnum:58 */
            0x3b, /* linux:59 (KEY_F1) -> linux:59 (KEY_F1) -> qnum:59 */
            0x3c, /* linux:60 (KEY_F2) -> linux:60 (KEY_F2) -> qnum:60 */
            0x3d, /* linux:61 (KEY_F3) -> linux:61 (KEY_F3) -> qnum:61 */
            0x3e, /* linux:62 (KEY_F4) -> linux:62 (KEY_F4) -> qnum:62 */
            0x3f, /* linux:63 (KEY_F5) -> linux:63 (KEY_F5) -> qnum:63 */
            0x40, /* linux:64 (KEY_F6) -> linux:64 (KEY_F6) -> qnum:64 */
            0x41, /* linux:65 (KEY_F7) -> linux:65 (KEY_F7) -> qnum:65 */
            0x42, /* linux:66 (KEY_F8) -> linux:66 (KEY_F8) -> qnum:66 */
            0x43, /* linux:67 (KEY_F9) -> linux:67 (KEY_F9) -> qnum:67 */
            0x44, /* linux:68 (KEY_F10) -> linux:68 (KEY_F10) -> qnum:68 */
            0x45, /* linux:69 (KEY_NUMLOCK) -> linux:69 (KEY_NUMLOCK) -> qnum:69 */
            0x46, /* linux:70 (KEY_SCROLLLOCK) -> linux:70 (KEY_SCROLLLOCK) -> qnum:70 */
            0x47, /* linux:71 (KEY_KP7) -> linux:71 (KEY_KP7) -> qnum:71 */
            0x48, /* linux:72 (KEY_KP8) -> linux:72 (KEY_KP8) -> qnum:72 */
            0x49, /* linux:73 (KEY_KP9) -> linux:73 (KEY_KP9) -> qnum:73 */
            0x4a, /* linux:74 (KEY_KPMINUS) -> linux:74 (KEY_KPMINUS) -> qnum:74 */
            0x4b, /* linux:75 (KEY_KP4) -> linux:75 (KEY_KP4) -> qnum:75 */
            0x4c, /* linux:76 (KEY_KP5) -> linux:76 (KEY_KP5) -> qnum:76 */
            0x4d, /* linux:77 (KEY_KP6) -> linux:77 (KEY_KP6) -> qnum:77 */
            0x4e, /* linux:78 (KEY_KPPLUS) -> linux:78 (KEY_KPPLUS) -> qnum:78 */
            0x4f, /* linux:79 (KEY_KP1) -> linux:79 (KEY_KP1) -> qnum:79 */
            0x50, /* linux:80 (KEY_KP2) -> linux:80 (KEY_KP2) -> qnum:80 */
            0x51, /* linux:81 (KEY_KP3) -> linux:81 (KEY_KP3) -> qnum:81 */
            0x52, /* linux:82 (KEY_KP0) -> linux:82 (KEY_KP0) -> qnum:82 */
            0x53, /* linux:83 (KEY_KPDOT) -> linux:83 (KEY_KPDOT) -> qnum:83 */
            0x54, /* linux:84 (unnamed) -> linux:84 (unnamed) -> qnum:84 */
            0x76, /* linux:85 (KEY_ZENKAKUHANKAKU) -> linux:85 (KEY_ZENKAKUHANKAKU) -> qnum:118 */
            0x56, /* linux:86 (KEY_102ND) -> linux:86 (KEY_102ND) -> qnum:86 */
            0x57, /* linux:87 (KEY_F11) -> linux:87 (KEY_F11) -> qnum:87 */
            0x58, /* linux:88 (KEY_F12) -> linux:88 (KEY_F12) -> qnum:88 */
            0x73, /* linux:89 (KEY_RO) -> linux:89 (KEY_RO) -> qnum:115 */
            0x78, /* linux:90 (KEY_KATAKANA) -> linux:90 (KEY_KATAKANA) -> qnum:120 */
            0x77, /* linux:91 (KEY_HIRAGANA) -> linux:91 (KEY_HIRAGANA) -> qnum:119 */
            0x79, /* linux:92 (KEY_HENKAN) -> linux:92 (KEY_HENKAN) -> qnum:121 */
            0x70, /* linux:93 (KEY_KATAKANAHIRAGANA) -> linux:93 (KEY_KATAKANAHIRAGANA) -> qnum:112 */
            0x7b, /* linux:94 (KEY_MUHENKAN) -> linux:94 (KEY_MUHENKAN) -> qnum:123 */
            0x5c, /* linux:95 (KEY_KPJPCOMMA) -> linux:95 (KEY_KPJPCOMMA) -> qnum:92 */
            0x9c, /* linux:96 (KEY_KPENTER) -> linux:96 (KEY_KPENTER) -> qnum:156 */
            0x9d, /* linux:97 (KEY_RIGHTCTRL) -> linux:97 (KEY_RIGHTCTRL) -> qnum:157 */
            0xb5, /* linux:98 (KEY_KPSLASH) -> linux:98 (KEY_KPSLASH) -> qnum:181 */
            0x54, /* linux:99 (KEY_SYSRQ) -> linux:99 (KEY_SYSRQ) -> qnum:84 */
            0xb8, /* linux:100 (KEY_RIGHTALT) -> linux:100 (KEY_RIGHTALT) -> qnum:184 */
            0x5b, /* linux:101 (KEY_LINEFEED) -> linux:101 (KEY_LINEFEED) -> qnum:91 */
            0xc7, /* linux:102 (KEY_HOME) -> linux:102 (KEY_HOME) -> qnum:199 */
            0xc8, /* linux:103 (KEY_UP) -> linux:103 (KEY_UP) -> qnum:200 */
            0xc9, /* linux:104 (KEY_PAGEUP) -> linux:104 (KEY_PAGEUP) -> qnum:201 */
            0xcb, /* linux:105 (KEY_LEFT) -> linux:105 (KEY_LEFT) -> qnum:203 */
            0xcd, /* linux:106 (KEY_RIGHT) -> linux:106 (KEY_RIGHT) -> qnum:205 */
            0xcf, /* linux:107 (KEY_END) -> linux:107 (KEY_END) -> qnum:207 */
            0xd0, /* linux:108 (KEY_DOWN) -> linux:108 (KEY_DOWN) -> qnum:208 */
            0xd1, /* linux:109 (KEY_PAGEDOWN) -> linux:109 (KEY_PAGEDOWN) -> qnum:209 */
            0xd2, /* linux:110 (KEY_INSERT) -> linux:110 (KEY_INSERT) -> qnum:210 */
            0xd3, /* linux:111 (KEY_DELETE) -> linux:111 (KEY_DELETE) -> qnum:211 */
            0xef, /* linux:112 (KEY_MACRO) -> linux:112 (KEY_MACRO) -> qnum:239 */
            0xa0, /* linux:113 (KEY_MUTE) -> linux:113 (KEY_MUTE) -> qnum:160 */
            0xae, /* linux:114 (KEY_VOLUMEDOWN) -> linux:114 (KEY_VOLUMEDOWN) -> qnum:174 */
            0xb0, /* linux:115 (KEY_VOLUMEUP) -> linux:115 (KEY_VOLUMEUP) -> qnum:176 */
            0xde, /* linux:116 (KEY_POWER) -> linux:116 (KEY_POWER) -> qnum:222 */
            0x59, /* linux:117 (KEY_KPEQUAL) -> linux:117 (KEY_KPEQUAL) -> qnum:89 */
            0xce, /* linux:118 (KEY_KPPLUSMINUS) -> linux:118 (KEY_KPPLUSMINUS) -> qnum:206 */
            0xc6, /* linux:119 (KEY_PAUSE) -> linux:119 (KEY_PAUSE) -> qnum:198 */
            0x8b, /* linux:120 (KEY_SCALE) -> linux:120 (KEY_SCALE) -> qnum:139 */
            0x7e, /* linux:121 (KEY_KPCOMMA) -> linux:121 (KEY_KPCOMMA) -> qnum:126 */
            0x72, /* linux:122 (KEY_HANGEUL) -> linux:122 (KEY_HANGEUL) -> qnum:114 */
            0x71, /* linux:123 (KEY_HANJA) -> linux:123 (KEY_HANJA) -> qnum:113 */
            0x7d, /* linux:124 (KEY_YEN) -> linux:124 (KEY_YEN) -> qnum:125 */
            0xdb, /* linux:125 (KEY_LEFTMETA) -> linux:125 (KEY_LEFTMETA) -> qnum:219 */
            0xdc, /* linux:126 (KEY_RIGHTMETA) -> linux:126 (KEY_RIGHTMETA) -> qnum:220 */
            0xdd, /* linux:127 (KEY_COMPOSE) -> linux:127 (KEY_COMPOSE) -> qnum:221 */
            0xe8, /* linux:128 (KEY_STOP) -> linux:128 (KEY_STOP) -> qnum:232 */
            0x85, /* linux:129 (KEY_AGAIN) -> linux:129 (KEY_AGAIN) -> qnum:133 */
            0x86, /* linux:130 (KEY_PROPS) -> linux:130 (KEY_PROPS) -> qnum:134 */
            0x87, /* linux:131 (KEY_UNDO) -> linux:131 (KEY_UNDO) -> qnum:135 */
            0x8c, /* linux:132 (KEY_FRONT) -> linux:132 (KEY_FRONT) -> qnum:140 */
            0xf8, /* linux:133 (KEY_COPY) -> linux:133 (KEY_COPY) -> qnum:248 */
            0x64, /* linux:134 (KEY_OPEN) -> linux:134 (KEY_OPEN) -> qnum:100 */
            0x65, /* linux:135 (KEY_PASTE) -> linux:135 (KEY_PASTE) -> qnum:101 */
            0xc1, /* linux:136 (KEY_FIND) -> linux:136 (KEY_FIND) -> qnum:193 */
            0xbc, /* linux:137 (KEY_CUT) -> linux:137 (KEY_CUT) -> qnum:188 */
            0xf5, /* linux:138 (KEY_HELP) -> linux:138 (KEY_HELP) -> qnum:245 */
            0x9e, /* linux:139 (KEY_MENU) -> linux:139 (KEY_MENU) -> qnum:158 */
            0xa1, /* linux:140 (KEY_CALC) -> linux:140 (KEY_CALC) -> qnum:161 */
            0x66, /* linux:141 (KEY_SETUP) -> linux:141 (KEY_SETUP) -> qnum:102 */
            0xdf, /* linux:142 (KEY_SLEEP) -> linux:142 (KEY_SLEEP) -> qnum:223 */
            0xe3, /* linux:143 (KEY_WAKEUP) -> linux:143 (KEY_WAKEUP) -> qnum:227 */
            0x67, /* linux:144 (KEY_FILE) -> linux:144 (KEY_FILE) -> qnum:103 */
            0x68, /* linux:145 (KEY_SENDFILE) -> linux:145 (KEY_SENDFILE) -> qnum:104 */
            0x69, /* linux:146 (KEY_DELETEFILE) -> linux:146 (KEY_DELETEFILE) -> qnum:105 */
            0x93, /* linux:147 (KEY_XFER) -> linux:147 (KEY_XFER) -> qnum:147 */
            0x9f, /* linux:148 (KEY_PROG1) -> linux:148 (KEY_PROG1) -> qnum:159 */
            0x97, /* linux:149 (KEY_PROG2) -> linux:149 (KEY_PROG2) -> qnum:151 */
            0x82, /* linux:150 (KEY_WWW) -> linux:150 (KEY_WWW) -> qnum:130 */
            0x6a, /* linux:151 (KEY_MSDOS) -> linux:151 (KEY_MSDOS) -> qnum:106 */
            0x92, /* linux:152 (KEY_SCREENLOCK) -> linux:152 (KEY_SCREENLOCK) -> qnum:146 */
            0x6b, /* linux:153 (KEY_DIRECTION) -> linux:153 (KEY_DIRECTION) -> qnum:107 */
            0xa6, /* linux:154 (KEY_CYCLEWINDOWS) -> linux:154 (KEY_CYCLEWINDOWS) -> qnum:166 */
            0xec, /* linux:155 (KEY_MAIL) -> linux:155 (KEY_MAIL) -> qnum:236 */
            0xe6, /* linux:156 (KEY_BOOKMARKS) -> linux:156 (KEY_BOOKMARKS) -> qnum:230 */
            0xeb, /* linux:157 (KEY_COMPUTER) -> linux:157 (KEY_COMPUTER) -> qnum:235 */
            0xea, /* linux:158 (KEY_BACK) -> linux:158 (KEY_BACK) -> qnum:234 */
            0xe9, /* linux:159 (KEY_FORWARD) -> linux:159 (KEY_FORWARD) -> qnum:233 */
            0xa3, /* linux:160 (KEY_CLOSECD) -> linux:160 (KEY_CLOSECD) -> qnum:163 */
            0x6c, /* linux:161 (KEY_EJECTCD) -> linux:161 (KEY_EJECTCD) -> qnum:108 */
            0xfd, /* linux:162 (KEY_EJECTCLOSECD) -> linux:162 (KEY_EJECTCLOSECD) -> qnum:253 */
            0x99, /* linux:163 (KEY_NEXTSONG) -> linux:163 (KEY_NEXTSONG) -> qnum:153 */
            0xa2, /* linux:164 (KEY_PLAYPAUSE) -> linux:164 (KEY_PLAYPAUSE) -> qnum:162 */
            0x90, /* linux:165 (KEY_PREVIOUSSONG) -> linux:165 (KEY_PREVIOUSSONG) -> qnum:144 */
            0xa4, /* linux:166 (KEY_STOPCD) -> linux:166 (KEY_STOPCD) -> qnum:164 */
            0xb1, /* linux:167 (KEY_RECORD) -> linux:167 (KEY_RECORD) -> qnum:177 */
            0x98, /* linux:168 (KEY_REWIND) -> linux:168 (KEY_REWIND) -> qnum:152 */
            0x63, /* linux:169 (KEY_PHONE) -> linux:169 (KEY_PHONE) -> qnum:99 */
            0, /* linux:170 (KEY_ISO) -> linux:170 (KEY_ISO) -> qnum:None */
            0x81, /* linux:171 (KEY_CONFIG) -> linux:171 (KEY_CONFIG) -> qnum:129 */
            0xb2, /* linux:172 (KEY_HOMEPAGE) -> linux:172 (KEY_HOMEPAGE) -> qnum:178 */
            0xe7, /* linux:173 (KEY_REFRESH) -> linux:173 (KEY_REFRESH) -> qnum:231 */
            0, /* linux:174 (KEY_EXIT) -> linux:174 (KEY_EXIT) -> qnum:None */
            0, /* linux:175 (KEY_MOVE) -> linux:175 (KEY_MOVE) -> qnum:None */
            0x88, /* linux:176 (KEY_EDIT) -> linux:176 (KEY_EDIT) -> qnum:136 */
            0x75, /* linux:177 (KEY_SCROLLUP) -> linux:177 (KEY_SCROLLUP) -> qnum:117 */
            0x8f, /* linux:178 (KEY_SCROLLDOWN) -> linux:178 (KEY_SCROLLDOWN) -> qnum:143 */
            0xf6, /* linux:179 (KEY_KPLEFTPAREN) -> linux:179 (KEY_KPLEFTPAREN) -> qnum:246 */
            0xfb, /* linux:180 (KEY_KPRIGHTPAREN) -> linux:180 (KEY_KPRIGHTPAREN) -> qnum:251 */
            0x89, /* linux:181 (KEY_NEW) -> linux:181 (KEY_NEW) -> qnum:137 */
            0x8a, /* linux:182 (KEY_REDO) -> linux:182 (KEY_REDO) -> qnum:138 */
            0x5d, /* linux:183 (KEY_F13) -> linux:183 (KEY_F13) -> qnum:93 */
            0x5e, /* linux:184 (KEY_F14) -> linux:184 (KEY_F14) -> qnum:94 */
            0x5f, /* linux:185 (KEY_F15) -> linux:185 (KEY_F15) -> qnum:95 */
            0x55, /* linux:186 (KEY_F16) -> linux:186 (KEY_F16) -> qnum:85 */
            0x83, /* linux:187 (KEY_F17) -> linux:187 (KEY_F17) -> qnum:131 */
            0xf7, /* linux:188 (KEY_F18) -> linux:188 (KEY_F18) -> qnum:247 */
            0x84, /* linux:189 (KEY_F19) -> linux:189 (KEY_F19) -> qnum:132 */
            0x5a, /* linux:190 (KEY_F20) -> linux:190 (KEY_F20) -> qnum:90 */
            0x74, /* linux:191 (KEY_F21) -> linux:191 (KEY_F21) -> qnum:116 */
            0xf9, /* linux:192 (KEY_F22) -> linux:192 (KEY_F22) -> qnum:249 */
            0x6d, /* linux:193 (KEY_F23) -> linux:193 (KEY_F23) -> qnum:109 */
            0x6f, /* linux:194 (KEY_F24) -> linux:194 (KEY_F24) -> qnum:111 */
            0x95, /* linux:195 (unnamed) -> linux:195 (unnamed) -> qnum:149 */
            0x96, /* linux:196 (unnamed) -> linux:196 (unnamed) -> qnum:150 */
            0x9a, /* linux:197 (unnamed) -> linux:197 (unnamed) -> qnum:154 */
            0x9b, /* linux:198 (unnamed) -> linux:198 (unnamed) -> qnum:155 */
            0xa7, /* linux:199 (unnamed) -> linux:199 (unnamed) -> qnum:167 */
            0xa8, /* linux:200 (KEY_PLAYCD) -> linux:200 (KEY_PLAYCD) -> qnum:168 */
            0xa9, /* linux:201 (KEY_PAUSECD) -> linux:201 (KEY_PAUSECD) -> qnum:169 */
            0xab, /* linux:202 (KEY_PROG3) -> linux:202 (KEY_PROG3) -> qnum:171 */
            0xac, /* linux:203 (KEY_PROG4) -> linux:203 (KEY_PROG4) -> qnum:172 */
            0xad, /* linux:204 (KEY_DASHBOARD) -> linux:204 (KEY_DASHBOARD) -> qnum:173 */
            0xa5, /* linux:205 (KEY_SUSPEND) -> linux:205 (KEY_SUSPEND) -> qnum:165 */
            0xaf, /* linux:206 (KEY_CLOSE) -> linux:206 (KEY_CLOSE) -> qnum:175 */
            0xb3, /* linux:207 (KEY_PLAY) -> linux:207 (KEY_PLAY) -> qnum:179 */
            0xb4, /* linux:208 (KEY_FASTFORWARD) -> linux:208 (KEY_FASTFORWARD) -> qnum:180 */
            0xb6, /* linux:209 (KEY_BASSBOOST) -> linux:209 (KEY_BASSBOOST) -> qnum:182 */
            0xb9, /* linux:210 (KEY_PRINT) -> linux:210 (KEY_PRINT) -> qnum:185 */
            0xba, /* linux:211 (KEY_HP) -> linux:211 (KEY_HP) -> qnum:186 */
            0xbb, /* linux:212 (KEY_CAMERA) -> linux:212 (KEY_CAMERA) -> qnum:187 */
            0xbd, /* linux:213 (KEY_SOUND) -> linux:213 (KEY_SOUND) -> qnum:189 */
            0xbe, /* linux:214 (KEY_QUESTION) -> linux:214 (KEY_QUESTION) -> qnum:190 */
            0xbf, /* linux:215 (KEY_EMAIL) -> linux:215 (KEY_EMAIL) -> qnum:191 */
            0xc0, /* linux:216 (KEY_CHAT) -> linux:216 (KEY_CHAT) -> qnum:192 */
            0xe5, /* linux:217 (KEY_SEARCH) -> linux:217 (KEY_SEARCH) -> qnum:229 */
            0xc2, /* linux:218 (KEY_CONNECT) -> linux:218 (KEY_CONNECT) -> qnum:194 */
            0xc3, /* linux:219 (KEY_FINANCE) -> linux:219 (KEY_FINANCE) -> qnum:195 */
            0xc4, /* linux:220 (KEY_SPORT) -> linux:220 (KEY_SPORT) -> qnum:196 */
            0xc5, /* linux:221 (KEY_SHOP) -> linux:221 (KEY_SHOP) -> qnum:197 */
            0x94, /* linux:222 (KEY_ALTERASE) -> linux:222 (KEY_ALTERASE) -> qnum:148 */
            0xca, /* linux:223 (KEY_CANCEL) -> linux:223 (KEY_CANCEL) -> qnum:202 */
            0xcc, /* linux:224 (KEY_BRIGHTNESSDOWN) -> linux:224 (KEY_BRIGHTNESSDOWN) -> qnum:204 */
            0xd4, /* linux:225 (KEY_BRIGHTNESSUP) -> linux:225 (KEY_BRIGHTNESSUP) -> qnum:212 */
            0xed, /* linux:226 (KEY_MEDIA) -> linux:226 (KEY_MEDIA) -> qnum:237 */
            0xd6, /* linux:227 (KEY_SWITCHVIDEOMODE) -> linux:227 (KEY_SWITCHVIDEOMODE) -> qnum:214 */
            0xd7, /* linux:228 (KEY_KBDILLUMTOGGLE) -> linux:228 (KEY_KBDILLUMTOGGLE) -> qnum:215 */
            0xd8, /* linux:229 (KEY_KBDILLUMDOWN) -> linux:229 (KEY_KBDILLUMDOWN) -> qnum:216 */
            0xd9, /* linux:230 (KEY_KBDILLUMUP) -> linux:230 (KEY_KBDILLUMUP) -> qnum:217 */
            0xda, /* linux:231 (KEY_SEND) -> linux:231 (KEY_SEND) -> qnum:218 */
            0xe4, /* linux:232 (KEY_REPLY) -> linux:232 (KEY_REPLY) -> qnum:228 */
            0x8e, /* linux:233 (KEY_FORWARDMAIL) -> linux:233 (KEY_FORWARDMAIL) -> qnum:142 */
            0xd5, /* linux:234 (KEY_SAVE) -> linux:234 (KEY_SAVE) -> qnum:213 */
            0xf0, /* linux:235 (KEY_DOCUMENTS) -> linux:235 (KEY_DOCUMENTS) -> qnum:240 */
            0xf1, /* linux:236 (KEY_BATTERY) -> linux:236 (KEY_BATTERY) -> qnum:241 */
            0xf2, /* linux:237 (KEY_BLUETOOTH) -> linux:237 (KEY_BLUETOOTH) -> qnum:242 */
            0xf3, /* linux:238 (KEY_WLAN) -> linux:238 (KEY_WLAN) -> qnum:243 */
            0xf4, /* linux:239 (KEY_UWB) -> linux:239 (KEY_UWB) -> qnum:244 */


            /*   Rest of the keycodes have no mapping so far

            0, /* linux:240 (KEY_UNKNOWN) -> linux:240 (KEY_UNKNOWN) -> qnum:None */
            0, /* linux:241 (KEY_VIDEO_NEXT) -> linux:241 (KEY_VIDEO_NEXT) -> qnum:None */
            0, /* linux:242 (KEY_VIDEO_PREV) -> linux:242 (KEY_VIDEO_PREV) -> qnum:None */
            0, /* linux:243 (KEY_BRIGHTNESS_CYCLE) -> linux:243 (KEY_BRIGHTNESS_CYCLE) -> qnum:None */
            0, /* linux:244 (KEY_BRIGHTNESS_ZERO) -> linux:244 (KEY_BRIGHTNESS_ZERO) -> qnum:None */
            0, /* linux:245 (KEY_DISPLAY_OFF) -> linux:245 (KEY_DISPLAY_OFF) -> qnum:None */
            0, /* linux:246 (KEY_WIMAX) -> linux:246 (KEY_WIMAX) -> qnum:None */
            0, /* linux:247 (unnamed) -> linux:247 (unnamed) -> qnum:None */
            0, /* linux:248 (unnamed) -> linux:248 (unnamed) -> qnum:None */
            0, /* linux:249 (unnamed) -> linux:249 (unnamed) -> qnum:None */
            0, /* linux:250 (unnamed) -> linux:250 (unnamed) -> qnum:None */
            0, /* linux:251 (unnamed) -> linux:251 (unnamed) -> qnum:None */
            0, /* linux:252 (unnamed) -> linux:252 (unnamed) -> qnum:None */
            0, /* linux:253 (unnamed) -> linux:253 (unnamed) -> qnum:None */
            0, /* linux:254 (unnamed) -> linux:254 (unnamed) -> qnum:None */
            0, /* linux:255 (unnamed) -> linux:255 (unnamed) -> qnum:None */
            0, /* linux:256 (BTN_0) -> linux:256 (BTN_0) -> qnum:None */
            0, /* linux:257 (BTN_1) -> linux:257 (BTN_1) -> qnum:None */
            0, /* linux:258 (BTN_2) -> linux:258 (BTN_2) -> qnum:None */
            0, /* linux:259 (BTN_3) -> linux:259 (BTN_3) -> qnum:None */
            0, /* linux:260 (BTN_4) -> linux:260 (BTN_4) -> qnum:None */
            0, /* linux:261 (BTN_5) -> linux:261 (BTN_5) -> qnum:None */
            0, /* linux:262 (BTN_6) -> linux:262 (BTN_6) -> qnum:None */
            0, /* linux:263 (BTN_7) -> linux:263 (BTN_7) -> qnum:None */
            0, /* linux:264 (BTN_8) -> linux:264 (BTN_8) -> qnum:None */
            0, /* linux:265 (BTN_9) -> linux:265 (BTN_9) -> qnum:None */
            0, /* linux:266 (unnamed) -> linux:None (unnamed) -> qnum:None */
            0, /* linux:267 (unnamed) -> linux:None (unnamed) -> qnum:None */
            0, /* linux:268 (unnamed) -> linux:None (unnamed) -> qnum:None */
            0, /* linux:269 (unnamed) -> linux:None (unnamed) -> qnum:None */
            0, /* linux:270 (unnamed) -> linux:None (unnamed) -> qnum:None */
            0, /* linux:271 (unnamed) -> linux:None (unnamed) -> qnum:None */
            0, /* linux:272 (BTN_LEFT) -> linux:272 (BTN_LEFT) -> qnum:None */
            0, /* linux:273 (BTN_RIGHT) -> linux:273 (BTN_RIGHT) -> qnum:None */
            0, /* linux:274 (BTN_MIDDLE) -> linux:274 (BTN_MIDDLE) -> qnum:None */
            0, /* linux:275 (BTN_SIDE) -> linux:275 (BTN_SIDE) -> qnum:None */
            0, /* linux:276 (BTN_EXTRA) -> linux:276 (BTN_EXTRA) -> qnum:None */
            0, /* linux:277 (BTN_FORWARD) -> linux:277 (BTN_FORWARD) -> qnum:None */
            0, /* linux:278 (BTN_BACK) -> linux:278 (BTN_BACK) -> qnum:None */
            0, /* linux:279 (BTN_TASK) -> linux:279 (BTN_TASK) -> qnum:None */
            0, /* linux:280 (unnamed) -> linux:None (unnamed) -> qnum:None */
            0, /* linux:281 (unnamed) -> linux:None (unnamed) -> qnum:None */
            0, /* linux:282 (unnamed) -> linux:None (unnamed) -> qnum:None */
            0, /* linux:283 (unnamed) -> linux:None (unnamed) -> qnum:None */
            0, /* linux:284 (unnamed) -> linux:None (unnamed) -> qnum:None */
            0, /* linux:285 (unnamed) -> linux:None (unnamed) -> qnum:None */
            0, /* linux:286 (unnamed) -> linux:None (unnamed) -> qnum:None */
            0, /* linux:287 (unnamed) -> linux:None (unnamed) -> qnum:None */
            0, /* linux:288 (BTN_TRIGGER) -> linux:288 (BTN_TRIGGER) -> qnum:None */
            0, /* linux:289 (BTN_THUMB) -> linux:289 (BTN_THUMB) -> qnum:None */
            0, /* linux:290 (BTN_THUMB2) -> linux:290 (BTN_THUMB2) -> qnum:None */
            0, /* linux:291 (BTN_TOP) -> linux:291 (BTN_TOP) -> qnum:None */
            0, /* linux:292 (BTN_TOP2) -> linux:292 (BTN_TOP2) -> qnum:None */
            0, /* linux:293 (BTN_PINKIE) -> linux:293 (BTN_PINKIE) -> qnum:None */
            0, /* linux:294 (BTN_BASE) -> linux:294 (BTN_BASE) -> qnum:None */
            0, /* linux:295 (BTN_BASE2) -> linux:295 (BTN_BASE2) -> qnum:None */
            0, /* linux:296 (BTN_BASE3) -> linux:296 (BTN_BASE3) -> qnum:None */
            0, /* linux:297 (BTN_BASE4) -> linux:297 (BTN_BASE4) -> qnum:None */
            0, /* linux:298 (BTN_BASE5) -> linux:298 (BTN_BASE5) -> qnum:None */
            0, /* linux:299 (BTN_BASE6) -> linux:299 (BTN_BASE6) -> qnum:None */
            0, /* linux:300 (unnamed) -> linux:None (unnamed) -> qnum:None */
            0, /* linux:301 (unnamed) -> linux:None (unnamed) -> qnum:None */
            0, /* linux:302 (unnamed) -> linux:None (unnamed) -> qnum:None */
            0, /* linux:303 (BTN_DEAD) -> linux:303 (BTN_DEAD) -> qnum:None */
            0, /* linux:304 (BTN_A) -> linux:304 (BTN_A) -> qnum:None */
            0, /* linux:305 (BTN_B) -> linux:305 (BTN_B) -> qnum:None */
            0, /* linux:306 (BTN_C) -> linux:306 (BTN_C) -> qnum:None */
            0, /* linux:307 (BTN_X) -> linux:307 (BTN_X) -> qnum:None */
            0, /* linux:308 (BTN_Y) -> linux:308 (BTN_Y) -> qnum:None */
            0, /* linux:309 (BTN_Z) -> linux:309 (BTN_Z) -> qnum:None */
            0, /* linux:310 (BTN_TL) -> linux:310 (BTN_TL) -> qnum:None */
            0, /* linux:311 (BTN_TR) -> linux:311 (BTN_TR) -> qnum:None */
            0, /* linux:312 (BTN_TL2) -> linux:312 (BTN_TL2) -> qnum:None */
            0, /* linux:313 (BTN_TR2) -> linux:313 (BTN_TR2) -> qnum:None */
            0, /* linux:314 (BTN_SELECT) -> linux:314 (BTN_SELECT) -> qnum:None */
            0, /* linux:315 (BTN_START) -> linux:315 (BTN_START) -> qnum:None */
            0, /* linux:316 (BTN_MODE) -> linux:316 (BTN_MODE) -> qnum:None */
            0, /* linux:317 (BTN_THUMBL) -> linux:317 (BTN_THUMBL) -> qnum:None */
            0, /* linux:318 (BTN_THUMBR) -> linux:318 (BTN_THUMBR) -> qnum:None */
            0, /* linux:319 (unnamed) -> linux:None (unnamed) -> qnum:None */
            0, /* linux:320 (BTN_TOOL_PEN) -> linux:320 (BTN_TOOL_PEN) -> qnum:None */
            0, /* linux:321 (BTN_TOOL_RUBBER) -> linux:321 (BTN_TOOL_RUBBER) -> qnum:None */
            0, /* linux:322 (BTN_TOOL_BRUSH) -> linux:322 (BTN_TOOL_BRUSH) -> qnum:None */
            0, /* linux:323 (BTN_TOOL_PENCIL) -> linux:323 (BTN_TOOL_PENCIL) -> qnum:None */
            0, /* linux:324 (BTN_TOOL_AIRBRUSH) -> linux:324 (BTN_TOOL_AIRBRUSH) -> qnum:None */
            0, /* linux:325 (BTN_TOOL_FINGER) -> linux:325 (BTN_TOOL_FINGER) -> qnum:None */
            0, /* linux:326 (BTN_TOOL_MOUSE) -> linux:326 (BTN_TOOL_MOUSE) -> qnum:None */
            0, /* linux:327 (BTN_TOOL_LENS) -> linux:327 (BTN_TOOL_LENS) -> qnum:None */
            0, /* linux:328 (unnamed) -> linux:None (unnamed) -> qnum:None */
            0, /* linux:329 (unnamed) -> linux:None (unnamed) -> qnum:None */
            0, /* linux:330 (BTN_TOUCH) -> linux:330 (BTN_TOUCH) -> qnum:None */
            0, /* linux:331 (BTN_STYLUS) -> linux:331 (BTN_STYLUS) -> qnum:None */
            0, /* linux:332 (BTN_STYLUS2) -> linux:332 (BTN_STYLUS2) -> qnum:None */
            0, /* linux:333 (BTN_TOOL_DOUBLETAP) -> linux:333 (BTN_TOOL_DOUBLETAP) -> qnum:None */
            0, /* linux:334 (BTN_TOOL_TRIPLETAP) -> linux:334 (BTN_TOOL_TRIPLETAP) -> qnum:None */
            0, /* linux:335 (BTN_TOOL_QUADTAP) -> linux:335 (BTN_TOOL_QUADTAP) -> qnum:None */
            0, /* linux:336 (BTN_GEAR_DOWN) -> linux:336 (BTN_GEAR_DOWN) -> qnum:None */
            0, /* linux:337 (BTN_GEAR_UP) -> linux:337 (BTN_GEAR_UP) -> qnum:None */
            0, /* linux:338 (unnamed) -> linux:None (unnamed) -> qnum:None */
            0, /* linux:339 (unnamed) -> linux:None (unnamed) -> qnum:None */
            0, /* linux:340 (unnamed) -> linux:None (unnamed) -> qnum:None */
            0, /* linux:341 (unnamed) -> linux:None (unnamed) -> qnum:None */
            0, /* linux:342 (unnamed) -> linux:None (unnamed) -> qnum:None */
            0, /* linux:343 (unnamed) -> linux:None (unnamed) -> qnum:None */
            0, /* linux:344 (unnamed) -> linux:None (unnamed) -> qnum:None */
            0, /* linux:345 (unnamed) -> linux:None (unnamed) -> qnum:None */
            0, /* linux:346 (unnamed) -> linux:None (unnamed) -> qnum:None */
            0, /* linux:347 (unnamed) -> linux:None (unnamed) -> qnum:None */
            0, /* linux:348 (unnamed) -> linux:None (unnamed) -> qnum:None */
            0, /* linux:349 (unnamed) -> linux:None (unnamed) -> qnum:None */
            0, /* linux:350 (unnamed) -> linux:None (unnamed) -> qnum:None */
            0, /* linux:351 (unnamed) -> linux:None (unnamed) -> qnum:None */
            0, /* linux:352 (KEY_OK) -> linux:352 (KEY_OK) -> qnum:None */
            0, /* linux:353 (KEY_SELECT) -> linux:353 (KEY_SELECT) -> qnum:None */
            0, /* linux:354 (KEY_GOTO) -> linux:354 (KEY_GOTO) -> qnum:None */
            0, /* linux:355 (KEY_CLEAR) -> linux:355 (KEY_CLEAR) -> qnum:None */
            0, /* linux:356 (KEY_POWER2) -> linux:356 (KEY_POWER2) -> qnum:None */
            0, /* linux:357 (KEY_OPTION) -> linux:357 (KEY_OPTION) -> qnum:None */
            0, /* linux:358 (KEY_INFO) -> linux:358 (KEY_INFO) -> qnum:None */
            0, /* linux:359 (KEY_TIME) -> linux:359 (KEY_TIME) -> qnum:None */
            0, /* linux:360 (KEY_VENDOR) -> linux:360 (KEY_VENDOR) -> qnum:None */
            0, /* linux:361 (KEY_ARCHIVE) -> linux:361 (KEY_ARCHIVE) -> qnum:None */
            0, /* linux:362 (KEY_PROGRAM) -> linux:362 (KEY_PROGRAM) -> qnum:None */
            0, /* linux:363 (KEY_CHANNEL) -> linux:363 (KEY_CHANNEL) -> qnum:None */
            0, /* linux:364 (KEY_FAVORITES) -> linux:364 (KEY_FAVORITES) -> qnum:None */
            0, /* linux:365 (KEY_EPG) -> linux:365 (KEY_EPG) -> qnum:None */
            0, /* linux:366 (KEY_PVR) -> linux:366 (KEY_PVR) -> qnum:None */
            0, /* linux:367 (KEY_MHP) -> linux:367 (KEY_MHP) -> qnum:None */
            0, /* linux:368 (KEY_LANGUAGE) -> linux:368 (KEY_LANGUAGE) -> qnum:None */
            0, /* linux:369 (KEY_TITLE) -> linux:369 (KEY_TITLE) -> qnum:None */
            0, /* linux:370 (KEY_SUBTITLE) -> linux:370 (KEY_SUBTITLE) -> qnum:None */
            0, /* linux:371 (KEY_ANGLE) -> linux:371 (KEY_ANGLE) -> qnum:None */
            0, /* linux:372 (KEY_ZOOM) -> linux:372 (KEY_ZOOM) -> qnum:None */
            0, /* linux:373 (KEY_MODE) -> linux:373 (KEY_MODE) -> qnum:None */
            0, /* linux:374 (KEY_KEYBOARD) -> linux:374 (KEY_KEYBOARD) -> qnum:None */
            0, /* linux:375 (KEY_SCREEN) -> linux:375 (KEY_SCREEN) -> qnum:None */
            0, /* linux:376 (KEY_PC) -> linux:376 (KEY_PC) -> qnum:None */
            0, /* linux:377 (KEY_TV) -> linux:377 (KEY_TV) -> qnum:None */
            0, /* linux:378 (KEY_TV2) -> linux:378 (KEY_TV2) -> qnum:None */
            0, /* linux:379 (KEY_VCR) -> linux:379 (KEY_VCR) -> qnum:None */
            0, /* linux:380 (KEY_VCR2) -> linux:380 (KEY_VCR2) -> qnum:None */
            0, /* linux:381 (KEY_SAT) -> linux:381 (KEY_SAT) -> qnum:None */
            0, /* linux:382 (KEY_SAT2) -> linux:382 (KEY_SAT2) -> qnum:None */
            0, /* linux:383 (KEY_CD) -> linux:383 (KEY_CD) -> qnum:None */
            0, /* linux:384 (KEY_TAPE) -> linux:384 (KEY_TAPE) -> qnum:None */
            0, /* linux:385 (KEY_RADIO) -> linux:385 (KEY_RADIO) -> qnum:None */
            0, /* linux:386 (KEY_TUNER) -> linux:386 (KEY_TUNER) -> qnum:None */
            0, /* linux:387 (KEY_PLAYER) -> linux:387 (KEY_PLAYER) -> qnum:None */
            0, /* linux:388 (KEY_TEXT) -> linux:388 (KEY_TEXT) -> qnum:None */
            0, /* linux:389 (KEY_DVD) -> linux:389 (KEY_DVD) -> qnum:None */
            0, /* linux:390 (KEY_AUX) -> linux:390 (KEY_AUX) -> qnum:None */
            0, /* linux:391 (KEY_MP3) -> linux:391 (KEY_MP3) -> qnum:None */
            0, /* linux:392 (KEY_AUDIO) -> linux:392 (KEY_AUDIO) -> qnum:None */
            0, /* linux:393 (KEY_VIDEO) -> linux:393 (KEY_VIDEO) -> qnum:None */
            0, /* linux:394 (KEY_DIRECTORY) -> linux:394 (KEY_DIRECTORY) -> qnum:None */
            0, /* linux:395 (KEY_LIST) -> linux:395 (KEY_LIST) -> qnum:None */
            0, /* linux:396 (KEY_MEMO) -> linux:396 (KEY_MEMO) -> qnum:None */
            0, /* linux:397 (KEY_CALENDAR) -> linux:397 (KEY_CALENDAR) -> qnum:None */
            0, /* linux:398 (KEY_RED) -> linux:398 (KEY_RED) -> qnum:None */
            0, /* linux:399 (KEY_GREEN) -> linux:399 (KEY_GREEN) -> qnum:None */
            0, /* linux:400 (KEY_YELLOW) -> linux:400 (KEY_YELLOW) -> qnum:None */
            0, /* linux:401 (KEY_BLUE) -> linux:401 (KEY_BLUE) -> qnum:None */
            0, /* linux:402 (KEY_CHANNELUP) -> linux:402 (KEY_CHANNELUP) -> qnum:None */
            0, /* linux:403 (KEY_CHANNELDOWN) -> linux:403 (KEY_CHANNELDOWN) -> qnum:None */
            0, /* linux:404 (KEY_FIRST) -> linux:404 (KEY_FIRST) -> qnum:None */
            0, /* linux:405 (KEY_LAST) -> linux:405 (KEY_LAST) -> qnum:None */
            0, /* linux:406 (KEY_AB) -> linux:406 (KEY_AB) -> qnum:None */
            0, /* linux:407 (KEY_NEXT) -> linux:407 (KEY_NEXT) -> qnum:None */
            0, /* linux:408 (KEY_RESTART) -> linux:408 (KEY_RESTART) -> qnum:None */
            0, /* linux:409 (KEY_SLOW) -> linux:409 (KEY_SLOW) -> qnum:None */
            0, /* linux:410 (KEY_SHUFFLE) -> linux:410 (KEY_SHUFFLE) -> qnum:None */
            0, /* linux:411 (KEY_BREAK) -> linux:411 (KEY_BREAK) -> qnum:None */
            0, /* linux:412 (KEY_PREVIOUS) -> linux:412 (KEY_PREVIOUS) -> qnum:None */
            0, /* linux:413 (KEY_DIGITS) -> linux:413 (KEY_DIGITS) -> qnum:None */
            0, /* linux:414 (KEY_TEEN) -> linux:414 (KEY_TEEN) -> qnum:None */
            0, /* linux:415 (KEY_TWEN) -> linux:415 (KEY_TWEN) -> qnum:None */
            0, /* linux:416 (KEY_VIDEOPHONE) -> linux:416 (KEY_VIDEOPHONE) -> qnum:None */
            0, /* linux:417 (KEY_GAMES) -> linux:417 (KEY_GAMES) -> qnum:None */
            0, /* linux:418 (KEY_ZOOMIN) -> linux:418 (KEY_ZOOMIN) -> qnum:None */
            0, /* linux:419 (KEY_ZOOMOUT) -> linux:419 (KEY_ZOOMOUT) -> qnum:None */
            0, /* linux:420 (KEY_ZOOMRESET) -> linux:420 (KEY_ZOOMRESET) -> qnum:None */
            0, /* linux:421 (KEY_WORDPROCESSOR) -> linux:421 (KEY_WORDPROCESSOR) -> qnum:None */
            0, /* linux:422 (KEY_EDITOR) -> linux:422 (KEY_EDITOR) -> qnum:None */
            0, /* linux:423 (KEY_SPREADSHEET) -> linux:423 (KEY_SPREADSHEET) -> qnum:None */
            0, /* linux:424 (KEY_GRAPHICSEDITOR) -> linux:424 (KEY_GRAPHICSEDITOR) -> qnum:None */
            0, /* linux:425 (KEY_PRESENTATION) -> linux:425 (KEY_PRESENTATION) -> qnum:None */
            0, /* linux:426 (KEY_DATABASE) -> linux:426 (KEY_DATABASE) -> qnum:None */
            0, /* linux:427 (KEY_NEWS) -> linux:427 (KEY_NEWS) -> qnum:None */
            0, /* linux:428 (KEY_VOICEMAIL) -> linux:428 (KEY_VOICEMAIL) -> qnum:None */
            0, /* linux:429 (KEY_ADDRESSBOOK) -> linux:429 (KEY_ADDRESSBOOK) -> qnum:None */
            0, /* linux:430 (KEY_MESSENGER) -> linux:430 (KEY_MESSENGER) -> qnum:None */
            0, /* linux:431 (KEY_DISPLAYTOGGLE) -> linux:431 (KEY_DISPLAYTOGGLE) -> qnum:None */
            0, /* linux:432 (KEY_SPELLCHECK) -> linux:432 (KEY_SPELLCHECK) -> qnum:None */
            0, /* linux:433 (KEY_LOGOFF) -> linux:433 (KEY_LOGOFF) -> qnum:None */
            0, /* linux:434 (KEY_DOLLAR) -> linux:434 (KEY_DOLLAR) -> qnum:None */
            0, /* linux:435 (KEY_EURO) -> linux:435 (KEY_EURO) -> qnum:None */
            0, /* linux:436 (KEY_FRAMEBACK) -> linux:436 (KEY_FRAMEBACK) -> qnum:None */
            0, /* linux:437 (KEY_FRAMEFORWARD) -> linux:437 (KEY_FRAMEFORWARD) -> qnum:None */
            0, /* linux:438 (KEY_CONTEXT_MENU) -> linux:438 (KEY_CONTEXT_MENU) -> qnum:None */
            0, /* linux:439 (KEY_MEDIA_REPEAT) -> linux:439 (KEY_MEDIA_REPEAT) -> qnum:None */
            0, /* linux:440 (unnamed) -> linux:None (unnamed) -> qnum:None */
            0, /* linux:441 (unnamed) -> linux:None (unnamed) -> qnum:None */
            0, /* linux:442 (unnamed) -> linux:None (unnamed) -> qnum:None */
            0, /* linux:443 (unnamed) -> linux:None (unnamed) -> qnum:None */
            0, /* linux:444 (unnamed) -> linux:None (unnamed) -> qnum:None */
            0, /* linux:445 (unnamed) -> linux:None (unnamed) -> qnum:None */
            0, /* linux:446 (unnamed) -> linux:None (unnamed) -> qnum:None */
            0, /* linux:447 (unnamed) -> linux:None (unnamed) -> qnum:None */
            0, /* linux:448 (KEY_DEL_EOL) -> linux:448 (KEY_DEL_EOL) -> qnum:None */
            0, /* linux:449 (KEY_DEL_EOS) -> linux:449 (KEY_DEL_EOS) -> qnum:None */
            0, /* linux:450 (KEY_INS_LINE) -> linux:450 (KEY_INS_LINE) -> qnum:None */
            0, /* linux:451 (KEY_DEL_LINE) -> linux:451 (KEY_DEL_LINE) -> qnum:None */
            0, /* linux:452 (unnamed) -> linux:None (unnamed) -> qnum:None */
            0, /* linux:453 (unnamed) -> linux:None (unnamed) -> qnum:None */
            0, /* linux:454 (unnamed) -> linux:None (unnamed) -> qnum:None */
            0, /* linux:455 (unnamed) -> linux:None (unnamed) -> qnum:None */
            0, /* linux:456 (unnamed) -> linux:None (unnamed) -> qnum:None */
            0, /* linux:457 (unnamed) -> linux:None (unnamed) -> qnum:None */
            0, /* linux:458 (unnamed) -> linux:None (unnamed) -> qnum:None */
            0, /* linux:459 (unnamed) -> linux:None (unnamed) -> qnum:None */
            0, /* linux:460 (unnamed) -> linux:None (unnamed) -> qnum:None */
            0, /* linux:461 (unnamed) -> linux:None (unnamed) -> qnum:None */
            0, /* linux:462 (unnamed) -> linux:None (unnamed) -> qnum:None */
            0, /* linux:463 (unnamed) -> linux:None (unnamed) -> qnum:None */
            0, /* linux:464 (KEY_FN) -> linux:464 (KEY_FN) -> qnum:None */
            0, /* linux:465 (KEY_FN_ESC) -> linux:465 (KEY_FN_ESC) -> qnum:None */
            0, /* linux:466 (KEY_FN_F1) -> linux:466 (KEY_FN_F1) -> qnum:None */
            0, /* linux:467 (KEY_FN_F2) -> linux:467 (KEY_FN_F2) -> qnum:None */
            0, /* linux:468 (KEY_FN_F3) -> linux:468 (KEY_FN_F3) -> qnum:None */
            0, /* linux:469 (KEY_FN_F4) -> linux:469 (KEY_FN_F4) -> qnum:None */
            0, /* linux:470 (KEY_FN_F5) -> linux:470 (KEY_FN_F5) -> qnum:None */
            0, /* linux:471 (KEY_FN_F6) -> linux:471 (KEY_FN_F6) -> qnum:None */
            0, /* linux:472 (KEY_FN_F7) -> linux:472 (KEY_FN_F7) -> qnum:None */
            0, /* linux:473 (KEY_FN_F8) -> linux:473 (KEY_FN_F8) -> qnum:None */
            0, /* linux:474 (KEY_FN_F9) -> linux:474 (KEY_FN_F9) -> qnum:None */
            0, /* linux:475 (KEY_FN_F10) -> linux:475 (KEY_FN_F10) -> qnum:None */
            0, /* linux:476 (KEY_FN_F11) -> linux:476 (KEY_FN_F11) -> qnum:None */
            0, /* linux:477 (KEY_FN_F12) -> linux:477 (KEY_FN_F12) -> qnum:None */
            0, /* linux:478 (KEY_FN_1) -> linux:478 (KEY_FN_1) -> qnum:None */
            0, /* linux:479 (KEY_FN_2) -> linux:479 (KEY_FN_2) -> qnum:None */
            0, /* linux:480 (KEY_FN_D) -> linux:480 (KEY_FN_D) -> qnum:None */
            0, /* linux:481 (KEY_FN_E) -> linux:481 (KEY_FN_E) -> qnum:None */
            0, /* linux:482 (KEY_FN_F) -> linux:482 (KEY_FN_F) -> qnum:None */
            0, /* linux:483 (KEY_FN_S) -> linux:483 (KEY_FN_S) -> qnum:None */
            0, /* linux:484 (KEY_FN_B) -> linux:484 (KEY_FN_B) -> qnum:None */
            0, /* linux:485 (unnamed) -> linux:None (unnamed) -> qnum:None */
            0, /* linux:486 (unnamed) -> linux:None (unnamed) -> qnum:None */
            0, /* linux:487 (unnamed) -> linux:None (unnamed) -> qnum:None */
            0, /* linux:488 (unnamed) -> linux:None (unnamed) -> qnum:None */
            0, /* linux:489 (unnamed) -> linux:None (unnamed) -> qnum:None */
            0, /* linux:490 (unnamed) -> linux:None (unnamed) -> qnum:None */
            0, /* linux:491 (unnamed) -> linux:None (unnamed) -> qnum:None */
            0, /* linux:492 (unnamed) -> linux:None (unnamed) -> qnum:None */
            0, /* linux:493 (unnamed) -> linux:None (unnamed) -> qnum:None */
            0, /* linux:494 (unnamed) -> linux:None (unnamed) -> qnum:None */
            0, /* linux:495 (unnamed) -> linux:None (unnamed) -> qnum:None */
            0, /* linux:496 (unnamed) -> linux:None (unnamed) -> qnum:None */
            0, /* linux:497 (KEY_BRL_DOT1) -> linux:497 (KEY_BRL_DOT1) -> qnum:None */
            0, /* linux:498 (KEY_BRL_DOT2) -> linux:498 (KEY_BRL_DOT2) -> qnum:None */
            0, /* linux:499 (KEY_BRL_DOT3) -> linux:499 (KEY_BRL_DOT3) -> qnum:None */
            0, /* linux:500 (KEY_BRL_DOT4) -> linux:500 (KEY_BRL_DOT4) -> qnum:None */
            0, /* linux:501 (KEY_BRL_DOT5) -> linux:501 (KEY_BRL_DOT5) -> qnum:None */
            0, /* linux:502 (KEY_BRL_DOT6) -> linux:502 (KEY_BRL_DOT6) -> qnum:None */
            0, /* linux:503 (KEY_BRL_DOT7) -> linux:503 (KEY_BRL_DOT7) -> qnum:None */
            0, /* linux:504 (KEY_BRL_DOT8) -> linux:504 (KEY_BRL_DOT8) -> qnum:None */
            0, /* linux:505 (KEY_BRL_DOT9) -> linux:505 (KEY_BRL_DOT9) -> qnum:None */
            0, /* linux:506 (KEY_BRL_DOT10) -> linux:506 (KEY_BRL_DOT10) -> qnum:None */
            0, /* linux:507 (unnamed) -> linux:None (unnamed) -> qnum:None */
            0, /* linux:508 (unnamed) -> linux:None (unnamed) -> qnum:None */
            0, /* linux:509 (unnamed) -> linux:None (unnamed) -> qnum:None */
            0, /* linux:510 (unnamed) -> linux:None (unnamed) -> qnum:None */
            0, /* linux:511 (unnamed) -> linux:None (unnamed) -> qnum:None */
            0, /* linux:512 (KEY_NUMERIC_0) -> linux:512 (KEY_NUMERIC_0) -> qnum:None */
            0, /* linux:513 (KEY_NUMERIC_1) -> linux:513 (KEY_NUMERIC_1) -> qnum:None */
            0, /* linux:514 (KEY_NUMERIC_2) -> linux:514 (KEY_NUMERIC_2) -> qnum:None */
            0, /* linux:515 (KEY_NUMERIC_3) -> linux:515 (KEY_NUMERIC_3) -> qnum:None */
            0, /* linux:516 (KEY_NUMERIC_4) -> linux:516 (KEY_NUMERIC_4) -> qnum:None */
            0, /* linux:517 (KEY_NUMERIC_5) -> linux:517 (KEY_NUMERIC_5) -> qnum:None */
            0, /* linux:518 (KEY_NUMERIC_6) -> linux:518 (KEY_NUMERIC_6) -> qnum:None */
            0, /* linux:519 (KEY_NUMERIC_7) -> linux:519 (KEY_NUMERIC_7) -> qnum:None */
            0, /* linux:520 (KEY_NUMERIC_8) -> linux:520 (KEY_NUMERIC_8) -> qnum:None */
            0, /* linux:521 (KEY_NUMERIC_9) -> linux:521 (KEY_NUMERIC_9) -> qnum:None */
            0, /* linux:522 (KEY_NUMERIC_STAR) -> linux:522 (KEY_NUMERIC_STAR) -> qnum:None */
            0, /* linux:523 (KEY_NUMERIC_POUND) -> linux:523 (KEY_NUMERIC_POUND) -> qnum:None */
            0, /* linux:524 (KEY_RFKILL) -> linux:524 (KEY_RFKILL) -> qnum:None */
            */
    )
}