/*
 * Copyright (c) 2021  Gaurav Ujjwal.
 *
 * SPDX-License-Identifier:  GPL-3.0-or-later
 *
 * See COPYING.txt for more details.
 */

package com.gaurav.avnc.vnc

/**
 * Implements mapping between Unicode code-points and X KeySyms.
 */
object XKeySymUnicode {

    /**
     * Returns X KeySym for [uChar].
     */
    fun getKeySymForUnicodeChar(uChar: Int): Int {
        return if (uChar < 0x100) uChar else uChar + 0x01000000
    }

    /**
     * Returns legacy X KeySym for given [uChar].
     * Returns 0 if no legacy KeySym is found.
     */
    fun getLegacyKeySymForUnicodeChar(uChar: Int): Int {

        // Check if character is outside of our map
        if (uChar < UnicodeToLegacyKeysym[0] || uChar > UnicodeToLegacyKeysym[UnicodeToLegacyKeysym.size - 2])
            return 0

        // Binary Search on the 'first column' of map
        var low = 0
        var high = (UnicodeToLegacyKeysym.size / 2) - 1

        while (low <= high) {
            val mid = (low + high) / 2
            val midChar = UnicodeToLegacyKeysym[mid * 2]

            when {
                uChar < midChar -> high = mid - 1
                uChar > midChar -> low = mid + 1
                else /* uChar == midChar */ -> return UnicodeToLegacyKeysym[mid * 2 + 1]
            }
        }

        return 0
    }

    /**
     * This array maps Unicode code points to X KeySyms that were used before the
     * introduction of Unicode KeySyms to X Windows System Protocol.
     *
     *
     * Source of the map:  https://www.cl.cam.ac.uk/~mgk25/ucs/keysym2ucs.c
     *
     *
     * This array is used as 2D array with stride = 2.
     * First column is kept sorted to allow binary search.
     */
    private val UnicodeToLegacyKeysym = intArrayOf(
            // Unicode, X KeySym
            0x100, 0x3C0,
            0x101, 0x3E0,
            0x102, 0x1C3,
            0x103, 0x1E3,
            0x104, 0x1A1,
            0x105, 0x1B1,
            0x106, 0x1C6,
            0x107, 0x1E6,
            0x108, 0x2C6,
            0x109, 0x2E6,
            0x10A, 0x2C5,
            0x10B, 0x2E5,
            0x10C, 0x1C8,
            0x10D, 0x1E8,
            0x10E, 0x1CF,
            0x10F, 0x1EF,
            0x110, 0x1D0,
            0x111, 0x1F0,
            0x112, 0x3AA,
            0x113, 0x3BA,
            0x116, 0x3CC,
            0x117, 0x3EC,
            0x118, 0x1CA,
            0x119, 0x1EA,
            0x11A, 0x1CC,
            0x11B, 0x1EC,
            0x11C, 0x2D8,
            0x11D, 0x2F8,
            0x11E, 0x2AB,
            0x11F, 0x2BB,
            0x120, 0x2D5,
            0x121, 0x2F5,
            0x122, 0x3AB,
            0x123, 0x3BB,
            0x124, 0x2A6,
            0x125, 0x2B6,
            0x126, 0x2A1,
            0x127, 0x2B1,
            0x128, 0x3A5,
            0x129, 0x3B5,
            0x12A, 0x3CF,
            0x12B, 0x3EF,
            0x12E, 0x3C7,
            0x12F, 0x3E7,
            0x130, 0x2A9,
            0x131, 0x2B9,
            0x134, 0x2AC,
            0x135, 0x2BC,
            0x136, 0x3D3,
            0x137, 0x3F3,
            0x138, 0x3A2,
            0x139, 0x1C5,
            0x13A, 0x1E5,
            0x13B, 0x3A6,
            0x13C, 0x3B6,
            0x13D, 0x1A5,
            0x13E, 0x1B5,
            0x141, 0x1A3,
            0x142, 0x1B3,
            0x143, 0x1D1,
            0x144, 0x1F1,
            0x145, 0x3D1,
            0x146, 0x3F1,
            0x147, 0x1D2,
            0x148, 0x1F2,
            0x14A, 0x3BD,
            0x14B, 0x3BF,
            0x14C, 0x3D2,
            0x14D, 0x3F2,
            0x150, 0x1D5,
            0x151, 0x1F5,
            0x152, 0x13BC,
            0x153, 0x13BD,
            0x154, 0x1C0,
            0x155, 0x1E0,
            0x156, 0x3A3,
            0x157, 0x3B3,
            0x158, 0x1D8,
            0x159, 0x1F8,
            0x15A, 0x1A6,
            0x15B, 0x1B6,
            0x15C, 0x2DE,
            0x15D, 0x2FE,
            0x15E, 0x1AA,
            0x15F, 0x1BA,
            0x160, 0x1A9,
            0x161, 0x1B9,
            0x162, 0x1DE,
            0x163, 0x1FE,
            0x164, 0x1AB,
            0x165, 0x1BB,
            0x166, 0x3AC,
            0x167, 0x3BC,
            0x168, 0x3DD,
            0x169, 0x3FD,
            0x16A, 0x3DE,
            0x16B, 0x3FE,
            0x16C, 0x2DD,
            0x16D, 0x2FD,
            0x16E, 0x1D9,
            0x16F, 0x1F9,
            0x170, 0x1DB,
            0x171, 0x1FB,
            0x172, 0x3D9,
            0x173, 0x3F9,
            0x178, 0x13BE,
            0x179, 0x1AC,
            0x17A, 0x1BC,
            0x17B, 0x1AF,
            0x17C, 0x1BF,
            0x17D, 0x1AE,
            0x17E, 0x1BE,
            0x192, 0x8F6,
            0x2C7, 0x1B7,
            0x2D8, 0x1A2,
            0x2D9, 0x1FF,
            0x2DB, 0x1B2,
            0x2DD, 0x1BD,
            0x385, 0x7AE,
            0x386, 0x7A1,
            0x388, 0x7A2,
            0x389, 0x7A3,
            0x38A, 0x7A4,
            0x38C, 0x7A7,
            0x38E, 0x7A8,
            0x38F, 0x7AB,
            0x390, 0x7B6,
            0x391, 0x7C1,
            0x392, 0x7C2,
            0x393, 0x7C3,
            0x394, 0x7C4,
            0x395, 0x7C5,
            0x396, 0x7C6,
            0x397, 0x7C7,
            0x398, 0x7C8,
            0x399, 0x7C9,
            0x39A, 0x7CA,
            0x39B, 0x7CB,
            0x39C, 0x7CC,
            0x39D, 0x7CD,
            0x39E, 0x7CE,
            0x39F, 0x7CF,
            0x3A0, 0x7D0,
            0x3A1, 0x7D1,
            0x3A3, 0x7D2,
            0x3A4, 0x7D4,
            0x3A5, 0x7D5,
            0x3A6, 0x7D6,
            0x3A7, 0x7D7,
            0x3A8, 0x7D8,
            0x3A9, 0x7D9,
            0x3AA, 0x7A5,
            0x3AB, 0x7A9,
            0x3AC, 0x7B1,
            0x3AD, 0x7B2,
            0x3AE, 0x7B3,
            0x3AF, 0x7B4,
            0x3B0, 0x7BA,
            0x3B1, 0x7E1,
            0x3B2, 0x7E2,
            0x3B3, 0x7E3,
            0x3B4, 0x7E4,
            0x3B5, 0x7E5,
            0x3B6, 0x7E6,
            0x3B7, 0x7E7,
            0x3B8, 0x7E8,
            0x3B9, 0x7E9,
            0x3BA, 0x7EA,
            0x3BB, 0x7EB,
            0x3BC, 0x7EC,
            0x3BD, 0x7ED,
            0x3BE, 0x7EE,
            0x3BF, 0x7EF,
            0x3C0, 0x7F0,
            0x3C1, 0x7F1,
            0x3C2, 0x7F3,
            0x3C3, 0x7F2,
            0x3C4, 0x7F4,
            0x3C5, 0x7F5,
            0x3C6, 0x7F6,
            0x3C7, 0x7F7,
            0x3C8, 0x7F8,
            0x3C9, 0x7F9,
            0x3CA, 0x7B5,
            0x3CB, 0x7B9,
            0x3CC, 0x7B7,
            0x3CD, 0x7B8,
            0x3CE, 0x7BB,
            0x401, 0x6B3,
            0x402, 0x6B1,
            0x403, 0x6B2,
            0x404, 0x6B4,
            0x405, 0x6B5,
            0x406, 0x6B6,
            0x407, 0x6B7,
            0x408, 0x6B8,
            0x409, 0x6B9,
            0x40A, 0x6BA,
            0x40B, 0x6BB,
            0x40C, 0x6BC,
            0x40E, 0x6BE,
            0x40F, 0x6BF,
            0x410, 0x6E1,
            0x411, 0x6E2,
            0x412, 0x6F7,
            0x413, 0x6E7,
            0x414, 0x6E4,
            0x415, 0x6E5,
            0x416, 0x6F6,
            0x417, 0x6FA,
            0x418, 0x6E9,
            0x419, 0x6EA,
            0x41A, 0x6EB,
            0x41B, 0x6EC,
            0x41C, 0x6ED,
            0x41D, 0x6EE,
            0x41E, 0x6EF,
            0x41F, 0x6F0,
            0x420, 0x6F2,
            0x421, 0x6F3,
            0x422, 0x6F4,
            0x423, 0x6F5,
            0x424, 0x6E6,
            0x425, 0x6E8,
            0x426, 0x6E3,
            0x427, 0x6FE,
            0x428, 0x6FB,
            0x429, 0x6FD,
            0x42A, 0x6FF,
            0x42B, 0x6F9,
            0x42C, 0x6F8,
            0x42D, 0x6FC,
            0x42E, 0x6E0,
            0x42F, 0x6F1,
            0x430, 0x6C1,
            0x431, 0x6C2,
            0x432, 0x6D7,
            0x433, 0x6C7,
            0x434, 0x6C4,
            0x435, 0x6C5,
            0x436, 0x6D6,
            0x437, 0x6DA,
            0x438, 0x6C9,
            0x439, 0x6CA,
            0x43A, 0x6CB,
            0x43B, 0x6CC,
            0x43C, 0x6CD,
            0x43D, 0x6CE,
            0x43E, 0x6CF,
            0x43F, 0x6D0,
            0x440, 0x6D2,
            0x441, 0x6D3,
            0x442, 0x6D4,
            0x443, 0x6D5,
            0x444, 0x6C6,
            0x445, 0x6C8,
            0x446, 0x6C3,
            0x447, 0x6DE,
            0x448, 0x6DB,
            0x449, 0x6DD,
            0x44A, 0x6DF,
            0x44B, 0x6D9,
            0x44C, 0x6D8,
            0x44D, 0x6DC,
            0x44E, 0x6C0,
            0x44F, 0x6D1,
            0x451, 0x6A3,
            0x452, 0x6A1,
            0x453, 0x6A2,
            0x454, 0x6A4,
            0x455, 0x6A5,
            0x456, 0x6A6,
            0x457, 0x6A7,
            0x458, 0x6A8,
            0x459, 0x6A9,
            0x45A, 0x6AA,
            0x45B, 0x6AB,
            0x45C, 0x6AC,
            0x45E, 0x6AE,
            0x45F, 0x6AF,
            0x5D0, 0xCE0,
            0x5D1, 0xCE1,
            0x5D2, 0xCE2,
            0x5D3, 0xCE3,
            0x5D4, 0xCE4,
            0x5D5, 0xCE5,
            0x5D6, 0xCE6,
            0x5D7, 0xCE7,
            0x5D8, 0xCE8,
            0x5D9, 0xCE9,
            0x5DA, 0xCEA,
            0x5DB, 0xCEB,
            0x5DC, 0xCEC,
            0x5DD, 0xCED,
            0x5DE, 0xCEE,
            0x5DF, 0xCEF,
            0x5E0, 0xCF0,
            0x5E1, 0xCF1,
            0x5E2, 0xCF2,
            0x5E3, 0xCF3,
            0x5E4, 0xCF4,
            0x5E5, 0xCF5,
            0x5E6, 0xCF6,
            0x5E7, 0xCF7,
            0x5E8, 0xCF8,
            0x5E9, 0xCF9,
            0x5EA, 0xCFA,
            0x60C, 0x5AC,
            0x61B, 0x5BB,
            0x61F, 0x5BF,
            0x621, 0x5C1,
            0x622, 0x5C2,
            0x623, 0x5C3,
            0x624, 0x5C4,
            0x625, 0x5C5,
            0x626, 0x5C6,
            0x627, 0x5C7,
            0x628, 0x5C8,
            0x629, 0x5C9,
            0x62A, 0x5CA,
            0x62B, 0x5CB,
            0x62C, 0x5CC,
            0x62D, 0x5CD,
            0x62E, 0x5CE,
            0x62F, 0x5CF,
            0x630, 0x5D0,
            0x631, 0x5D1,
            0x632, 0x5D2,
            0x633, 0x5D3,
            0x634, 0x5D4,
            0x635, 0x5D5,
            0x636, 0x5D6,
            0x637, 0x5D7,
            0x638, 0x5D8,
            0x639, 0x5D9,
            0x63A, 0x5DA,
            0x640, 0x5E0,
            0x641, 0x5E1,
            0x642, 0x5E2,
            0x643, 0x5E3,
            0x644, 0x5E4,
            0x645, 0x5E5,
            0x646, 0x5E6,
            0x647, 0x5E7,
            0x648, 0x5E8,
            0x649, 0x5E9,
            0x64A, 0x5EA,
            0x64B, 0x5EB,
            0x64C, 0x5EC,
            0x64D, 0x5ED,
            0x64E, 0x5EE,
            0x64F, 0x5EF,
            0x650, 0x5F0,
            0x651, 0x5F1,
            0x652, 0x5F2,
            0xE01, 0xDA1,
            0xE02, 0xDA2,
            0xE03, 0xDA3,
            0xE04, 0xDA4,
            0xE05, 0xDA5,
            0xE06, 0xDA6,
            0xE07, 0xDA7,
            0xE08, 0xDA8,
            0xE09, 0xDA9,
            0xE0A, 0xDAA,
            0xE0B, 0xDAB,
            0xE0C, 0xDAC,
            0xE0D, 0xDAD,
            0xE0E, 0xDAE,
            0xE0F, 0xDAF,
            0xE10, 0xDB0,
            0xE11, 0xDB1,
            0xE12, 0xDB2,
            0xE13, 0xDB3,
            0xE14, 0xDB4,
            0xE15, 0xDB5,
            0xE16, 0xDB6,
            0xE17, 0xDB7,
            0xE18, 0xDB8,
            0xE19, 0xDB9,
            0xE1A, 0xDBA,
            0xE1B, 0xDBB,
            0xE1C, 0xDBC,
            0xE1D, 0xDBD,
            0xE1E, 0xDBE,
            0xE1F, 0xDBF,
            0xE20, 0xDC0,
            0xE21, 0xDC1,
            0xE22, 0xDC2,
            0xE23, 0xDC3,
            0xE24, 0xDC4,
            0xE25, 0xDC5,
            0xE26, 0xDC6,
            0xE27, 0xDC7,
            0xE28, 0xDC8,
            0xE29, 0xDC9,
            0xE2A, 0xDCA,
            0xE2B, 0xDCB,
            0xE2C, 0xDCC,
            0xE2D, 0xDCD,
            0xE2E, 0xDCE,
            0xE2F, 0xDCF,
            0xE30, 0xDD0,
            0xE31, 0xDD1,
            0xE32, 0xDD2,
            0xE33, 0xDD3,
            0xE34, 0xDD4,
            0xE35, 0xDD5,
            0xE36, 0xDD6,
            0xE37, 0xDD7,
            0xE38, 0xDD8,
            0xE39, 0xDD9,
            0xE3A, 0xDDA,
            0xE3F, 0xDDF,
            0xE40, 0xDE0,
            0xE41, 0xDE1,
            0xE42, 0xDE2,
            0xE43, 0xDE3,
            0xE44, 0xDE4,
            0xE45, 0xDE5,
            0xE46, 0xDE6,
            0xE47, 0xDE7,
            0xE48, 0xDE8,
            0xE49, 0xDE9,
            0xE4A, 0xDEA,
            0xE4B, 0xDEB,
            0xE4C, 0xDEC,
            0xE4D, 0xDED,
            0xE50, 0xDF0,
            0xE51, 0xDF1,
            0xE52, 0xDF2,
            0xE53, 0xDF3,
            0xE54, 0xDF4,
            0xE55, 0xDF5,
            0xE56, 0xDF6,
            0xE57, 0xDF7,
            0xE58, 0xDF8,
            0xE59, 0xDF9,
            0x11A8, 0xED4,
            0x11A9, 0xED5,
            0x11AA, 0xED6,
            0x11AB, 0xED7,
            0x11AC, 0xED8,
            0x11AD, 0xED9,
            0x11AE, 0xEDA,
            0x11AF, 0xEDB,
            0x11B0, 0xEDC,
            0x11B1, 0xEDD,
            0x11B2, 0xEDE,
            0x11B3, 0xEDF,
            0x11B4, 0xEE0,
            0x11B5, 0xEE1,
            0x11B6, 0xEE2,
            0x11B7, 0xEE3,
            0x11B8, 0xEE4,
            0x11B9, 0xEE5,
            0x11BA, 0xEE6,
            0x11BB, 0xEE7,
            0x11BC, 0xEE8,
            0x11BD, 0xEE9,
            0x11BE, 0xEEA,
            0x11BF, 0xEEB,
            0x11C0, 0xEEC,
            0x11C1, 0xEED,
            0x11C2, 0xEEE,
            0x11EB, 0xEF8,
            0x11F0, 0xEF9,
            0x11F9, 0xEFA,
            0x2002, 0xAA2,
            0x2003, 0xAA1,
            0x2004, 0xAA3,
            0x2005, 0xAA4,
            0x2007, 0xAA5,
            0x2008, 0xAA6,
            0x2009, 0xAA7,
            0x200A, 0xAA8,
            0x2012, 0xABB,
            0x2013, 0xAAA,
            0x2014, 0xAA9,
            0x2015, 0x7AF,
            0x2017, 0xCDF,
            0x2018, 0xAD0,
            0x2019, 0xAD1,
            0x201A, 0xAFD,
            0x201C, 0xAD2,
            0x201D, 0xAD3,
            0x201E, 0xAFE,
            0x2020, 0xAF1,
            0x2021, 0xAF2,
            0x2022, 0xAE6,
            0x2025, 0xAAF,
            0x2026, 0xAAE,
            0x2032, 0xAD6,
            0x2033, 0xAD7,
            0x2038, 0xAFC,
            0x203E, 0x47E,
            0x20A9, 0xEFF,
            0x20AC, 0x20AC,
            0x20AC, 0x13A4,
            0x2105, 0xAB8,
            0x2116, 0x6B0,
            0x2117, 0xAFB,
            0x211E, 0xAD4,
            0x2122, 0xAC9,
            0x2153, 0xAB0,
            0x2154, 0xAB1,
            0x2155, 0xAB2,
            0x2156, 0xAB3,
            0x2157, 0xAB4,
            0x2158, 0xAB5,
            0x2159, 0xAB6,
            0x215A, 0xAB7,
            0x215B, 0xAC3,
            0x215C, 0xAC4,
            0x215D, 0xAC5,
            0x215E, 0xAC6,
            0x2190, 0x8FB,
            0x2191, 0x8FC,
            0x2192, 0x8FD,
            0x2193, 0x8FE,
            0x21D2, 0x8CE,
            0x21D4, 0x8CD,
            0x2202, 0x8EF,
            0x2207, 0x8C5,
            0x2218, 0xBCA,
            0x221A, 0x8D6,
            0x221D, 0x8C1,
            0x221E, 0x8C2,
            0x2227, 0x8DE,
            0x2227, 0xBA9,
            0x2228, 0x8DF,
            0x2228, 0xBA8,
            0x2229, 0x8DC,
            0x2229, 0xBC3,
            0x222A, 0xBD6,
            0x222A, 0x8DD,
            0x222B, 0x8BF,
            0x2234, 0x8C0,
            0x223C, 0x8C8,
            0x2243, 0x8C9,
            0x2260, 0x8BD,
            0x2261, 0x8CF,
            0x2264, 0x8BC,
            0x2265, 0x8BE,
            0x2282, 0xBDA,
            0x2282, 0x8DA,
            0x2283, 0xBD8,
            0x2283, 0x8DB,
            0x22A2, 0xBDC,
            0x22A3, 0xBFC,
            0x22A4, 0xBCE,
            0x22A5, 0xBC2,
            0x2308, 0xBD3,
            0x230A, 0xBC4,
            0x2315, 0xAFA,
            0x2320, 0x8A4,
            0x2321, 0x8A5,
            0x2329, 0xABC,
            0x232A, 0xABE,
            0x2395, 0xBCC,
            0x239B, 0x8AB,
            0x239D, 0x8AC,
            0x239E, 0x8AD,
            0x23A0, 0x8AE,
            0x23A1, 0x8A7,
            0x23A3, 0x8A8,
            0x23A4, 0x8A9,
            0x23A6, 0x8AA,
            0x23A8, 0x8AF,
            0x23AC, 0x8B0,
            0x23B7, 0x8A1,
            0x23BA, 0x9EF,
            0x23BB, 0x9F0,
            0x23BC, 0x9F2,
            0x23BD, 0x9F3,
            0x2409, 0x9E2,
            0x240A, 0x9E5,
            0x240B, 0x9E9,
            0x240C, 0x9E3,
            0x240D, 0x9E4,
            0x2424, 0x9E8,
            0x2500, 0x9F1,
            0x2500, 0x8A3,
            0x2502, 0x8A6,
            0x2502, 0x9F8,
            0x250C, 0x9EC,
            0x250C, 0x8A2,
            0x2510, 0x9EB,
            0x2514, 0x9ED,
            0x2518, 0x9EA,
            0x251C, 0x9F4,
            0x2524, 0x9F5,
            0x252C, 0x9F7,
            0x2534, 0x9F6,
            0x253C, 0x9EE,
            0x2592, 0x9E1,
            0x25AA, 0xAE7,
            0x25AB, 0xAE1,
            0x25AC, 0xADB,
            0x25AD, 0xAE2,
            0x25AE, 0xADF,
            0x25AF, 0xACF,
            0x25B2, 0xAE8,
            0x25B3, 0xAE3,
            0x25B6, 0xADD,
            0x25B7, 0xACD,
            0x25BC, 0xAE9,
            0x25BD, 0xAE4,
            0x25C0, 0xADC,
            0x25C1, 0xACC,
            0x25C6, 0x9E0,
            0x25CB, 0xBCF,
            0x25CB, 0xACE,
            0x25CF, 0xADE,
            0x25E6, 0xAE0,
            0x2606, 0xAE5,
            0x260E, 0xAF9,
            0x2613, 0xACA,
            0x261C, 0xAEA,
            0x261E, 0xAEB,
            0x2640, 0xAF8,
            0x2642, 0xAF7,
            0x2663, 0xAEC,
            0x2665, 0xAEE,
            0x2666, 0xAED,
            0x266D, 0xAF6,
            0x266F, 0xAF5,
            0x2713, 0xAF3,
            0x2717, 0xAF4,
            0x271D, 0xAD9,
            0x2720, 0xAF0,
            0x3001, 0x4A4,
            0x3002, 0x4A1,
            0x300C, 0x4A2,
            0x300D, 0x4A3,
            0x309B, 0x4DE,
            0x309C, 0x4DF,
            0x30A1, 0x4A7,
            0x30A2, 0x4B1,
            0x30A3, 0x4A8,
            0x30A4, 0x4B2,
            0x30A5, 0x4A9,
            0x30A6, 0x4B3,
            0x30A7, 0x4AA,
            0x30A8, 0x4B4,
            0x30A9, 0x4AB,
            0x30AA, 0x4B5,
            0x30AB, 0x4B6,
            0x30AD, 0x4B7,
            0x30AF, 0x4B8,
            0x30B1, 0x4B9,
            0x30B3, 0x4BA,
            0x30B5, 0x4BB,
            0x30B7, 0x4BC,
            0x30B9, 0x4BD,
            0x30BB, 0x4BE,
            0x30BD, 0x4BF,
            0x30BF, 0x4C0,
            0x30C1, 0x4C1,
            0x30C3, 0x4AF,
            0x30C4, 0x4C2,
            0x30C6, 0x4C3,
            0x30C8, 0x4C4,
            0x30CA, 0x4C5,
            0x30CB, 0x4C6,
            0x30CC, 0x4C7,
            0x30CD, 0x4C8,
            0x30CE, 0x4C9,
            0x30CF, 0x4CA,
            0x30D2, 0x4CB,
            0x30D5, 0x4CC,
            0x30D8, 0x4CD,
            0x30DB, 0x4CE,
            0x30DE, 0x4CF,
            0x30DF, 0x4D0,
            0x30E0, 0x4D1,
            0x30E1, 0x4D2,
            0x30E2, 0x4D3,
            0x30E3, 0x4AC,
            0x30E4, 0x4D4,
            0x30E5, 0x4AD,
            0x30E6, 0x4D5,
            0x30E7, 0x4AE,
            0x30E8, 0x4D6,
            0x30E9, 0x4D7,
            0x30EA, 0x4D8,
            0x30EB, 0x4D9,
            0x30EC, 0x4DA,
            0x30ED, 0x4DB,
            0x30EF, 0x4DC,
            0x30F2, 0x4A6,
            0x30F3, 0x4DD,
            0x30FB, 0x4A5,
            0x30FC, 0x4B0,
            0x3131, 0xEA1,
            0x3132, 0xEA2,
            0x3133, 0xEA3,
            0x3134, 0xEA4,
            0x3135, 0xEA5,
            0x3136, 0xEA6,
            0x3137, 0xEA7,
            0x3138, 0xEA8,
            0x3139, 0xEA9,
            0x313A, 0xEAA,
            0x313B, 0xEAB,
            0x313C, 0xEAC,
            0x313D, 0xEAD,
            0x313E, 0xEAE,
            0x313F, 0xEAF,
            0x3140, 0xEB0,
            0x3141, 0xEB1,
            0x3142, 0xEB2,
            0x3143, 0xEB3,
            0x3144, 0xEB4,
            0x3145, 0xEB5,
            0x3146, 0xEB6,
            0x3147, 0xEB7,
            0x3148, 0xEB8,
            0x3149, 0xEB9,
            0x314A, 0xEBA,
            0x314B, 0xEBB,
            0x314C, 0xEBC,
            0x314D, 0xEBD,
            0x314E, 0xEBE,
            0x314F, 0xEBF,
            0x3150, 0xEC0,
            0x3151, 0xEC1,
            0x3152, 0xEC2,
            0x3153, 0xEC3,
            0x3154, 0xEC4,
            0x3155, 0xEC5,
            0x3156, 0xEC6,
            0x3157, 0xEC7,
            0x3158, 0xEC8,
            0x3159, 0xEC9,
            0x315A, 0xECA,
            0x315B, 0xECB,
            0x315C, 0xECC,
            0x315D, 0xECD,
            0x315E, 0xECE,
            0x315F, 0xECF,
            0x3160, 0xED0,
            0x3161, 0xED1,
            0x3162, 0xED2,
            0x3163, 0xED3,
            0x316D, 0xEEF,
            0x3171, 0xEF0,
            0x3178, 0xEF1,
            0x317F, 0xEF2,
            0x3181, 0xEF3,
            0x3184, 0xEF4,
            0x3186, 0xEF5,
            0x318D, 0xEF6,
            0x318E, 0xEF7
    )
}