package com.futsch1.medtimer;


import static org.junit.jupiter.api.Assertions.assertEquals;
import static org.junit.jupiter.api.Assertions.assertNotNull;
import static org.junit.jupiter.api.Assertions.assertNull;

import android.graphics.Color;

import com.futsch1.medtimer.database.FullMedicine;
import com.futsch1.medtimer.database.JSONMedicineBackup;
import com.futsch1.medtimer.database.Medicine;
import com.futsch1.medtimer.database.Reminder;
import com.futsch1.medtimer.database.Tag;

import org.junit.jupiter.api.Test;

import java.util.ArrayList;
import java.util.Collections;
import java.util.List;

class JSONMedicineBackupUnitTest {

    // Generated by CodiumAI

    // creates a backup object with a version number and a medicines array
    @Test
    void testCreatesBackupWithVersionAndMedicinesArray() {
        JSONMedicineBackup jsonMedicineBackup = new JSONMedicineBackup();
        List<FullMedicine> medicinesWithReminders = new ArrayList<>();
        FullMedicine medicineWithReminders = new FullMedicine();
        Reminder reminder = new Reminder(0);
        reminder.reminderId = 14;
        reminder.timeInMinutes = 60;
        reminder.amount = "1";
        reminder.consecutiveDays = 1;
        reminder.pauseDays = 0;
        reminder.cycleStartDay = 12;
        reminder.instructions = "Take with water";
        reminder.periodStart = 13;
        reminder.days.set(0, false);
        reminder.activeDaysOfMonth = 7;
        reminder.linkedReminderId = 15;
        reminder.automaticallyTaken = true;
        reminder.intervalStartTimeOfDay = 1;
        reminder.intervalEndTimeOfDay = 2;
        medicineWithReminders.reminders = Collections.singletonList(reminder);
        medicineWithReminders.medicine = new Medicine("Medicine A");
        medicineWithReminders.medicine.useColor = true;
        medicineWithReminders.medicine.color = Color.RED;
        medicineWithReminders.medicine.iconId = 5;
        medicineWithReminders.medicine.unit = "pills";
        medicineWithReminders.medicine.sortOrder = 4.5;
        medicineWithReminders.medicine.notes = "Generic ingredient: Dihydrogen monoxide";
        medicineWithReminders.tags = new ArrayList<>();
        medicineWithReminders.tags.add(new Tag("Tag A"));
        medicinesWithReminders.add(medicineWithReminders);

        String result = jsonMedicineBackup.createBackupAsString(5, medicinesWithReminders);

        assertNotNull(result);

        // @formatter:off
        assertEquals("""
{
  "version": 5,
  "list": [
    {
      "medicine": {
        "name": "Medicine A",
        "color": -65536,
        "useColor": true,
        "notificationImportance": 3,
        "iconId": 5,
        "outOfStockReminder": "OFF",
        "amount": 0.0,
        "outOfStockReminderThreshold": 0.0,
        "refillSizes": [],
        "unit": "pills",
        "sortOrder": 4.5,
        "notes": "Generic ingredient: Dihydrogen monoxide"
      },
      "tags": [
        {
          "name": "Tag A"
        }
      ],
      "reminders": [
        {
          "reminderId": 14,
          "timeInMinutes": 60,
          "consecutiveDays": 1,
          "pauseDays": 0,
          "instructions": "Take with water",
          "cycleStartDay": 12,
          "amount": "1",
          "days": [
            false,
            true,
            true,
            true,
            true,
            true,
            true
          ],
          "active": true,
          "periodStart": 13,
          "periodEnd": 0,
          "activeDaysOfMonth": 7,
          "linkedReminderId": 15,
          "intervalStart": 0,
          "intervalStartsFromProcessed": false,
          "variableAmount": false,
          "automaticallyTaken": true,
          "intervalStartTimeOfDay": 1,
          "intervalEndTimeOfDay": 2,
          "windowedInterval": false
        }
      ]
    }
  ]
}""", result);
        // @formatter:on

        List<FullMedicine> parsedReminders = jsonMedicineBackup.parseBackup(result);
        assertNotNull(parsedReminders);
        compareListFullMedicine(parsedReminders, medicinesWithReminders);
    }

    private void compareListFullMedicine(List<FullMedicine> actual, List<FullMedicine> expected) {
        assertEquals(expected.size(), actual.size());
        for (int i = 0; i < actual.size(); i++) {
            compareFullMedicine(actual.get(i), expected.get(i));
        }
    }

    private void compareFullMedicine(FullMedicine medicineWithReminders, FullMedicine medicineWithReminders1) {
        assertEquals(medicineWithReminders.reminders.size(), medicineWithReminders1.reminders.size());
        for (int i = 0; i < medicineWithReminders.reminders.size(); i++) {
            compareReminder(medicineWithReminders.reminders.get(i), medicineWithReminders1.reminders.get(i));
        }
        assertEquals(medicineWithReminders.medicine.name, medicineWithReminders1.medicine.name);
        assertEquals(medicineWithReminders.medicine.useColor, medicineWithReminders1.medicine.useColor);
        assertEquals(medicineWithReminders.medicine.color, medicineWithReminders1.medicine.color);
    }

    private void compareReminder(Reminder reminder, Reminder reminder1) {
        assertEquals(reminder.timeInMinutes, reminder1.timeInMinutes);
        assertEquals(reminder.consecutiveDays, reminder1.consecutiveDays);
        assertEquals(reminder.instructions, reminder1.instructions);
        assertEquals(reminder.amount, reminder1.amount);
    }

    // iterates over the list of FullMedicine and adds each one to the medicines array as a JSONObject
    @Test
    void testIteratesOverMedicinesWithRemindersAndAddsToMedicinesArray() {
        JSONMedicineBackup jsonMedicineBackup = new JSONMedicineBackup();
        List<FullMedicine> medicinesWithReminders = new ArrayList<>();
        FullMedicine medicineWithReminders1 = new FullMedicine();
        Reminder reminder1 = new Reminder(0);
        reminder1.reminderId = 1;
        reminder1.timeInMinutes = 60;
        reminder1.amount = "1";
        reminder1.consecutiveDays = 1;
        reminder1.pauseDays = 0;
        reminder1.cycleStartDay = 1234;
        reminder1.instructions = "Take with water";
        reminder1.activeDaysOfMonth = 0xFFFF_FFFF;
        medicineWithReminders1.reminders = Collections.singletonList(reminder1);
        medicineWithReminders1.medicine = new Medicine("Medicine A");
        medicineWithReminders1.medicine.useColor = true;
        medicineWithReminders1.medicine.color = Color.RED;
        medicineWithReminders1.medicine.notificationImportance = 4;
        medicineWithReminders1.medicine.sortOrder = 3.44;
        medicinesWithReminders.add(medicineWithReminders1);

        FullMedicine medicineWithReminders2 = new FullMedicine();
        Reminder reminder2 = new Reminder(0);
        reminder2.reminderId = 2;
        reminder2.timeInMinutes = 120;
        reminder2.amount = "2";
        reminder2.consecutiveDays = 2;
        reminder2.pauseDays = 1;
        reminder2.cycleStartDay = 4567;
        reminder2.instructions = "Take after meal";
        reminder2.days.set(6, false);
        reminder2.activeDaysOfMonth = 255;
        reminder2.linkedReminderId = 234;
        reminder2.intervalStart = 256;
        reminder2.intervalStartsFromProcessed = true;
        reminder2.variableAmount = true;
        medicineWithReminders2.reminders = new ArrayList<>() {{
            add(reminder1);
            add(reminder2);
        }};
        medicineWithReminders2.medicine = new Medicine("Medicine B");
        medicineWithReminders2.medicine.useColor = false;
        medicineWithReminders2.medicine.color = Color.BLUE;
        medicineWithReminders2.medicine.notificationImportance = 5;
        medicineWithReminders2.medicine.iconId = 7;
        medicineWithReminders2.medicine.amount = 17.0;
        medicineWithReminders2.medicine.outOfStockReminderThreshold = 17.5;
        medicinesWithReminders.add(medicineWithReminders2);

        String result = jsonMedicineBackup.createBackupAsString(4, medicinesWithReminders);

        assertNotNull(result);

        // @formatter:off
        assertEquals("""
{
  "version": 4,
  "list": [
    {
      "medicine": {
        "name": "Medicine A",
        "color": -65536,
        "useColor": true,
        "notificationImportance": 4,
        "iconId": 0,
        "outOfStockReminder": "OFF",
        "amount": 0.0,
        "outOfStockReminderThreshold": 0.0,
        "refillSizes": [],
        "unit": "",
        "sortOrder": 3.44,
        "notes": ""
      },
      "reminders": [
        {
          "reminderId": 1,
          "timeInMinutes": 60,
          "consecutiveDays": 1,
          "pauseDays": 0,
          "instructions": "Take with water",
          "cycleStartDay": 1234,
          "amount": "1",
          "days": [
            true,
            true,
            true,
            true,
            true,
            true,
            true
          ],
          "active": true,
          "periodStart": 0,
          "periodEnd": 0,
          "activeDaysOfMonth": -1,
          "linkedReminderId": 0,
          "intervalStart": 0,
          "intervalStartsFromProcessed": false,
          "variableAmount": false,
          "automaticallyTaken": false,
          "intervalStartTimeOfDay": 480,
          "intervalEndTimeOfDay": 1320,
          "windowedInterval": false
        }
      ]
    },
    {
      "medicine": {
        "name": "Medicine B",
        "color": -16776961,
        "useColor": false,
        "notificationImportance": 5,
        "iconId": 7,
        "outOfStockReminder": "OFF",
        "amount": 17.0,
        "outOfStockReminderThreshold": 17.5,
        "refillSizes": [],
        "unit": "",
        "sortOrder": 1.0,
        "notes": ""
      },
      "reminders": [
        {
          "reminderId": 1,
          "timeInMinutes": 60,
          "consecutiveDays": 1,
          "pauseDays": 0,
          "instructions": "Take with water",
          "cycleStartDay": 1234,
          "amount": "1",
          "days": [
            true,
            true,
            true,
            true,
            true,
            true,
            true
          ],
          "active": true,
          "periodStart": 0,
          "periodEnd": 0,
          "activeDaysOfMonth": -1,
          "linkedReminderId": 0,
          "intervalStart": 0,
          "intervalStartsFromProcessed": false,
          "variableAmount": false,
          "automaticallyTaken": false,
          "intervalStartTimeOfDay": 480,
          "intervalEndTimeOfDay": 1320,
          "windowedInterval": false
        },
        {
          "reminderId": 2,
          "timeInMinutes": 120,
          "consecutiveDays": 2,
          "pauseDays": 1,
          "instructions": "Take after meal",
          "cycleStartDay": 4567,
          "amount": "2",
          "days": [
            true,
            true,
            true,
            true,
            true,
            true,
            false
          ],
          "active": true,
          "periodStart": 0,
          "periodEnd": 0,
          "activeDaysOfMonth": 255,
          "linkedReminderId": 234,
          "intervalStart": 256,
          "intervalStartsFromProcessed": true,
          "variableAmount": true,
          "automaticallyTaken": false,
          "intervalStartTimeOfDay": 480,
          "intervalEndTimeOfDay": 1320,
          "windowedInterval": false
        }
      ]
    }
  ]
}""", result);
        // @formatter:on

        List<FullMedicine> parsedReminders = jsonMedicineBackup.parseBackup(result);
        assertNotNull(parsedReminders);
        compareListFullMedicine(parsedReminders, medicinesWithReminders);
    }

    @Test
    void testParseJsonBackupMissingFields() {
        // Arrange
        JSONMedicineBackup jsonMedicineBackup = new JSONMedicineBackup();
        String validJsonBackup = "{\"version\": 1, \"medicinesWithReminders\": [{\"reminders\": [], \"medicine\": {}}]}";

        // Act
        List<FullMedicine> result = jsonMedicineBackup.parseBackup(validJsonBackup);

        // Assert
        assertNull(result);
    }

    @Test
    void testParseInvalidJsonBackup() {
        // Arrange
        JSONMedicineBackup jsonMedicineBackup = new JSONMedicineBackup();
        String invalidJsonBackup = "invalid_json";

        // Act
        List<FullMedicine> result = jsonMedicineBackup.parseBackup(invalidJsonBackup);

        // Assert
        assertNull(result);
    }

    @Test
    void testParseJsonBackupWithMissingFields() {
        // Arrange
        JSONMedicineBackup jsonMedicineBackup = new JSONMedicineBackup();
        String jsonBackupWithMissingFields = "{\"version\": 1}";

        // Act
        List<FullMedicine> result = jsonMedicineBackup.parseBackup(jsonBackupWithMissingFields);

        // Assert
        assertNull(result);
    }

}

