package com.fadcam.ui;

import android.animation.Animator;
import android.animation.AnimatorListenerAdapter;
import android.animation.ObjectAnimator;
import android.animation.ValueAnimator;
import android.content.Context;
import android.content.Intent;
import android.content.pm.PackageInfo;
import android.content.pm.PackageManager;
import android.graphics.Color;
import android.content.res.ColorStateList;
import android.net.Uri;
import android.os.Bundle;
import android.text.Html;
import android.text.Spanned;
import android.text.SpannableStringBuilder;
import android.view.LayoutInflater;
import android.view.View;
import android.view.ViewGroup;
import android.widget.ImageView;
import android.widget.LinearLayout;
import android.widget.ScrollView;
import android.widget.TextView;
import android.graphics.drawable.Drawable;

import androidx.annotation.NonNull;
import androidx.appcompat.app.AlertDialog;
import androidx.cardview.widget.CardView;
import androidx.core.text.HtmlCompat;
import androidx.fragment.app.Fragment;
import androidx.core.content.ContextCompat;
import androidx.recyclerview.widget.RecyclerView;

import com.fadcam.Constants;
import com.fadcam.Log;
import com.fadcam.R;
import com.fadcam.SharedPreferencesManager;
import com.google.android.material.button.MaterialButton;
import com.google.android.material.card.MaterialCardView;
import com.google.android.material.dialog.MaterialAlertDialogBuilder;

import org.json.JSONArray;
import org.json.JSONException;
import org.json.JSONObject;

import java.io.BufferedReader;
import java.io.InputStreamReader;
import java.net.HttpURLConnection;
import java.net.URL;
import java.util.concurrent.ExecutorService;
import java.util.concurrent.Executors;

public class AboutFragment extends BaseFragment {

    private View view;
    private ExecutorService executorService;
    private AlertDialog loadingDialog;
    private MaterialAlertDialogBuilder alertDialogBuilder;


    @Override
    public View onCreateView(LayoutInflater inflater, ViewGroup container, Bundle savedInstanceState) {
        view = inflater.inflate(R.layout.fragment_about, container, false);
        
        // Initialize with executor service
        executorService = Executors.newSingleThreadExecutor();
        
        // Initialize views and setup interactions
        initializeViews();
        
        // Apply theme-specific UI adjustments for Snow Veil theme
        SharedPreferencesManager sharedPreferencesManager = SharedPreferencesManager.getInstance(requireContext());
        String currentTheme = sharedPreferencesManager.sharedPreferences.getString(com.fadcam.Constants.PREF_APP_THEME, Constants.DEFAULT_APP_THEME);
        if ("Snow Veil".equals(currentTheme)) {
            applySnowVeilThemeToUI(view);
        }
        
        return view;
    }

    private void initializeViews() {
        // Header back button handling (new)
        View back = view.findViewById(R.id.back_button);
        if (back != null) {
            back.setOnClickListener(v -> OverlayNavUtil.dismiss(requireActivity()));
        }
        ImageView appIcon = view.findViewById(R.id.app_icon);
        TextView appName = view.findViewById(R.id.app_name);
        TextView appVersion = view.findViewById(R.id.app_version);
        TextView appDescription = view.findViewById(R.id.app_description);
        MaterialCardView fadSecInfoCard = view.findViewById(R.id.fadsec_info_card);
        MaterialButton sourceCodeButton = view.findViewById(R.id.source_code_button);
        MaterialButton donateButton = view.findViewById(R.id.donate_button);
        MaterialButton checkUpdatesButton = view.findViewById(R.id.check_updates_button);
        TextView emailText = view.findViewById(R.id.email_text);
        TextView discordText = view.findViewById(R.id.discord_text);
        MaterialCardView privacyInfoCard = view.findViewById(R.id.privacy_info_card);
        ScrollView scrollView = view.findViewById(R.id.scroll_view);

        // Initialize footer elements
        ImageView ivFadSecLabLogo = view.findViewById(R.id.ivFadSecLabLogo);
        TextView tvAboutFooter = view.findViewById(R.id.tvAboutFooter);

        int colorHeading = resolveThemeColor(R.attr.colorHeading);
        int colorButton = resolveThemeColor(R.attr.colorButton);
        int colorDialog = resolveThemeColor(R.attr.colorDialog);
        int colorOnPrimary = resolveThemeColor(android.R.attr.textColorPrimary);
        int colorOnSurface = resolveThemeColor(android.R.attr.textColorSecondary);
        
        // Get current theme directly from Constants to ensure consistent theme naming
        String currentTheme = com.fadcam.SharedPreferencesManager.getInstance(requireContext()).sharedPreferences.getString(com.fadcam.Constants.PREF_APP_THEME, Constants.DEFAULT_APP_THEME);
        
        // Define theme-specific colors - use theme-specific colors from resources instead of hardcoded values
        int themeTextColor;
        if ("Midnight Dusk".equals(currentTheme)) {
            themeTextColor = ContextCompat.getColor(requireContext(), R.color.colorPrimary); // Light purple (#cfbafd)
        } else if ("Crimson Bloom".equals(currentTheme)) {
            themeTextColor = ContextCompat.getColor(requireContext(), R.color.red_theme_primary); // Red primary color
        } else if ("Premium Gold".equals(currentTheme)) {
            themeTextColor = ContextCompat.getColor(requireContext(), R.color.gold_theme_primary); // Gold
        } else if ("Silent Forest".equals(currentTheme)) {
            themeTextColor = ContextCompat.getColor(requireContext(), R.color.silentforest_theme_primary); // Green
        } else if ("Shadow Alloy".equals(currentTheme)) {
            themeTextColor = ContextCompat.getColor(requireContext(), R.color.shadowalloy_theme_primary); // Silver
        } else if ("Pookie Pink".equals(currentTheme)) {
            themeTextColor = ContextCompat.getColor(requireContext(), R.color.pookiepink_theme_primary); // Pink
        } else if ("Snow Veil".equals(currentTheme)) {
            themeTextColor = ContextCompat.getColor(requireContext(), R.color.snowveil_theme_text_primary); // Black
        } else if ("Faded Night".equals(currentTheme)) {
            themeTextColor = Color.WHITE; // Default for Faded Night
        } else {
            // Default to Crimson Bloom if no matches
            themeTextColor = ContextCompat.getColor(requireContext(), R.color.red_theme_primary);
        }

        appIcon.setImageResource(R.mipmap.ic_launcher);
        appName.setText(getString(R.string.app_name));
        // Set app name to theme color instead of default colorHeading
        appName.setTextColor(themeTextColor);
        
        try {
            PackageManager packageManager = requireContext().getPackageManager();
            String packageName = requireContext().getPackageName();
            PackageInfo packageInfo = packageManager.getPackageInfo(packageName, 0);
            String versionName = packageInfo.versionName;
            int versionCode = packageInfo.versionCode;
            
            // Format: "Version Name (versionCode)\npackage.name"
            String versionText = getString(R.string.version_format, versionName) + " (" + versionCode + ")\n" + packageName;
            appVersion.setText(versionText);
            appVersion.setTextColor(Color.WHITE);
        } catch (Exception e) {
            appVersion.setText("");
        }
        String appDesc = getString(R.string.app_description);

        // The text with format markers for highlights and links
        String formattedAppDesc = "";
        
        // Setup highlight colors that match the theme colors
        String highlightColorHex;
        if ("Midnight Dusk".equals(currentTheme)) {
            highlightColorHex = String.format("#%06X", (0xFFFFFF & ContextCompat.getColor(requireContext(), R.color.colorPrimary)));
        } else if ("Crimson Bloom".equals(currentTheme)) {
            highlightColorHex = String.format("#%06X", (0xFFFFFF & ContextCompat.getColor(requireContext(), R.color.red_theme_primary)));
        } else if ("Premium Gold".equals(currentTheme)) {
            highlightColorHex = String.format("#%06X", (0xFFFFFF & ContextCompat.getColor(requireContext(), R.color.gold_theme_primary)));
        } else if ("Silent Forest".equals(currentTheme)) {
            highlightColorHex = String.format("#%06X", (0xFFFFFF & ContextCompat.getColor(requireContext(), R.color.silentforest_theme_primary)));
        } else if ("Shadow Alloy".equals(currentTheme)) {
            highlightColorHex = String.format("#%06X", (0xFFFFFF & ContextCompat.getColor(requireContext(), R.color.shadowalloy_theme_primary)));
        } else if ("Pookie Pink".equals(currentTheme)) {
            highlightColorHex = String.format("#%06X", (0xFFFFFF & ContextCompat.getColor(requireContext(), R.color.pookiepink_theme_primary)));
        } else if ("Snow Veil".equals(currentTheme)) {
            highlightColorHex = String.format("#%06X", (0xFFFFFF & ContextCompat.getColor(requireContext(), R.color.snowveil_theme_text_primary)));
        } else {
            // Faded Night or default
            highlightColorHex = "#FFFFFF"; // White for Faded Night
        }
        
        // Replace any hardcoded color with the theme-specific one
        appDesc = appDesc.replaceAll("#cfbafd", highlightColorHex);
        
        appDescription.setText(Html.fromHtml(appDesc, Html.FROM_HTML_MODE_LEGACY));
        appDescription.setTextColor(Color.WHITE);
        checkUpdatesButton.setTextColor(Color.WHITE);
        checkUpdatesButton.setIconTint(ColorStateList.valueOf(Color.WHITE));
        checkUpdatesButton.setStrokeColor(ColorStateList.valueOf(colorButton));
        fadSecInfoCard.setCardBackgroundColor(colorDialog);
        fadSecInfoCard.setStrokeColor(colorButton);
        TextView fadSecInfoText = fadSecInfoCard.findViewById(R.id.fadsec_info_text);
        if (fadSecInfoText != null) fadSecInfoText.setTextColor(Color.WHITE);
        sourceCodeButton.setBackgroundTintList(ColorStateList.valueOf(ContextCompat.getColor(requireContext(), R.color.gray_button_filled)));
        sourceCodeButton.setStrokeColor(ColorStateList.valueOf(ContextCompat.getColor(requireContext(), R.color.gray)));
        sourceCodeButton.setTextColor(Color.WHITE);
        sourceCodeButton.setIconTint(ColorStateList.valueOf(Color.WHITE));
        int gold = ContextCompat.getColor(requireContext(), R.color.gold);
        int black = ContextCompat.getColor(requireContext(), R.color.black);
        donateButton.setBackgroundTintList(ColorStateList.valueOf(gold));
        donateButton.setTextColor(black);
        donateButton.setIconTint(ColorStateList.valueOf(black));
        emailText.setTextColor(Color.WHITE);
        discordText.setTextColor(Color.WHITE);
        privacyInfoCard.setCardBackgroundColor(colorDialog);
        privacyInfoCard.setStrokeColor(colorButton);
        LinearLayout privacyHeader = privacyInfoCard.findViewById(R.id.privacy_info_header);
        if (privacyHeader != null) {
            TextView privacyTitle = privacyHeader.findViewById(R.id.privacy_info_title);
            if (privacyTitle != null) privacyTitle.setTextColor(themeTextColor);
            ImageView expandIcon = privacyHeader.findViewById(R.id.expand_icon);
            if (expandIcon != null) expandIcon.setColorFilter(themeTextColor);
        }
        TextView privacyInfoContent = privacyInfoCard.findViewById(R.id.privacy_info_content);
        if (privacyInfoContent != null) privacyInfoContent.setTextColor(Color.WHITE);
        String[] questions = getResources().getStringArray(R.array.questions_array);
        String[] answers = getResources().getStringArray(R.array.answers_array);
        
        // Setup footer text with Palestine and Pakistan flags
        if (tvAboutFooter != null) {
            String footer = "Made with Palestine at FadSec Lab in Pakistan";
            android.text.SpannableString spannable = new android.text.SpannableString(footer);
            
            // Palestine flag image
            Drawable palestine = androidx.core.content.res.ResourcesCompat.getDrawable(getResources(), R.drawable.palestine, null);
            if (palestine != null) {
                int size = (int) android.util.TypedValue.applyDimension(android.util.TypedValue.COMPLEX_UNIT_DIP, 18, getResources().getDisplayMetrics());
                palestine.setBounds(0, 0, size, size);
                android.text.style.ImageSpan palestineSpan = new android.text.style.ImageSpan(palestine, android.text.style.ImageSpan.ALIGN_BOTTOM);
                int palestineIndex = footer.indexOf("Palestine");
                if (palestineIndex != -1) {
                    spannable.setSpan(palestineSpan, palestineIndex, palestineIndex + "Palestine".length(), android.text.Spanned.SPAN_EXCLUSIVE_EXCLUSIVE);
                }
            }
            
            // Pakistan flag image
            Drawable pakistan = androidx.core.content.res.ResourcesCompat.getDrawable(getResources(), R.drawable.pakistan, null);
            if (pakistan != null) {
                int size = (int) android.util.TypedValue.applyDimension(android.util.TypedValue.COMPLEX_UNIT_DIP, 18, getResources().getDisplayMetrics());
                pakistan.setBounds(0, 0, size, size);
                android.text.style.ImageSpan pakistanSpan = new android.text.style.ImageSpan(pakistan, android.text.style.ImageSpan.ALIGN_BOTTOM);
                int pakistanIndex = footer.indexOf("Pakistan");
                if (pakistanIndex != -1) {
                    spannable.setSpan(pakistanSpan, pakistanIndex, pakistanIndex + "Pakistan".length(), android.text.Spanned.SPAN_EXCLUSIVE_EXCLUSIVE);
                }
            }
            
            // FadSec Lab clickable, bold, and red
            int fadSecLabStart = footer.indexOf("FadSec Lab");
            int fadSecLabEnd = fadSecLabStart + "FadSec Lab".length();
            if (fadSecLabStart >= 0) {
                spannable.setSpan(new android.text.style.StyleSpan(android.graphics.Typeface.BOLD), fadSecLabStart, fadSecLabEnd, android.text.Spanned.SPAN_EXCLUSIVE_EXCLUSIVE);
                spannable.setSpan(new android.text.style.ClickableSpan() {
                    @Override
                    public void onClick(@NonNull View widget) {
                        Intent browserIntent = new Intent(Intent.ACTION_VIEW, Uri.parse("https://github.com/fadsec-lab"));
                        widget.getContext().startActivity(browserIntent);
                    }
                    @Override
                    public void updateDrawState(android.text.TextPaint ds) {
                        super.updateDrawState(ds);
                        ds.setColor(Color.parseColor("#E43C3C")); // Red color
                        ds.setUnderlineText(false); // No underline
                    }
                }, fadSecLabStart, fadSecLabEnd, android.text.Spanned.SPAN_EXCLUSIVE_EXCLUSIVE);
            }
            
            tvAboutFooter.setText(spannable);
            tvAboutFooter.setTextColor(isSnowVeilTheme() ? Color.BLACK : Color.WHITE);
            tvAboutFooter.setMovementMethod(android.text.method.LinkMovementMethod.getInstance());
        }
        
        // Check current theme to determine answers color
        String currentThemeAnswers = com.fadcam.SharedPreferencesManager.getInstance(requireContext()).sharedPreferences.getString(com.fadcam.Constants.PREF_APP_THEME, Constants.DEFAULT_APP_THEME);
        
        // Use theme-specific colors for answers
        String answerColorHex;
        if ("Midnight Dusk".equals(currentThemeAnswers)) {
            answerColorHex = String.format("#%06X", (0xFFFFFF & ContextCompat.getColor(requireContext(), R.color.colorPrimary)));
        } else if ("Crimson Bloom".equals(currentThemeAnswers)) {
            answerColorHex = String.format("#%06X", (0xFFFFFF & ContextCompat.getColor(requireContext(), R.color.red_theme_primary)));
        } else if ("Premium Gold".equals(currentThemeAnswers)) {
            answerColorHex = String.format("#%06X", (0xFFFFFF & ContextCompat.getColor(requireContext(), R.color.gold_theme_primary)));
        } else if ("Silent Forest".equals(currentThemeAnswers)) {
            answerColorHex = String.format("#%06X", (0xFFFFFF & ContextCompat.getColor(requireContext(), R.color.silentforest_theme_primary)));
        } else if ("Shadow Alloy".equals(currentThemeAnswers)) {
            answerColorHex = String.format("#%06X", (0xFFFFFF & ContextCompat.getColor(requireContext(), R.color.shadowalloy_theme_primary)));
        } else if ("Pookie Pink".equals(currentThemeAnswers)) {
            answerColorHex = String.format("#%06X", (0xFFFFFF & ContextCompat.getColor(requireContext(), R.color.pookiepink_theme_primary)));
        } else if ("Snow Veil".equals(currentThemeAnswers)) {
            answerColorHex = String.format("#%06X", (0xFFFFFF & ContextCompat.getColor(requireContext(), R.color.snowveil_theme_text_primary)));
        } else {
            answerColorHex = "#FFFFFF"; // White for Faded Night
        }
        
        StringBuilder qnaContent = new StringBuilder();
        for (int i = 0; i < questions.length; i++) {
            qnaContent.append("<b><font color='#FFFFFF'>").append(questions[i]).append("</font></b><br>")
                    .append("<font color='" + answerColorHex + "'>").append(answers[i]).append("</font><br><br>");
        }
        privacyInfoContent.setText(Html.fromHtml(qnaContent.toString(), Html.FROM_HTML_MODE_LEGACY));
        sourceCodeButton.setOnClickListener(v -> openUrl("https://github.com/fadsec-lab/"));
        donateButton.setOnClickListener(v -> {
            KoFiSupportBottomSheet bottomSheet = new KoFiSupportBottomSheet();
            bottomSheet.show(getParentFragmentManager(), "KoFiSupportBottomSheet");
        });
        checkUpdatesButton.setOnClickListener(v -> checkForUpdates());
        emailText.setOnClickListener(v -> sendEmail());
        discordText.setOnClickListener(v -> openUrl("https://discord.gg/kvAZvdkuuN"));
        setupPrivacyInfo(privacyInfoCard, scrollView);
        view.findViewById(R.id.check_updates_button).setOnClickListener(v -> checkForUpdates());

        // Find text elements with specific content and set their colors
        ScrollView scrollViewObj = view.findViewById(R.id.scroll_view);
        if (scrollViewObj != null && scrollViewObj.getChildCount() > 0) {
            ViewGroup rootLayout = (ViewGroup) scrollViewObj.getChildAt(0);
            for (int i = 0; i < rootLayout.getChildCount(); i++) {
                View child = rootLayout.getChildAt(i);
                if (child instanceof TextView) {
                    TextView textView = (TextView) child;
                    CharSequence text = textView.getText();
                    if (text != null) {
                        String textStr = text.toString();
                        // Set copyright text color
                        if (textStr.contains(getString(R.string.copyright_info))) {
                            textView.setTextColor(themeTextColor);
                        }
                    }
                }
            }
        }

        // Initialize the alert dialog builder just once
        alertDialogBuilder = new MaterialAlertDialogBuilder(requireContext())
                .setView(R.layout.loading_dialog)
                .setCancelable(false);
    }

    private void setupPrivacyInfo(MaterialCardView cardView, ScrollView scrollView) {
        String[] questions = getResources().getStringArray(R.array.questions_array);
        String[] answers = getResources().getStringArray(R.array.answers_array);
        
        // Check current theme to determine answers color
        String currentTheme = com.fadcam.SharedPreferencesManager.getInstance(requireContext()).sharedPreferences.getString(com.fadcam.Constants.PREF_APP_THEME, Constants.DEFAULT_APP_THEME);
        
        // Use theme-specific colors for answers
        String answerColorHex;
        if ("Midnight Dusk".equals(currentTheme)) {
            answerColorHex = String.format("#%06X", (0xFFFFFF & ContextCompat.getColor(requireContext(), R.color.colorPrimary)));
        } else if ("Crimson Bloom".equals(currentTheme)) {
            answerColorHex = String.format("#%06X", (0xFFFFFF & ContextCompat.getColor(requireContext(), R.color.red_theme_primary)));
        } else if ("Premium Gold".equals(currentTheme)) {
            answerColorHex = String.format("#%06X", (0xFFFFFF & ContextCompat.getColor(requireContext(), R.color.gold_theme_primary)));
        } else if ("Silent Forest".equals(currentTheme)) {
            answerColorHex = String.format("#%06X", (0xFFFFFF & ContextCompat.getColor(requireContext(), R.color.silentforest_theme_primary)));
        } else if ("Shadow Alloy".equals(currentTheme)) {
            answerColorHex = String.format("#%06X", (0xFFFFFF & ContextCompat.getColor(requireContext(), R.color.shadowalloy_theme_primary)));
        } else if ("Pookie Pink".equals(currentTheme)) {
            answerColorHex = String.format("#%06X", (0xFFFFFF & ContextCompat.getColor(requireContext(), R.color.pookiepink_theme_primary)));
        } else if ("Snow Veil".equals(currentTheme)) {
            answerColorHex = String.format("#%06X", (0xFFFFFF & ContextCompat.getColor(requireContext(), R.color.snowveil_theme_text_primary)));
        } else {
            answerColorHex = "#FFFFFF"; // White for Faded Night
        }
        
        // Determine question text color based on theme
        String questionColorHex = "#FFFFFF"; // Default white for most themes
        if ("Snow Veil".equals(currentTheme)) {
            questionColorHex = "#000000"; // Black for Snow Veil theme
        }
        
        StringBuilder qnaContent = new StringBuilder();
        for (int i = 0; i < questions.length; i++) {
            qnaContent.append("<b><font color='").append(questionColorHex).append("'>").append(questions[i]).append("</font></b><br>")
                    .append("<font color='" + answerColorHex + "'>").append(answers[i]).append("</font><br><br>");
        }
        TextView privacyInfoContent = cardView.findViewById(R.id.privacy_info_content);
        privacyInfoContent.setText(Html.fromHtml(qnaContent.toString(), Html.FROM_HTML_MODE_LEGACY));

        ImageView expandIcon = cardView.findViewById(R.id.expand_icon);
        LinearLayout headerLayout = (LinearLayout) expandIcon.getParent();

        headerLayout.setOnClickListener(v -> {
            boolean isVisible = privacyInfoContent.getVisibility() == View.VISIBLE;

            if (!isVisible) {
                privacyInfoContent.setVisibility(View.VISIBLE);
            }

            privacyInfoContent.measure(View.MeasureSpec.makeMeasureSpec(cardView.getWidth(), View.MeasureSpec.EXACTLY),
                    View.MeasureSpec.makeMeasureSpec(0, View.MeasureSpec.UNSPECIFIED));
            int startHeight = isVisible ? privacyInfoContent.getHeight() : 0;
            int endHeight = isVisible ? 0 : privacyInfoContent.getMeasuredHeight();

            ValueAnimator heightAnimator = ValueAnimator.ofInt(startHeight, endHeight);
            heightAnimator.addUpdateListener(animation -> {
                privacyInfoContent.getLayoutParams().height = (int) animation.getAnimatedValue();
                privacyInfoContent.requestLayout();
            });
            heightAnimator.setDuration(300);

            heightAnimator.addListener(new AnimatorListenerAdapter() {
                @Override
                public void onAnimationStart(Animator animation) {
                    if (!isVisible) {
                        scrollView.post(() -> {
                            scrollView.post(() -> {
                                scrollView.postDelayed(() -> {
                                    int cardTop = cardView.getTop();
                                    ValueAnimator scrollAnimator = ValueAnimator.ofInt(scrollView.getScrollY(), cardTop);
                                    scrollAnimator.addUpdateListener(scrollAnimation -> {
                                        scrollView.scrollTo(0, (int) scrollAnimation.getAnimatedValue());
                                    });
                                    scrollAnimator.setDuration(400); // Match the card animation duration
                                    scrollAnimator.start();
                                }, 400); // Delay to match the expansion duration
                            });

                        });
                    }
                }

                @Override
                public void onAnimationEnd(Animator animation) {
                    if (isVisible) {
                        privacyInfoContent.setVisibility(View.GONE);
                    }
                }
            });

            heightAnimator.start();

            // Add animation for icon rotation
            float startRotation = isVisible ? 180f : 0f;
            float endRotation = isVisible ? 0f : 180f;
            ObjectAnimator iconAnimator = ObjectAnimator.ofFloat(expandIcon, "rotation", startRotation, endRotation);
            iconAnimator.setDuration(300);
            iconAnimator.start();
        });
    }

    private String getAppVersionForUpdates() {
        try {
            PackageManager pm = requireActivity().getPackageManager();
            PackageInfo pInfo = pm.getPackageInfo(requireActivity().getPackageName(), 0);
            return pInfo.versionName;
        } catch (PackageManager.NameNotFoundException e) {
            e.printStackTrace();
            return "N/A";
        }
    }

    private void openUrl(String url) {
        Intent intent = new Intent(Intent.ACTION_VIEW, Uri.parse(url));
        startActivity(intent);
    }

    private void sendEmail() {
        Intent intent = new Intent(Intent.ACTION_SENDTO);
        intent.setData(Uri.parse("mailto:fadedhood@proton.me"));
        intent.putExtra(Intent.EXTRA_SUBJECT, "FadCam Feedback");
        startActivity(intent);
    }


    // Code for checking updates


    private void checkForUpdates() {
        showLoadingDialog(getString(R.string.up_to_date_loading));
        ExecutorService updateExecutor = Executors.newSingleThreadExecutor();
        updateExecutor.execute(() -> {
            try {
                java.net.URL url = new java.net.URL("https://github.com/anonfaded/FadCam/releases/latest");
                java.net.HttpURLConnection connection = (java.net.HttpURLConnection) url.openConnection();
                connection.setRequestMethod("GET");
                connection.setInstanceFollowRedirects(false); // Do not follow redirects
                connection.connect();
                String location = connection.getHeaderField("Location");
                connection.disconnect();
                String latestVersion = null;
                String tagUrl = "https://github.com/anonfaded/FadCam/releases/latest";
                if (location != null && location.contains("/tag/")) {
                    int tagIndex = location.lastIndexOf("/tag/");
                    tagUrl = location;
                    latestVersion = location.substring(tagIndex + 5).replace("v", "").trim();
                }
                String currentVersion = getAppVersionForUpdates();
                final String finalCurrentVersion = currentVersion;
                final String finalLatestVersion = latestVersion;
                final String finalTagUrl = tagUrl;
                requireActivity().runOnUiThread(() -> {
                    dismissLoadingDialog();
                    if (finalLatestVersion != null && isUpdateAvailable(finalCurrentVersion, finalLatestVersion)) {
                        String changelog = ""; // Not available via this method
                        UpdateAvailableBottomSheet.newInstance(finalLatestVersion, changelog, finalTagUrl)
                            .show(getParentFragmentManager(), "UpdateAvailableBottomSheet");
                    } else {
                        showUpToDateDialog();
                    }
                });
            } catch (Exception e) {
                e.printStackTrace();
                requireActivity().runOnUiThread(() -> {
                    dismissLoadingDialog();
                    showErrorDialog("Failed to check for updates. Please try again later.");
                });
            }
        });
    }

    private boolean isUpdateAvailable(String currentVersion, String latestVersion) {
        boolean currentIsBeta = currentVersion.contains("beta");

        currentVersion = currentVersion.replace("-beta", "");
        String[] current = currentVersion.split("\\.");
        String[] latest = latestVersion.split("\\.");

        for (int i = 0; i < Math.min(current.length, latest.length); i++) {
            int currentPart = Integer.parseInt(current[i]);
            int latestPart = Integer.parseInt(latest[i]);

            if (latestPart > currentPart) {
                return true;
            } else if (latestPart < currentPart) {
                return false;
            }
        }

        return latest.length > current.length || (latest.length == current.length && currentIsBeta);
    }

    private void showLoadingDialog(String message) {
        requireActivity().runOnUiThread(() -> {
            TextView messageView = new TextView(requireContext());
            messageView.setText(message);
            
            // Check if we're using Snow Veil theme
            String currentTheme = SharedPreferencesManager.getInstance(requireContext()).sharedPreferences.getString(com.fadcam.Constants.PREF_APP_THEME, Constants.DEFAULT_APP_THEME);
            boolean isSnowVeilTheme = "Snow Veil".equals(currentTheme);
            
            // Set text color based on theme
            messageView.setTextColor(isSnowVeilTheme ? Color.BLACK : Color.WHITE);
            
            messageView.setPadding(48, 32, 48, 32);
            messageView.setTextSize(16);
            MaterialAlertDialogBuilder builder = themedDialogBuilder(requireContext());
            builder.setView(messageView);
            builder.setCancelable(false);
            loadingDialog = builder.create();
            loadingDialog.show();
        });
    }

    private void dismissLoadingDialog() {
        if (loadingDialog != null && loadingDialog.isShowing()) {
            loadingDialog.dismiss();
        }
    }





    private void showUpToDateDialog() {
        TextView messageView = new TextView(requireContext());
        messageView.setText(getString(R.string.up_to_date_description));
        
        // Check if we're using Snow Veil theme
        String currentTheme = SharedPreferencesManager.getInstance(requireContext()).sharedPreferences.getString(com.fadcam.Constants.PREF_APP_THEME, Constants.DEFAULT_APP_THEME);
        boolean isSnowVeilTheme = "Snow Veil".equals(currentTheme);
        
        // Set text color based on theme
        messageView.setTextColor(isSnowVeilTheme ? Color.BLACK : Color.WHITE);
        
        messageView.setPadding(48, 32, 48, 32);
        messageView.setTextSize(16);
        
        MaterialAlertDialogBuilder builder = themedDialogBuilder(requireContext())
                .setTitle(getString(R.string.up_to_date))
                .setView(messageView)
                .setPositiveButton("OK", null);
        
        AlertDialog dialog = builder.create();
        dialog.setOnShowListener(dialogInterface -> setDialogButtonColors(dialog));
        dialog.show();
    }

    private void showErrorDialog(String message) {
        // Check if we're using Snow Veil theme
        String currentTheme = SharedPreferencesManager.getInstance(requireContext()).sharedPreferences.getString(com.fadcam.Constants.PREF_APP_THEME, Constants.DEFAULT_APP_THEME);
        boolean isSnowVeilTheme = "Snow Veil".equals(currentTheme);
        
        MaterialAlertDialogBuilder builder = themedDialogBuilder(requireContext())
                .setTitle("Error")
                .setMessage(message)
                .setPositiveButton("OK", null);
        
        AlertDialog dialog = builder.create();
        dialog.setOnShowListener(dialogInterface -> setDialogButtonColors(dialog));
        dialog.show();
        
        // If using Snow Veil theme, make sure the message text is black
        if (isSnowVeilTheme && dialog.findViewById(android.R.id.message) instanceof TextView) {
            ((TextView) dialog.findViewById(android.R.id.message)).setTextColor(Color.BLACK);
        }
    }

    private void openUpdateUrl(String url) {
        Intent intent = new Intent(Intent.ACTION_VIEW, Uri.parse(url));
        startActivity(intent);
    }


    private int resolveThemeColor(int attr) {
        android.util.TypedValue typedValue = new android.util.TypedValue();
        requireContext().getTheme().resolveAttribute(attr, typedValue, true);
        return typedValue.data;
    }

    private MaterialAlertDialogBuilder themedDialogBuilder(Context context) {
        // Get current theme
        SharedPreferencesManager spm = SharedPreferencesManager.getInstance(context);
        String currentTheme = spm.sharedPreferences.getString(com.fadcam.Constants.PREF_APP_THEME, Constants.DEFAULT_APP_THEME);
        
        // Select the appropriate dialog theme based on current app theme
        int dialogTheme;
        if ("Crimson Bloom".equals(currentTheme)) {
            dialogTheme = R.style.ThemeOverlay_FadCam_Red_Dialog;
        } else if ("Faded Night".equals(currentTheme)) {
            dialogTheme = R.style.ThemeOverlay_FadCam_Amoled_MaterialAlertDialog;
        } else if ("Snow Veil".equals(currentTheme)) {
            dialogTheme = R.style.ThemeOverlay_FadCam_SnowVeil_Dialog;
        } else {
            dialogTheme = R.style.ThemeOverlay_FadCam_Dialog;
        }
        
        return new MaterialAlertDialogBuilder(context, dialogTheme);
    }
    
    // Helper method to make dialog buttons white or black based on theme
    private void setDialogButtonColors(AlertDialog dialog) {
        if (dialog == null) return;
        
        // Check if we're using Snow Veil theme
        String currentTheme = SharedPreferencesManager.getInstance(requireContext()).sharedPreferences.getString(com.fadcam.Constants.PREF_APP_THEME, Constants.DEFAULT_APP_THEME);
        boolean isSnowVeilTheme = "Snow Veil".equals(currentTheme);
        
        // Set button colors based on theme
        int buttonColor = isSnowVeilTheme ? Color.BLACK : Color.WHITE;
        
        dialog.getButton(AlertDialog.BUTTON_POSITIVE).setTextColor(buttonColor);
        dialog.getButton(AlertDialog.BUTTON_NEGATIVE).setTextColor(buttonColor);
        dialog.getButton(AlertDialog.BUTTON_NEUTRAL).setTextColor(buttonColor);
    }

    // New method to handle Snow Veil theme UI adjustments
    private void applySnowVeilThemeToUI(View rootView) {
        // Find all icon views and set them to black for better contrast
        tintAllIcons(rootView, Color.BLACK);
        
        // Ensure all text views have proper contrast
        ensureTextContrast(rootView);
        
        // Apply specific adjustments to MaterialCardViews
        applySnowVeilToCards(rootView);
    }
    
    private void applySnowVeilToCards(View rootView) {
        try {
            // Find MaterialCardViews in the fragment
            MaterialCardView fadsecInfoCard = rootView.findViewById(R.id.fadsec_info_card);
            MaterialCardView privacyInfoCard = rootView.findViewById(R.id.privacy_info_card);
            
            // Apply Snow Veil theme styles to cards
            if (fadsecInfoCard != null) {
                fadsecInfoCard.setCardBackgroundColor(Color.WHITE);
                fadsecInfoCard.setStrokeColor(Color.LTGRAY);
                
                // Find text views inside the card and set color
                TextView fadsecInfoText = fadsecInfoCard.findViewById(R.id.fadsec_info_text);
                if (fadsecInfoText != null) {
                    fadsecInfoText.setTextColor(Color.BLACK);
                }
            }
            
            if (privacyInfoCard != null) {
                privacyInfoCard.setCardBackgroundColor(Color.WHITE);
                privacyInfoCard.setStrokeColor(Color.LTGRAY);
                
                // Find views inside the privacy card
                TextView privacyTitle = privacyInfoCard.findViewById(R.id.privacy_info_title);
                TextView privacyContent = privacyInfoCard.findViewById(R.id.privacy_info_content);
                ImageView expandIcon = privacyInfoCard.findViewById(R.id.expand_icon);
                
                if (privacyTitle != null) {
                    privacyTitle.setTextColor(Color.BLACK);
                }
                
                if (privacyContent != null) {
                    privacyContent.setTextColor(Color.BLACK);
                }
                
                if (expandIcon != null) {
                    expandIcon.setColorFilter(Color.BLACK);
                }
            }
            
            // Apply to other UI elements
            MaterialButton sourceCodeButton = rootView.findViewById(R.id.source_code_button);
            MaterialButton donateButton = rootView.findViewById(R.id.donate_button);
            MaterialButton checkUpdatesButton = rootView.findViewById(R.id.check_updates_button);
            
            if (sourceCodeButton != null) {
                sourceCodeButton.setTextColor(Color.WHITE);
                sourceCodeButton.setIconTint(ColorStateList.valueOf(Color.WHITE));
                sourceCodeButton.setStrokeColor(ColorStateList.valueOf(ContextCompat.getColor(requireContext(), R.color.gray)));
                sourceCodeButton.setBackgroundTintList(ColorStateList.valueOf(ContextCompat.getColor(requireContext(), R.color.gray_button_filled)));
            }
            
            if (donateButton != null) {
                donateButton.setTextColor(Color.BLACK);
                donateButton.setIconTint(ColorStateList.valueOf(Color.BLACK));
            }
            
            if (checkUpdatesButton != null) {
                checkUpdatesButton.setTextColor(Color.BLACK);
                checkUpdatesButton.setIconTint(ColorStateList.valueOf(Color.BLACK));
                checkUpdatesButton.setStrokeColor(ColorStateList.valueOf(Color.BLACK));
            }
            
            // Find and apply black tint ONLY to the email icon
            TextView emailText = rootView.findViewById(R.id.email_text);
            if (emailText != null) {
                // For the email icon, use direct approach to set the drawable tint
                Drawable[] emailDrawables = emailText.getCompoundDrawablesRelative();
                if (emailDrawables[0] != null) { // Start drawable
                    Drawable emailIcon = emailDrawables[0].mutate(); // Create a mutable copy
                    emailIcon.setColorFilter(Color.BLACK, android.graphics.PorterDuff.Mode.SRC_IN);
                    emailText.setCompoundDrawablesRelativeWithIntrinsicBounds(emailIcon, null, null, null);
                }
                emailText.setTextColor(Color.BLACK);
            }
            
        } catch (Exception e) {
            Log.e("AboutFragment", "Error applying Snow Veil theme to cards: " + e.getMessage());
        }
    }
    
    private void setTextRecursive(ViewGroup viewGroup, int color) {
        for (int i = 0; i < viewGroup.getChildCount(); i++) {
            View child = viewGroup.getChildAt(i);
            if (child instanceof TextView) {
                ((TextView) child).setTextColor(color);
            } else if (child instanceof ViewGroup) {
                setTextRecursive((ViewGroup) child, color);
            }
        }
    }
    
    // Helper method to tint all icons in the view hierarchy
    private void tintAllIcons(View view, int color) {
        if (view instanceof ViewGroup) {
            ViewGroup viewGroup = (ViewGroup) view;
            for (int i = 0; i < viewGroup.getChildCount(); i++) {
                tintAllIcons(viewGroup.getChildAt(i), color);
            }
        } else if (view instanceof ImageView) {
            // Skip applying tint to the app icon
            if (view.getId() == R.id.app_icon) {
                return; // Don't apply tint to app logo
            }
            
            ((ImageView) view).setColorFilter(color);
        } else if (view instanceof MaterialButton) {
            MaterialButton button = (MaterialButton) view;
            if (button.getIcon() != null) {
                button.setIconTint(ColorStateList.valueOf(color));
            }
        }
    }
    
    // Helper method to ensure text has proper contrast
    private void ensureTextContrast(View view) {
        if (view instanceof ViewGroup) {
            ViewGroup viewGroup = (ViewGroup) view;
            for (int i = 0; i < viewGroup.getChildCount(); i++) {
                ensureTextContrast(viewGroup.getChildAt(i));
            }
        } else if (view instanceof TextView) {
            TextView textView = (TextView) view;
            // Only override if the text is very light (poor contrast against white)
            int currentColor = textView.getCurrentTextColor();
            if (isLightColor(currentColor)) {
                textView.setTextColor(Color.BLACK);
            }
        }
    }
    
    // Helper method to determine if a color is light
    private boolean isLightColor(int color) {
        double darkness = 1 - (0.299 * Color.red(color) + 0.587 * Color.green(color) + 0.114 * Color.blue(color)) / 255;
        return darkness < 0.5; // If darkness is less than 0.5, it's a light color
    }

    // Helper method to check if Snow Veil theme is active
    private boolean isSnowVeilTheme() {
        String currentTheme = com.fadcam.SharedPreferencesManager.getInstance(requireContext()).sharedPreferences.getString(com.fadcam.Constants.PREF_APP_THEME, Constants.DEFAULT_APP_THEME);
        return "Snow Veil".equals(currentTheme);
    }
}
